# -*- coding: UTF-8 -*-
import expparser
import traceback
from cbb.frame.dsl import adapter
from cbb.frame.dsl.base_operator import BaseOperator


class ParserDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        parse_key = (expparser.Literal("vertical_parser_list")
                     | expparser.Literal("horizontal_parser")
                     | expparser.Literal("vertical_parser")).setResultsName("type")
        pattern = c + parse_key
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        if toks.type == "vertical_parser":
            return adapter.get_vertical_ret(self.dsl_context.last_data)
        elif toks.type == "horizontal_parser":
            return adapter.get_horizontal_ret(self.dsl_context.last_data)
        else:
            return adapter.get_vertical_list_ret(self.dsl_context.last_data)


class RegexDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        not_char = expparser.CharsNotIn("'")
        parse_key = expparser.Literal("regex").setResultsName("type")
        quota = expparser.Suppress("'")
        regex = quota + not_char.setResultsName("regex") + quota
        expression = c + parse_key + regex
        expression.setParseAction(self.base_action)
        return expression

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        data = self.dsl_context.last_data
        if not data:
            return None
        if not isinstance(data, list):
            return self._search(toks.regex, data)

        ret = []
        for item in data:
            sections = self._search(toks.regex, item)
            if sections:
                ret.append(sections)
        return ret

    @staticmethod
    def _search(pattern, text):
        matched = expparser.re.search(pattern, text)
        if not matched:
            return ""
        elem = matched.groupdict()
        if not elem:
            elem = matched.groups()
        return elem


class SplitlinesDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        parse_key = expparser.Literal("splitlines").setResultsName("type")
        pattern = c + parse_key
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        return self.dsl_context.last_data.splitlines()


class SplitDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        parse_key = (expparser.Literal("split")).setResultsName("type")
        a = expparser.CharsNotIn("'")
        b = expparser.Suppress("'")
        symbol = b + a.setResultsName("symbol") + b
        pattern = c + parse_key + symbol
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        data = self.dsl_context.last_data
        if not isinstance(data, list):
            return data.split(toks.symbol)
        ret = []
        for item in data:
            ret.append(item.split(toks.symbol))
        return ret


class GetKeyDsl(BaseOperator):
    def __init__(self, dsl_context, data=None):
        BaseOperator.__init__(self, dsl_context, data)

    def get(self, alone=True):
        c = expparser.Suppress("|")
        parse_key = (expparser.Literal("get_key")).setResultsName("type")
        filed_name = expparser.Regex(r"[^']+").setResultsName("filed_name")
        pattern = parse_key + expparser.Suppress("('") + filed_name + expparser.Suppress("')")
        if alone:
            pattern = c + pattern
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        field_name = toks.filed_name
        # 用类型+字段名作为key
        key = toks.type + toks.filed_name
        if isinstance(self.data, list):
            ret = []
            for item in self.data:
                ret.append(item.get(field_name, ''))
            toks[key] = ret
            return ret

        # 将提取到的值存入toks中
        toks[key] = self.data.get(field_name, '')
        return toks[key]


class GetIndexDsl(BaseOperator):
    def __init__(self, dsl_context, data=None):
        BaseOperator.__init__(self, dsl_context, data)

    def get(self, alone=True):
        c = expparser.Suppress("|")
        parse_key = (expparser.Literal("get_index")).setResultsName("type")
        index = expparser.Word(expparser.nums).setResultsName("index")
        pattern = parse_key + expparser.Suppress("(") + index + expparser.Suppress(")")
        if alone:
            pattern = c + pattern
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        # 用类型+字段名作为key
        index = int(toks.index)
        key = toks.type + toks.index
        if isinstance(self.data, list):
            ret = []
            for item in self.data:
                ret.append(item[index])
            toks[key] = ret
            return ret
        toks[key] = self.data[index]
        return self.data[index]


class FilterDsl(BaseOperator):
    """
    过滤操作:从list中筛选出符合条件的数据，没有返回[]
    """
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        operate_key = expparser.Literal("filter").setResultsName("type")
        express = expparser.Combine(expparser.OneOrMore(expparser.CharsNotIn("|")))\
            .setResultsName("inner_express")
        pattern = c + operate_key + express
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        ret = []
        for one_data in self.data:
            pattern = ExpressDsl(self.dsl_context, one_data).get()
            try:
                pattern.parseString(toks.inner_express)
            except Exception:
                self.logger.error("parse filter error."
                                  + traceback.format_exc())
                continue
            if self.dsl_context.last_data:
                ret.append(one_data)
        return ret


class ExpressDsl(BaseOperator):
    """
    逻辑运算操作
    支持嵌套get运算
    """
    def __init__(self, dsl_context, data=None):
        BaseOperator.__init__(self, dsl_context, data)

    def get(self):
        operator = (expparser.Literal(">")
                    | expparser.Literal("==")
                    | expparser.Literal("<")
                    | expparser.Literal("!=")).setResultsName("operator")
        operator_num = (expparser.Word(expparser.nums) | expparser.Suppress("'") + expparser.CharsNotIn("'")
                        + expparser.Suppress("'")).setResultsName("operator_num")
        get_key_dsl = \
            expparser.Combine(GetKeyDsl(self.dsl_context, self.data).get(False)) \
            + operator + operator_num
        get_index_dsl = \
            GetIndexDsl(self.dsl_context, self.data).get(False) \
            + operator + operator_num
        pattern = get_key_dsl | get_index_dsl
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        express = []
        for item in toks:
            if item in toks:
                express.append(toks[item])
            else:
                express.append(item)
        if len(express) == 3 and \
                (not express[0].isdigit() or not express[2].isdigit()):
            express[0] = "'" + express[0] + "'"
            express[2] = "'" + express[2] + "'"
        return eval(" ".join(express))


class LogicDsl(BaseOperator):
    """
    逻辑运算操作
    支持嵌套get运算
    """
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        parse_key = expparser.Literal("logic").setResultsName("type")
        express = ExpressDsl(self.dsl_context).get()
        pattern = c + parse_key + express
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        """
        toks既是数组，也是字典
        toks ['logic', 'get_keyfault_page_num', '>', '20']
        filed_name = {str} 'fault_page_num'
        get_keyfault_page_num = {str} '10'
        operator = {str} '>'
        operator_num = {str} '20'
        type = {str} 'get_key'
        """
        self.logger.info("enter {} action ".format(toks.type))
        return self.dsl_context.last_data


class BreakDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        c = expparser.Suppress("|")
        ret_key = (expparser.Literal("return") | expparser.Literal("raise")).setResultsName("type")
        ret_value = expparser.Literal("true") | expparser.Literal("false")
        condition_key = expparser.Literal("if")
        condition_value = expparser.Literal("empty")

        pattern = c + ret_key + ret_value + condition_key + condition_value
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        return


class EchoDsl(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)

    def get(self):
        echo_key = expparser.Literal("echo").setResultsName("type")
        pattern = echo_key + expparser.CharsNotIn("|")
        pattern.setParseAction(self.base_action)
        return pattern

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        return self.dsl_context.args[0]
