# coding:utf-8
""" 
@version: Toolkit V200R005C00
@time: 2018/09/09 
@file: common.py 
@function: 
@modify: 
"""
import functools
import traceback
import threading
import time
import re
from cbb.frame.cli import cliUtil

errInfoDict = {"query.result.abnormal":
    {
        "zh": u"\n获取信息失败",
        "en": "\nThere is an error in getting information",
    }}


def getErrorLogFunc(logger):
    errorLogFun = None
    if logger and hasattr(logger, 'error'):
        errorLogFun = getattr(logger, 'error', None)
    elif logger and hasattr(logger, 'logError'):
        errorLogFun = getattr(logger, 'logError', None)

    return errorLogFun


def wrapInspectException(lan='en', logger=None, originalInfo=''):
    """封装巡检项执行脚本异常，执行异常时返回未检测。

    如果要带回原始信息，则需要在巡检脚本中将原始信息变量定义为全局变量。

    :param lan: 语言信息
    :param logger: 日志对象，方便记录巡检异常堆栈信息。
    :param originalInfo: 原始信息（异常出现过程时，可能执行了部分检查过程，这些过程的执行结果可以返回到报告中）。
    :return:
    """
    recordErrorLogFunc = getErrorLogFunc(logger)

    def wrapper(func):
        @functools.wraps(func)
        def inner(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except:
                if recordErrorLogFunc:
                    recordErrorLogFunc("exception caught:" + traceback.format_exc())
                return 'NOCHECK', originalInfo, errInfoDict.get("query.result.abnormal").get(lan)

        return inner

    return wrapper


def wrapAllException(func):
    """封装所有异常信息，保证执行函数不会抛出任何异常.

    :param func:
    :return:
    """

    @functools.wraps(func)
    def inner(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except:
            return

    return inner


def wrapAllExceptionLogged(logger=None):
    """封装所有异常信息，并记录日志， 保证执行函数不会抛出任何异常.

    :param logger:
    :return:
    """

    recordErrorLogFunc = getErrorLogFunc(logger)

    def wrapper(func):
        @functools.wraps(func)
        def inner(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if recordErrorLogFunc:
                    recordErrorLogFunc('Exception caught:%s' % e)
                    recordErrorLogFunc('Exception trace:%s) ' % traceback.format_exc())
            except:
                if recordErrorLogFunc:
                    recordErrorLogFunc('Java exception caught?')
                    recordErrorLogFunc('Java exception trace:%s) ' % traceback.format_exc())
                return

        return inner

    return wrapper


def fakeProgress(py_java_env, totalSeconds=1 * 30, logger=None, interval=1):
    """根据预估总体执行时间，模拟进度条.

    :param py_java_env:
    :param totalSeconds: 预估总体执行时间
    :param interval: 进度刷新时间间隔（每次刷新，进度会均匀增长）
    :param logger: 日志对象
    :return:
    """

    def wrapper(func):
        event = threading.Event()
        t = threading.Thread(target=refreshProgress, args=(py_java_env, totalSeconds, interval, logger, event))

        @functools.wraps(func)
        def inner(*args, **kwargs):
            try:
                t.start()
                ret = func(*args, **kwargs)
            finally:
                event.set()
            return ret

        return inner

    return wrapper


def refreshProgress(py_java_env, totalSeconds, intervalSeconds, logger, event):
    """调用UI的进度刷新对象的方法，更新进度.

    :param py_java_env:java上下文
    :param totalSeconds: 预估执行时间
    :param intervalSeconds:进度刷新间隔时间
    :param logger:日志对象
    :return:
    """
    recordErrorLogFunc = getErrorLogFunc(logger)

    def setProgress(progress):
        if observer:
            try:
                observer.updateProgress(int(progress))
            except:
                if recordErrorLogFunc:
                    recordErrorLogFunc('Update progress exception:%s' % traceback.format_exc())

    observer = py_java_env.get("progressObserver")
    cnt = totalSeconds / intervalSeconds
    intervalCnt = 0
    setProgress(1)
    while not event.isSet():
        time.sleep(intervalSeconds)
        intervalCnt += 1
        progress = int(100.0 / cnt * intervalCnt)
        progress = progress if progress < 100 else 99
        setProgress(progress)

    setProgress(100)


def is_support_read_only_user_enter_debug(product_version, hot_patch):
    """
    判断是否支持只读用户进去debug,如果支持返回true
    :param product_version:
    :param hot_patch:
    :return:
    """
    # 支持的版本列表，后续可以在此列表中新增
    support_read_only_version = {'V300R006C20': 30}
    if product_version in support_read_only_version:
        support_patch = support_read_only_version.get(product_version)

        pattern_hot_patch = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
        match_hot_path = pattern_hot_patch.search(hot_patch)
        if match_hot_path and support_patch <= int(match_hot_path.group(1)):
            return True
    return False


def get_product_version_by_upgrade_package(cli, lang):
    """
    @summary: 通过show upgrade package命令获取设备版本，
          适用于OEM场景与需要获取spc版本的需求
    """
    productVersion = ""
    checkRet, softwareVersionList, hotPatchVersionList = parse_upgrade_package(
        cli, lang)
    if checkRet[0] is not True:
        return (checkRet[0], checkRet[1], checkRet[2], productVersion)

    flag, productVersion, errMsg = get_current_version(softwareVersionList, lang)
    return (flag, checkRet[1], errMsg, productVersion)


def get_current_version(software_version_list, lang):
    '''
    @summary: 通过software信息字典列表获取版本信息
    '''
    currentVersion = ""
    errMsg = ""
    for controller in software_version_list:
        currentVersion = controller.get("Current Version")
        return (True, currentVersion, "")

    errMsg = cliUtil.getMsg(lang, "cannot.get.product.version.info")
    return (False, "", errMsg)


def parse_upgrade_package(cli, lang):
    '''
    @summary: 执行show upgrade packge命令，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型,并返回CLI回显
    @param cli: cli对象
    @param lang: 语言lang
    '''
    cmd = "show upgrade package"
    software_version_list = []
    hotPatchVersionList = []

    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag is not True:
        return ((flag, cliRet, errMsg), software_version_list, hotPatchVersionList)

    software_version_index = cliRet.find("Software Version")
    hot_patch_version_index = cliRet.find("HotPatch Version")
    software_version_list = cliUtil.getHorizontalCliRet(
        cliRet[(software_version_index):hot_patch_version_index])
    hotPatchVersionList = cliUtil.getHorizontalCliRet(
        cliRet[(hot_patch_version_index):])

    if len(software_version_list) == 0 or len(hotPatchVersionList) == 0:
        errMsg += cliUtil.getMsg(lang, "cannot.get.upgrade.package.info")
        return ((False, cliRet, errMsg), software_version_list, hotPatchVersionList)

    for software_version in software_version_list:
        if software_version["Current Version"] == "--":
            flag = False
            errMsg += cliUtil.getMsg(lang, "cannot.get.contrller.version.info",
                                     software_version.get("Name"))

    if not is_controller_version_same(software_version_list):
        errMsg += cliUtil.getMsg(lang, "controller.version.not.consistence")
        return ((False, cliRet, errMsg), software_version_list, hotPatchVersionList)

    return ((flag, cliRet, errMsg), software_version_list, hotPatchVersionList)


def is_controller_version_same(software_version_list):
    '''
    @summary:校验控制器版本是否一致
    @param software_version_list: 控制器版本列表
    @return:
        True:控制器版本一致
        False:控制器软件版本不一致
    '''
    VersionSet = set(softwareVersion.get("Current Version") for softwareVersion in software_version_list)
    if len(VersionSet) != 1:
        return False
    return True


def get_engine_ctrl_node_info(cli, logger, lang):
    """
    @summary: 获取除当前节点外的引擎其他健康节点信息，当前节点信息。
    @param cli: cli连接
    @return: (currentCtrlNodeId, currentEngineCtrlIdList)
        currentNodeId: 当前节点ID
        engineCtrlNodeMappingDict: 除当前节点外的引擎其他健康节点信息
        nodeNum
    """
    nodeNum = 2
    cmd = 'sys showcls'
    # 当前节点
    currentCtrlNodeId = None
    currentEngine = None
    engine_ctrl_node_mapping_dict = {}

    flag, cliRet, errMsg = execute_one_debug_Command(cli, cmd, lang)
    if flag is not True:
        logger.logInfo("get node info Failed. errMsg: %s" % errMsg)
        return flag, errMsg, cliRet, currentCtrlNodeId, engine_ctrl_node_mapping_dict, currentEngine, nodeNum

    ctrl_info_dict_list = cliUtil.getVerticalCliRet(cliRet)
    engine_info_dict_list = cliUtil.getHorizontalNostandardCliRet(cliRet)

    for ctrl_info in ctrl_info_dict_list:
        if ctrl_info.get("local node id", "") != "":
            currentCtrlNodeId = ctrl_info.get("local node id", "")
            break

    for engine_info in engine_info_dict_list:
        engine = engine_info.get("engine", "")
        if engine == "":
            continue

        status = engine_info.get("status", "")
        if status == "" or status != "normal":
            continue

        ctrlNodeId = engine_info.get("id", "")
        if ctrlNodeId == "":
            continue

        tmpList = engine_ctrl_node_mapping_dict.get(engine, [])
        tmpList.append(ctrlNodeId)
        engine_ctrl_node_mapping_dict[engine] = tmpList

        if ctrlNodeId == currentCtrlNodeId:
            currentEngine = engine

    for engine_id in engine_ctrl_node_mapping_dict:
        nodeNum = max(len(engine_ctrl_node_mapping_dict.get(engine_id, [])), nodeNum)

    # 当前引擎包含的控制器节点
    return (True, "", cliRet, currentCtrlNodeId, engine_ctrl_node_mapping_dict, currentEngine, nodeNum)


def execute_one_debug_Command(cli, cmd, lang):
    """
    @summary: 执行一条debug命令。命令执行完后恢复到CLI模式。
    """
    switch = True
    try:
        # V3R6及其之后版本版本，需要打开developer模式开关才能进入developer模式
        (flag, cliRet, errMsg), switch = cliUtil.open_developer_switch_v(cli, lang)
        if flag is False:
            return (False, cliRet, errMsg)
        flag, cliRet, errMsg = cliUtil.excuteCmdInDebugModel(cli, cmd, True, lang)
        return (flag, cliRet, errMsg)
    finally:
        if switch is False:
            cliUtil.closeDeveloperSwitch(cli, lang)
        # 退出到cli模式
        cliUtil.enterCliModeFromSomeModel(cli, lang)


def query_micro_system_work_mode(cli, lang):
    cmd = "show system work_mode"
    __, ret, __ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    res = re.findall("System mode:\s*(.*)", ret)
    if res:
        return res[0].strip()
    return None


def is_micro_pass_though_mode(cli, lang):
    res = query_micro_system_work_mode(cli, lang)
    return "pass-through mode" == res, res


def is_cold_patch(patch_version):
    """
    用于-1/-2补丁集场景，判断是否为-1补丁（-1为冷不丁），
    -1补丁版本为SPC001，不含SPH
    :param patch_version:
    :return:
    """
    if "SPH" not in patch_version.upper():
        return True
    return False


class UnCheckException(Exception):
    """
    @summary: 未检查异常自定义类
    """
    def __init__(self, error_msg, cli_ret):
        self.error_msg = error_msg
        self.cli_ret = cli_ret
