# -*- coding: UTF-8 -*-
from common.constant import SCRIPT_RET_CODE
from common.result import getErrMsgFromLanguageFile
from funcFactory.cliParser import getInfoByCliParser
import os


def AddlistToDict(list, keyName):
    '''
    @summary: 将集合list转换为以keyName为键的字
    @param list: 集合
    @param keyName: 字典的键
    @return: 字典
    '''
    map = {}
    map[keyName] = list
    return map


def getCmdNodeInfo(cmdNode):
    '''
    @summary: 从cmd节点中获取cli命令信息
    @param cmdNode: cmd节点解析后的字典
    @return: (cli命令，cli类型，需要获取的回显字段)
    '''
    command = cmdNode["elementAttrbutes"]["command"]
    cliType = cmdNode["elementAttrbutes"]["type"]
    
    collectItemTitleList = []        
    attrNodeList = cmdNode["elementChildEles"][0]["elementChildEles"] 
    for attrNode in attrNodeList:
        collectItemTitleList.append(attrNode["elementText"])
    return (command, cliType, collectItemTitleList)


def getBaseCmdNodeInfo(baseCmdNode):
    '''
    @summary: 从baseCmd节点中获取cli命令信息
    @param cmdNode: baseCmd节点解析后的字典
    @return: (cli命令，cli类型，需要获取的回显字段(作为其他cli命令的参数)
    '''
    command = baseCmdNode["elementAttrbutes"]["command"]
    cliType = baseCmdNode["elementAttrbutes"]["type"]
    paramName = baseCmdNode["elementAttrbutes"]["paramName"]
    
    return (command, cliType, paramName)


def setDetailedErrMsg(errMsgKey, failMsg, cliRet, context):
    '''
    @summary: 将CLI执行时的错误信息和回显根据中英文环境构造为一个字符串；
            并以errMsg为key，该字符串为value构造一个dict
    '''
    errMsg = getErrMsgFromLanguageFile(context, errMsgKey)
    
    if cliRet == "" or cliRet == None:
        return {errMsg:failMsg}
    else:
        return {errMsg:cliRet}

def setDetailedSuccMsg(succMsgKey,context):
    '''
    @summary: 收集成功的项需要将其设置到界面上
    '''
    succMsg = getErrMsgFromLanguageFile(context, succMsgKey)
    return {succMsg:""}


def collectData(context, xmInfo):     
    '''
    @summary: 根据每个收集项的收集类型和使用的cli类型收集数据
    @param xmInfo: 解析xml后的信息(一个嵌套的字典)
    @return: None
    '''
    #收集项失败数
    failNum = 0
    #收集项的数据
    allRetDataMap = {}
    #失败的收集项构成的字符串，用于工具界面显
    allErrMsg = {}
    
    collectItemNodes = xmInfo["elementChildEles"]
    
    collectNum = float(len(collectItemNodes))
    currentCollectNum = 0
    listener = context.get('listener')
    globalDict = globals()
    for node in collectItemNodes:
        #根据cmd4IBMS.xml中配置的类型自动调用dataCollectUtil.py中的同名方法来收集
        collectItemType = node["elementAttrbutes"]["type"]
        collectItemMethod = globalDict[collectItemType]
        (code, retDataMap, cliRet, failMsg) = collectItemMethod(context, node)
        
        collectItemName = node["elementAttrbutes"]["name"]
        #全部收集失败
        if code == SCRIPT_RET_CODE.FAIL:
            failNum += 1
            retDataMap = {collectItemName:[]}
            errMsgKey = "collect.fail.msg.%s" % collectItemName
            detailedErrMsgMap = setDetailedErrMsg(errMsgKey, failMsg, cliRet, context)
            allErrMsg.update(detailedErrMsgMap)
        #部分收集成功
        if code == SCRIPT_RET_CODE.PART_SUCCESS:
            failNum += 1
            errMsgKey = "collect.partfail.msg.%s" % collectItemName
            detailedErrMsgMap = setDetailedErrMsg(errMsgKey, failMsg, cliRet, context)
            allErrMsg.update(detailedErrMsgMap)
        #success, but no data
        if code == SCRIPT_RET_CODE.SUCCESS and retDataMap == None:
            retDataMap = {collectItemName:[]}
            
        #收集成功，将收集项设置到界面上
        if code == SCRIPT_RET_CODE.SUCCESS:
            succMsgKey = "collect.succ.msg.%s" % collectItemName
            detailedSuccMap = setDetailedSuccMsg(succMsgKey,context)
            allErrMsg.update(detailedSuccMap)
        
        allRetDataMap.update(retDataMap)
        
        #收集进度
        currentCollectNum += 1
        progress = int((currentCollectNum / collectNum) * 100)
        listener.refleshProgress(progress)
    
    #全部收集成功   
    if failNum == 0:
        code = SCRIPT_RET_CODE.SUCCESS
    #全部收集失败
    elif failNum == len(collectItemNodes):
        code = SCRIPT_RET_CODE.FAIL
    #部分收集成功
    else:
        code = SCRIPT_RET_CODE.PART_SUCCESS
    
    return (code, allRetDataMap, allErrMsg)
    

def single(context, collectItemNode):
    '''
    @summary: 通过单条cli命令收集收集项
    @param collectItemNode: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    '''
    cmdNode = collectItemNode["elementChildEles"][0]
    (command, cliType, collectItemTitleList) = getCmdNodeInfo(cmdNode)
    
    collectItemName = collectItemNode["elementAttrbutes"]["name"]
    #retDataList:[{},{},...]
    (isSuccess, retDataList, cliRet, errMsg) = getInfoByCliParser(context, command, cliType, collectItemTitleList, collectItemName)
    
    #success, but no data
    if isSuccess and retDataList == None:
        return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
    
    #收集失败
    if not isSuccess:
        return (SCRIPT_RET_CODE.FAIL, None, cliRet, errMsg)
    #收集成功
    else:
        collectItemName = collectItemNode["elementAttrbutes"]["name"]
        return (SCRIPT_RET_CODE.SUCCESS, AddlistToDict(retDataList, collectItemName), cliRet, errMsg)


def dependency(context, collectItemNode):
    '''
    @summary: 通过两条cli命令收集收集项，其中第二条cli的参数来自于第一条cli的回显
    @param collectItemNode: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    '''
    
    
    (baseCommand, baseCliType, paramName) = ("", "", "")
    (command, cliType, collectItemTitleList) = ("", "", [])
    
    #获取<baseCmd><cmd>中的内容
    cmdNodes = collectItemNode["elementChildEles"]
    for cmdNode in cmdNodes:
        if cmdNode["elementName"] == "baseCmd":
            (baseCommand, baseCliType, paramName) = getBaseCmdNodeInfo(cmdNode)
        else:
            (command, cliType, collectItemTitleList) = getCmdNodeInfo(cmdNode)
            
    #执行<baseCmd>的cli命令，最终获取的数据将作<cmd>的cli命令的参
    baseCollectItemTitleList = []
    baseCollectItemTitleList.append(paramName)
    (isSuccess, paramMapList, cliRet, errMsg) = getInfoByCliParser(context, baseCommand, baseCliType, baseCollectItemTitleList)
    
    #success, but no data
    if isSuccess and paramMapList == None:
        return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
    
    #收集失败
    if not isSuccess:
        return (SCRIPT_RET_CODE.FAIL, None, cliRet, errMsg)
        
    #循环执行<cmd>中的命令，并<baseCmd>获取的结果作为参
    failNum = 0
    retDataMapList = [] 
    alllCliRet = ""
    
    collectItemName = collectItemNode["elementAttrbutes"]["name"]
    titleListMap = context["titleListMap"]
    
    for paramMap in paramMapList:
        param = paramMap[paramName]
        collectItemTitleList = titleListMap[collectItemName]
        (isSuccess, retDataList, cliRet, errMsg) = getInfoByCliParser(context, command % param, cliType, collectItemTitleList, collectItemName)
        #收集失败
        if not isSuccess:
            failNum += 1
            alllCliRet = alllCliRet + cliRet
        elif retDataList != None:
            retDataMapList.extend(retDataList)
    
    #全部收集成功   
    if failNum == 0:
        return (SCRIPT_RET_CODE.SUCCESS, AddlistToDict(retDataMapList, collectItemName), alllCliRet, errMsg)
    
    elif failNum == len(paramMapList):#全部收集失败
        return (SCRIPT_RET_CODE.FAIL, None, alllCliRet, errMsg)

    else:#部分收集成功
        return (SCRIPT_RET_CODE.PART_SUCCESS, AddlistToDict(retDataMapList, collectItemName), alllCliRet, errMsg)


def many(context, collectItemNode):
    '''
    @summary: 通过多条条cli命令收集收集项，此方法只适用于cli回显单个表格数据的情况
    @param collectItemNode: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    '''
    cmdNodes = collectItemNode["elementChildEles"]

    retDataMap = {}
    for cmdNode in cmdNodes:
        #执行命令
        (command, cliType, collectItemTitleList) = getCmdNodeInfo(cmdNode)
        (isSuccess, retDataList, cliRet, errMsg) = getInfoByCliParser(context, command, cliType, collectItemTitleList)
        #success, but no data
        if isSuccess and retDataList == None:
            return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
        
        #只要有一条命令执行失败就认为收集失败
        if not isSuccess:
            return (SCRIPT_RET_CODE.FAIL, None, cliRet, errMsg)
        #将获取到的cli回显解析后的map追加到retDataMap
        retDataMap.update(retDataList[0])
          
    retDataMapList = []
    retDataMapList.append(retDataMap)
    
    collectItemName = collectItemNode["elementAttrbutes"]["name"]
    return (SCRIPT_RET_CODE.SUCCESS, AddlistToDict(retDataMapList, collectItemName), cliRet, errMsg)


