# -*- coding: UTF-8 -*-
from common.commonFunction import excute_cmd, excute_cmd_in_developer
from common.log import Log
from frame.cli.cliUtil import get_horizontal_cli_ret, get_vertical_cli_ret, get_splited_cli_ret


def log_when_cli_excute_failed(command, err_msg, cli_ret):
    Log.error("execute cli %s failed, because %s, and the cli return is %s", command, err_msg, cli_ret)
    return


def get_info_by_cli_parser(context, command, cli_type, collect_item_title_list):
    """
    @summary: 调用cli解析器获取cli回显，并过滤出collectItemTitleList中的内容
    @param command: cli命令
    @param cli_type: cli命令回显结构
    @param collect_item_title_list: 获取的cli回显的字典
    @return: (cli命令，cli类型，需要获取的回显字段)
    """
    cli_parser_name = "parse_%s" % cli_type
    cli_parser = globals()[cli_parser_name]
    # 根据cmd4IBMS.xml<cmd><baseCmd>节点中配置的cli命令自动调用cliParser.py中的同名方法来收集数据
    is_success, dict_list = cli_parser(context, command)
    if not is_success or dict_list is None:
        return is_success, dict_list

    # 根据cmd4IBMS.xml<str>节点中配置的收集数据从cli回显中筛选出最终的数据
    dict_list = ret_data_filter(dict_list, collect_item_title_list)
    return is_success, dict_list


def ret_data_filter(all_data, filt_data):
    """
    @summary: 筛选完整的cli回显中包含在<attr>节点中的数据
    @param all_data: cli回显(一个字典列表)
    @param filt_data: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    for one_data in all_data:
        one_data_keys = one_data.keys()
        for key in one_data_keys:
            if key not in filt_data:
                one_data.pop(key)

    return all_data


def parse_vertical_table(context, command):
    """
    @summary: 执行cli命令并将回显解析成[{},{},...]，此种cli回显为竖向列表
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    # 1. 将cli回显解析为字典列表
    # 2. cli执行成功，但cli解析为空，即cli未获取到信息
    dict_list = get_vertical_cli_ret(cli_ret)

    return is_success, dict_list


def parse_transverse_table(context, command):
    """
    @summary: 执行cli命令并将回显解析成[{},{},...]，此种cli回显为横向列表
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    # 1. 将cli回显解析为字典列表
    # 2. cli执行成功，但cli解析为空，即cli未获取到信息
    dict_list = get_horizontal_cli_ret(cli_ret)

    return is_success, dict_list


def parse_mix_table(context, command):
    """
    @summary: 执行cli命令并将回显解析成[{},{},...]，此种cli回显为多个“----“分隔的竖向表格
                                    但第一个表格与其它表格字段不一致，此方法用来获取其中第一个表格的内容
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """

    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)

    # cli执行成功，但cli解析为空，即cli未获取到信息
    if dict_list:
        dict_list = [dict_list[0]]
    return is_success, dict_list


def parse_software(context, command):
    """
    @summary: 执行show upgrade package,获取其中的部分内容
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    hot_patch_version_index = cli_ret.find("HotPatch Version")

    software_version = get_splited_cli_ret(cli_ret[:hot_patch_version_index], "Software Version")
    software_version_list = get_horizontal_cli_ret(software_version)

    hot_patch_version = get_splited_cli_ret(cli_ret[hot_patch_version_index:], "HotPatch Version")
    hot_patch_version_list = get_horizontal_cli_ret(hot_patch_version)

    for soft_version in software_version_list:
        for hot_patch_version in hot_patch_version_list:
            if soft_version["Name"] == hot_patch_version["Name"]:
                soft_version["patchVersion"] = hot_patch_version["Current Version"]

    return is_success, software_version_list


def parse_vertical_table_no_parse_elc_label(context, command):
    """
    @summary: 解析竖向列表结构的CLI回显,其中电子标签作为一个整体,不再解析
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    # 将cli回显解析为字典列
    is_parse_elc_label = False
    dict_list = get_vertical_cli_ret(cli_ret, is_parse_elc_label)
    # cli执行成功，但cli解析为空，即cli未获取到信息

    return is_success, dict_list


def parse_vertical_table_in_developer(context, command):
    """
    @summary: 解析竖向列表结构的developer命令回显
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = excute_cmd_in_developer(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息

    return is_success, dict_list


def parse_transverse_table_in_developer(context, command):
    """
    @summary: 解析横向列表结构的developer命令回显
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = excute_cmd_in_developer(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    # 将cli回显解析为字典列
    dict_list = get_horizontal_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息

    return is_success, dict_list


def get_license_or_feature_cli_ret(context, command, isLicenseFeature):
    """
    @summary: 解析show license命令回显，并将其解析为license基本信息，license特性两个字典列表
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, []

    region_index = cli_ret.find("Region")
    feature_index = cli_ret.find("Feature Name")
    if isLicenseFeature:
        cli_ret = cli_ret[region_index:]
    else:
        cli_ret = cli_ret[:feature_index]
    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息

    return is_success, dict_list


def parse_license(context, command):
    """
    @summary: 执行show license命令回显，并将回显的上部解析为license的基本信息
    """
    is_license_feature = False
    return get_license_or_feature_cli_ret(context, command, is_license_feature)


def parse_license_feature(context, command):
    """
    @summary: 执行show license命令回显，并将回显的下部部解析为license特性的基本信息
    """
    is_license_feature = True
    return get_license_or_feature_cli_ret(context, command, is_license_feature)


def parse_upgrade_package(context, command):
    """
    @summary: 执行show upgrade packge命令回显，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型
    """
    dict_list = []

    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, None

    software_version_index = cli_ret.find("Software Version")
    hot_patch_version_index = cli_ret.find("HotPatch Version")
    software_version_list = get_horizontal_cli_ret(cli_ret[(software_version_index):hot_patch_version_index])
    hot_patch_version_list = get_horizontal_cli_ret(cli_ret[(hot_patch_version_index):])
    Log.info("softwareVersionList list is %s" % str(software_version_list))
    Log.info("hotPatchVersionList list is %s" % str(hot_patch_version_list))

    if not software_version_list or not hot_patch_version_list:
        return is_success, dict_list

    for software_version in software_version_list:
        software_version["Version Type"] = "Software Version"

    for hotPatchVersion in hot_patch_version_list:
        hotPatchVersion["Version Type"] = "HotPatch Version"

    Log.info("softwareVersionList2 list is %s" % str(software_version_list))
    Log.info("hotPatchVersionList2 list is %s" % str(hot_patch_version_list))

    software_version_list.extend(hot_patch_version_list)
    dict_list = software_version_list
    Log.info("upgradePackage list is %s" % str(dict_list))

    return is_success, dict_list


class VersionTypeHandler:

    def __init__(self):
        self.next_handler = None

    def set_next(self, handler):
        pass

    def handle_request(self, request):
        pass


class ProductVersionHandler(VersionTypeHandler):
    def __init__(self, cli_ret, controller_version_index):
        super().__init__()
        self.cli_ret = cli_ret
        self.controller_version_index = controller_version_index

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if request == "productVersion":
            cli_ret = self.cli_ret[:self.controller_version_index]
            dict_list = get_vertical_cli_ret(cli_ret)
            return dict_list
        else:
            return self.next_handler.handle_request(request)


class ControllerVersionHandler(VersionTypeHandler):
    def __init__(self, cli_ret, controller_version_index, expansion_version_index):
        super().__init__()
        self.cli_ret = cli_ret
        self.controller_version_index = controller_version_index
        self.expansion_version_index = expansion_version_index

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if request == "controllerVersion":
            cli_ret = self.cli_ret[self.controller_version_index:self.expansion_version_index]
            dict_list = get_vertical_cli_ret(cli_ret)
            return dict_list
        else:
            return self.next_handler.handle_request(request)


class ExpansionVersionHandler(VersionTypeHandler):

    def __init__(self, cli_ret, expansion_version_index, bbu_version_index):
        super().__init__()
        self.cli_ret = cli_ret
        self.expansion_version_index = expansion_version_index
        self.bbu_version_index = bbu_version_index

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if request == "expansionVersion":
            cli_ret = self.cli_ret[self.expansion_version_index:self.bbu_version_index]
            dict_list = get_horizontal_cli_ret(cli_ret)
            return dict_list
        else:
            return self.next_handler.handle_request(request)


class BbuVersionHandler(VersionTypeHandler):
    def __init__(self, cli_ret, bbu_version_index):
        super().__init__()
        self.cli_ret = cli_ret
        self.bbu_version_index = bbu_version_index

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if request == "bbuVersion":
            cli_ret = self.cli_ret[self.bbu_version_index:]
            dict_list = get_horizontal_cli_ret(cli_ret)
            return dict_list
        else:
            return self.next_handler.handle_request(request)


def process_request(request, cli_ret, controller_version_index, expansion_version_index, bbu_version_index):
    handler_a = ProductVersionHandler(cli_ret, controller_version_index)
    handler_b = ControllerVersionHandler(cli_ret, controller_version_index, expansion_version_index)
    handler_c = ExpansionVersionHandler(cli_ret, expansion_version_index, bbu_version_index)
    handler_d = BbuVersionHandler(cli_ret, bbu_version_index)

    handler_a.set_next(handler_b)
    handler_b.set_next(handler_c)
    handler_d.set_next(handler_d)

    return handler_a.handle_request(request)


def get_version_cli_ret(context, command, version_type):
    """
    @summary: 执行show version all命令回显，并根据versionType解析不同类型的version
    """

    dict_list = []

    is_success, cli_ret, err_msg = excute_cmd(context, command)
    if not is_success:
        log_when_cli_excute_failed(command, err_msg, cli_ret)
        return is_success, dict_list

    controller_version_index = cli_ret.find("Controller:")
    expansion_version_index = cli_ret.find("Expansion Module:")
    bbu_version_index = cli_ret.find("BBU:")

    result = process_request(version_type, controller_version_index, expansion_version_index, bbu_version_index)

    if not result:
        return is_success, []
    return is_success, dict_list


def parse_product_version(context, command):
    """
    @summary: 执行show version all命令回显，获取product Version
    """
    version_type = "productVersion"
    return get_version_cli_ret(context, command, version_type)


def parse_controller_version(context, command):
    """
    @summary: 执行show version all命令回显，controller Version
    """
    version_type = "controllerVersion"
    return get_version_cli_ret(context, command, version_type)


def parse_expansion_version(context, command):
    """
    @summary: 执行show version all命令回显，expansion Version
    """
    version_type = "expansionVersion"
    return get_version_cli_ret(context, command, version_type)


def parse_bbu_version(context, command):
    """
    @summary: 执行show version all命令回显，bbu Version
    """
    version_type = "bbuVersion"
    return get_version_cli_ret(context, command, version_type)
