# -*- coding: UTF-8 -*-
import os
from dataclasses import dataclass, field

from common.constant import SCRIPT_RET_CODE
from funcFactory.cliParser import get_info_by_cli_parser
from common.result import get_err_msg_from_language_file


def add_list_to_dict(target_list, key_name):
    """
    @summary: 将集合list转换为以keyName为键的字
    @param target_list: 集合
    @param key_name: 字典的键
    @return: 字典
    """
    ret_map = dict()
    ret_map[key_name] = target_list
    return ret_map


def get_cmd_node_info(cmd_node):
    """
    @summary: 从cmd节点中获取cli命令信息
    @param cmd_node: cmd节点解析后的字典
    @return: cli命令，cli类型，需要获取的回显字段
    """
    command = cmd_node["elementAttrbutes"]["command"]
    cli_type = cmd_node["elementAttrbutes"]["type"]

    collect_item_title_list = []
    attr_node_list = cmd_node["elementChildEles"][0]["elementChildEles"]
    for attr_node in attr_node_list:
        collect_item_title_list.append(attr_node["elementText"])
    return command, cli_type, collect_item_title_list


def get_base_cmd_node_info(base_cmd_node):
    """
    @summary: 从baseCmd节点中获取cli命令信息
    @param cmdNode: baseCmd节点解析后的字典
    @return: cli命令，cli类型，需要获取的回显字段(作为其他cli命令的参数
    """
    command = base_cmd_node["elementAttrbutes"]["command"]
    cli_type = base_cmd_node["elementAttrbutes"]["type"]
    param_name = base_cmd_node["elementAttrbutes"]["paramName"]

    return command, cli_type, param_name


def collect_fail_num(collection, global_dict, context, listener):
    for node in collection.collect_item_nodes:
        # 根据cmd4IBMS.xml中配置的类型自动调用dataCollectUtil.py中的同名方法来收集
        collect_item_type = node["elementAttrbutes"]["type"]
        collect_item_method = global_dict[collect_item_type]
        (code, ret_data_map) = collect_item_method(context, node)

        collect_item_name = node["elementAttrbutes"]["name"]
        # 全部收集失败
        if code == SCRIPT_RET_CODE.FAIL:
            collection.fail_num += 1
            err_msg_key = "collect.fail.msg.%s" % collect_item_name
            err_msg = get_err_msg_from_language_file(context, err_msg_key) + os.linesep
            ret_data_map = {collect_item_name: []}
            all_err_msg = all_err_msg + err_msg
        # 部分收集成功
        if code == SCRIPT_RET_CODE.PART_SUCCESS:
            collection.fail_num += 1
            err_msg_key = "collect.partfail.msg.%s" % collect_item_name
            err_msg = get_err_msg_from_language_file(context, err_msg_key) + os.linesep
            all_err_msg = all_err_msg + err_msg
        # success, but no data
        if code == SCRIPT_RET_CODE.SUCCESS and ret_data_map is None:
            ret_data_map = {collect_item_name: []}

        collection.all_ret_data_map.update(ret_data_map)

        # 收集进度
        collection.current_collect_num += 1
        progress = int((collection.current_collect_num / collection.collect_num) * 100)
        listener.refleshProgress(progress)


def get_collect_status(collection_object):
    # 全部收集成功
    if collection_object.fail_num == 0:
        code = SCRIPT_RET_CODE.SUCCESS
    # 全部收集失败
    elif collection_object.fail_num == len(collection_object.collect_item_nodes):
        code = SCRIPT_RET_CODE.FAIL
    # 部分收集成功
    else:
        code = SCRIPT_RET_CODE.PART_SUCCESS

    return code


@dataclass
class Collection:
    # 收集项失败数
    fail_num: int
    # 收集项的数据
    all_ret_data_map: dict
    # 失败的收集项构成的字符串，用于工具界面显
    all_err_msg: str

    collect_item_nodes: dict
    collect_num: float
    current_collect_num: int


def collect_data(context, xm_info):
    """
    @summary: 根据每个收集项的收集类型和使用的cli类型收集数据
    @param context: dict
    @param xm_info: 解析xml后的信息(一个嵌套的字典)
    @return: None
    """

    collection = Collection()

    # 收集项失败数
    collection.fail_num = 0
    # 收集项的数据
    collection.all_ret_data_map = {}
    # 失败的收集项构成的字符串，用于工具界面显
    collection.all_err_msg = ""

    collection.collect_item_nodes = xm_info["elementChildEles"]
    collection.collect_num = float(len(collection.collect_item_nodes))
    collection.current_collect_num = 0

    listener = context.get('listener')
    global_dict = globals()

    collect_fail_num(collection, global_dict, context, listener)
    code = get_collect_status(collection)

    return code, collection.all_ret_data_map, collection.all_err_msg


def single(context, collect_item_node):
    """
    @summary: 通过单条cli命令收集收集�?
    @param collect_item_node: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    """
    cmd_node = collect_item_node["elementChildEles"][0]
    (command, cli_type, collect_item_title_list) = get_cmd_node_info(cmd_node)

    # ret_data_list:[{},{},...]
    (is_success, ret_data_list) = get_info_by_cli_parser(context, command, cli_type, collect_item_title_list)

    # success, but no data
    if is_success and ret_data_list is None:
        return SCRIPT_RET_CODE.SUCCESS, None

    # 收集失败
    if not is_success:
        return SCRIPT_RET_CODE.FAIL, None
    # 收集成功
    else:
        collect_item_name = collect_item_node["elementAttrbutes"]["name"]
        return SCRIPT_RET_CODE.SUCCESS, add_list_to_dict(ret_data_list, collect_item_name)


def get_success_ret_data_list(param_map_list, cmd_node_info, context):
    # 循环执行<cmd>中的命令，并<baseCmd>获取的结果作为参
    fail_num = 0
    ret_data_map_list = []
    for param_map in param_map_list:
        param = param_map[cmd_node_info.param_name]
        is_success, ret_data_list = get_info_by_cli_parser(context, cmd_node_info.command % param,
                                                             cmd_node_info.cli_type,
                                                             cmd_node_info.collect_item_title_list)
        # 收集失败
        if not is_success:
            fail_num += 1
        elif ret_data_list is not None:
            ret_data_map_list.extend(ret_data_list)
    return fail_num, ret_data_map_list


def get_dependency_collect_status(fail_num, ret_data_map_list, collect_item_name, param_map_list):
    if fail_num == 0:
        return SCRIPT_RET_CODE.SUCCESS, add_list_to_dict(ret_data_map_list, collect_item_name)

    elif fail_num == len(param_map_list):  # 全部收集失败
        return SCRIPT_RET_CODE.FAIL, None

    else:  # 部分收集成功
        return SCRIPT_RET_CODE.PART_SUCCESS, add_list_to_dict(ret_data_map_list, collect_item_name)


@dataclass
class CmdNodeInfo:
    base_command: str = ''
    base_cli_type: str = ''
    param_name: str = ''
    command: str = ''
    cli_type: str = ''
    collect_item_title_list: list = field(default_factory=list)


def dependency(context, collect_item_node):
    """
    @summary: 通过两条cli命令收集收集项，其中第二条cli的参数来自于第一条cli的回显
    @param collect_item_node: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    """

    cmd_node_info = CmdNodeInfo()
    # 获取<baseCmd><cmd>中的内容
    cmd_nodes = collect_item_node["elementChildEles"]
    for cmd_node in cmd_nodes:
        if cmd_node["elementName"] == "baseCmd":
            (cmd_node_info.base_command, cmd_node_info.base_cli_type,
             cmd_node_info.param_name) = get_base_cmd_node_info(
                cmd_node)
        else:
            (cmd_node_info.command, cmd_node_info.cli_type, cmd_node_info.collect_item_title_list) = get_cmd_node_info(
                cmd_node)

    # 执行<baseCmd>的cli命令，最终获取的数据将作<cmd>的cli命令的参
    base_collect_item_title_list = list()
    base_collect_item_title_list.append(cmd_node_info.param_name)
    is_success, param_map_list = get_info_by_cli_parser(context, cmd_node_info.base_command,
                                                        cmd_node_info.base_cli_type,
                                                        base_collect_item_title_list)

    # success, but no data
    if is_success and param_map_list is None:
        return SCRIPT_RET_CODE.SUCCESS, None

    # 收集失败
    if not is_success:
        return SCRIPT_RET_CODE.FAIL, None
    fail_num, ret_data_map_list = get_success_ret_data_list(param_map_list, cmd_node_info, context)

    collect_item_name = collect_item_node["elementAttrbutes"]["name"]
    return get_dependency_collect_status(fail_num, ret_data_map_list, collect_item_name, param_map_list)


def many(context, collect_item_node):
    """
    @summary: 通过多条条cli命令收集收集项，此方法只适用于cli回显单个表格数据的情况
    @param collect_item_node: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    """
    cmd_nodes = collect_item_node["elementChildEles"]

    ret_data_map = {}
    for cmd_node in cmd_nodes:
        # 执行命令
        (command, cli_type, collect_item_title_list) = get_cmd_node_info(cmd_node)
        (is_success, ret_data_list) = get_info_by_cli_parser(context, command, cli_type, collect_item_title_list)
        # success, but no data
        if is_success and ret_data_list is None:
            return SCRIPT_RET_CODE.SUCCESS, None

        # 只要有一条命令执行失败就认为收集失败
        if not is_success:
            return SCRIPT_RET_CODE.FAIL, None
        # 将获取到的cli回显解析后的map追加到retDataMap
        ret_data_map.update(ret_data_list[0])

    ret_data_map_list = list()
    ret_data_map_list.append(ret_data_map)

    collect_item_name = collect_item_node["elementAttrbutes"]["name"]
    return SCRIPT_RET_CODE.SUCCESS, add_list_to_dict(ret_data_map_list, collect_item_name)
