# -*- coding:utf-8 -*-
import copy
import re

from common import cliCmdManager
from common import utils
from common.cBase import cFastTypeDict
from common.cTV2R2 import cHandleTypeList
from common.commonFunction import getResource
from common.constant import PosMap
from common.constant import SCRIPT_RET_CODE

isLiceFileExst = True  # license是否存在
succStateDict = {}  # 命令执行成功状态字典


def execute(context):
    '''
    @summary: collect standard items
    @date: 2014-07-25
    @param context: context object
    @return: (result code, error message) as (integer, string)
    '''
    logger = context.get('logger')
    logger.info('[collectStandard]Collect standard item start!')
    sshConnector = context['sshConnector']
    resource = getResource(context)
    global isLiceFileExst
    global succStateDict

    # 获取标准收集项标题映射和barcode映射
    stdItemCfgMap = context.get('stdItemCfgMap')
    stdItemTitleMap = get_item_title_map(stdItemCfgMap)
    stdItemBarcodeMap = get_item_field_map(stdItemCfgMap, 'BarCode')

    # 获取标准收集项命令字典
    stdItemCmdDict = context.get('stdItemCmdDict')

    if not stdItemTitleMap or not stdItemCmdDict:
        logger.error('[collectStandard]Invalid device node object!')
        return (SCRIPT_RET_CODE.FAIL, resource.getString('err.collect.necessary.info.failed'))

    # 获取标准收集项数据格式
    stdItemDataMap = get_data_map_struct(stdItemTitleMap)

    # 初始化成功状态字典
    succStateDict = init_succ_state_dict(stdItemTitleMap)

    devNode = context.get('devNode')
    if not devNode:
        logger.error('[collectStandard]Invalid device node object!')
        return (SCRIPT_RET_CODE.FAIL, resource.getString('err.init.dev.failed'))
    else:
        devModel = devNode.getDeviceProductModel(context)

    for collectItem in stdItemTitleMap:
        cmdList = stdItemCmdDict.get(collectItem)
        if not cmdList:
            logger.warn('[collectStandard] command list is null')
            continue
        logger.info('[collectStandard]collectItem=' + str(collectItem))
        logger.info('[collectStandard]cmdList=' + str(cmdList))

        # Software不符合通用处理方式，需要单独处理
        if 'Software' == collectItem:
            logger.info('[collectStandard] collect Software...')
            sysInfoDictList = stdItemDataMap.get('System')
            # 若系统信息不存在则获取并赋值
            if not sysInfoDictList:
                cliCmd = stdItemCmdDict.get('System')[0]
                sysInfoDictList = get_sys_info_dict_list(sshConnector, cliCmd, logger)
                stdItemDataMap['System'] = sysInfoDictList
            retDictList = collect_software(sshConnector, cmdList, logger, sysInfoDictList[0])
            stdItemDataMap[collectItem].extend(retDictList)

            # 统计单个收集项总个数
            count_single_item_sum(context, stdItemDataMap, collectItem)
            continue

        # 通用处理流程：发命令后解析回文获取数据
        for cliCmd in cmdList:
            # 发送命令获取回文
            isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, logger)
            if not isSucc:
                succStateDict[collectItem] = False
                logger.error('[collectStandard] Execute command [' + cliCmd + '] failed!')
                break
            if 'License' == collectItem:
                # license不存在
                if re.search('The License file does not exist', cmdRet, re.IGNORECASE):
                    isLiceFileExst = False

            # 判断数据有效性
            isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
            if not isExecSuc:
                logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
                continue
            elif not isParsable:
                logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
                continue

            # 解析回文
            formatFunction = cFastTypeDict(cmdRet, defaultSep='|')
            dataInfoDictList = formatFunction.handle()[1]
            logger.info('[collectStandard]dataInfoDictList=' + str(dataInfoDictList))

            # 如果dataInfoDictList为空则换一种解析方式
            if not dataInfoDictList:
                formatFunction = cHandleTypeList(cmdRet)
                ret = formatFunction.handle()
                if len(ret) < 2:
                    continue
                dataInfoDictList = ret[1]
                logger.info('[collectStandard]dataInfoDictList=' + str(dataInfoDictList))

            # 更新收集需要的license信息
            if 'License' == collectItem:
                sysInfoDictList = stdItemDataMap.get('System')
                if not sysInfoDictList:
                    cliCmd = stdItemCmdDict.get('System')[0]
                    sysInfoDictList = get_sys_info_dict_list(sshConnector, cliCmd, logger)
                    stdItemDataMap['System'] = sysInfoDictList
                devSN = sysInfoDictList[0].get('Device Serial Number', '')
                logger.info('[collectStandard] update license devSN=' + devSN)
                update_license_info(devSN, dataInfoDictList, logger)
            stdItemDataMap[collectItem].extend(dataInfoDictList)

        # 统计单个收集项总个数
        count_single_item_sum(context, stdItemDataMap, collectItem)

        # 清除Barcode为空的项
        itemDictList = stdItemDataMap.get(collectItem)
        clean_null_data(collectItem, itemDictList, stdItemBarcodeMap)

        # 调整收集项location
        adjust_item_loc_info(collectItem, itemDictList, context, devModel)

    # 更新柜信息字典列表
    update_rack_dict_list(stdItemDataMap, stdItemCfgMap, context)
    # 更新量纲信息字典列表
    update_dime_dict_list(stdItemDataMap, stdItemCfgMap, context)

    context['titleListMap'] = stdItemTitleMap
    context['retDataMap'] = stdItemDataMap

    # 判断收集结果：成功、失败or部分成功
    collectResult, errMsg = judge_collect_result(context, stdItemDataMap)
    return (collectResult, errMsg)


def count_single_item_sum(context, itemDataMap, collectItem):
    '''
    @summary: get single item sum
    @date: 2014-10-09
    @param context: context object
    @param itemDataMap: item data map
    @param collectItem: collect item
    @return: single item sum
    '''
    logger = context.get('logger')
    itemDictList = itemDataMap.get(collectItem)
    componentNumKey = collectItem + 'Num'
    context[componentNumKey] = len(itemDictList)
    logger.info('[collectStandard]' + componentNumKey + ':' + str(context[componentNumKey]))


def get_item_field_map(itemCfgMap, fieldName):
    '''
    @summary: get item field map
        e.g. if fieldName is barcode, the field map like {'Enclosure': 'Serial Number', 'Controller': 'BarCode'}
    @date: 2014-10-08
    @param itemCfgMap: item configuration map
    @param fieldName: field name
    @return: item field map
    '''
    itemFieldMap = {}
    for title in itemCfgMap:
        cfgDictList = itemCfgMap.get(title)
        for cfgDict in cfgDictList:
            xmlTagVal = cfgDict.get('xmlTag')
            if fieldName == xmlTagVal:
                itemFieldMap[title] = cfgDict.get('attrId')
    return itemFieldMap


def get_item_title_map(itemCfgMap):
    '''
    @summary: get item title map
        e.g. {'Enclosure':['Serial Number', 'Type'], 'System':['Device Type', 'System Name']}
    @date: 2014-10-08
    @param itemCfgMap: item configuration map
    @return: item title map
    '''
    itemTitleMap = {}
    for title in itemCfgMap:
        itemTitleMap[title] = []
        cfgDictList = itemCfgMap.get(title)
        for cfgDict in cfgDictList:
            itemTitleMap[title].append(cfgDict.get('attrId'))
    return itemTitleMap


def update_dime_dict_list(stdItemDataMap, stdItemCfgMap, context):
    '''
    @summary: update the list of dimension information dictionary 
    @date: 2014-09-29
    @param stdItemDataMap: standard item data map
    @param stdItemCfgMap: collect item configuration map
    @param context: context object
    @return: 
    '''
    collectItem = 'Dimension'
    logger = context.get('logger')
    enclosureDictList = stdItemDataMap.get('Enclosure')
    stdItemTypeMap = get_item_field_map(stdItemCfgMap, 'Type')
    stdItemNameMap = get_item_field_map(stdItemCfgMap, 'Name')
    stdItemValMap = get_item_field_map(stdItemCfgMap, 'Value')
    dimensionDictList = []
    enclTypeNumDict = {}

    # 获取所有框类型个数字典
    type = stdItemTypeMap.get('Enclosure')
    for enclosureDict in enclosureDictList:
        enclosureType = enclosureDict.get(type)  # 获取框类型
        enclTypeNumDict[enclosureType] = enclTypeNumDict.get(enclosureType, 0) + 1
    logger.info('[collectStandard] enclTypeNumDict=' + str(enclTypeNumDict))

    # 按返回格式组装数据
    name = stdItemNameMap.get('Dimension')
    value = stdItemValMap.get('Dimension')
    for enclType in enclTypeNumDict:
        dimensionDict = {}
        dimensionDict[name] = enclType
        dimensionDict[value] = str(enclTypeNumDict.get(enclType, 'NA'))
        dimensionDictList.append(dimensionDict)
    logger.info('[collectStandard] dimensionDictList=' + str(dimensionDictList))
    stdItemDataMap[collectItem] = dimensionDictList
    count_single_item_sum(context, stdItemDataMap, collectItem)


def update_rack_dict_list(stdItemDataMap, stdItemCfgMap, context):
    '''
    @summary: update the list of rack information dictionary 
    @date: 2014-09-29
    @param stdItemDataMap: standard item data map
    @param stdItemCfgMap: collect item configuration map
    @param context: context object
    @return: 
    '''
    collectItem = 'Rack'
    logger = context.get('logger')
    enclosureDictList = stdItemDataMap.get('Enclosure')
    logger.info('[collectStandard] enclosureDictList=' + str(enclosureDictList))
    stdItemLocMap = get_item_field_map(stdItemCfgMap, 'Location')
    enclLocField = stdItemLocMap.get('Enclosure')
    rackLocField = stdItemLocMap.get('Rack')
    rackDictList = []

    for enclosureDict in enclosureDictList:
        enclosureLoc = enclosureDict.get(enclLocField)  # 获取框位置
        rackLoc = enclosureLoc.split(',')[0]  # 获取柜位置
        rackDict = {rackLocField: rackLoc}
        if rackDict not in rackDictList:
            rackDictList.append(rackDict)
    stdItemDataMap[collectItem] = rackDictList
    count_single_item_sum(context, stdItemDataMap, collectItem)


def get_sys_info_dict_list(sshConnector, cliCmd, logger):
    '''
    @summary: get the list of system information dictionary
    @date: 2014-09-16
    @param sshConnector: the connector of SSH
    @param cliCmd: command of system information
    @param logger: a logger provided by tool
    @return: the list of system information dictionary
    '''
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, logger)
    if not isSucc:
        succStateDict['System'] = False
        logger.error('[collectStandard] Execute command [' + cliCmd + '] failed!')
        return []

    # 判断数据有效性
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []

    # 解析回文
    formatFunction = cFastTypeDict(cmdRet, defaultSep='|')
    dataInfoDictList = formatFunction.handle()[1]
    logger.info('[collectStandard]dataInfoDictList=' + str(dataInfoDictList))

    # 如果dataInfoDictList为空则换一种解析方式
    if not dataInfoDictList:
        formatFunction = cHandleTypeList(cmdRet)
        ret = formatFunction.handle()
        if len(ret) < 2:
            return []
        dataInfoDictList = ret[1]
        logger.info('[collectStandard]dataInfoDictList=' + str(dataInfoDictList))

    return dataInfoDictList


def init_succ_state_dict(stdItemTitleMap):
    '''
    @summary: initialize the dict of success state 
    @date: 2014-09-24
    @param stdItemTitleMap: standard item title map
    @return: the dict of success state 
    '''
    tmpDataMap = copy.deepcopy(stdItemTitleMap)
    for key in tmpDataMap:
        tmpDataMap[key] = True
    return tmpDataMap


def get_data_map_struct(stdItemTitleMap):
    '''
    @summary: get standard item data map structure
            e.g. {'System':[],'Enclosure':[]}
    @date: 2014-09-09
    @param stdItemTitleMap: standard item title map
    @return: standard item data map structure
    '''
    tmpDataMap = copy.deepcopy(stdItemTitleMap)
    for key in tmpDataMap:
        tmpDataMap[key] = []
    return tmpDataMap


def update_license_info(devSN, dataInfoDictList, logger):
    '''
    @summary: update license
    @date: 2014-09-02
    @param devSN: Device Serial Number
    @param dataInfoDictList: the list of data dictionary
    @param logger: a logger provided by tool
    @return: 
    '''
    for dataInfoDict in dataInfoDictList:
        dataInfoDict['Device Serial Number'] = devSN
    logger.info('[collectStandard] license data list is ' + str(dataInfoDictList))


def clean_null_data(collectItem, dataInfoDictList, stdItemBarcodeMap):
    '''
    @summary: clean null data
    @date: 2014-09-04
    @param collectItem: collect item
    @param dataInfoDictList: the list of data dictionary
    @param stdItemBarcodeMap: barcode map
    @return: 
    '''
    # 在IBMS系统中barcode唯一，不能为空或者“--”
    barcode = stdItemBarcodeMap.get(collectItem)
    tmpDictList = copy.copy(dataInfoDictList)
    for dataInfoDict in tmpDictList:
        for key in dataInfoDict.keys():
            if key == barcode and (not dataInfoDict.get(key) or '--' == dataInfoDict.get(key)):
                dataInfoDictList.remove(dataInfoDict)

    # 清除值为空字符串的字段
    for dataInfoDict in dataInfoDictList:
        tmpDict = copy.copy(dataInfoDict)
        for key in tmpDict.keys():
            if not dataInfoDict.get(key):
                dataInfoDict.pop(key)


def update_enclosure(dataInfoDictList, logger):
    global ctrlEnclId
    for dataInfoDict in dataInfoDictList:
        enclosureID = dataInfoDict.get('Enclosure ID')
        updateKey = 'Enclosure ID'
        if not enclosureID:
            logger.warn('[collectStandard]Enclosure ID is null.')
        else:
            type = dataInfoDict.get('Type')
            if 'Controller' == type:
                ctrlEnclId = enclosureID
            rack = 'rack=-1'
            shelf = 'shelf=' + enclosureID
            dataInfoDict[updateKey] = rack + ',' + shelf


def update_controller(dataInfoDictList, logger, ctrlOrBBUslotposMap, devModel):
    for dataInfoDict in dataInfoDictList:
        ctrlId = dataInfoDict.get('Controller ID', '')
        updateKey = 'Controller ID'
        if not ctrlId:
            logger.warn('[collectStandard]ctrlId is null.')
        else:
            rack = 'rack=-1'
            shelf = 'shelf=' + ctrlEnclId
            subshelf = 'subshelf=-1'
            slotpos = ctrlOrBBUslotposMap.get(devModel, 'slotpos=0')
            slot = 'slot=' + ctrlId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def update_expansion_module(dataInfoDictList, logger):
    for dataInfoDict in dataInfoDictList:
        enclIdOfExp = dataInfoDict.get('Enclosure ID', '')
        ctrlIdOfExp = dataInfoDict.get('Controller ID', '')
        updateKey = 'Controller ID'
        if not enclIdOfExp or not ctrlIdOfExp:
            logger.warn('[collectStandard]IdOfExp is null.')
        else:
            rack = 'rack=-1'
            shelf = 'shelf=' + enclIdOfExp
            subshelf = 'subshelf=-1'
            slotpos = 'slotpos=1'
            slot = 'slot=' + ctrlIdOfExp
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def update_interface_module(dataInfoDictList, logger):
    for dataInfoDict in dataInfoDictList:
        ctrlIdOfIntf = dataInfoDict.get('Controller ID', '')
        intfId = dataInfoDict.get('Interface Module ID', '')
        updateKey = 'Interface Module ID'
        if not ctrlIdOfIntf or not intfId:
            logger.warn('[collectStandard]IdOfIntf is null.')
        else:
            rack = 'rack=-1'
            shelf = 'shelf=' + ctrlEnclId
            subshelf = 'subshelf=-1'
            slotpos = 'slotpos=1'
            slot = 'slot=' + ctrlIdOfIntf + '.' + intfId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def update_power_module(dataInfoDictList, logger):
    for dataInfoDict in dataInfoDictList:
        enclIdOfPower = dataInfoDict.get('Enclosure ID', '')
        powerId = dataInfoDict.get('Power ID', '')
        updateKey = 'Power ID'
        if not enclIdOfPower or not powerId:
            logger.warn('[collectStandard]IdOfPower is null.')
        else:
            rack = 'rack=-1'
            shelf = 'shelf=' + enclIdOfPower
            subshelf = 'subshelf=-1'
            slotpos = 'slotpos=1'
            slot = 'slot=' + powerId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def update_bbu(dataInfoDictList, logger, ctrlOrBBUslotposMap, devModel):
    for dataInfoDict in dataInfoDictList:
        ctrlIdOfBbu = dataInfoDict.get('Controller ID', '')
        bbuId = dataInfoDict.get('BBU ID', '')
        updateKey = 'BBU ID'
        if not bbuId:
            logger.warn('[collectStandard]IdOfBBU is null.')
        else:
            rack = 'rack=-1'
            shelf = 'shelf=' + ctrlEnclId
            subshelf = 'subshelf=-1'
            slotpos = ctrlOrBBUslotposMap.get(devModel, 'slotpos=0')
            slot = 'slot='
            if not ctrlIdOfBbu:
                slot += bbuId
            else:
                slot += ctrlIdOfBbu + '.' + bbuId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def update_disk(dataInfoDictList, logger):
    for dataInfoDict in dataInfoDictList:
        diskLoc = dataInfoDict.get('Disk Location', '')
        updateKey = 'Disk Location'
        if not diskLoc:
            logger.warn('[collectStandard]IdOfDisk is null.')
        else:
            diskLocArr = diskLoc.replace('(', '').replace(')', '').split(',')
            rack = 'rack=-1'
            shelf = 'shelf=' + diskLocArr[0]
            subshelf = 'subshelf=-1'
            slotpos = 'slotpos=0'
            slot = 'slot=' + diskLocArr[1]
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def update_fan(dataInfoDictList, logger):
    for dataInfoDict in dataInfoDictList:
        enclIdOfFan = dataInfoDict.get('Enclosure ID', '')
        fanId = dataInfoDict.get('Fan ID', '')
        updateKey = 'Fan ID'
        if not enclIdOfFan or not fanId:
            logger.warn('[collectStandard]IdOfFan is null.')
        else:
            rack = 'rack=-1'
            shelf = 'shelf=' + enclIdOfFan
            subshelf = 'subshelf=-1'
            slotpos = 'slotpos=1'
            slot = 'slot=' + fanId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot


def adjust_item_loc_info(collectItem, dataInfoDictList, context, devModel):
    '''
    @summary: adjust item location info
    @date: 2014-08-08
    @param collectItem: collect item
    @param dataInfoDictList: the list of data dictionary
    @param context: context object
    @param devModel: device model
    @return: 
    '''

    logger = context.get('logger')
    ctrlOrBBUslotposMap = PosMap.ctrlOrBBUslotposMap4TV1

    logger.info('[collectStandard]Adjusting item location:' + collectItem)

    options = {
        'Enclosure': (update_enclosure, [dataInfoDictList, logger]),
        'Controller': (update_controller, [dataInfoDictList, logger, ctrlOrBBUslotposMap, devModel]),
        'ExpansionModule': (update_expansion_module, [dataInfoDictList, logger]),
        'InterfaceModule': (update_interface_module, [dataInfoDictList, logger]),
        'PowerModule': (update_power_module, [dataInfoDictList, logger]),
        'BBU': (update_bbu, [dataInfoDictList, logger]),
        'Disk': (update_disk, [dataInfoDictList, logger]),
        'Fan': (update_fan, [dataInfoDictList, logger])
    }

    if collectItem:
        selected_option = options.get(collectItem, '')
        if selected_option:
            function = selected_option[0]
            variables = selected_option[1]
            function(*variables)
        else:
            logger.info('no corresponding option :', collectItem)
    else:
        logger.info('[collectStandard]No need to adjusting item location:' + collectItem)


def system_version_issues_check(is_succ, cmd_ret, cmd4syst, logger):
    if not is_succ:
        succStateDict['Software'] = False
        logger.error('[collectStandard] Execute command [' + cmd4syst + '] failed!')
        return []
    isExecSuc, isParsable = utils.checkCliInfoValid(cmd_ret, True)
    if not isExecSuc:
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []

    return [is_succ, cmd_ret]


def check_is_succ(is_succ, cmd_ret):
    if is_succ:
        isExecSuc, isParsable = utils.checkCliInfoValid(cmd_ret, True)

        # 回文有效才解析
        if isExecSuc and isParsable:
            formatFunction = cHandleTypeList(cmd_ret)
            ret = formatFunction.handle()
            if len(ret) == 2:
                tmpRetDictList = ret[1]
                return tmpRetDictList
    return []


def cold_patch_version_check(is_succ, cmd_ret, ret_dict):
    tmpRetDictList = check_is_succ(is_succ, cmd_ret)
    if tmpRetDictList:
        for tmpRetDict in tmpRetDictList:
            ColdPatchPackageVersion = tmpRetDict.get('Package Version', '')

            # 取有效值
            if ColdPatchPackageVersion and '--' != ColdPatchPackageVersion:
                break
        ret_dict['ColdPatchPackageVersion'] = ColdPatchPackageVersion


def hot_patch_version_check(is_succ, cmd_ret, ret_dict):
    tmpRetDictList = check_is_succ(is_succ, cmd_ret)
    if tmpRetDictList:
        for tmpRetDict in tmpRetDictList:
            HotPatchPackageVersion = tmpRetDict.get('Package Version', '')
            if HotPatchPackageVersion and '--' != HotPatchPackageVersion:
                break
        ret_dict['HotPatchPackageVersion'] = HotPatchPackageVersion


def collect_software(sshConnector, cmdList, logger, sysInfoDict):
    '''
    @summary: collect software
    @date: 2014-08-08
    @param sshConnector: the connector of SSH
    @param cmdList: command list
    @param logger: a logger provided by tool
    @return: the dictionary list of result
    '''
    ret_dict = {}
    cmd4syst = cmdList[0]  # 系统
    cmd4coldP = cmdList[1]  # 冷补丁
    cmd4hotP = cmdList[2]  # 热补丁

    # 1.system version
    is_succ, cmd_ret = cliCmdManager.execCmd(sshConnector, cmd4syst, logger)

    check_res = system_version_issues_check(is_succ, cmd_ret, cmd4syst, logger)
    if not check_res:
        return check_res

    # 解析回文
    formatFunction = cHandleTypeList(cmd_ret)
    ret = formatFunction.handle()
    if len(ret) < 2:
        logger.error('[collectStandard] It occurs exception when parse cmd_ret!')
        return []
    SoftwarePackageVersion = ''
    tmpRetDictList = ret[1]
    for tmpRetDict in tmpRetDictList:
        SoftwarePackageVersion = tmpRetDict.get('Package Version', '')

        # 取含有效值的SoftwarePackageVersion
        if SoftwarePackageVersion and '--' != SoftwarePackageVersion:
            break
    ret_dict['SoftwarePackageVersion'] = SoftwarePackageVersion
    ret_dict['Device Serial Number'] = sysInfoDict.get('Device Serial Number', '')
    ret_dict['Device Type'] = sysInfoDict.get('Device Type', '')

    # 2.cold patch version
    is_succ, cmd_ret = cliCmdManager.execCmd(sshConnector, cmd4coldP, logger)
    cold_patch_version_check(is_succ, cmd_ret, ret_dict)

    # 3.hot patch version
    is_succ, cmd_ret = cliCmdManager.execCmd(sshConnector, cmd4hotP, logger)
    hot_patch_version_check(is_succ, cmd_ret, ret_dict)

    return [ret_dict]


def judge_collect_result(context, stdItemDataMap):
    '''
    @summary: Judge collect result.
    @date: 2014-08-12
    @param context: context object
    @param stdItemDataMap: the data map of standard item 
    @return: (return code, error message) as (integer, string)
    '''
    logger = context.get('logger')
    resource = getResource(context)
    errMsgList = []
    collectResutList = []

    # 单项全部成功：命令执行成功 and 总个数与收集个数相等
    # 单项全部失败：命令执行失败并且收集个数为0 or 命令执行成功并且总个数不为0并且收集个数为0
    # 单项部分成功：命令执行失败并且收集个数不为0 or 命令执行成功并且总个数与收集个数不等并且都不为0
    for itemName in stdItemDataMap:
        isSucc = succStateDict.get(itemName)
        allNum = context.get(itemName + 'Num', 0)  # 单项统计总个数
        collNum = len(stdItemDataMap.get(itemName))  # 单项实际收集到的个数
        logger.info('[collectStandard]' + itemName + ': isSucc=' + str(isSucc) + ';allNum=' + str(allNum)
                    + ';collNum=' + str(collNum))

        if isSucc and allNum == collNum:  # 单项全部成功
            logger.info('[collectStandard]Collect all ' + itemName + ' information')
            errMsgList.append(resource.getString('info.collect.' + itemName + '.info.success'))
            collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
        elif not isSucc and collNum == 0 or isSucc and allNum != 0 and collNum == 0:  # 单项全部失败
            logger.error('[collectStandard]Collect all ' + itemName + ' information failed.')
            errMsgList.append(resource.getString('err.collect.' + itemName + '.info.fail'))
            collectResutList.append(SCRIPT_RET_CODE.FAIL)
        else:  # 单项部分成功
            logger.warn('[collectStandard]Collect part ' + itemName + ' information')
            errMsgList.append(resource.getString('err.collect.part.' + itemName + '.info.fail'))
            collectResutList.append(SCRIPT_RET_CODE.PART_SUCCESS)

    # 判断总体收集成功、失败还是部分成功
    if collectResutList.count(SCRIPT_RET_CODE.SUCCESS) == len(stdItemDataMap):
        logger.info('[collectStandard]Collect all standard item information success.')
        return (SCRIPT_RET_CODE.SUCCESS, '')
    elif collectResutList.count(SCRIPT_RET_CODE.FAIL) == len(stdItemDataMap):
        logger.error('[collectStandard]Collect all standard item information failed!')
        return (SCRIPT_RET_CODE.FAIL, '\n'.join(errMsgList))
    else:
        logger.warn('[collectStandard]Collect part standard item information success.')
        return (SCRIPT_RET_CODE.PART_SUCCESS, '\n'.join(errMsgList))
