# -*- coding:utf-8 -*-
import copy
import os
from typing import List

from defusedxml import ElementTree as ET

from common import cliCmdManager
from common import utils
from common.cTV2R2 import cHandleTypeList
from common.commonFunction import getResource
from common.constant import SCRIPT_RET_CODE
from common.devNode import DevNode


def execute(context):
    '''
    @summary: Initialize device node.
    @date: 2014-07-28
    @param context: context object
    @return: 
    '''

    # 初始化各需要的变量
    logger = context.get('logger')
    logger.info('[prepare] start to prepare handle for TV1...')
    resource = getResource(context)
    devNode = context.get('devNode')
    logger.info('[prepare] devNode=' + str(devNode))
    if not devNode:
        devNode = DevNode(context)
        context['sshConnector'] = devNode.getSshConnector(context)
        context['devNode'] = devNode
        logger.info('[prepare] devNode=' + str(devNode))

    # 判断SSH连接状态
    logger.info('[prepare] Judge SSH connection status....')
    if not is_ssh_connection_normal(context):
        errMsg = resource.getString('err.create.SSH.connection.failed')
        return (SCRIPT_RET_CODE.FAIL, errMsg)

    # 判断系统状态
    logger.info('[prepare] Judge system status...')
    if not devNode.isSystemNormal4TV1(context):
        errMsg = resource.getString('err.sys.status.abnormal')
        return (SCRIPT_RET_CODE.FAIL, errMsg)

    # 解析标准收集项命令配置文件cmd4IBMS.xml
    cfgFile4Standard = context.get('path') + os.sep + 'script' + os.sep + 'config' + os.sep + 'cmd4IBMS.xml'
    errCode, errMsg = parse_cmd_cfg_file(context, cfgFile4Standard)

    # 解析生成标准项配置文件xmlarchivecfg.xml
    if not errMsg:
        xmlarchivecfgPath = context.get('path') + os.sep + 'xmlarchivecfg.xml'
        context['stdItemCfgMap'] = get_item_cfg_map(context, xmlarchivecfgPath)

    logger.info('[prepare] Finish preparing handle.')
    return (errCode, errMsg)


def is_ssh_connection_normal(context):
    '''
    @summary: Check if the SSH connection is normal
    @date: 2014-08-18
    @param context: context object
    @return: 
    '''
    logger = context.get('logger')
    sshConnector = context.get('sshConnector')
    if not sshConnector.getConnectionNoException():
        logger.error('[prepare] Get SSH connection failed!')
        return False
    else:
        logger.info('[prepare] Get SSH connection successfully!')
        return True


def param_to_cmd(paramTagList, context, linkSym) -> List[str]:
    for paramTag in paramTagList:
        paramDict = paramTag.attrib
        paraName = paramDict.get('paraName').strip()
        paraValList = []

        # 场景1：param标签含有LoopBaseCmd
        paraValCmd = paramDict.get('LoopBaseCmd').strip()
        if paraValCmd:
            fieldName = paramDict.get('loop').strip()
            paraValList = get_para_val_by_cli(context, paraValCmd, fieldName)

        # 场景2：param标签含有rangeList
        paraValRange = paramDict.get('rangeList').strip()
        if paraValRange:
            paraValList = paraValRange.split('|')

        # 拼接参数
        tmpList = []
        for paraVal in paraValList:
            tmpList.append(paraName + linkSym + paraVal)
        if not paramList:
            paramList = tmpList
            continue
        paramList = utils.getTranslatedCartesianRet(paramList, tmpList)

    return paramList


def get_item_cmd_list(itemTagList, context, linkSym, itemCmdListDict):
    for itemTag in itemTagList:
        # 获取收集项名称
        itemName = itemTag.attrib['name'].strip()

        # 获取收集项命令列表
        itemCmdList = []
        simpleCmdList = []  # 单命令（无参数）列表
        cmdTagList = itemTag.findall('cmd')
        for cmdTag in cmdTagList:
            paramTagList = cmdTag.findall('param')
            baseCmd = cmdTag.attrib['command'].strip()

            # 无参数的直接加入命令列表
            if not paramTagList:
                simpleCmdList.append(baseCmd)
                continue

            # 有参数的解析参数并拼凑成完整的命令
            paramList = param_to_cmd(paramTagList, context, linkSym)

            # 将当前cmd标签拼接成完整命令
            curTagFullCmdList = utils.getTranslatedCartesianRet([baseCmd], paramList)
            itemCmdList.extend(curTagFullCmdList)

        itemCmdList.extend(simpleCmdList)
        itemCmdListDict[itemName] = itemCmdList


def parse_cmd_cfg_file(context, cmdCfgFile):
    '''
    @summary: load and parse command configure file, then load to memory
    @date: 2014-07-28
    @param context: context object
    @param cmdCfgFile: command configuration file
    @return: (result code, error message) as (integer, string)
    '''
    logger = context.get('logger')
    resource = getResource(context)
    itemCmdListDict = {}
    errMsg = ''
    retCode = SCRIPT_RET_CODE.SUCCESS
    linkSym = '='  # 参数名与值的连接符，有些环境是空格' '

    # 加载配置和解析配置文件
    root = ET.parse(cmdCfgFile)
    itemTagList = root.findall('item')
    try:
        get_item_cmd_list(itemTagList, context, linkSym, itemCmdListDict)
    except:
        logger.error('[prepare]Parse IBMS command configure file caught exception.')
        errMsg = resource.getString('err.parse.IBMS.cfg.failed')
        retCode = SCRIPT_RET_CODE.FAIL
    else:
        logger.info('[prepare]Standard itemTag name-cmdList:' + str(itemCmdListDict))
        context['stdItemCmdDict'] = itemCmdListDict
    finally:
        return (retCode, errMsg)


def get_para_val_by_cli(context, cmd, fieldName):
    '''
    @summary: Get the value of parameter.
    @date: 2014-07-28
    @param context: context object
    @param cmd: command
    @param fieldName: field name
    @return: the value list of parameter
    '''

    sshConnector = context['sshConnector']
    logger = context.get('logger')

    # 发送命令
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger)
    if not isSucc:
        logger.error('[prepare] Execute base loop command [' + cmd + '] failed!')
        return []

    # 检查数据有效性
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        logger.error('[prepare]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[prepare]Executed command successfully but the result is not parsable.')
        return []

    # 解析回文
    formatFunction = cHandleTypeList(cmdRet, [fieldName])
    fieldInfoDictList = formatFunction.handle()[1]

    # 获取ID值
    fieldValList = [fieldInfoDict.get(fieldName, '') for fieldInfoDict in fieldInfoDictList]
    tmpValList = copy.copy(fieldValList)

    # 去除空ID
    for tmpVal in tmpValList:
        if '' == tmpVal:
            fieldValList.remove(tmpVal)

    logger.info('[prepare]FieldValueList:' + str(fieldValList))
    return fieldValList


def get_item_cfg_map(xmlarchivecfgPath):
    '''
    @summary: get collect item configuration map
    @date: 2014-09-09
    @param xmlarchivecfgPath: the path of file named xmlarchivecfg.xml
    @return: collect item configuration map
    '''
    root = ET.parse(xmlarchivecfgPath)
    dataTypeList = root.getiterator('DataType')

    stdItemCfgMap = {}
    for dataType in dataTypeList:
        # 收集项名称
        typeId = dataType.attrib['typeId']
        stdItemCfgMap[typeId] = []
        dataAttrList = dataType.findall('DataAttr')
        for dataAttr in dataAttrList:
            # 具体的收集子项配置字典
            dataAttrDict = dataAttr.attrib
            stdItemCfgMap[typeId].append(dataAttrDict)
    return stdItemCfgMap
