﻿# -*- coding:utf-8 -*-
from com.huawei.ism.tlv.lang import UnsignedInt32
from com.huawei.ism.tlv.bean import Param
from com.huawei.ism.tlv.docoder import ParamType
from com.huawei.ism.tlv.bean import Record
from com.huawei.ism.tlv import TLVUtils
from com.huawei.ism.exception import IsmException
from com.huawei.ism.tool.framework.pubservice.exception import ConnectionException
from com.huawei.ism.tool.framework.platform.exception import ToolException

from connectionPara import TlvPara
from connectionPara import SshPara
from cliCmdManager import execCmd
from tlvCmdManager import getBatch, invoke

from constant import T_V2_FactoryCfg,Status,TV2R2_TLV_CMD,ObjectType,SystemRoleType,CMD_DEFAULT_TIMEOUT
from constant import RET_CODE,ProductModel,CmdConstants,ExpboardType,EnclosureType
import commonFunction 

class DevNode():
    def __init__(self, context):
        dev = context.get("dev")
        self.logger = context.get('logger')
        self.deviceName = dev.getDeviceName()
        self.sn = dev.getDeviceSerialNumber()
        self.localIP = dev.getIp()
        self.peerIP = dev.getPeerIp()
        self.sshPort = dev.getPort()
        self.tlvPort = T_V2_FactoryCfg.TLV_PORT
        self.userName = dev.getLoginUser().getUserName()
        self.userPassword = dev.getLoginUser().getPassword()
        self.devPwd = dev.getDeveloperPwd()
        self.lang = context.get('lang')
        
        self.sshPara = SshPara(self.userName, 
                               self.userPassword, 
                               self.localIP, 
                               self.sshPort)
        self.tlvPara = TlvPara(self.userName, 
                               self.userPassword, 
                               self.localIP, 
                               True, 
                               self.tlvPort)
        
    def __str__(self):
        return ('sn:' + str(self.sn)
                + '\n' + 'localIP:' + str(self.localIP)
                + '\n' + 'peerIP:' + str(self.peerIP)
                + '\n' + 'sshPort:' + str(self.sshPort)
                + '\n' + 'tlvPort:' + str(self.tlvPort)
                + '\n' + 'userName:' + str(self.userName)
                + '\n' + 'lang:' + str(self.lang))
                
    def __CreateSshConnector(self, context):
        '''
        Function describe: Crate SSH connector.
        Input:            
        Return value:     (None)
                            
        Revision History: 1. Created 2014-07-28
        '''
        self.sshConnector = commonFunction.createSshConnector(self.sshPara, context, self.logger) 
        return     
    
    def __CreateTlvConnector(self, context):
        '''
        Function describe: Crate TLV connector.
        Input:            
        Return value:     (None)
                            
        Revision History: 1. Created 2014-07-28
        '''
        
        self.tlvConnector = commonFunction.createTlvConnector(self.tlvPara, context)
        return
    
    def getSshConnector(self, context):
        '''
        Function describe: Crate SSH connector.
        Input:            
        Return value:     (None)
                            
        Revision History: 1. Created 2014-07-28
        '''
        try:
            return self.sshConnector
        except:
            self.__CreateSshConnector(context)
            return self.sshConnector
        else:
            pass
        
    def getTlvConnector(self, context):
        '''
        Function describe: Crate SSH connector.
        Input:            
        Return value:     (None)
                            
        Revision History: 1. Created 2014-07-28
        '''
        try:
            return self.tlvConnector
        except:
            self.logger.warn('Got TLV connector failed, try to create it now.')
            self.__CreateTlvConnector(context)
            return self.tlvConnector
        else:
            pass    
    
    def isSystemNormal(self, context):
        dev = context.get("dev")
        userName = dev.getLoginUser().getUserName()
        userPasswd = dev.getLoginUser().getPassword()
        isSysNormal = True
        sshConnector = self.getSshConnector(context)
        healthSta, runningSta = commonFunction.getSystemStatus(sshConnector, self.logger, userName, userPasswd)
        if not healthSta:
            self.logger.error('Health status is not normal.')
            isSysNormal = False
        
        if not runningSta or (runningSta.lower() != 'running'):
            self.logger.error('Running status is not normal.')
            isSysNormal = False  
            
        return  isSysNormal
        
    def getSysMgtCfgList(self, context):
        sshConnector = self.getSshConnector(context)
        dev = context.get("dev")
        userName = dev.getLoginUser().getUserName()
        userPasswd = dev.getLoginUser().getPassword()
        return commonFunction.getMgtCfgList(sshConnector, self.logger, userName, userPasswd)
    
    def getNodeNumber(self, context):
        '''
        Function describe: Get the number of node in a cluster.
        Input:            
        Return value:     (int, string)
                            int :number of node in a cluster， -1 implies query failed.
                            String: error message if there is.
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        #Object type parameter.
        msgParam0 = Param(0, ParamType.ENUM, UnsignedInt32(ObjectType.CONTROLLER))
        params = TLVUtils.paramList(msgParam0)
        
        try:
            tlvRet = invoke(tlvConnector, TV2R2_TLV_CMD.COUNT, params, CmdConstants.TLV_DEFAULT_TIMEOUT, logger)
            if not tlvRet[0] or not tlvRet[1]:
                logger.error('Query controller number failed by TLV.')
                return (-1, tlvRet[1])
            
            rec = tlvRet[1]
            nodeNum = rec.getParamIntValue(10).intValue()
        except Exception, e:
            logger.error('Query controller number caught exception:' + str(e))
            return (-1, str(e))
        else:
            logger.info('Query controller number:' + str(nodeNum))
            return (nodeNum, '')
    

    def getNodeNumberOfExp(self, tlvConnector):
        '''
        Function describe: Get the number of node in a cluster.
        Input:            
        Return value:     (int, string)
                            int :number of node in a cluster， -1 implies query failed.
                            String: error message if there is.
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        
        #Object type parameter.
        msgParam0 = Param(0, ParamType.ENUM, UnsignedInt32(ObjectType.CONTROLLER))
        params = TLVUtils.paramList(msgParam0)
        
        try:
            tlvRet = invoke(tlvConnector, TV2R2_TLV_CMD.COUNT, params, CmdConstants.TLV_DEFAULT_TIMEOUT, logger)
            if not tlvRet[0] or not tlvRet[1]:
                logger.error('Query controller number failed by TLV.')
                return (-1, tlvRet[1])
            
            rec = tlvRet[1]
            nodeNum = rec.getParamIntValue(10).intValue()
        except Exception, e:
            logger.error('Query controller number caught exception:' + str(e))
            return (-1, str(e))
        else:
            logger.info('Query controller number:' + str(nodeNum))
            return (nodeNum, '')
    
    
    def getExpboardNumber(self, context):
        '''
        Function describe: Get the number of expand board in a cluster.
        Input:            
        Return value:     (int, string)
                            int :number of node in a cluster， -1 implies query failed.
                            String: error message if there is.
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        #Object type parameter.
        expboardNum, errMsg = commonFunction.getExpboardNumber(tlvConnector, logger)
        if expboardNum < 0:
            logger.info('Queried expand board number failed:' + str(errMsg))
            return -1
            
        return expboardNum
    
    def getEnclosureNumber(self, context):
        '''
        Function describe: Get the number of expand board in a cluster.
        Input:            
        Return value:     (int, string)
                            int :number of node in a cluster， -1 implies query failed.
                            String: error message if there is.
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        enclosureNum, errMsg = commonFunction.getEnclosureNumber(tlvConnector, logger)
        if enclosureNum < 0:
            logger.info('Queried enclosure number failed:' + str(errMsg))
            return -1
            
        return enclosureNum
        
    def getSysInfoByTlv(self, context, devSN = ''):
        '''
        Function describe:    Get system information of an engine.
        Input: devSN, if Null, then get all's.           
        Return value:  
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        #Object type parameter.
        msgParam0 = Param(0, ParamType.ENUM, UnsignedInt32(ObjectType.SYSTEM))
        if devSN:
            msgParam1 = Param(1, ParamType.STRING, devSN)
            params = TLVUtils.paramList(msgParam0, msgParam1)
        else:
            params = TLVUtils.paramList(msgParam0)
        
        systemInfo = {'sn':'', 
                    'systemName':'',
                    'healthStatus':None,
                    'runningStatus':None,
                    'productModel':None,
                    'productVersion':None}    
        try:
            tlvRet = invoke(tlvConnector, TV2R2_TLV_CMD.GET, params, CMD_DEFAULT_TIMEOUT, logger)
            if not tlvRet[0] or not tlvRet[1]:
                logger.error('Query controller information failed by TLV.')
                return (RET_CODE.EXEC_FAILED,  {})
            
            rec = tlvRet[1]        
                
            sn = rec.getParamStrValue(1)
            systemName = rec.getParamStrValue(2)
            healthStatus = rec.getParamIntValue(7).intValue()
            runningStatus = rec.getParamIntValue(8).intValue()
            productModel = rec.getParamIntValue(64).intValue()
            productVersion = rec.getParamStrValue(65)
            
            healthStatus = Status.healthStatusMap.get(healthStatus)
            runningStatus = Status.runningStatusMap.get(runningStatus)
            
            #ProductModel is a class, but productModel is an integer.
            productModel = ProductModel.modelMap.get(productModel)
            
            systemInfo['sn'] = sn
            systemInfo['systemName'] = systemName
            systemInfo['healthStatus'] = healthStatus
            systemInfo['runningStatus'] = runningStatus
            systemInfo['productModel'] = productModel
            systemInfo['productVersion'] = productVersion
        except Exception, e:
            logger.error('Query system information caught exception:' + str(e))
            return (RET_CODE.PARSE_TLV_EXCEPT, {})
        else:
            logger.info('Queried information of system successfully:' + str(systemInfo))
            return (RET_CODE.RET_SUCCESS, systemInfo)
    
    def getCtrlsInfoByTlv(self, context):
        '''
        Function describe:    Get controller information of an engine.
        Input: devSN, if Null, then get all's.           
        Return value:  
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        #Object type parameter.
        msgParam0 = Param(0, ParamType.ENUM, UnsignedInt32(ObjectType.CONTROLLER))
        params = TLVUtils.paramList(msgParam0)
        
        ctrlList = []
        ctrlInfo = {'id':None, 
                    'name':None,
                    'location':None,
                    'healthStatus':None,
                    'runningStatus':None,
                    'softVer':None,
                    'temperature':None,
                    'elabel': None,
                    'pcbVer':None,
                    'sesVer':None,
                    'bmcVer':None,
                    'logicVer':None,
                    'biosVer':None,
                    'memorySize':None,
                    'cpuInfo':None,
                    'role':None,
                    'cpuUsage':None,
                    'memoryUsage':None,
                    'voltage':None,
                    'dirtyDataRate':None                   
                    }    
        try:
            tlvRet = getBatch(tlvConnector, TV2R2_TLV_CMD.GET_BATCH_NEXT, params, CMD_DEFAULT_TIMEOUT, logger)
            if not tlvRet[0] or not tlvRet[1]:
                logger.error('Query controller information failed by TLV.')
                return (RET_CODE.EXEC_FAILED,  ctrlList)
            
            recs = tlvRet[1]        
            recNum = recs.size()
            logger.info('Record size: ' + str(recNum))
            logger.info('Records:' + str(recs))
            
            for index in range(0, recNum):
                try:
                    rec = recs.get(index)
                    logger.info('This record:' + str(rec))
                    ctrlID = rec.getParamStrValue(1)
                    name = rec.getParamStrValue(2)
                    location = rec.getParamStrValue(6)
                    healthStatus = rec.getParamIntValue(7).intValue()
                    runningStatus = rec.getParamIntValue(8).intValue()
                    softVer = rec.getParamStrValue(64)
                    temperature = rec.getParamIntValue(65).intValue()
                    
                    elabel = rec.getParamStrValue(67)
                    pcbVer = rec.getParamStrValue(68)
                    sesVer = rec.getParamStrValue(69)
                    bmcVer = rec.getParamStrValue(70)
                    logicVer = rec.getParamStrValue(71)
                    biosVer = rec.getParamStrValue(72)
                    memorySize = rec.getParamIntValue(74).intValue()
                    cpuInfo = rec.getParamStrValue(75)
                    role = rec.getParamIntValue(76).intValue()
                    cpuUsage = rec.getParamIntValue(77).intValue()
                    memoryUsage = rec.getParamIntValue(78).intValue()
                    voltage = rec.getParamIntValue(79).intValue() / 10.0
                    dirtyDataRate =  rec.getParamIntValue(81).intValue() 

                except Exception, e:
                    logger.error('Parse controller information record caught exception:' + str(e))
                    continue
                else:
                    healthStatus = Status.healthStatusMap.get(healthStatus)
                    runningStatus = Status.runningStatusMap.get(runningStatus)
                    role = SystemRoleType.systemRoleMap.get(role)
                    #ProductModel is a class, but productModel is an integer.
                    
                    ctrlInfo['id'] =  ctrlID
                    ctrlInfo['name'] = name
                    ctrlInfo['location'] = location
                    ctrlInfo['healthStatus'] = healthStatus
                    ctrlInfo['runningStatus'] = runningStatus
                    ctrlInfo['softVer'] = softVer
                    ctrlInfo['temperature'] = temperature
                    ctrlInfo['elabel'] = elabel
                    ctrlInfo['pcbVer'] = pcbVer
                    ctrlInfo['sesVer'] = sesVer
                    ctrlInfo['bmcVer'] = bmcVer
                    ctrlInfo['logicVer'] = logicVer
                    ctrlInfo['biosVer'] = biosVer
                    ctrlInfo['memorySize'] = memorySize
                    ctrlInfo['cpuInfo'] = cpuInfo
                    ctrlInfo['role'] = role
                    ctrlInfo['cpuUsage'] = cpuUsage
                    ctrlInfo['memoryUsage'] = memoryUsage
                    ctrlInfo['voltage'] = voltage
                    ctrlInfo['dirtyDataRate'] = dirtyDataRate
                    ctrlList.append(ctrlInfo)
                finally:
                    ctrlInfo = {}
        except Exception, e:
            logger.error('Query controller information caught exception:' + str(e))
            return (RET_CODE.PARSE_TLV_EXCEPT, ctrlList)
        else:
            logger.info('Queried information of system successfully:' + str(ctrlList))
            return (RET_CODE.RET_SUCCESS, ctrlList)
        
            
    def getEnclosureByTlv(self, context):
        '''
        Function describe:    Get all enclosures of a cluster.
        Input:            
        Return value:     [EnclosureDict{'ID': id
                                        'Logic Type': logicType,
                                        'Health Status': healthStatus,
                                        'Running Status': runningStatus,
                                        'Type':type,
                                        'Temperature(Celsius)': temperature}]
                            {} implies query failed.
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        enclosureList = []
        
        #Object type parameter.
        msgParam0 = Param(0, ParamType.ENUM, UnsignedInt32(ObjectType.ENCLOSURE))
        params = TLVUtils.paramList(msgParam0)
        
        try:
            tlvRet = getBatch(tlvConnector, TV2R2_TLV_CMD.GET_BATCH_NEXT, params, CmdConstants.TLV_DEFAULT_TIMEOUT, logger)
            if not tlvRet[0]:
                logger.error('Query enclosure by TLV failed.')
                return {}
            
            recs = tlvRet[1]
            recNum = recs.size()
            logger.info('Number of enclosure is:' + str(recNum))
            
            thisEnclosure = {}
            
            for index in range(0, recNum):
                try:
                    rec = recs.get(index)
                    enclosureId = rec.getParamStrValue(1)
                    logicType = rec.getParamIntValue(65).intValue()
                    healStatus = rec.getParamIntValue(7).intValue()
                    runStatus = rec.getParamIntValue(8).intValue()
                    enclosureType = rec.getParamIntValue(0).intValue()
                    temperature = rec.getParamIntValue(67).intValue()
                except Exception, e:
                    logger.error('Parse enclosure record caught exception:' + str(e))
                    continue
                else:
                    logicType = EnclosureType.logicTypeMap.get(logicType)
                    healStatus = Status.healthStatusMap.get(healStatus)
                    runStatus = Status.runningStatusMap.get(runStatus)
                    enclosureType = EnclosureType.typeMap.get(enclosureType)
                    
                    thisEnclosure = {'ID':enclosureId,
                                    'Logic Type':logicType,
                                    'Health Status':healStatus,
                                    'Running Status':runStatus,
                                    'Type':enclosureType,
                                    'Temperature(Celsius)':temperature}
                    
                    logger.info('This enclosure info:' + str(thisEnclosure))
                    enclosureList.append(thisEnclosure)
                finally:
                    thisEnclosure = {}

        except IsmException, e:
            logger.error("Query enclosure caught IsmException: "  + e.getErrorId())
            return enclosureList
        except ConnectionException, e:
            logger.error("Query enclosure caught ConnectionException: " + e.getErrorId())
            return enclosureList
        except Exception, e:
            logger.error('Query enclosure caught exception:' + str(e))
            return enclosureList
        else:
            logger.info('Query enclosure successfully:' + str(enclosureList))

        return enclosureList

    def getExpboardByTlv(self, context):
        '''
        Function describe:    Get all expand board of a cluster.
        Input:            
        Return value:     [EnclosureDict{'ID': id,-----------------------1
                                        'name': name,     ---------------2
                                        'Health Status': healthStatus,---7
                                        'Running Status': runningStatus,-8
                                        'Type':model,--------------------64
                                        'logicVer':logicVer,-------------65
                                        'pcbVer':pcbVer,-----------------66
                                        'sesVer':sesVer,-----------------67
                                        'elabel':elabel,-----------------68
                                        'Voltage(V)': vol}]--------------69
                            {} implies query failed.
        Revision History: 1. Created 2014-07-28
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        expboardList = []
        
        #Object type parameter.
        msgParam0 = Param(0, ParamType.ENUM, UnsignedInt32(ObjectType.EXPBOARD))
        params = TLVUtils.paramList(msgParam0)
        
        try:
            tlvRet = getBatch(tlvConnector, TV2R2_TLV_CMD.GET_BATCH_NEXT, params, CmdConstants.TLV_DEFAULT_TIMEOUT, logger)
            if not tlvRet[0]:
                logger.error('Query expand board by TLV failed.')
                return expboardList
            
            recs = tlvRet[1]
            recNum = recs.size()
            logger.info('Number of expand board is:' + str(recNum))
            
            thisExpboard = {}
            
            for index in range(0, recNum):
                try:
                    rec = recs.get(index)
                    expboardId = rec.getParamStrValue(1)
                    expboardName = rec.getParamStrValue(2)
                    location = rec.getParamStrValue(6)
                    healStatus = rec.getParamIntValue(7).intValue()
                    runStatus = rec.getParamIntValue(8).intValue()
                    expandboardType = rec.getParamIntValue(64).intValue()
                    
                    logicVer = rec.getParamStrValue(65)
                    pcbVer   = rec.getParamStrValue(66)
                    sesVer   = rec.getParamStrValue(67)
                    elabel   = rec.getParamStrValue(68)
                    
                    voltage = rec.getParamIntValue(69).intValue() / 10.0
                except Exception, e:
                    logger.error('Parse enclosure record caught exception:' + str(e))
                    continue
                else:
                    healStatus = Status.healthStatusMap.get(healStatus)
                    runStatus = Status.runningStatusMap.get(runStatus)
                    expandboardType = ExpboardType.expboardTypeMap.get(expandboardType)
                    
                    thisExpboard = {'ID': expboardId,
                                    'name': expboardName,
                                    'location':location,
                                    'Health Status': healStatus,
                                    'Running Status': runStatus,
                                    'Type':expandboardType,
                                    'logicVer':logicVer,
                                    'pcbVer':pcbVer,
                                    'sesVer':sesVer,
                                    'elabel':elabel,
                                    'Voltage(V)': voltage}
                    
                    logger.info('This expboard info:' + str(thisExpboard))
                    expboardList.append(thisExpboard)
                finally:
                    thisExpboard = {}

        except IsmException, e:
            logger.error("Query expand board caught IsmException: "  + e.getErrorId())
            return expboardList
        except ConnectionException, e:
            logger.error("Query expand board caught ConnectionException: " + e.getErrorId())
            return expboardList
        except Exception, e:
            logger.error('Query expand board caught exception:' + str(e))
            return expboardList
        else:
            logger.info('Query expand board successfully:' + str(expboardList))

        return expboardList
    
    def getDeviceProductModel(self, context):
        '''
        Function describe:    Get the product model of controller by CLI.
        Return value:         tuple(boolean, string)
                                      boolean-true if command executed success, false if command executed failed.
                                      string-device product model, e.g., S5500T.
        Revision History:     1. Created 2014-07-28 
        '''
        logger = self.logger
        sshConnector = self.getSshConnector(context)
		
        dev = context.get("dev")
        userName = dev.getLoginUser().getUserName()
        userPasswd = dev.getLoginUser().getPassword()
      
        return commonFunction.getDeviceProductModel(sshConnector, logger, userName, userPasswd)    
    
    def getDeviceFullSystemVersion(self, context):
        '''
        Function describe:    Get the product model of controller by CLI.
        Return value:         tuple(boolean, string)
                                      boolean-true if command executed success, false if command executed failed.
                                      string-device product model, e.g., S5500T.
        Revision History:     1. Created 2014-07-28 
        '''
        logger = self.logger
        sshConnector = self.getSshConnector(context)
		
        dev = context.get("dev")
        userName = dev.getLoginUser().getUserName()
        userPasswd = dev.getLoginUser().getPassword()
      
        return commonFunction.getDeviceProductFullVersion(sshConnector, logger, userName, userPasswd)      
    
    def getDeviceMemorySize(self):
        ''''
        Function describe:    Get memory size of controller.
        Return value:         tuple(boolean, string)
                                      boolean-true if command executed success, false if command executed failed.
                                      string-device memory size, e.g., 8.000GB.
        Revision History:     1. Created 2014-07-28
        '''
        logger = self.logger
        dev = context.get("dev")
        userName = dev.getLoginUser().getUserName()
        userPasswd = dev.getLoginUser().getPassword()
        
        (exeSucc, memSize) = commonFunction.getDeviceMemorySize(self.sshConnector, logger, userName, userPasswd)
        if not exeSucc:
            logger.error('Query memory size failed.')
            return (False, '')
        else:
            logger.info('Query memory size:' + str(memSize))
            return (True, memSize)
        
    def getBBUNumber(self, context):
        '''
        Function describe: Get the number of BBU.
        Input:            
        Return value:     (int):number of fan in a cluster， -1 implies query failed.
        Revision History: 1. Created 2014-08-01
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        BBUNum, errMsg = commonFunction.getBBUNumber(tlvConnector, logger)
        if BBUNum < 0:
            logger.info('Queried BBU number failed:' + str(errMsg))
            return -1
            
        return BBUNum
    
    def getFanNumber(self, context):
        '''
        Function describe: Get the number of FAN.
        Input:            
        Return value:     (int):number of fan in a cluster， -1 implies query failed.
        Revision History: 1. Created 2014-08-01
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        fanNum, errMsg = commonFunction.getFanNumber(tlvConnector, logger)
        if fanNum < 0:
            logger.info('Queried fan number failed:' + str(errMsg))
            return -1
            
        return fanNum

    def getVmmsNumber(self, context):
        '''
        Function describe: Get the number of vmms Num.
        Input:            
        Return value:     (int):number of vmms in a cluster， -1 implies query failed.
        Revision History: 1. Created 2014-08-01
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        vmmsNum, errMsg = commonFunction.getVmmsNumber(tlvConnector, logger)
        if vmmsNum < 0:
            logger.info('Queried Vmms number failed:' + str(errMsg))
            return -1
            
        return vmmsNum
    
    def getDiskNumber(self, context):
        '''
        Function describe: Get the number of DISK.
        Input:            
        Return value:     (int):number of disk in a cluster， -1 implies query failed.
        Revision History: 1. Created 2014-08-01
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        diskNum, errMsg = commonFunction.getDiskNumber(tlvConnector, logger)
        if diskNum < 0:
            logger.info('Queried disk number failed:' + str(errMsg))
            return -1
            
        return diskNum
    
    def getIntfModuleNumber(self, context):
        '''
        Function describe: Get the number of interface module in a cluster.
        Input:            
        Return value:     (int):number of interface module in a cluster， -1 implies query failed.
        Revision History: 1. Created 2014-08-01
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        intfModuleNum, errMsg = commonFunction.getIntfModuleNumber(tlvConnector, logger)
        if intfModuleNum < 0:
            logger.info('Queried interface module number failed:' + str(errMsg))
            return -1
            
        return intfModuleNum

    def getPowerNumer(self, context):
        '''
        Function describe: Get the number of power in a cluster.
        Input:            
        Return value:     (int):number of power in a cluster， -1 implies query failed.
        Revision History: 1. Created 2014-08-01
        '''
        logger = self.logger
        tlvConnector = self.getTlvConnector(context)
        
        powerNum, errMsg = commonFunction.getPowerNumber(tlvConnector, logger)
        if powerNum < 0:
            logger.info('Queried power number failed:' + str(errMsg))
            return -1
            
        return powerNum
    