# -*- coding: UTF-8 -*-
import re

from common.log import Log
from frame.cli import cliResource
from pymodule.packages.utils.device_series_util import MAX_RETRYS, DEVELOPER_MODEL_FLAG, CLI_RET_END_FLAG, \
    STOR_DEV_INFO_DICT
from pymodule.packages.utils.query_util import is_chinese, is_none_license, query_result_with_no_record, \
    is_internal_error


def get_msg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    """
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    """
    err_msg = "\n--"

    try:
        if not resource.has_key(msg):
            return err_msg

        locale_dict = resource.get(msg)
        if not locale_dict.has_key(lang):
            return err_msg

        locale_msg = locale_dict.get(lang)
        if "%s" in locale_msg or "%i" in locale_msg:
            return locale_msg % args
        else:
            return locale_msg

    except Exception as ex:
        Log.error("Failed to get message: %s", ex)
        return err_msg


def is_none_object(cli_ret):
    """
    @summary: 判断回显结果里面是否包含查询的对象
    @param cli_ret: cli回显
    @return: 
        True: cli查询的对象为空
        False: cli查询的对象不为空
    """
    cli_ret_list = cli_ret.splitlines()
    for line in cli_ret_list:
        lower_line = line.lower()
        if "error" in lower_line and "not exist" in lower_line:
            return True

    return False


def get_user_name(cli, lang):
    """
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: flag, ret, errMsg
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    """
    cli_ret = ""
    err_msg = ""

    try:
        cmd = "show system general"
        check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
        if not check_ret[0]:
            return check_ret

        cli_ret = check_ret[1]
        line_list = cli_ret.splitlines()

        if not line_list:
            err_msg = get_msg(lang, "cannot.get.sys.info")
            return True, cli_ret, err_msg

        last_line = line_list[-1]
        if CLI_RET_END_FLAG in last_line:
            user_name = last_line.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            return True, user_name, err_msg

    except Exception as ex:
        Log.error("Failed to get user name: %s", ex)
        err_msg = get_msg(lang, "cannot.get.sys.info")
        return False, cli_ret, err_msg

    return False, cli_ret, err_msg


def get_user_privilege(cli, lang):
    """
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: flag, ret, errMsg
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    """
    cli_ret = ""
    err_msg = ""

    try:
        check_ret = get_user_name(cli, lang)
        if not check_ret:
            return check_ret

        user_name = check_ret[1]
        cmd = "show user user_name=%s" % user_name
        check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
        if not check_ret[0]:
            return check_ret

        cli_ret = check_ret[1]
        cli_ret_lines_list = get_horizontal_cli_ret(cli_ret)

        if not cli_ret_lines_list:
            err_msg = get_msg(lang, "cannot.get.user.info")
            return False, cli_ret, err_msg

        for line in cli_ret_lines_list:
            name = line["Name"]
            if name == user_name:
                user_privilege = line["Level"]
                return True, user_privilege, err_msg

    except Exception as ex:
        Log.error("Failed to get user privilege: %s", ex)
        err_msg = get_msg(lang, "cannot.get.user.info")
        return False, cli_ret, err_msg

    return False, cli_ret, err_msg


def has_admin_or_super_admin_privilege(cli, lang):
    """
    @summary: 判断用户是否具有管理员或超级管理员权限
    @param cli: cli对象
    @param cli: 语言lang
    @return: flag, ret, errMsg
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时
                True: 具有管理员或超级管理员权限
                False： 不具有管理员或超级管理员权限
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    """
    check_ret = get_user_privilege(cli, lang)
    if not check_ret[0]:
        return check_ret

    user_level = check_ret[1]
    if not user_level or user_level.lower() not in ["super_admin", "admin"]:
        return True, False, ""
    else:
        return True, True, ""


def enter_developer_mode(cli, lang):
    """
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @param devPwd: 语言devPwd
    @return: flag, cliRet, errMsg
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    """
    err_msg = ""
    cli_ret = ""

    try:
        check_ret = has_admin_or_super_admin_privilege(cli, lang)
        if not check_ret[0]:
            return check_ret

        is_admin_or_super_admin_privilege = check_ret[1]
        if not is_admin_or_super_admin_privilege:
            err_msg = get_msg(lang, "cannot.enter.to.developer.model")
            return False, cli_ret, err_msg

        cmd = "change user_mode current_mode user_mode=developer"
        check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
        if not check_ret[0]:
            return check_ret

        if is_in_developer_mode(check_ret[1]):
            return True, check_ret, err_msg
        else:
            err_msg = get_msg(lang, "cannot.access.developer.mode")
            developer_mode2_cli_mode(cli)
            return False, check_ret, err_msg

    except Exception as ex:
        Log.error("Failed to get: %s", ex)
        err_msg = get_msg(lang, "cannot.access.developer.mode")
        return False, check_ret, err_msg


def developer_mode2_cli_mode(cli):
    """
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    """
    index = 0
    try:
        cli_ret = cli.execCmd("show system general")
        while "Password is wrong" in cli_ret or "Password:" in cli_ret or is_in_developer_mode(cli_ret):
            index += 1
            cli_ret = cli.execCmd("exit")
            if "Are you sure to exit?(y/n):" in cli_ret:
                cli_ret = cli.execCmd("n")
            if index > MAX_RETRYS:
                break
    except Exception as ex:
        Log.error("Failed to get developer_mode2_cli_mode: %s", ex)

    return None


def is_in_developer_mode(cli_ret):
    """
    @summary: 判断当前是否在developer模式下
    @param cli_ret: cli回显
    @return: 
        True: 当前在developer模式下
        False: 当前不在developer模式下
    """
    if re.search(DEVELOPER_MODEL_FLAG, cli_ret, re.IGNORECASE):
        return True
    return False


def check_line_in_black_list(cli_line, lang):
    """
    @summary: 判断cli语句行是否在黑名单中
    @param cli_line: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    """
    err_msg = ""

    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cli_line:
            if is_chinese(lang):
                err_msg = dictItems.get("msg_zh")
            else:
                err_msg = dictItems.get("msg_en")
            return True, err_msg
    return False, err_msg


class CheckMode:

    def __init__(self):
        self.next_handler = None

    def set_next(self, handler):
        pass

    def handle_request(self, request):
        pass


class CheckModeCliRet(CheckMode):
    def __init__(self, lang):
        super().__init__()
        self.lang = lang

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if not request:
            err_msg = get_msg(self.lang, "cli.result.is.empty")
            return False, request, err_msg
        return self.next_handler.handle_request(request)


class CheckModeQueryRecord(CheckMode):
    def __init__(self, err_msg):
        super().__init__()
        self.err_msg = err_msg

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if query_result_with_no_record(request):
            return True, request, self.err_msg
        return self.next_handler.handle_request(request)


class CheckModeLicense(CheckMode):
    def __init__(self):
        super().__init__()

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if is_none_license(request):
            return True, request, self.err_msg
        return self.next_handler.handle_request(request)


class CheckModeInternalError(CheckMode):
    def __init__(self, lang):
        super().__init__()
        self.lang = lang

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if is_internal_error(request):
            err_msg = get_msg(self.lang, "cli.excute.find.internal.error")
            return False, request, err_msg
        return self.next_handler.handle_request(request)


class CheckModeObject(CheckMode):
    def __init__(self, lang):
        super().__init__()
        self.lang = lang

    def set_next(self, handler):
        self.next_handler = handler

    def handle_request(self, request):
        if is_none_object(request):
            err_msg = get_msg(self.lang, "object.does.not.exit")
            return False, request, err_msg
        return False, "-1", "-1"


def process_request(cli_ret, lang, err_msg):
    handler_a = CheckModeCliRet(lang)
    handler_b = CheckModeQueryRecord(err_msg)
    handler_c = CheckModeLicense(err_msg)
    handler_d = CheckModeInternalError(lang)
    handler_e = CheckModeObject(lang)

    handler_a.set_next(handler_b)
    handler_b.set_next(handler_c)
    handler_d.set_next(handler_e)

    return handler_a.handle_request(cli_ret)


def execute_cmd_in_cli_mode(cli, cmd, log_result, lang):
    """
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param log_result: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: flag, cliRet, errMsg
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    """
    cli_ret = ""

    if cli is None:
        err_msg = get_msg(lang, "device.connect.abnormal")
        return False, cli_ret, err_msg

    try:
        if log_result:
            cli_ret = cli.execCmd(cmd)
        else:
            cli_ret = cli.execCmdNoLog(cmd)
    except Exception as ex:
        Log.error("Failed to do cli.exeCmd: %s", ex)
        err_msg = get_msg(lang, "system.status.abnormal")
        return False, cli_ret, err_msg

    _, request, err_msg = process_request(cli_ret, lang)
    if err_msg != '-1':
        return False, request, err_msg

    line_list = cli_ret.splitlines()
    for line in line_list:
        check_ret = check_line_in_black_list(line, lang)
        if check_ret[0]:
            err_msg = check_ret[1]
            return False, cli_ret, err_msg

    return True, cli_ret, err_msg


def excute_cmd_in_developer_mode(cli, cmd, is_has_log, lang):
    """
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param is_has_log: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param devPwd: devPwd
    @return: flag, ret, errMsg
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    """
    enter_developer_check_ret = enter_developer_mode(cli, lang)
    if not enter_developer_check_ret[0]:
        developer_mode2_cli_mode(cli)
        return enter_developer_check_ret

    check_ret = execute_cmd_in_cli_mode(cli, cmd, is_has_log, lang)
    cli_ret = check_ret[1]

    if is_in_developer_mode(cli_ret):
        developer_mode2_cli_mode(cli)

    return check_ret


def get_product_model_version(cli, lang):
    """
    @summary: 获取产品类型和产品版本
    @param cli: cli对象
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型 产品版本 
            flag为False时，cli回显
        errMsg: 错误消息
    """
    cmd = "show system general"
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_list = cli_ret.splitlines()
    product_model = ""
    product_version = ""
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Product Model":
            product_model = field_value
        elif field_name == "Product Version":
            product_version = field_value

        if not product_model and not product_version and product_model != "--" and product_version != "--":
            return True, " ".join((product_model, product_version)), ""

    return False, cli_ret, get_msg(lang, "cannot.get.product.version.info")


def get_product_model(cli, lang):
    """
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    """
    cmd = "show system general"
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_list = cli_ret.splitlines()
    product_model = ""
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Product Model":
            product_model = field_value

        if not product_model and product_model != "--":
            return True, product_model, ""

    return False, cli_ret, get_msg(lang, "cannot.get.product.model.info")


def get_product_version(cli, lang, context):
    """
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    """
    product_version = ""
    dev_node = context["dev"]
    if dev_node is not None:
        product_version = dev_node.getProductVersion()
        del dev_node
    if product_version != "" and product_version is not None:
        return True, product_version, ""

    cmd = "show system general"
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_list = cli_ret.splitlines()
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Product Version":
            product_version = field_value

        if not product_version and product_version != "--":
            return True, product_version, ""

    return False, cli_ret, get_msg(lang, "cannot.get.product.model.info")


def get_storage_type(cli, lang):
    """
    @summary: 获取存储设备类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，存储设备类型
            flag为False时，cli回显
        errMsg: 错误消息
    """
    get_product_model_ret = get_product_model(cli, lang)
    if not get_product_model_ret[0]:
        return get_product_model_ret

    pdt_model = get_product_model_ret[1]

    if not STOR_DEV_INFO_DICT.has_key(pdt_model):
        err_msg = get_msg(lang, "cannot.get.storage.type.info")
        return False, "", err_msg

    storage_type = STOR_DEV_INFO_DICT.get(pdt_model)
    return True, storage_type, ""


def get_horizontal_cli_ret(cli_ret):
    """
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cli_ret: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    try:
        headline = ""
        i = 0
        cli_ret_list = cli_ret.encode("utf8").splitlines()
        for line in cli_ret_list:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cli_ret_list) - 1:
            return []

        title = cli_ret_list[i - 1]
        field_words = cli_ret_list[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0

        while start_pos <= len(headline):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        dict_list = []
        for line in field_words:
            if line.find(":/>") >= 0 or (len(tuple_idxs) >= 2 and len(line) <= tuple_idxs[-2][1]):
                break
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dict_list.append(dict(zip(keys, vals)))

        return dict_list
    except Exception as ex:
        Log.error("Failed to get horizontal cli ret: %s", ex)
        return []


def get_vertical_cli_ret(cliRet, is_parse_elc_label=True):
    """
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    """
    cli_ret_list = cliRet.encode("utf8").splitlines()
    dict_list = []
    line_dict = {}
    elc_label = ""
    for line in cli_ret_list:
        if CLI_RET_END_FLAG in line:
            break

        if re.search("^-+\r*\n*$", line):
            if not is_parse_elc_label and line_dict.has_key("Electronic Label"):
                line_dict["Electronic Label"] = line_dict["Electronic Label"] + elc_label
                elc_label = ""

            dict_list.append(line_dict.copy())
            line_dict.clear()

        fields = line.split(" : ")
        if len(fields) < 2:
            if is_parse_elc_label and "=" in line:
                fields = line.split("=")
            elif "=" in line and re.search("^\s{16,}", line):
                elc_label += line
                continue
            else:
                continue

        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")

        if line_dict.has_key(key):
            key += "_" + str(str(line_dict.keys()).count(key + "_") + 1)
        line_dict.setdefault(key, value)

    if line_dict:
        if not is_parse_elc_label and line_dict.has_key("Electronic Label"):
            line_dict["Electronic Label"] = line_dict["Electronic Label"] + elc_label
        dict_list.append(line_dict.copy())

    return dict_list


def get_splited_cli_ret(cli_ret, split_mark):
    """
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cli_ret: cli回显
    @param split_mark: 分割标记
    """
    cli_ret_list = cli_ret.split(split_mark)
    if len(cli_ret_list) < 2:
        return ""

    splited_cli_ret = cli_ret_list[1]
    match = re.search("^.*:(/>)*\r*\n*$", splited_cli_ret, re.MULTILINE)
    if match:
        splited_cli_ret = splited_cli_ret[0:match.start()]

    if not splited_cli_ret.strip():
        return ""
    else:
        return splited_cli_ret


def is_engine_id(value):
    """
    @summary: 根据id判断是否为引擎ID
    @param value: 回显中的ID
    @return: 
        True: 是引擎ID
        False: 不是引擎ID
    """
    if "ENG" in value or "CTE" in value:
        return True
    else:
        return False


def is_disk_id(value):
    """
    @summary: 根据id判断是否为硬盘ID
    @param value: 回显中的ID
    @return: 
        True: 是硬盘ID
        False: 不是硬盘ID
    """
    if "DAE" in value:
        return True
    else:
        return False


def is_data_switch_id(value):
    """
    @summary: 根据id判断是否为交换机ID
    @param value: 回显中的ID
    @return: 
        True: 是交换机ID
        False: 不是交换机ID
    """
    if "DSW" in value:
        return True
    else:
        return False


def is_high_density_disk(enclosure_id, cli, lang):
    """
    @summary: 判断硬盘框是否为高密框
    @param enclosure_id: 框ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，是高密框
            flag为False时，不是高密框
        errMsg: 错误时的消息
    """
    if not is_disk_id(enclosure_id):
        return True, False, ""

    check_ret = get_high_density_disk_enclosure_id_list(cli, lang)
    if not check_ret[0]:
        return check_ret

    high_density_disk_enclosure_id_list = check_ret[1]
    if enclosure_id in high_density_disk_enclosure_id_list:
        return True, True, ""
    else:
        return True, False, ""


def get_enclosure_height(cli, enclosure_id, lang):
    """
    @summary: 根据框ID获取框高度(U)
    @param cli: cli对象
    @param enclosure_id: 框ID
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的高度(U)
            flag为False时，cli回显
        errMsg: 错误消息
    """

    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Height(U)" % enclosure_id
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_list = cli_ret.splitlines()
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Height(U)":
            enclosure_height = field_value
            return True, enclosure_height, ""

    err_msg = get_msg(lang, "cannot.get.enclosure.info")
    return False, cli_ret, err_msg


def get_enclosure_type(cli, enclosure_id, lang):
    """
    @summary: 根据框ID获取框类型
    @param cli: cli对象
    @param enclosure_id: 框ID
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的类型
            flag为False时，cli回显
        errMsg: 错误消息
    """

    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Type" % enclosure_id
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_list = cli_ret.splitlines()
    for line in cli_ret_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Type":
            enclosure_type = field_value
            return True, enclosure_type, ""

    err_msg = get_msg(lang, "cannot.get.enclosure.info")
    return False, cli_ret, err_msg


def get_controller_id_list(cli, lang):
    """
    @summary: 获取设备所有控制器ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，设备所有控制器ID列表
            flag为False时，cli回显
        errMsg: 错误消息
    """
    cmd = "show controller general|filterColumn include columnList=Controller"
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    cli_ret_lines_list = cli_ret.splitlines()
    control_list = []

    for line in cli_ret_lines_list:
        fields = line.split(":")
        if len(fields) < 2:
            continue

        field_name = fields[0].strip()
        field_value = fields[1].strip()

        if field_name == "Controller":
            ctrl_id = field_value
            control_list.append(ctrl_id)

    if not control_list:
        err_msg = get_msg(lang, "cannot.get.controller.info")
        return False, cli_ret, err_msg

    return True, control_list, ""


def get_high_density_disk_enclosure_id_list(cli, lang):
    """
    @summary: 获取高密硬盘框ID集合
    @param cli: cli对象
    @param lang: 语言lang
    @return: flag, ret, errMsg
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，高密硬盘框ID集合
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    """
    cmd = "show enclosure"
    check_ret = execute_cmd_in_cli_mode(cli, cmd, True, lang)
    if not check_ret[0]:
        return check_ret

    cli_ret = check_ret[1]
    high_density_disk_enclosure_id_list = []
    cli_ret_lines_list = get_horizontal_cli_ret(cli_ret)

    if not cli_ret_lines_list:
        err_msg = get_msg(lang, "cannot.get.enclosure.info")
        return False, cli_ret, err_msg

    for line in cli_ret_lines_list:
        enclosure_type = line["Type"]
        enclosure_id = line["ID"]
        if "4U 75" in enclosure_type:
            high_density_disk_enclosure_id_list.append(enclosure_id)

    return True, high_density_disk_enclosure_id_list, ""
