# -*- coding: UTF-8 -*-
from frame.cli.cliUtil import get_horizontal_cli_ret, get_vertical_cli_ret, get_splited_cli_ret
from utils.cli_parser_util import log_when_cli_execute_failed
from utils.common_function_util import execute_cmd, execute_cmd_in_developer
from utils.log_util import Log


def parse_vertical_table(context, command):
    """
    @summary: 执行cli命令并将回显解析成[{},{},...]，此种cli回显为竖向列表
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    return is_success, dict_list


def parse_transverse_table(context, command):
    """
    @summary: 执行cli命令并将回显解析成[{},{},...]，此种cli回显为横向列表
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    # 将cli回显解析为字典列
    dict_list = get_horizontal_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    return is_success, dict_list


def parse_mix_table(context, command):
    """
    @summary: 执行cli命令并将回显解析成[{},{},...]，此种cli回显为多个“----“分隔的竖向表格
                                    但第一个表格与其它表格字段不一致，此方法用来获取其中第一个表格的内容
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)

    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    else:
        dict_list = [dict_list[0]]
    return is_success, dict_list


def parse_software(context, command):
    """
    @summary: 执行show upgrade package,获取其中的部分内容
    @param command: cli命令
    @return: (cli执行是否成功，cli回显解析后的列表，cli执行失败时的原因)
    """
    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    hot_patch_version_index = cli_ret.find("HotPatch Version")

    software_version = get_splited_cli_ret(cli_ret[:hot_patch_version_index], "Software Version")
    software_version_list = get_horizontal_cli_ret(software_version)

    hot_patch_version = get_splited_cli_ret(cli_ret[hot_patch_version_index:], "HotPatch Version")
    hot_patch_version_list = get_horizontal_cli_ret(hot_patch_version)

    for soft_version in software_version_list:
        for hot_patch_version in hot_patch_version_list:
            if soft_version["Name"] == hot_patch_version["Name"]:
                soft_version["patchVersion"] = hot_patch_version["Current Version"]

    return is_success, software_version_list


def parse_vertical_table_no_parse_elc_label(context, command):
    """
    @summary: 解析竖向列表结构的CLI回显,其中电子标签作为一个整体,不再解析
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    # 将cli回显解析为字典列
    is_parse_elc_label = False
    dict_list = get_vertical_cli_ret(cli_ret, is_parse_elc_label)
    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    return is_success, dict_list


def parse_vertical_table_in_developer(context, command):
    """
    @summary: 解析竖向列表结构的developer命令回显
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = execute_cmd_in_developer(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    return is_success, dict_list


def parse_transverse_table_in_developer(context, command):
    """
    @summary: 解析横向列表结构的developer命令回显
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = execute_cmd_in_developer(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    # 将cli回显解析为字典列
    dict_list = get_horizontal_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    return is_success, dict_list


def get_license_or_feature_cli_ret(context, command, is_license_feature):
    """
    @summary: 解析show license命令回显，并将其解析为license基本信息，license特性两个字典列表
    @param allData: cli回显(一个字典列表)
    @param filtData: <attr>节点中配置的收集数据
    @return: 收集项的数据（一个字典列表）
    """
    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    region_index = cli_ret.find("Region")
    feature_index = cli_ret.find("Feature Name")
    if is_license_feature:
        cli_ret = cli_ret[region_index:]
    else:
        cli_ret = cli_ret[:feature_index]
    # 将cli回显解析为字典列
    dict_list = get_vertical_cli_ret(cli_ret)
    # cli执行成功，但cli解析为空，即cli未获取到信息
    if not dict_list:
        dict_list = None
    return is_success, dict_list


def parse_license(context, command):
    """
    @summary: 执行show license命令回显，并将回显的上部解析为license的基本信息
    """
    is_license_feature = False
    return get_license_or_feature_cli_ret(context, command, is_license_feature)


def parse_license_feature(context, command):
    """
    @summary: 执行show license命令回显，并将回显的下部部解析为license特性的基本信息
    """
    is_license_feature = True
    return get_license_or_feature_cli_ret(context, command, is_license_feature)


def parse_upgrade_package(context, command):
    """
    @summary: 执行show upgrade packge命令回显，将Software Version与HotPatch Version的
              回显存放到同一个字典列表中，并增加Version Type键来区分Version类型
    """

    is_success, cli_ret, err_msg = execute_cmd(context, command)
    if not is_success:
        log_when_cli_execute_failed(command, err_msg, cli_ret)
        return is_success, None

    software_version_index = cli_ret.find("Software Version")
    hot_patch_version_index = cli_ret.find("HotPatch Version")
    software_version_list = get_horizontal_cli_ret(cli_ret[software_version_index:hot_patch_version_index])
    hot_patch_version_list = get_horizontal_cli_ret(cli_ret[hot_patch_version_index:])
    Log.info("softwareVersionList list is %s" % str(software_version_list))
    Log.info("hotPatchVersionList list is %s" % str(hot_patch_version_list))

    if not software_version_list or not hot_patch_version_list:
        dict_list = None
        return is_success, dict_list

    for software_version in software_version_list:
        software_version["Version Type"] = "Software Version"

    for hot_patch_version in hot_patch_version_list:
        hot_patch_version["Version Type"] = "HotPatch Version"

    Log.info("softwareVersionList2 list is %s" % str(software_version_list))
    Log.info("hotPatchVersionList2 list is %s" % str(hot_patch_version_list))

    software_version_list.extend(hot_patch_version_list)
    dict_list = software_version_list
    Log.info("upgradePackage list is %s" % str(dict_list))

    return is_success, dict_list
