# -*- coding: UTF-8 -*-
import os
from collections import namedtuple

from common.constant import SCRIPT_RET_CODE
from common.result import get_err_msg_from_language_file
from utils.cli_parser_util import get_info_by_cli_parser
from utils.data_collect_util import get_cmd_node_info
from utils.log_util import Log

ERROR_CODE = -1


def get_base_cmd_node_info(base_cmd_node):
    """
    @summary: 从baseCmd节点中获取cli命令信息
    @param base_cmd_node: baseCmd节点解析后的字典
    @return: (cli命令，cli类型，需要获取的回显字段(作为其他cli命令的参数)
    """
    command = base_cmd_node["elementAttrbutes"]["command"]
    cli_type = base_cmd_node["elementAttrbutes"]["type"]
    param_name = base_cmd_node["elementAttrbutes"]["paramName"]

    return command, cli_type, param_name


def check_collection_status(fail_num, collect_item_nodes):
    collect_item_nodes_count = len(collect_item_nodes)

    # 全部收集成功
    if fail_num == 0:
        return SCRIPT_RET_CODE.SUCCESS
    # 全部收集失败
    elif fail_num == collect_item_nodes_count:
        return SCRIPT_RET_CODE.FAIL
    # 部分收集成功
    else:
        return SCRIPT_RET_CODE.PART_SUCCESS


def process_data(params):
    error_result = (-1, -1, -1)

    ret_code_status_dict = {
        SCRIPT_RET_CODE.FAIL: (
            process_when_ret_code_fail,
            [params.fail_num, params.collect_item_name, params.context, params.all_err_msg]),
        SCRIPT_RET_CODE.PART_SUCCESS: (
            process_when_ret_code_partial_success,
            [params.fail_num, params.collect_item_name, params.context, params.ret_data_map,
             params.all_err_msg]),
        SCRIPT_RET_CODE.SUCCESS: (
            process_when_ret_code_success,
            [params.fail_num, params.collect_item_name, params.ret_data_map, params.all_err_msg])
    }

    try:
        selected_option = ret_code_status_dict.get(params.code, "")
        function = selected_option[0]
        variables = selected_option[1]
        return function(*variables)
    except Exception as e:
        Log.error("Invalid code: %s", params.code)
        Log.error("Fail to get ret_code_status: %s", e)
        return error_result


# 全部收集失败
def process_when_ret_code_fail(fail_num, collect_item_name, context, all_err_msg):
    fail_num += 1
    err_msg_key = "collect.fail.msg.%s" % collect_item_name
    err_msg = get_err_msg_from_language_file(context, err_msg_key) + os.linesep
    ret_data_map = {collect_item_name: []}
    all_err_msg = all_err_msg + err_msg
    return fail_num, ret_data_map, all_err_msg


# 部分收集成功
def process_when_ret_code_partial_success(fail_num, collect_item_name, context, ret_data_map, all_err_msg):
    fail_num += 1
    err_msg_key = "collect.partfail.msg.%s" % collect_item_name
    err_msg = get_err_msg_from_language_file(context, err_msg_key) + os.linesep
    all_err_msg = all_err_msg + err_msg
    return fail_num, ret_data_map, all_err_msg


# success, but no data
def process_when_ret_code_success(fail_num, collect_item_name, ret_data_map, all_err_msg):
    if ret_data_map is None:
        ret_data_map = {collect_item_name: list()}
    return fail_num, ret_data_map, all_err_msg


FuncParams = namedtuple("FuncParams",
                        ['code', 'fail_num', 'collect_item_name', 'context', 'ret_data_map', 'all_err_msg'])


def collect_data(context, xm_info):
    """
    @summary: 根据每个收集项的收集类型和使用的cli类型收集数据
    @param xm_info: 解析xml后的信息(一个嵌套的字典)
    @return: None
    """
    # 收集项失败数
    fail_num = 0
    # 收集项的数据
    all_ret_data_map = {}
    # 失败的收集项构成的字符串，用于工具界面显
    all_err_msg = ""

    collect_item_nodes = xm_info["elementChildEles"]

    collect_num = float(len(collect_item_nodes))
    current_collect_num = 0
    listener = context.get('listener')
    global_dict = globals()

    error_result = (-1, -1, -1)

    for node in collect_item_nodes:
        # 根据cmd4IBMS.xml中配置的类型自动调用dataCollectUtil.py中的同名方法来收集
        collect_item_type = node["elementAttrbutes"]["type"]
        collect_item_method = global_dict[collect_item_type]
        code, ret_data_map = collect_item_method(context, node)

        collect_item_name = node["elementAttrbutes"]["name"]

        params = FuncParams(code, fail_num, collect_item_name, context, ret_data_map, all_err_msg)

        fail_num, ret_data_map, all_err_msg = process_data(params)
        if fail_num == ERROR_CODE:
            return error_result

        all_ret_data_map.update(ret_data_map)

        # 收集进度
        current_collect_num += 1
        progress = int((current_collect_num / collect_num) * 100)
        listener.refleshProgress(progress)

    code = check_collection_status(fail_num, collect_item_nodes)

    success_result = (code, all_ret_data_map, all_err_msg)

    return success_result


def single(context, collect_item_node):
    """
    @summary: 通过单条cli命令收集收集�?
    @param collect_item_node: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    """
    cmd_node = collect_item_node["elementChildEles"][0]
    command, cli_type, collect_item_title_list = get_cmd_node_info(cmd_node)

    # retDataList:[{},{},...]
    is_success, ret_data_list = get_info_by_cli_parser(context, command, cli_type, collect_item_title_list)

    # success, but no data
    if is_success and ret_data_list is None:
        return SCRIPT_RET_CODE.SUCCESS, None

    # 收集失败
    if not is_success:
        return SCRIPT_RET_CODE.FAIL, None
    # 收集成功
    else:
        collect_item_name = collect_item_node["elementAttrbutes"]["name"]
        return SCRIPT_RET_CODE.SUCCESS, {collect_item_name: ret_data_list}


def dependency(context, collect_item_node):
    """
    @summary: 通过两条cli命令收集收集项，其中第二条cli的参数来自于第一条cli的回显
    @param collect_item_node: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    """
    base_command, base_cli_type, param_name = ("", "", "")
    command, cli_type, collect_item_title_list = ("", "", [])

    # 获取<baseCmd><cmd>中的内容
    cmd_nodes = collect_item_node["elementChildEles"]
    for cmdNode in cmd_nodes:
        if cmdNode["elementName"] == "baseCmd":
            base_command, base_cli_type, param_name = get_base_cmd_node_info(cmdNode)
        else:
            command, cli_type, collect_item_title_list = get_cmd_node_info(cmdNode)

    # 执行<baseCmd>的cli命令，最终获取的数据将作<cmd>的cli命令的参
    base_collect_item_title_list = list()
    base_collect_item_title_list.append(param_name)
    is_success, param_map_list = get_info_by_cli_parser(context, base_command, base_cli_type,
                                                        base_collect_item_title_list)

    # success, but no data
    if is_success and param_map_list is None:
        return SCRIPT_RET_CODE.SUCCESS, None

    # 收集失败
    if not is_success:
        return SCRIPT_RET_CODE.FAIL, None

    # 循环执行<cmd>中的命令，并<baseCmd>获取的结果作为参
    fail_num = 0
    ret_data_map_list = []
    for paramMap in param_map_list:
        param = paramMap[param_name]
        is_success, ret_data_list = get_info_by_cli_parser(context, command % param, cli_type, collect_item_title_list)
        # 收集失败
        if not is_success:
            fail_num += 1
        elif ret_data_list is not None:
            ret_data_map_list.extend(ret_data_list)

    collect_item_name = collect_item_node["elementAttrbutes"]["name"]
    # 全部收集成功
    param_map_list_count = len(param_map_list)

    if fail_num == 0:
        return SCRIPT_RET_CODE.SUCCESS, {collect_item_name: ret_data_map_list}

    elif fail_num == param_map_list_count:  # 全部收集失败
        return SCRIPT_RET_CODE.FAIL, None

    else:  # 部分收集成功
        return SCRIPT_RET_CODE.PART_SUCCESS, {collect_item_name: ret_data_map_list}


def many(context, collect_item_node):
    """
    @summary: 通过多条条cli命令收集收集项，此方法只适用于cli回显单个表格数据的情况
    @param collect_item_node: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    """
    cmd_nodes = collect_item_node["elementChildEles"]

    ret_data_map = {}
    for cmd_node in cmd_nodes:
        # 执行命令
        command, cli_type, collect_item_title_list = get_cmd_node_info(cmd_node)
        is_success, ret_data_list = get_info_by_cli_parser(context, command, cli_type, collect_item_title_list)

        # 只要有一条命令执行失败就认为收集失败
        if not is_success:
            return SCRIPT_RET_CODE.FAIL, None

        # success, but no data
        if ret_data_list is None:
            return SCRIPT_RET_CODE.SUCCESS, None

        # 将获取到的cli回显解析后的map追加到retDataMap
        ret_data_map.update(ret_data_list[0])

    collect_item_name = collect_item_node["elementAttrbutes"]["name"]
    return SCRIPT_RET_CODE.SUCCESS, {collect_item_name: [ret_data_map]}
