# -*- coding:utf-8 -*-
import copy
import re
import traceback
from common.exception import DisconnectionException
from common.commonFunction import getResource
from common.commonFunction import handleS5000InterfaceCli
from common.constant import SCRIPT_RET_CODE
from common.constant import PosMap
from common import utils
from common import cliCmdManager
from common.cBase import cFastTypeDict
from common.cTV2R2 import cHandleTypeList


isLiceFileExst = True #license是否存在
succStateDict = {} #命令执行成功状态字典
ctrlEnclId = ''
G_cmdExecRetDict = {} #所有收集项对应命令执行后的回显


def execute(context):
    '''
    @summary: collect standard items
    @date: 2014-07-25
    @param context: context object
    @return: (result code, error message) as (integer, string)
    '''
    logger = context.get('logger')
    logger.info('[collectStandard]Collect standard item start!')
    sshConnector = context['sshConnector']
    resource = getResource(context)
    global isLiceFileExst
    global succStateDict
    global G_cmdExecRetDict #全局的存放收集项所有的回显信息的字典
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin1
    try:
        #xmlarchivecfg.xml生成的字典
        stdItemCfgMap = context.get('stdItemCfgMap')
        logger.info(str(stdItemCfgMap))
        
        #返回给框架的title字典:{收集项名称：[收集项返回的所有CLI回显字段名的列表]}
        stdItemTitleMap = getItemTitleMap(stdItemCfgMap)
        
        #获取标准收集项名称和barcode映射,{'Enclosure': 'Serial Number', 
        #                        'Controller': 'BarCode', 
        #                        'Disk': 'BarCode', ....}
        stdItemBarcodeMap = getItemFieldMap(stdItemCfgMap, 'BarCode')
        logger.info(str(stdItemBarcodeMap))
        
        #获取标准收集项命令字典
        stdItemCmdDict = context.get('stdItemCmdDict')
        
        if not stdItemTitleMap or not stdItemCmdDict:
            logger.error('[collectStandard]Invalid device node object!')
            return (SCRIPT_RET_CODE.FAIL, resource.getString('err.collect.necessary.info.failed'), {})
        
        #获取标准收集项数据格式
        stdItemDataMap = getDataMapStruct(stdItemTitleMap)
        
        #初始化成功状态字典
        succStateDict = initSuccStateDict(stdItemTitleMap)
        
        devNode = context.get('devNode')
        if not devNode:
            logger.error('[collectStandard]Invalid device node object!')
            return (SCRIPT_RET_CODE.FAIL, resource.getString('err.init.dev.failed'), {})
        else:
            devModel = devNode.getDeviceProductModel(context)
        
        for collectItem in stdItemTitleMap:
            G_cmdExecRetDict[collectItem] = '' #存放单个收集项执行命令后的回显
            cmdList = stdItemCmdDict.get(collectItem)
            if not cmdList:
                logger.warn('[collectStandard] command list is null:' + collectItem)
                continue
            logger.info('[collectStandard]collectItem=' + str(collectItem))
            logger.info('[collectStandard]cmdListLen=' + str(len(cmdList)))
            
            #Software不符合通用处理方式，需要单独处理
            if 'Software' == collectItem:
                logger.info('[collectStandard] collect Software...')
                sysInfoDictList = stdItemDataMap.get('System')
                #若系统信息不存在则获取并赋值
                if not sysInfoDictList:
                    cliCmd = stdItemCmdDict.get('System')[0]
                    sysInfoDictList = getSysInfoDictList(sshConnector, cliCmd, logger)
                    stdItemDataMap['System'] = sysInfoDictList
        
                retDictList = collectSoftware(sshConnector, cmdList, logger, sysInfoDictList[0])
                stdItemDataMap[collectItem].extend(retDictList)
                
                #统计单个收集项总个数
                countSingleItemSum(context, stdItemDataMap, collectItem)
                continue
            
            #兼容同一个功能使用不同命令执行成功情况的标志位，例如：
            isCtrlCmdExeSucc = False 
            isExpCmdExeSucc = False
            isCollectLicense = True
            
            #通用处理流程：发命令后解析回文获取数据
            for cliCmd in cmdList:
                #发送命令获取回文
                isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, logger)
                G_cmdExecRetDict[collectItem] += '\n' + cmdRet #统计对应收集项的回显数据
                if not isSucc:
                    succStateDict[collectItem] = False
                    logger.error('[collectStandard] Execute command [' + cliCmd + '] failed!')
                    break
                if 'License' == collectItem:
                    #license不存在
                    if re.search('The License file does not exist', cmdRet, re.IGNORECASE):
                        isLiceFileExst = False
                    
                #判断数据有效性
                isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
                if not isExecSuc:
                    logger.error('[collectStandard]Failed to execute command and the result is not parsable:' + collectItem)
                    continue
                elif not isParsable:
                    logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
                    continue
                
                #兼容带有info后缀的命令在2000的设备上都可以执行成功的情况
                if 'Controller' == collectItem:
                    isCtrlCmdExeSucc = True
                if 'ExpansionModule' == collectItem:
                    isExpCmdExeSucc = True
                if 'InterfaceModule' == collectItem:
                    logger.info('Parsing interface CLI now!')
                    dataInfoDictList = handleS5000InterfaceCli(cmdRet, logger)
                else:
                    #解析回文
                    formatFunction = cFastTypeDict(cmdRet, defaultSep='|')
                    dataInfoDictList = formatFunction.handle()[1]
                    
                logger.info('[collectStandard]dataInfoDictList[' + collectItem + "][" + cliCmd + ']' + str(dataInfoDictList))
                    
                #如果dataInfoDictList为空则换一种解析方式
                if not dataInfoDictList:
                    formatFunction = cHandleTypeList(cmdRet)
                    ret = formatFunction.handle()
                    if len(ret) < 2:
                        continue
                    else:
                        dataInfoDictList = ret[1]
                        
                #针对硬盘的逻辑的状态为"Isolation"、"Fault"的情况，使用"showdisk -s x -sl y"格式的命令不能查询到“Disk Location“字段
                if "Disk" == collectItem:
                    for dataInfoDict in dataInfoDictList:
                        diskLoc = dataInfoDict.get('Disk Location', '')
                        updateKey = 'Disk Location'
                        if not diskLoc:
                            logger.warn('[collectStandard]IdOfDisk is null,command is:' + cliCmd)
                            diskLoc = cliCmd.split()
                            frameNo = diskLoc[2]
                            soltNo = diskLoc[4]
                            dataInfoDict[updateKey] = "(" + frameNo + "," + soltNo + ")"
                            logger.warn('[collectStandard]Disk info:' + frameNo + "," + soltNo)
                if 'System' == collectItem:
                    if not dataInfoDictList[0]['System Name']:
                        logger.warn('System name is NULL, fill with default name:[Device Model]')
                        dataInfoDictList[0]['System Name'] = dataInfoDictList[0]['Device Model']
                    
                #更新收集需要的license信息
                if 'License' == collectItem:
                    sysInfoDictList = stdItemDataMap.get('System')
                    if not sysInfoDictList:
                        cliCmd = stdItemCmdDict.get('System')[0]
                        sysInfoDictList = getSysInfoDictList(sshConnector, cliCmd, logger)
                        stdItemDataMap['System'] = sysInfoDictList
                    devSN = sysInfoDictList[0].get('Device Serial Number', '')
                    logger.info('[collectStandard] update license devSN=' + devSN)
                    updateLicenseInfo(devSN, dataInfoDictList, logger)
     
                stdItemDataMap[collectItem].extend(dataInfoDictList)
                
                #S2300V1R1C02版本，Controller收集项两个命令都能收集成功：showctrlboardinfo/showctrlboard
                if isCtrlCmdExeSucc:
                    logger.info('Controller item information is collect successfully!')
                    break
                if isExpCmdExeSucc:
                    logger.info('Expboard item information is collect successfully!')
                    break
                
            #统计单个收集项总个数
            if "Disk" == collectItem:
                #将"showdisk -p"命令执行的回显添加到详细信息中
                isSucc, showdiskCmdRet = cliCmdManager.execCmd(sshConnector, "showdisk -p", logger)
                G_cmdExecRetDict["Disk"] = showdiskCmdRet + "\n" + G_cmdExecRetDict["Disk"] #
                #【特殊处理】硬盘部件的个数的统计
                componentNumKey = collectItem + 'Num'
                context[componentNumKey] = len(cmdList)
                logger.info('[collectStandard][Disk]' + componentNumKey + ':' + str(context[componentNumKey]))
            else:
                countSingleItemSum(context, stdItemDataMap, collectItem)
                    
            #清除Barcode为空的项
            itemDictList = stdItemDataMap.get(collectItem)
            cleanNullData(collectItem, itemDictList, stdItemBarcodeMap)
            logger.info(collectItem + '[collectStandard]' + str(itemDictList))
            
            #调整收集项locatio
            adjustItemLocInfo(collectItem, itemDictList, context, devModel,)
            
        #更新柜信息字典列表
        updateRackDictList(stdItemDataMap, stdItemCfgMap, context)
        
        #更新量纲信息字典列表
        updateDimeDictList(stdItemDataMap, stdItemCfgMap, context)
        
        context['titleListMap'] = stdItemTitleMap
        context['retDataMap'] = stdItemDataMap
        logger.info("----[CollectInfo]Begin-----")
        logger.info(str(stdItemTitleMap))
        logger.info(str(stdItemDataMap))
        logger.info("----[CollectInfo]End-----")
        
        #判断收集结果：成功、失败or部分成功
        collectResult, errMsg, errMsgDetailDict = judgeCollectResult(context, stdItemDataMap)
        return (collectResult, errMsg, errMsgDetailDict)
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin2
    except DisconnectionException, de:
        logger.error('[collectStandard] Maybe the network is unavailable or device is abnormal, details:' + traceback.format_exc())
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg, {})
    # 处理不同阶段网络断开问题（涉及的问题单为： ）End


def countSingleItemSum(context, itemDataMap, collectItem):
    '''
    @summary: get single item sum
    @date: 2014-10-09
    @param context: context object
    @param itemDataMap: item data map
    @param collectItem: collect item
    @return: single item sum
    '''
    logger = context.get('logger')
    itemDictList = itemDataMap.get(collectItem)
    componentNumKey = collectItem + 'Num'
    context[componentNumKey] = len(itemDictList)
    logger.info('[collectStandard]' + componentNumKey + ':' + str(context[componentNumKey]))


def getItemFieldMap(itemCfgMap, fieldName):
    '''
    @summary: get item field map
        e.g. if fieldName is barcode, the field map like {'Enclosure': 'Serial Number', 'Controller': 'BarCode'}
    @date: 2014-10-08
    @param itemCfgMap: item configuration map
    @param fieldName: field name
    @return: item field map
    '''
    
    itemFieldMap = {}
    for title in itemCfgMap:
        cfgDictList = itemCfgMap.get(title)
        for cfgDict in cfgDictList:
            xmlTagVal = cfgDict.get('xmlTag')
            if fieldName == xmlTagVal:
                itemFieldMap[title] = cfgDict.get('attrId')
    return itemFieldMap


def getItemTitleMap(itemCfgMap):
    '''
    @summary: get item title map
        e.g. {'Enclosure':['Serial Number', 'Type'], 'System':['Device Type', 'System Name']}
    @date: 2014-10-08
    @param itemCfgMap: item configuration map
    @return: item title map
    '''
    itemTitleMap = {}
    for title in itemCfgMap:
        itemTitleMap[title] = []
        cfgDictList = itemCfgMap.get(title)
        for cfgDict in cfgDictList:
            itemTitleMap[title].append(cfgDict.get('attrId'))
    return itemTitleMap


def updateDimeDictList(stdItemDataMap, stdItemCfgMap, context):
    '''
    @summary: update the list of dimension information dictionary 
    @date: 2014-09-29
    @param stdItemDataMap: standard item data map
    @param stdItemCfgMap: collect item configuration map
    @param context: context object
    @return: 
    '''
    collectItem = 'Dimension'
    logger = context.get('logger')
    enclosureDictList = stdItemDataMap.get('Enclosure')
    stdItemTypeMap = getItemFieldMap(stdItemCfgMap, 'Type')
    stdItemNameMap = getItemFieldMap(stdItemCfgMap, 'Name')
    stdItemValMap = getItemFieldMap(stdItemCfgMap, 'Value')
    dimensionDictList = []
    enclTypeNumDict = {}
    
    #获取所有框类型个数字典
    type = stdItemTypeMap.get('Enclosure')
    for enclosureDict in enclosureDictList:
        enclosureType = enclosureDict.get(type)#获取框类型
        enclTypeNumDict[enclosureType] = enclTypeNumDict.get(enclosureType, 0) + 1
    logger.info('[collectStandard] enclTypeNumDict=' + str(enclTypeNumDict))
    
    #按返回格式组装数据
    name = stdItemNameMap.get('Dimension')
    value = stdItemValMap.get('Dimension')
    for enclType in enclTypeNumDict:
        dimensionDict = {}
        dimensionDict[name] = enclType
        dimensionDict[value] = str(enclTypeNumDict.get(enclType, 'NA'))
        dimensionDictList.append(dimensionDict)
    logger.info('[collectStandard] dimensionDictList=' + str(dimensionDictList))
    stdItemDataMap[collectItem] = dimensionDictList
    countSingleItemSum(context, stdItemDataMap, collectItem)


def updateRackDictList(stdItemDataMap, stdItemCfgMap, context):
    '''
    @summary: update the list of rack information dictionary 
    @date: 2014-09-29
    @param stdItemDataMap: standard item data map
    @param stdItemCfgMap: collect item configuration map
    @param context: context object
    @return: 
    '''
    collectItem = 'Rack'
    logger = context.get('logger')
    enclosureDictList = stdItemDataMap.get('Enclosure')
    logger.info('[collectStandard] enclosureDictList=' + str(enclosureDictList))
    stdItemLocMap = getItemFieldMap(stdItemCfgMap, 'Location')
    logger.info(str(stdItemLocMap))
    
    enclLocField = stdItemLocMap.get('Enclosure')
    rackLocField = stdItemLocMap.get('Rack')
    rackDictList = []
    
    for enclosureDict in enclosureDictList:
        enclosureLoc = enclosureDict.get(enclLocField)#获取框位置
        logger.info('Enclosure Location:' + str(enclosureLoc))
        rackLoc = enclosureLoc.split(',')[0]#获取柜位置
        rackDict = {rackLocField: rackLoc}
        if rackDict not in rackDictList:
            rackDictList.append(rackDict)
    stdItemDataMap[collectItem] = rackDictList
    countSingleItemSum(context, stdItemDataMap, collectItem)


def getSysInfoDictList(sshConnector, cliCmd, logger):
    '''
    @summary: get the list of system information dictionary
    @date: 2014-09-16
    @param sshConnector: the connector of SSH
    @param cliCmd: command of system information
    @param logger: a logger provided by tool
    @return: the list of system information dictionary
    '''
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, logger)
    if not isSucc:
        succStateDict['System'] = False
        logger.error('[collectStandard] Execute command [' + cliCmd + '] failed!')
        return []
    
    #判断数据有效性
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []
    
    #解析回文
    formatFunction = cFastTypeDict(cmdRet, defaultSep='|')
    dataInfoDictList = formatFunction.handle(isHandleELabel=False, childSep="^")[1]
    logger.info('[collectStandard]dataInfoDictList=' + str(dataInfoDictList))
    
    #如果dataInfoDictList为空则换一种解析方式
    if not dataInfoDictList:
        formatFunction = cHandleTypeList(cmdRet)
        ret = formatFunction.handle()
        if len(ret) < 2:
            return []
        dataInfoDictList = ret[1]
        logger.info('[collectStandard]dataInfoDictList=' + str(dataInfoDictList))
    
    return dataInfoDictList


def initSuccStateDict(stdItemTitleMap):
    '''
    @summary: initialize the dict of success state 
    @date: 2014-09-24
    @param stdItemTitleMap: standard item title map
    @return: the dict of success state 
    '''
    tmpDataMap = copy.deepcopy(stdItemTitleMap)
    for key in tmpDataMap:
        tmpDataMap[key] = True
    return tmpDataMap


def getDataMapStruct(stdItemTitleMap):
    '''
    @summary: get standard item data map structure
            e.g. {'System':[],'Enclosure':[]}
    @date: 2014-09-09
    @param stdItemTitleMap: standard item title map
    @return: standard item data map structure
    '''
    tmpDataMap = copy.deepcopy(stdItemTitleMap)
    for key in tmpDataMap:
        tmpDataMap[key] = []
    return tmpDataMap


def updateLicenseInfo(devSN, dataInfoDictList, logger):
    '''
    @summary: update license
    @date: 2014-09-02
    @param devSN: Device Serial Number
    @param dataInfoDictList: the list of data dictionary
    @param logger: a logger provided by tool
    @return: 
    '''
    for dataInfoDict in dataInfoDictList:
        dataInfoDict['Device Serial Number'] = devSN
        dataInfoDict['Authorization type'] = 'COMM'
    logger.info('[collectStandard] license data list is ' + str(dataInfoDictList))


def cleanNullData(collectItem, dataInfoDictList, stdItemBarcodeMap):
    '''
    @summary: clean null data
    @date: 2014-09-04
    @param collectItem: collect item
    @param dataInfoDictList: the list of data dictionary
    @param stdItemBarcodeMap: barcode map
    @return: 
    '''
    #在IBMS系统中barcode唯一，不能为空或者“--”
    barcode = stdItemBarcodeMap.get(collectItem)
    tmpDictList = copy.copy(dataInfoDictList)
    for dataInfoDict in tmpDictList:
        for key in dataInfoDict.keys():
            if key == barcode and (not dataInfoDict.get(key) or '--' == dataInfoDict.get(key)):
                dataInfoDictList.remove(dataInfoDict)
    
    #清除值为空字符串的字段
    for dataInfoDict in dataInfoDictList:
        tmpDict = copy.copy(dataInfoDict)
        for key in tmpDict.keys():
            if not dataInfoDict.get(key):
                dataInfoDict.pop(key)


def adjustItemLocInfo(collectItem, dataInfoDictList, context, devModel):
    '''
    @summary: adjust item location info
    @date: 2014-08-08
    @param collectItem: collect item
    @param dataInfoDictList: the list of data dictionary
    @param context: context object
    @param devModel: device model
    @return: 
    '''
    sshConnector = context['sshConnector']
    logger = context.get('logger')
    ctrlOrBBUslotposMap = PosMap.ctrlOrBBUslotposMap4S5000
    
    global ctrlEnclId
    logger.info('[collectStandard]Adjusting item location:' + collectItem)
    if 'Enclosure' == collectItem:
        for dataInfoDict in dataInfoDictList:
            enclosureID = dataInfoDict.get('Subrack ID')
            updateKey = 'Subrack ID'
            if not enclosureID:
                logger.warn('[collectStandard]Subrack ID is NULL.')
            else:
                type = dataInfoDict.get('Type')
                if 'Controller' == type:
                    ctrlEnclId = enclosureID
                rack = 'rack=-1'
                shelf = 'shelf=' + enclosureID
                dataInfoDict[updateKey] = rack + ',' + shelf
    elif 'Controller' == collectItem:
        for dataInfoDict in dataInfoDictList:# 
            ctrlId = dataInfoDict.get('Controller Board ID', '')
            updateKey = 'Controller Board ID'
            if not ctrlId:
                logger.warn('[collectStandard]ctrlId is null.')
                ctrlId = dataInfoDict.get('Controller ID', '')
                updateKey = 'Controller ID'
  
            rack = 'rack=-1'
            shelf = 'shelf=' + ctrlEnclId
            subshelf = 'subshelf=-1'
            slotpos = ctrlOrBBUslotposMap.get(devModel, 'slotpos=0')
            slot = 'slot=' + ctrlId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
    elif 'ExpansionModule' == collectItem:
        for dataInfoDict in dataInfoDictList:
            enclIdOfExp = dataInfoDict.get('Subrack ID', '')
            expBoardID = dataInfoDict.get('Controller ID', '')
            updateKey = 'Controller ID'
            if not expBoardID:
                expBoardID = dataInfoDict.get('Expander Board ID', '')
                updateKey = 'Expander Board ID'
                
            if not enclIdOfExp or not expBoardID:
                logger.warn('[collectStandard]IdOfExp is null.')
            else:
                rack = 'rack=-1'
                shelf = 'shelf=' + enclIdOfExp
                subshelf = 'subshelf=-1'
                slotpos = 'slotpos=1'
                slot = 'slot=' + expBoardID
                dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
    elif 'InterfaceModule' == collectItem:
        for dataInfoDict in dataInfoDictList:
            ctrlIdOfIntf = dataInfoDict.get('Controller', '')
            intfId = dataInfoDict.get('Interface ID', '')
            updateKey = 'Interface ID'
            if not ctrlIdOfIntf:
                ctrlIdOfIntf = dataInfoDict.get('Controller ID', '')
                logger.warn('[collectStandard]IdOfCtrl is NULL:' + str(ctrlIdOfIntf))

            if not intfId:
                logger.warn('[collectStandard]IdOfIntf is null.')
            else:
                rack = 'rack=-1'
                shelf = 'shelf=' + ctrlEnclId
                subshelf = 'subshelf=-1'
                slotpos = 'slotpos=1'
                slot = 'slot=' + ctrlIdOfIntf + '.' + intfId
                dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
    elif 'PowerModule' == collectItem:
        for dataInfoDict in dataInfoDictList:
            enclIdOfPower = dataInfoDict.get('Subrack ID', '')
            powerId = dataInfoDict.get('Power ID', '')
            updateKey = 'Power ID'
            if not enclIdOfPower or not powerId:
                logger.warn('[collectStandard]IdOfPower is null.')
                powerLocation = dataInfoDict.get('Power Location', '')
                updateKey = 'Power Location'
                enclIdOfPower = powerLocation.strip().split(',')[0].split('(')[1].strip()
                powerId = powerLocation.strip().split(',')[1].split(')')[0].strip()
                
            rack = 'rack=-1'
            shelf = 'shelf=' + enclIdOfPower
            subshelf = 'subshelf=-1'
            slotpos = 'slotpos=1'
            slot = 'slot=' + powerId
            dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
    elif 'BBU' == collectItem:
        for dataInfoDict in dataInfoDictList:
            ctrlIdOfBbu = dataInfoDict.get('Controller ID', '')
            if not ctrlIdOfBbu:
                logger.warn('The controller ID where BBU resides is NULL:' + ctrlEnclId)
                ctrlIdOfBbu = ctrlEnclId
                
            bbuId = dataInfoDict.get('BBU ID', '')
            updateKey = 'BBU ID'
            # 【 】解决BBU单板中无Location信息问题   Begin
            if not bbuId:
                bbuId = dataInfoDict.get('Bbu ID', '')
                updateKey = "Bbu ID"
            # 【 】解决BBU单板中无Location信息问题   End
            if not bbuId:
                logger.warn('[collectStandard]IdOfBBU is null.')
                bbuId = dataInfoDict.get('Battery ID', '')
                updateKey = 'Battery ID'
                
            if not bbuId:
                logger.error('[collectStandard]IdOfBBUis null.')
            else:
                rack = 'rack=-1'
                shelf = 'shelf=' + ctrlEnclId
                subshelf = 'subshelf=-1'
                slotpos = ctrlOrBBUslotposMap.get(devModel, 'slotpos=0')
                slot = 'slot=' + bbuId
                dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
    elif 'Disk' == collectItem:
        for dataInfoDict in dataInfoDictList:
            diskLoc = dataInfoDict.get('Disk Location', '')
            updateKey = 'Disk Location'
            if not diskLoc:
                logger.warn('[collectStandard]IdOfDisk is null.')
            else:
                diskLocArr = diskLoc.replace('(', '').replace(')', '').split(',')
                rack = 'rack=-1'
                shelf = 'shelf=' + diskLocArr[0].strip()
                subshelf = 'subshelf=-1'
                slotpos = 'slotpos=0'
                slot = 'slot=' + diskLocArr[1].strip()
                dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
            logger.error('[collectStandard]DiskZhou:' + str(dataInfoDict))
           
                
    elif 'Fan' == collectItem:
        for dataInfoDict in dataInfoDictList:
            enclIdOfFan = dataInfoDict.get('Subrack ID', '')
            fanId = dataInfoDict.get('Fan Group ID', '')
            updateKey = 'Fan Group ID'
            if not enclIdOfFan or not fanId:
                logger.warn('[collectStandard]IdOfFan is null.')
            else:
                rack = 'rack=-1'
                shelf = 'shelf=' + enclIdOfFan
                subshelf = 'subshelf=-1'
                slotpos = 'slotpos=1'
                slot = 'slot=' + fanId
                dataInfoDict[updateKey] = rack + ',' + shelf + ',' + subshelf + ',' + slotpos + ',' + slot
    else:
        logger.info('[collectStandard]No need to adjusting item location:' + collectItem)


def collectSoftware(sshConnector, cmdList, logger, sysInfoDict):
    '''
    @summary: collect software
    @date: 2014-08-08
    @param sshConnector: the connector of SSH
    @param cmdList: command list
    @param logger: a logger provided by tool
    @return: the dictionary list of result
    '''
    retDict = {}
    cmd4syst = cmdList[0] #系统
    global G_cmdExecRetDict #声明为全局变量
    G_cmdExecRetDict['Software'] = '' #存放软性收集项的回显信息
    
    #1.system version
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd4syst, logger)
    G_cmdExecRetDict['Software'] += cmdRet #软件信息收集项的回显数据的统计
    if not isSucc:
        succStateDict['Software'] = False
        logger.error('[collectStandard] Execute command [' + cmd4syst + '] failed!')
        return []
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []
    
    #解析回文
    formatFunction = cFastTypeDict(cmdRet, defaultSep="|")
    ret = formatFunction.handle()
    if not ret[0]:
        logger.error('[collectStandard] It occurs exception when parse cmdRet!')
        return []
    
    SoftwarePackageVersion = ''
    tmpRetDictList = ret[1]
    for tmpRetDict in tmpRetDictList:
        SoftwarePackageVersion = tmpRetDict.get('Software Version', '')
        
        #取含有效值的SoftwarePackageVersion
        if SoftwarePackageVersion and '--' != SoftwarePackageVersion:
            break
        
    retDict['SoftwarePackageVersion'] = SoftwarePackageVersion
    retDict['Device Serial Number'] = sysInfoDict.get('Device Serial Number', '')
    retDict['Device Type'] = sysInfoDict.get('Device Model', '')
    
    return [retDict]


def judgeCollectResult(context, stdItemDataMap):
    '''
    @summary: Judge collect result.
    @date: 2014-08-12
    @param context: context object
    @param stdItemDataMap: the data map of standard item 
    @return: (return code, error message) as (integer, string)
    '''
    logger = context.get('logger')
    resource = getResource(context)
    errMsgList = []
    collectResutList = []
    errMsgDetailDict = {}#存放所有收集项回显数据字典
    
    #单项全部成功：命令执行成功 and 总个数与收集个数相等
    #单项全部失败：命令执行失败并且收集个数为0 or 命令执行成功并且总个数不为0并且收集个数为0
    #单项部分成功：命令执行失败并且收集个数不为0 or 命令执行成功并且总个数与收集个数不等并且都不为0
    for itemName in stdItemDataMap:
        isSucc = succStateDict.get(itemName)
        cmdRet = G_cmdExecRetDict[itemName].strip() #获取每个收集项对应的回显信息
        allNum = context.get(itemName + 'Num', 0)#单项统计总个数
        collNum = len(stdItemDataMap.get(itemName))#单项实际收集到的个数
        logger.info('[collectStandard]' + itemName + ': isSucc=' + str(isSucc) + ';allNum=' + str(allNum)
                    + ';collNum=' + str(collNum))
        
        if isSucc and allNum == collNum:#单项全部成功
            logger.info('[collectStandard]Collect all ' + itemName + ' information')
            errMsg = resource.getString('info.collect.' + itemName + '.info.success')
            errMsgList.append(errMsg)
            collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
            errMsgDetailDict[errMsg] = ''
        elif not isSucc and collNum == 0 or isSucc and allNum != 0 and collNum == 0:#单项全部失败
            logger.error('[collectStandard]Collect all ' + itemName + ' information failed.')
            errMsg = resource.getString('err.collect.' + itemName + '.info.fail')
            errMsgList.append(errMsg)
            collectResutList.append(SCRIPT_RET_CODE.FAIL)
            errMsgDetailDict[errMsg] = cmdRet
        else:#单项部分成功
            logger.warn('[collectStandard]Collect part ' + itemName + ' information')
            errMsg = resource.getString('err.collect.part.' + itemName + '.info.fail')
            errMsgList.append(errMsg)
            collectResutList.append(SCRIPT_RET_CODE.PART_SUCCESS)
            errMsgDetailDict[errMsg] = cmdRet
    
    #判断总体收集成功、失败还是部分成功
    if collectResutList.count(SCRIPT_RET_CODE.SUCCESS) == len(stdItemDataMap):
        logger.info('[collectStandard]Collect all standard item information success.')
        return (SCRIPT_RET_CODE.SUCCESS, '', errMsgDetailDict)
    elif collectResutList.count(SCRIPT_RET_CODE.FAIL) == len(stdItemDataMap):
        logger.error('[collectStandard]Collect all standard item information failed!')
        return (SCRIPT_RET_CODE.FAIL, '\n'.join(errMsgList), errMsgDetailDict)
    else:
        logger.warn('[collectStandard]Collect part standard item information success.')
        return (SCRIPT_RET_CODE.PART_SUCCESS, '\n'.join(errMsgList), errMsgDetailDict)


        
