# -*- coding:utf-8 -*-
from defusedxml import ElementTree as ET
import traceback
import os
import copy
from common.exception import DisconnectionException
from common.devNode import DevNode
from common import commonFunction
from common.commonFunction import getResource
from common.constant import SCRIPT_RET_CODE
from common import cliCmdManager
from common.cTV2R2 import cHandleTypeList
from common import utils

def execute(context):
    '''
    @summary: Initialize device node.
    @date: 2014-07-28
    @param context: context object
    @return: 
    '''
    
    #初始化各需要的变量
    logger = context.get('logger')
    logger.info('[prepare] start to prepare handle for S5000...')
    resource = getResource(context)
    devNode = context.get('devNode')
    logger.info('[prepare] devNode=' + str(devNode))
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin1
    try:
        if not devNode:
            devNode = DevNode(context)
            context['sshConnector'] = devNode.getSshConnector(context)
            context['devNode'] = devNode
            logger.info('[prepare] devNode=' + str(devNode))
        
        #判断SSH连接状态
        logger.info('[prepare] Judge SSH connection status....')
        if not isSshConnectionNormal(context):
            errMsg = resource.getString('err.create.SSH.connection.failed')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        # 问题单 修复“S2600管理员账户、只读账户收集时脚本执行失败”的缺陷  begin
        #判断用户权限是否为超级管理员
        logger.info('[prepare] Judge whether current account is admin....')
        if not utils.isAccountAdmin(context):
            errMsg = resource.getString('err.account.not.admin')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        #判断是否设置调试密码
        logger.info('[prepare] Judge whether debug password exist or not....')
        if not utils.isDebugPwdExistent(context):
            errMsg = resource.getString('err.debug.passwd.not.input')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        #判断调试密码是否正确
        logger.info('[prepare] Judge whether debug password is right....')
        if not utils.isDebugPwdRight(context):
            errMsg = resource.getString('err.debug.passwd.wrong')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        # 问题单 修复“S2600管理员账户、只读账户收集时脚本执行失败”的缺陷  end
        
        #判断系统状态
        logger.info('[prepare] Judge system status...')
        if not devNode.isSystemNormal(context):
            errMsg = resource.getString('err.sys.status.abnormal')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        #解析标准收集项命令配置文件cmd4IBMS.xml
        cfgFile4Standard = context.get('path') + os.sep + 'script' + os.sep + 'config' + os.sep + 'cmd4IBMS.xml'
        errCode, errMsg = parseCmdCfgFile(context, cfgFile4Standard)
        
        #解析生成标准项配置文件xmlarchivecfg.xml
        if not errMsg:
            xmlarchivecfgPath = context.get('path') + os.sep + 'xmlarchivecfg.xml'
            context['stdItemCfgMap'] = getItemCfgMap(context, xmlarchivecfgPath)
        
        logger.info('[prepare] Finish preparing handle.')
        return (errCode, errMsg)
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin2
    except DisconnectionException, de:
        logger.error('[prepare] Maybe the network is unavailable or device is abnormal, details:' + traceback.format_exc())
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg)
    # 处理不同阶段网络断开问题（涉及的问题单为： ）End1


def isSshConnectionNormal(context):
    '''
    @summary: Check if the SSH connection is normal
    @date: 2014-08-18
    @param context: context object
    @return: 
    '''
    logger = context.get('logger')
    sshConnector = context.get('sshConnector')
    if not sshConnector.getConnectionNoException():
        logger.error('[prepare] Get SSH connection failed!')
        return False
    else:
        logger.info('[prepare] Get SSH connection successfully!')
        return True


def parseCmdCfgFile(context, cmdCfgFile):
    '''
    @summary: load and parse command configure file, then load to memory
    @date: 2014-07-28
    @param context: context object
    @param cmdCfgFile: command configuration file
    @return: (result code, error message) as (integer, string)
    '''
    logger = context.get('logger')
    resource = getResource(context)
    itemCmdListDict = {}
    errMsg = ''
    retCode = SCRIPT_RET_CODE.SUCCESS
    linkSym = '='#参数名与值的连接符，有些环境是空格' '
    
    #加载配置和解析配置文件
    root = ET.parse(cmdCfgFile)
    itemTagList = root.findall('item')
    try:
        for itemTag in itemTagList:
            #获取收集项名称
            itemName = itemTag.attrib['name'].strip()
            
            #获取收集项命令列表
            itemCmdList = []
            simpleCmdList = [] #单命令（无参数）列表
            cmdTagList = itemTag.findall('cmd')
            for cmdTag in cmdTagList:
                paramTagList = cmdTag.findall('param')
                baseCmd = cmdTag.attrib['command'].strip()
                
                #无参数的直接加入命令列表
                if not paramTagList:
                    simpleCmdList.append(baseCmd)
                    continue
                
                #有参数的解析参数并拼凑成完整的命令
                fullCmd = baseCmd
                paramList = []
                for paramTag in paramTagList:
                    paramDict = paramTag.attrib
                    paraName = paramDict.get('paraName').strip()
                    paraValList = []
                    
                    #场景1：param标签含有LoopBaseCmd
                    paraValCmd = paramDict.get('LoopBaseCmd').strip()
                    if paraValCmd:
                        fieldName = paramDict.get('loop').strip()
                        paraValList = getParaValByCLI(context, paraValCmd, fieldName)
                    
                    #场景2：param标签含有rangeList
                    paraValRange = paramDict.get('rangeList').strip()
                    if paraValRange:
                        paraValList = paraValRange.split('|')
                    
                    #拼接参数
                    tmpList = []
                    for paraVal in paraValList:
                        tmpList.append(paraName + linkSym + paraVal)
                    if not paramList:
                        paramList = tmpList
                        continue
                    paramList = utils.getTranslatedCartesianRet(paramList, tmpList)
                    
                #将当前cmd标签拼接成完整命令
                curTagFullCmdList = utils.getTranslatedCartesianRet([baseCmd], paramList)
                itemCmdList.extend(curTagFullCmdList)
            
            itemCmdList.extend(simpleCmdList)
            
            #单板中的硬盘详细信息的命令重新组合处理(使用showdisk -p命令获取硬盘的位置，然后组合单个硬盘的信息查询的命令)
            handleDiskDetailInfoCmd(itemName, itemCmdList, context)
            
            itemCmdListDict[itemName] = itemCmdList
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin3
    except DisconnectionException, de:
        logger.error('[prepare] Catch except in parseCmdCfgFile, details:' + traceback.format_exc())
        raise de
    # 处理不同阶段网络断开问题（涉及的问题单为： ）End2
    except:
        logger.error('[prepare]Parse IBMS command configure file caught exception.')
        errMsg = resource.getString('err.parse.IBMS.cfg.failed')
        retCode = SCRIPT_RET_CODE.FAIL
        return (retCode, errMsg)
    else:
        logger.info('[prepare]Standard itemTag name-cmdList:' + str(itemCmdListDict))
        context['stdItemCmdDict'] = itemCmdListDict
        return (retCode, errMsg)

def handleDiskDetailInfoCmd(itemName, itemCmdList, context):
    '''
    @summary: handle the single disk info 
    @date: 2015-01-26
    @param context: context object,itemName:collection item name,itemCmdList:command list of collection item
    @return: no value
    '''
    sshConnector = context['sshConnector']
    logger = context.get("logger")
    if itemName.strip() in "Disk":
        #是Disk项时候执行命令获取对应的硬盘位置
        isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, "showdisk -p", logger)
        if not isSucc:
            logger.error('[collectStandard] Execute command [showdisk -p] failed!')
            return []
        
        isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
        if not isExecSuc:
            logger.error('[handleDiskDetailInfoCmd]Failed to execute command and the result is not parsable.')
            return []
        elif not isParsable:
            logger.warn('[handleDiskDetailInfoCmd]Executed command successfully but the result is not parsable.')
            return []
        
        #解析回文
        formatFunction = cHandleTypeList(cmdRet)
        ret = formatFunction.handle()
        if not ret[0]:
            logger.error('[handleDiskDetailInfoCmd] It occurs exception when parse cmdRet!')
            return []
        
        diskLocation = ''
        tmpRetDictList = ret[1]
        
        for tmpRetDict in tmpRetDictList:
            diskLocation = tmpRetDict.get('Disk Location', '')
            #组合成新的命令，添加到命令列表中
            frameNo = diskLocation.split(',')[0].split('(')[1].strip()
            diskNo = diskLocation.split(',')[1].split(')')[0].strip()
            newCmd = "showdisk -s " + frameNo + " -sl " + diskNo
            itemCmdList.append(newCmd)
        itemCmdList.remove("showdisk -p")#删除无关的参考命令"showdisk -p"



def getParaValByCLI(context, cmd, fieldName):
    '''
    @summary: Get the value of parameter.
    @date: 2014-07-28
    @param context: context object
    @param cmd: command
    @param fieldName: field name
    @return: the value list of parameter
    '''
     
    sshConnector = context['sshConnector']
    logger = context.get('logger')
    
    #发送命令
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger)
    if not isSucc:
        logger.error('[prepare] Execute base loop command [' + cmd + '] failed!')
        return []
    
    #检查数据有效性
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        logger.error('[prepare]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[prepare]Executed command successfully but the result is not parsable.')
        return []
    
    #解析回文
    formatFunction = cHandleTypeList(cmdRet, [fieldName])
    fieldInfoDictList = formatFunction.handle()[1]
    
    #获取ID值
    fieldValList = [fieldInfoDict.get(fieldName, '') for fieldInfoDict in fieldInfoDictList]
    tmpValList = copy.copy(fieldValList)
    
    #去除空ID
    for tmpVal in tmpValList:
        if '' == tmpVal:
            fieldValList.remove(tmpVal)
    
    logger.info('[prepare]FieldValueList:' + str(fieldValList))
    return fieldValList


def getItemCfgMap(context, xmlarchivecfgPath):
    '''
    @summary: get collect item configuration map
    @date: 2014-09-09
    @param context: context object
    @param xmlarchivecfgPath: the path of file named xmlarchivecfg.xml
    @return: collect item configuration map
    '''
    root = ET.parse(xmlarchivecfgPath)
    dataTypeList = root.getiterator('DataType')
    
    stdItemCfgMap = {}
    for dataType in dataTypeList:
        #收集项名称
        typeId = dataType.attrib['typeId']
        stdItemCfgMap[typeId] = []
        dataAttrList = dataType.findall('DataAttr')
        for dataAttr in dataAttrList:
            #具体的收集子项配置字典
            dataAttrDict = dataAttr.attrib
            stdItemCfgMap[typeId].append(dataAttrDict)
    return stdItemCfgMap
    

