# -*- coding:utf-8 -*-
from defusedxml import ElementTree as ET
import os
import copy
import traceback
from common.devNode import DevNode
from common import commonFunction
from common.commonFunction import getResource
from common.constant import SCRIPT_RET_CODE
from common import cliCmdManager
from common.cTV1R1 import cHandleTypeList
from common import utils
from common.log import Log
from common.contextUtil import getSshConnector, getLogger
from common.exception import DisconnectionException

def execute(context):
    '''
    @summary: Initialize device node.
    @date: 2014-07-28
    @param context: context object
    @return: 
    '''
    # 处理不同阶段网络断开问题
    try:
        #初始化各需要的变量
        initLog(context)
        Log.info('start to prepare handle for TV1...')
        resource = getResource(context)
        devNode = context.get('devNode')
        Log.info('devNode=' + str(devNode))
        if not devNode:
            devNode = DevNode(context)
            context['sshConnector'] = devNode.getSshConnector(context)
            context['devNode'] = devNode
            Log.info('devNode=' + str(devNode))
        
        #判断SSH连接状态
        Log.info('Judge SSH connection status....')
        if not isSshConnectionNormal(context):
            errMsg = resource.getString('err.create.SSH.connection.failed')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        #判断系统状态
        Log.info('Judge system status...')
        if not devNode.isSystemNormal4TV1(context):
            errMsg = resource.getString('err.sys.status.abnormal')
            return (SCRIPT_RET_CODE.FAIL, errMsg)
        
        #解析标准收集项命令配置文件cmd4IBMS.xml
        cfgFile4Standard = os.sep.join([context.get('path'), 'script', 'config', 'cmd4IBMS.xml'])
        errCode, errMsg = parseCmdCfgFile(context, cfgFile4Standard)
        
        #解析生成标准项配置文件xmlarchivecfg.xml
        if not errMsg:
            xmlarchivecfgPath = context.get('path') + os.sep + 'xmlarchivecfg.xml'
            context['stdItemCfgMap'] = getItemCfgMap(context, xmlarchivecfgPath)
        
        cfgFile4Extend = os.sep.join([context.get('path'), 'script', 'config', 'cmd4Excel.xml'])
        errCode, errMsg = parseCmdCfgFile(context, cfgFile4Extend)
        
        Log.info('Finish preparing handle.')
        return (errCode, errMsg)
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin2
    except DisconnectionException, de:
        Log.error('Maybe the network is unavailable or device is abnormal, details:' + traceback.format_exc())
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg)
    # 处理不同阶段网络断开问题（涉及的问题单为： ）End1


def initLog(context):
    '''
    @summary: initialize log 
    @param context: context object
    '''
    Log(context)
    Log.updateScriptFlag('prepare')


def isSshConnectionNormal(context):
    '''
    @summary: Check if the SSH connection is normal
    @date: 2014-08-18
    @param context: context object
    @return: 
    '''
    logger = getLogger(context)
    sshConnector = getSshConnector(context)
    if not sshConnector.getConnectionNoException():
        logger.error('[prepare] Get SSH connection failed!')
        return False
    else:
        logger.info('[prepare] Get SSH connection successfully!')
        return True


def parseCmdCfgFile(context, cmdCfgFile):
    '''
    @summary: load and parse command configure file, then load to memory
    @date: 2014-07-28
    @param context: context object
    @param cmdCfgFile: command configuration file
    @return: (result code, error message) as (integer, string)
    '''
    logger = getLogger(context)
    resource = getResource(context)
    itemCmdListDict = {}
    errMsg = ''
    retCode = SCRIPT_RET_CODE.SUCCESS
    linkSym = ' '#参数名与值的连接符，有些环境是空格' '
    
    #获取配置文件类型
    cfgFileType = cmdCfgFile.split(os.sep)[-1].split('.')[0].split('4')[1]
    
    #加载配置和解析配置文件
    root = ET.parse(cmdCfgFile)
    itemTagList = root.findall('item')
    try:
        for itemTag in itemTagList:
            #获取收集项名称
            itemName = itemTag.attrib['name'].strip()
            
            #获取收集项命令列表
            itemCmdList = []
            simpleCmdList = [] #单命令（无参数）列表
            cmdTagList = itemTag.findall('cmd')
            for cmdTag in cmdTagList:
                paramTagList = cmdTag.findall('param')
                baseCmd = cmdTag.attrib['command'].strip()
                
                #无参数的直接加入命令列表
                if not paramTagList:
                    simpleCmdList.append(baseCmd)
                    continue
                
                #有参数的解析参数并拼凑成完整的命令
                fullCmd = baseCmd
                paramList = []
                for paramTag in paramTagList:
                    paramDict = paramTag.attrib
                    paraName = paramDict.get('paraName').strip()
                    paraValList = []
                    
                    #场景1：param标签含有LoopBaseCmd
                    paraValCmd = paramDict.get('LoopBaseCmd')
                    if paraValCmd:
                        fieldName = paramDict.get('loop').strip()
                        paraValCmd = paraValCmd.strip()
                        paraValList = getParaValByCLI(context, paraValCmd, fieldName)
                    else:
                        #场景2：param标签含有rangeList
                        paraValRange = paramDict.get('rangeList')
                        if paraValRange:
                            paraValRange = paraValRange.strip()
                            paraValList = paraValRange.split('|')
                    
                    #拼接参数
                    tmpList = []
                    for paraVal in paraValList:
                        tmpList.append(paraName + linkSym + paraVal)
                    if not paramList:
                        paramList = tmpList
                        continue
                    paramList = utils.getTranslatedCartesianRet(paramList, tmpList)
                    
                #将当前cmd标签拼接成完整命令
                curTagFullCmdList = utils.getTranslatedCartesianRet([baseCmd], paramList)
                itemCmdList.extend(curTagFullCmdList)
            
            itemCmdList.extend(simpleCmdList)
            itemCmdListDict[itemName] = itemCmdList
    # 处理不同阶段网络断开问题（涉及的问题单为： ）Begin3
    except DisconnectionException, de:
        logger.error('[prepare]Catch except in parseCmdCfgFile, details:' + traceback.format_exc())
        raise de
    # 处理不同阶段网络断开问题（涉及的问题单为： ）End2
    except:
        logger.error('[prepare]Parse %s command configure file caught exception.' % cfgFileType)
        logger.error(''+traceback.format_exc())
        errMsg = resource.getString('err.parse.%s.cfg.failed' % cfgFileType)
        retCode = SCRIPT_RET_CODE.FAIL
        return (retCode, errMsg)
    else:
        logger.info('[prepare]Collect %s itemTag name-cmdList:%s' % (cfgFileType, str(itemCmdListDict)))
        if 'IBMS' == cfgFileType:
            context['stdItemCmdDict'] = itemCmdListDict
        elif 'Excel' == cfgFileType:
            context['extItemCmdDict'] = itemCmdListDict
        return (retCode, errMsg)


def getParaValByCLI(context, cmd, fieldName):
    '''
    @summary: Get the value of parameter.
    @date: 2014-07-28
    @param context: context object
    @param cmd: command
    @param fieldName: field name
    @return: the value list of parameter
    '''
     
    sshConnector = getSshConnector(context)
    logger = getLogger(context)
    
    #发送命令
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger)
    if not isSucc:
        logger.error('[prepare] Execute base loop command [' + cmd + '] failed!')
        return []
    
    #检查数据有效性
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        logger.error('[prepare]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        logger.warn('[prepare]Executed command successfully but the result is not parsable.')
        return []
    
    #解析回文
    formatFunction = cHandleTypeList(cmdRet,[fieldName])
    fieldInfoDictList = formatFunction.handle()
    
    #获取ID值
    fieldValList = [fieldInfoDict.get(fieldName) for fieldInfoDict in fieldInfoDictList]
    tmpValList = copy.copy(fieldValList)
    
    #去除空ID
    for tmpVal in tmpValList:
        if '' == tmpVal:
            fieldValList.remove(tmpVal)
    
    logger.info('[prepare]FieldValueList:' + str(fieldValList))
    return fieldValList


def getItemCfgMap(context, xmlarchivecfgPath):
    '''
    @summary: get collect item configuration map
    @date: 2014-09-09
    @param context: context object
    @param xmlarchivecfgPath: the path of file named xmlarchivecfg.xml
    @return: collect item configuration map
    '''
    root = ET.parse(xmlarchivecfgPath)
    dataTypeList = root.getiterator('DataType')
    
    stdItemCfgMap = {}
    for dataType in dataTypeList:
        #收集项名称
        typeId = dataType.attrib['typeId']
        stdItemCfgMap[typeId] = []
        dataAttrList = dataType.findall('DataAttr')
        for dataAttr in dataAttrList:
            #具体的收集子项配置字典
            dataAttrDict = dataAttr.attrib
            stdItemCfgMap[typeId].append(dataAttrDict)
    return stdItemCfgMap
    

