# -*- coding: UTF-8 -*-
import re
from frame.cli import cliResource
from common.log import Log
from common.contextUtil import  getLogger
MAX_RETRYS = 5
CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEVELOPER_MODEL_FLAG = "developer:/>"
CLI_RET_END_FLAG = ":/>"
CLI_NOT_EXSIT_FLAG = "^"
import traceback
from common.cliParser import CliParser

def getMsg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"
    
    try:
        if not resource.has_key(msg):
            return errMsg
        
        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg
        
        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg
    
def isChinese(lang):
    '''
    @summary: 判断lang是否为中文
    @param lang: 语言lang
    @return: 
        True: 中文
        False: 非中文
    '''
    if lang == "zh":
        return True
    return False

def isNoneLicense(cliRet):
    '''
    @summary: 判断回显结果里面是否包含需要license
    @param cliRet: cli回显
    @return: 
        True: cli回显包含需要license
        False: cli回显不包含需要license
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and ("error" in lowerLine or "suggestion" in lowerLine):
            return True
    
    return False

def isCmdExsit(cliRet):
    '''
    @summary: 根据回显判断阵列上CLI命令是否存在
    @param cliRet: cli回显
    @return: 
        True: cli存在
        False: cli不存在
    '''
    cliRetList = cliRet.splitlines()
    if len(cliRetList) > 1 and CLI_NOT_EXSIT_FLAG in cliRetList[1]:
        return False
    return True

def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    if re.search(CLI_EXECUTE_CMD_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

def isInternalError(cliRet):
    '''
    @summary: 判断回显结果是否包含内部错误信息
    @param cliRet: cli回显
    @return: 
        True: cli回显包含内部错误信息
        False: cli回显不包含内部错误信息
    '''
    lowerCliRet = cliRet.lower()
    if "internal error" in lowerCliRet:
        return True
    else:
        return False
    
def isNoneObject(cliRet):
    '''
    @summary: 判断回显结果里面是否包含查询的对象
    @param cliRet: cli回显
    @return: 
        True: cli查询的对象为空
        False: cli查询的对象不为空
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "error" in lowerLine and "not exist" in lowerLine:
            return True
    
    return False

    
def getUserName(cli, lang):
    '''
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""
    userName = ""

    try:
        cmd = "showsys"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if not checkRet[0]: 
            return checkRet
        
        cliRet = checkRet[1]
        lineList = cliRet.splitlines()
        
        if len(lineList) == 0:
            errMsg = getMsg(lang, "cannot.get.sys.info")
            return (False, cliRet, errMsg)
        
        lastLine = lineList[-1]
        if CLI_RET_END_FLAG in lastLine:
            userName = lastLine.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            return (True, userName, errMsg)
    
    except:
        errMsg = getMsg(lang, "cannot.get.sys.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def getUserPrivilege(context, cli, lang):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang) 
        if not checkRet:
            return checkRet
        
        userName = checkRet[1]
        Log.info("userName is : %s" %(userName) )
        cmd = "showuser -u %s" % userName
        flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, True, lang)
        if not flag: 
            return flag, cliRet, errMsg
        logger = getLogger(context)
        
        modelParse = CliParser(cliRet, logger, cliType='dict')
        sysInfoDic = modelParse.genDict()
        userPrivilege = sysInfoDic.get('Level')
        return (True, userPrivilege, errMsg)

    except Exception:
        Log.error('Failed to get userPrivilege :' + traceback.format_exc())
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, errMsg)   
    return (False, cliRet, errMsg)


def checkLineInBlackList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""
    
    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
        
    return (False, errMsg)

def excuteCmdInCliMode(cli, cmd, logResult, lang):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param logResult: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""
    
    if cli is None:
        errMsg = getMsg(lang, "dev.conn.failure")
        return (False, cliRet, errMsg)
    
    try:
        if logResult:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)
    except:
        errMsg = getMsg(lang, "dev.conn.failure")
        return (False, cliRet, errMsg)
    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)
    
    if not isCmdExsit(cliRet):
        errMsg = getMsg(lang, "cmd.not.exsit")
        return (False, cliRet, errMsg)
    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    if isNoneLicense(cliRet):
        return (True, cliRet, errMsg)
    if isInternalError(cliRet):
        errMsg = getMsg(lang, "cli.excute.find.internal.error")
        return (False, cliRet, errMsg)
    if isNoneObject(cliRet):
        errMsg = getMsg(lang, "object.does.not.exit")
        return (False, cliRet, errMsg)
    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (False, cliRet, errMsg)  
    return (True, cliRet, errMsg)


   
def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        dictList = []
        for line in field_words:
            if line.find(":/>") >= 0 or (len(tuple_idxs) >= 2 and len(line) <= tuple_idxs[-2][1]):
                break
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getVerticalCliRet(cliRet, isParseElcLabel=True):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    elcLabel = ""
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
        
        if re.search("^-+\r*\n*$", line):
            if not isParseElcLabel and lineDict.has_key("Electronic Label"):
                lineDict["Electronic Label"] = lineDict["Electronic Label"] + elcLabel
                elcLabel = ""
                
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            if isParseElcLabel and "=" in line:
                fields = line.split("=")
            elif "=" in line and re.search("^\s{16,}", line):
                elcLabel += line
                continue
            else:
                continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        if not isParseElcLabel and lineDict.has_key("Electronic Label"):
            lineDict["Electronic Label"] = lineDict["Electronic Label"] + elcLabel
        dictList.append(lineDict.copy())
    
    return dictList

def getSplitedCliRet(cliRet, splitMark):
    '''
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cliRet: cli回显
    @param splitMark: 分割标记 
    '''
    cliRetlist = cliRet.split(splitMark)
    if len(cliRetlist) < 2:
        return ""
    
    splitedCliRet = cliRetlist[1]
    match = re.search("^.*:(/>)*\r*\n*$", splitedCliRet, re.MULTILINE)
    if match:
        splitedCliRet = splitedCliRet[0:match.start()]
    
    if len(splitedCliRet.strip()) == 0:
        return ""
    else:
        return splitedCliRet

