# -*- coding:utf-8 -*-

import traceback
import os
import re
from common.log import Log
from common import cliCmdManager
from common import utils
from common.cBase import cFastTypeDict
from common import commonFunction
from common.commonFunction import getResource, isSshConnectionNormal
from common.constant import SCRIPT_RET_CODE, CmdRetOrientation
from common.contextUtil import getSshConnector, getExtItemCmdDict, getLogger
from common.exception import DisconnectionException


G_cmdExecRetDict = {}
G_IS_CMD_SUCC = 'isCmdSucc'
G_IS_RET_VALID = 'isRetValid'
G_IS_RET_PARSABLE = 'isRetParsable'
G_CMDRET = 'cmdRet'


def execute(context):
    '''
    @summary: the entrance of main method
    @param context: the context object provided by tool framework 
    '''
    Log.updateScriptFlag('collectExtend')
    Log.info('Start to collect extend item...')
    try:
        resource = getResource(context)
        #判断SSH连接是否正常
        if not isSshConnectionNormal(context):
            errMsg = resource.getString('dev.conn.failure')
            return (SCRIPT_RET_CODE.FAIL, errMsg, {})
        
        #1.收集和封装数据
        extTitleListMap, extRetDataMap = collectData(context)
        
        #2.装载数据
        context['titleListMap'] = extTitleListMap
        context['retDataMap'] = extRetDataMap
        
        #3.判断收集结果
        resultCode, errMsg, errMsgDetailDict = judgeCollectResult(context, extRetDataMap)
        if SCRIPT_RET_CODE.SUCCESS != resultCode:
            return (resultCode, errMsg, errMsgDetailDict)
        return (resultCode, errMsg, {})
    except DisconnectionException, de:
        Log.error('Maybe the network is unavailable or device is abnormal, details:' + traceback.format_exc())
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg, {})
    except:
        Log.error('Failed to collect extend item, details:' + traceback.format_exc())
        errMsg = resource.getString('err.collect.Extend.info.fail')
        return (SCRIPT_RET_CODE.FAIL, errMsg, {})


def collectData(context):
    '''
    @summary: collect data
    @param context: context object
    @return: (titleListMap, retDataMap) as (dict, dict)
    '''
    titleListMap = {}
    retDataMap = {}
    global G_cmdExecRetDict
    
    #1.获取收集项及命令
    extItemCmdDict = getExtItemCmdDict(context)
    sshConnector = getSshConnector(context)
    resource = getResource(context)
    
    #2.发命令收集并解析回文
    #3.提取并封装数据
    for itemName in extItemCmdDict.keys():
        cmdList = extItemCmdDict.get(itemName)
        dataInfoDictList = []
        retDataMap[itemName] = []
        titleListMap[itemName] = []
        initCmdExecRetDict(itemName)
        if not cmdList:
            Log.warn('The command list of %s is null !' % itemName)
            continue
        
        Log.info('Collecting %s...' % itemName)
        if 'upgrade package' == itemName:
            cmd = cmdList[0]
            retDictList = collectSoftware(context, cmd)
            retDataMap[itemName].extend(retDictList)
            titleList = getTitleList(retDictList)
            titleListMap[itemName] = titleList
            Log.info('retDataMap[%s]=%s' % (itemName, str(retDataMap.get(itemName))))
            continue
        elif 'license' == itemName:
            cmd = cmdList[0]
            retDictList = collectLicense(context, cmd)
            retDataMap[itemName].extend(retDictList)
            titleListMap[itemName] = getTitleList(retDictList)
            Log.info('retDataMap[%s]=%s' % (itemName, str(retDataMap.get(itemName))))
            continue
        elif 'license feature' == itemName:
            cmd = cmdList[0]
            retDictList = collectLicenseFeature(context, cmd)
            retDataMap[itemName].extend(retDictList)
            titleListMap[itemName] = getTitleList(retDictList)
            Log.info('retDataMap[%s]=%s' % (itemName, str(retDataMap.get(itemName))))
            continue
        elif itemName in ['fc initiator', 'iscsi initiator']:
            cmd = cmdList[0]
            retDictList = collectInitiator(context, cmd, itemName)
            retDataMap[itemName].extend(retDictList)
            titleListMap[itemName] = getTitleList(retDictList)
            Log.info('retDataMap[%s]=%s' % (itemName, str(retDataMap.get(itemName))))
            continue
        for cliCmd in cmdList:
            isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, getLogger(context))
            if not isCmdExecSucc:
                oldCmdRet = G_cmdExecRetDict.get(itemName).get(G_CMDRET)
                errMsg = resource.getString('dev.conn.failure')
                tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:oldCmdRet + errMsg}
                G_cmdExecRetDict.get(itemName).update(tmpDict)
                Log.error('Failed to execute command [%s] !' % cliCmd)
                Log.error('Maybe the network is abnormal, stop collecting this item:' + itemName)
                raise Exception(errMsg)
            isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
            if not isRetValid:
                oldCmdRet = G_cmdExecRetDict.get(itemName).get(G_CMDRET)
                tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:oldCmdRet + cmdRet}
                G_cmdExecRetDict.get(itemName).update(tmpDict)
                Log.error('The result of command [%s] is invalid and not parsable.' % cliCmd)
                continue
            elif not isParsable:
                oldCmdRet = G_cmdExecRetDict.get(itemName).get(G_CMDRET)
                tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:oldCmdRet + cmdRet}
                G_cmdExecRetDict.get(itemName).update(tmpDict)
                Log.warn('The result of command [%s] is valid but not parsable.' % cliCmd)
                continue
            
            if itemName in CmdRetOrientation.horizontalLst:
                dataInfoDictList = utils.formatHDict2List(cmdRet)
            else:
                formatFunction = cFastTypeDict(cmdRet, defaultSep=":")
                dataInfoDictList = formatFunction.handle(isHandleELabel=False)[1]
            if not dataInfoDictList:
                Log.info('The item [%s] data is None' % itemName)
            Log.info('dataInfoDictList=' + str(dataInfoDictList))
            retDataMap[itemName].extend(dataInfoDictList)
        
        titleListMap[itemName] = getTitleList(dataInfoDictList)
    return (titleListMap, retDataMap)


def initCmdExecRetDict(itemName):
    '''
    @summary: initial the global dictionary of command executed result 
    @param itemName: the collect item name
    '''
    G_cmdExecRetDict[itemName] = {G_IS_CMD_SUCC:True, G_IS_RET_VALID:True, G_IS_RET_PARSABLE:True, G_CMDRET:''}


def clearDictValue(tmpDict):
    '''
    @summary: clear values of the dictionary
    @param tmpDict: the dictionary which need to be handled
    '''
    for key in tmpDict.keys():
        tmpDict.update({key: ''})


def combineDictList(addonDictList, generalDictList):
    '''
    @summary: combine two dictionary list
    @param addonDictList: the dictionary list which need to be added
    @param generalDictList: the base dictionary list
    @return: the combined dictionary list of two dictionary list
    '''
    retDictList = []
    need2HandleDictList = []
    for dct in generalDictList:
        featureID = dct.get('Feature ID')
        for tmpDct in addonDictList:
            tmpFeatureID = tmpDct.get('Feature ID')
            if tmpFeatureID == featureID:
                dct.update(tmpDct)
                retDictList.append(dct)
        if dct not in retDictList:
            need2HandleDictList.append(dct)
    
    for need2HandleDict in need2HandleDictList:
        tmpDict = copy.copy(retDictList[0])
        clearDictValue(tmpDict)#
        tmpDict.update(need2HandleDict)
        retDictList.append(tmpDict)
    return retDictList


def collectInitiator(context, cmd, itemName):
    '''
    @summary: collect initiator
    @param context: context object
    @param cmd: command
    @param itemName: the name of collect item
    @return: the dictionary list of result
    '''
    global G_cmdExecRetDict
    sshConnector = getSshConnector(context)
    resource = getResource(context)
    retDictList = []
    
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, getLogger(context))
    if not isCmdExecSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('Failed to execute command [%s] !' % cmd)
        return []
    
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('The result of command [%s] is invalid and not parsable.' % cmd)
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.warn('The result of command [%s] is valid but not parsable.' % cmd)
        return []
    
    fcInitiatorDictList = []
    iscsiInitiatorDictList = []
    fcInitiatorInfo = ''
    iscsiInitiatorInfo = ''
    fcInitiatorSwichOn = False
    iscsiInitiatorSwichOn = False
    
    cmdRetLineList = cmdRet.splitlines()
    for line in cmdRetLineList:
        if -1 != line.find('WWN') and -1 != line.find('Running Status'):
            fcInitiatorSwichOn = True
            iscsiInitiatorSwichOn = False
        if -1 != line.find('iSCSI IQN') and -1 != line.find('Running Status'):
            fcInitiatorSwichOn = False
            iscsiInitiatorSwichOn = True
        if -1 != line.find(':/>'):
            break
        if fcInitiatorSwichOn:
            fcInitiatorInfo += line + os.linesep
        if iscsiInitiatorSwichOn:
            iscsiInitiatorInfo += line + os.linesep
    if fcInitiatorInfo:
        fcInitiatorDictList = utils.formatHDict2List(fcInitiatorInfo)
    if iscsiInitiatorInfo:
        iscsiInitiatorDictList = utils.formatHDict2List(iscsiInitiatorInfo)
    
    if 'fc initiator' == itemName:
        return fcInitiatorDictList
    elif 'iscsi initiator' == itemName:
        return iscsiInitiatorDictList


def collectLicenseFeature(context, cmd):
    '''
    @summary: collect license feature
    @param context: context object
    @param cmd: command
    @return: the dictionary list of result
    '''
    global G_cmdExecRetDict
    itemName = 'license feature'
    sshConnector = getSshConnector(context)
    resource = getResource(context)
    retDictList = []
    
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, getLogger(context))
    if not isCmdExecSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('Failed to execute command [%s] !' % cmd)
        return []
    
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('The result of command [%s] is invalid and not parsable.' % cmd)
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.warn('The result of command [%s] is valid but not parsable.' % cmd)
        return []
    
    pattern = 'Region.*?(Feature\sName.+Maximal\sResource\sNumber.+?)?(Feature\sName.+):/>'
    ret4Search = re.search(pattern, cmdRet, re.DOTALL)
    if not ret4Search:
        Log.warn('Failed to search license feature')
        return []
    generalFeature = ret4Search.group(2)
    addonFeature = ret4Search.group(1)
    generalRetDictList = utils.formatVDict2List(generalFeature)
    Log.info('generalRetDictList=' + str(generalRetDictList))
    retDictList = generalRetDictList
    if addonFeature:
        addonRetDictList = utils.formatVDict2List(addonFeature)
        Log.info('addonRetDictList=' + str(addonRetDictList))
        retDictList = combineDictList(addonRetDictList, generalRetDictList)
    Log.info('combined retDictList=' + str(retDictList))
    return retDictList

def collectLicense(context, cmd):
    '''
    @summary: collect license
    @param context: context object
    @param cmd: command
    @return: the dictionary list of result
    '''
    global G_cmdExecRetDict
    itemName = 'license'
    sshConnector = getSshConnector(context)
    resource = getResource(context)
    retDictList = []
    
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, getLogger(context))
    if not isCmdExecSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('Failed to execute command [%s] !' % cmd)
        return []
    
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('The result of command [%s] is invalid and not parsable.' % cmd)
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.warn('The result of command [%s] is valid but not parsable.' % cmd)
        return []
    cliRetLine = cmdRet.splitlines()
    licenseInfo = ''
    for line in cliRetLine:
        if 'Feature Name' in line:
            break
        licenseInfo += line + os.linesep
    retDictList = utils.formatVDict2List(licenseInfo)
    Log.info('license retDictList=' + str(retDictList))
    return retDictList


def collectSoftware(context, cmd):
    '''
    @summary: collect software
    @param context: context object
    @param cmd: command
    @return: the dictionary list of result
    '''
    global G_cmdExecRetDict
    itemName = 'upgrade package'
    sshConnector = getSshConnector(context)
    resource = getResource(context)
    retDictList = []
    
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, getLogger(context))
    if not isCmdExecSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('Failed to execute command [%s] !' % cmd)
        return []
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.error('The result of command [%s] is invalid and not parsable.' % cmd)
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(itemName).update(tmpDict)
        Log.warn('The result of command [%s] is valid but not parsable.' % cmd)
        return []
    
    cmdRetList = cmdRet.splitlines()
    softwareStr = ''
    hotPatchStr = ''
    isSoftware = False
    isHotPatch = False
    softVer = 'Software Version'
    hotPVer = 'HotPatch Version'
    for line in cmdRetList:
        if re.search(softVer, line, re.IGNORECASE):
            isSoftware = True
            isHotPatch = False
        elif re.search(hotPVer, line, re.IGNORECASE):
            isSoftware = False
            isHotPatch = True
        elif re.search(':/>', line, re.IGNORECASE):
            isSoftware = False
            isHotPatch = False
        if isSoftware:
            softwareStr += line + os.linesep
        elif isHotPatch:
            hotPatchStr += line + os.linesep
    
    softwareDictList = utils.formatHDict2List(softwareStr)
    hotPatchDictList = utils.formatHDict2List(hotPatchStr)
    
    for softwareDict in softwareDictList:
        softwareDict.update({'Version Type': softVer})
    Log.info('softwareDictList=' + str(softwareDictList))
    
    for hotPatchDict in hotPatchDictList:
        hotPatchDict.update({'Version Type': hotPVer})
    Log.info('hotPatchDictList=' + str(hotPatchDictList))
    
    upgradePkgDictList = softwareDictList
    upgradePkgDictList.extend(hotPatchDictList)
    Log.info('upgradePkgDictList=' + str(upgradePkgDictList))
    return upgradePkgDictList


def getTitleList(dataInfoDictList):
    '''
    @summary: get the list of title
    @param dataInfoDictList: data information dictionary list
    @return: the list of title
    '''
    Log.info('getTitleList source: dataInfoDictList=' + str(dataInfoDictList))
    if not dataInfoDictList:
        return []
    return dataInfoDictList[0].keys()


def judgeCollectResult(context, extRetDataMap):
    '''
    @summary: Judge collect result.
    @param context: context object
    @param extRetDataMap: the data map of standard item 
    @return: (result code, error message) as (integer, string)
    '''
    resource = getResource(context)
    errMsgList = []
    collectResutList = []
    Log.info('G_cmdExecRetDict=%s' % unicode(G_cmdExecRetDict))
    errMsgDetailDict = {}
    
    #判断单项收集结果：成功、失败or部分成功
    for itemName in extRetDataMap:
        collNum = len(extRetDataMap.get(itemName))#单项实际收集到的个数
        Log.info('[%s] collNum=%s' % (itemName, collNum))
        isRetValid = G_cmdExecRetDict.get(itemName).get(G_IS_RET_VALID)
        cmdRet = G_cmdExecRetDict.get(itemName).get(G_CMDRET)
        
        if True == isRetValid:#单项收集全部成功
            Log.info('Collect %s information successful.' % itemName)
            errMsg = resource.getString('info.collect.%s.info.success' % itemName)
            errMsgList.append(errMsg)
            errMsgDetailDict[errMsg] = ''
            collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
        elif False == isRetValid and 0 == collNum:#单项收集全部失败
            Log.info('Collect %s information failed.' % itemName)
            errMsg = resource.getString('err.collect.%s.info.fail' % itemName)
            errMsgList.append(errMsg)
            errMsgDetailDict[errMsg] = cmdRet
            collectResutList.append(SCRIPT_RET_CODE.FAIL)
        else:#单项收集部分成功
            Log.info('Collect part %s information failed.' % itemName)
            errMsg = resource.getString('err.collect.part.%s.info.fail' % itemName)
            errMsgList.append(errMsg)
            errMsgDetailDict[errMsg] = cmdRet
            collectResutList.append(SCRIPT_RET_CODE.PART_SUCCESS)
    
    Log.info('errMsgDetailDict=' + unicode(errMsgDetailDict))
    
    #判断总体收集成功、失败还是部分成功
    if collectResutList.count(SCRIPT_RET_CODE.SUCCESS) == len(extRetDataMap):
        Log.info('Collect all extend item information success.')
        return (SCRIPT_RET_CODE.SUCCESS, '', errMsgDetailDict)
    elif collectResutList.count(SCRIPT_RET_CODE.FAIL) == len(extRetDataMap):
        Log.error('Collect all extend item information failed!')
        return (SCRIPT_RET_CODE.FAIL, os.linesep.join(errMsgList), errMsgDetailDict)
    else:
        Log.warn('Collect part extend item information success.')
        return (SCRIPT_RET_CODE.PART_SUCCESS, os.linesep.join(errMsgList), errMsgDetailDict)
