# -*- coding:utf-8 -*-
import re
import os
import copy
import traceback
from common import commonFunction
from common.commonFunction import getResource, isSshConnectionNormal
from common.constant import SCRIPT_RET_CODE
from common.constant import StandItemCmd
from common.cliCmdManager  import execCmd
from common import utils
from common import cliCmdManager
from common.cBase import cFastTypeDict
from common.log import Log
from common.contextUtil import getSshConnector, getExtItemCmdDict, getLogger
from common.exception import DisconnectionException

globalRackInfoDict = {}
isLiceFileExst = True #license是否存在
G_cmdExecRetDict = {}
G_IS_CMD_SUCC = 'isCmdSucc'
G_IS_RET_VALID = 'isRetValid'
G_IS_RET_PARSABLE = 'isRetParsable'
G_CMDRET = 'cmdRet'

def execute(context):
    '''
    @summary: collect standard items
    @date: 2014-07-25
    @param context: context object
    @return: (result code, error message) as (integer, string)
    '''
    
    Log.updateScriptFlag('collectStandard')
    logger = getLogger(context)
    logger.info('Collect standard item start!')
    resource = getResource(context)
    sshConnector = getSshConnector(context)
    #判断SSH连接是否正常
    if not isSshConnectionNormal(context):
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg, {})
    
    systemFieldList = ['Product Model', 'SN', 'System Name', 'Product Version']
    rackFieldList = ['rackLocation']
    enclosureFieldList = ['Logic Type', 'SN', 'ID']
    controllerFieldList = ['BoardType', 'BarCode', 'Item', 'Description', 'Manufactured', 'VendorName', 'Location', 'Logic Version', 'PCB Version', 'IssueNumber', 'CLEICode', 'BOM']
    expansionModuleFieldList = ['BoardType', 'BarCode', 'Item', 'Description', 'Manufactured', 'VendorName', 'ID', 'Logic Version', 'PCB Version', 'IssueNumber', 'CLEICode', 'BOM']
    intfModuleFieldList = ['BoardType', 'BarCode', 'Item', 'Description', 'Manufactured', 'VendorName', 'ID', 'Logic Version', 'PCB Version', 'IssueNumber', 'CLEICode', 'BOM']
    powerModuleFieldList = ['Type', 'Serial Number', 'Model', 'Produce Date', 'Manufacturer', 'ID', 'Version']
    bbuFieldList = ['BoardType', 'BarCode', 'Item', 'Description', 'Manufactured', 'VendorName', 'ID', 'Firmware Version', 'IssueNumber', 'CLEICode', 'BOM']
    diskFieldList = ['Type', 'Bar Code', 'Item', 'Model', 'Manufacturer', 'ID', 'Firmware Version']
    fanFieldList = ['BoardType', 'BarCode', 'Description', 'Manufactured', 'VendorName', 'ID', 'IssueNumber', 'CLEICode', 'BOM']
    softwareFieldList = ['SN', 'SoftwareCurrentVersion', 'Type', 'HotPatchCurrentVersion']
    licenseList = ['License SN', 'LICENSE ESN', 'LICENSE SERVICE AUTH TYPE']
    dimensionList = ['Type', 'Num']
    
    stdItemTitleMap = {"System":systemFieldList,
                       "Rack":rackFieldList,
                       "Enclosure":enclosureFieldList,
                       'Controller':controllerFieldList,
                       'ExpansionModule':expansionModuleFieldList,
                       'InterfaceModule':intfModuleFieldList,
                       'PowerModule':powerModuleFieldList,
                       'BBU':bbuFieldList,
                       'Disk':diskFieldList,
                       'Fan':fanFieldList,
                       'Software':softwareFieldList,
                       'License': licenseList,
                       'Dimension': dimensionList}
    
    stdItemDataMap = {"System":[],
                       'Rack':[],
                       "Enclosure":[],
                       'Controller':[],
                       'ExpansionModule':[],
                       'InterfaceModule':[],
                       'PowerModule':[],
                       'BBU':[],
                       'Disk':[],
                       'Fan':[],
                       'Software':[],
                       'License':[],
                       'Dimension': []}
    
    global G_cmdExecRetDict
    devNode = context.get('devNode')
    if not devNode:
        logger.error('Invalid device node object!')
        return (SCRIPT_RET_CODE.FAIL, resource.getString('err.init.dev.failed'), {})
    else:
        devModel = devNode.getDeviceProductModel(context)
        
    logger.info("context.get('stdItemDict')=" + str(context.get('stdItemDict')))
    
    try:
        for collectItem in stdItemTitleMap:
            initCmdExecRetDict(collectItem)
            cmdList = context.get('stdItemDict').get(collectItem)
            if not cmdList:
                logger.warn('[collectStandard] command list is null')
                continue
            logger.info('collectItem=' + str(collectItem))
            logger.info('cmdList=' + str(cmdList))
            if 'Software' == collectItem:
                logger.info('[collectStandard] collect Software...')
                cmd = 'show upgrade package'
                retDictList = collectSoftware(context, cmd, logger)
                stdItemDataMap[collectItem].extend(retDictList)
                continue
            elif 'License' == collectItem:
                logger.info('[collectStandard] collect License...')
                cmdLst = ['show license', 'show license_active']
                retDictList = collectLicense(context, cmdLst, logger)
                stdItemDataMap[collectItem].extend(retDictList)
                continue
            for cliCmd in cmdList:
                isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, logger)
                if not isCmdExecSucc:
                    errMsg = resource.getString('dev.conn.failure')
                    oldCmdRet = G_cmdExecRetDict.get(collectItem).get(G_CMDRET)
                    tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:oldCmdRet + errMsg}
                    G_cmdExecRetDict.get(collectItem).update(tmpDict)
                    logger.error('[collectStandard] Execute command [' + cliCmd + '] failed!')
                    logger.error('Maybe the network is abnormal, stop collecting this item:' + str(collectItem))
                    raise Exception('dev.conn.failure')
                isRetValid, isParsable = utils.checkCliInfoValid(cmdRet, True)
                if not isRetValid:
                    oldCmdRet = G_cmdExecRetDict.get(collectItem).get(G_CMDRET)
                    tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:oldCmdRet + cmdRet}
                    G_cmdExecRetDict.get(collectItem).update(tmpDict)
                    logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
                    continue
                elif not isParsable:
                    oldCmdRet = G_cmdExecRetDict.get(collectItem).get(G_CMDRET)
                    tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:oldCmdRet + cmdRet}
                    G_cmdExecRetDict.get(collectItem).update(tmpDict)
                    logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
                    continue
                else:
                    formatFunction = cFastTypeDict(cmdRet, defaultSep=":")
                    dataInfoDictList = formatFunction.handle()[1]
                    adjustItemLocInfo(collectItem, dataInfoDictList, context, devModel)
                    stdItemDataMap[collectItem].extend(dataInfoDictList)
        updateRackDictList(stdItemDataMap, context)
        updateDimeDictList(stdItemDataMap, context)
        
        context['titleListMap'] = stdItemTitleMap
        context['retDataMap'] = stdItemDataMap
        resultCode, errMsg, errMsgDetailDict = judgeCollectResult(context, stdItemDataMap)
        if SCRIPT_RET_CODE.SUCCESS != resultCode:
            return (resultCode, errMsg, errMsgDetailDict)
        return (resultCode, errMsg, {})
    except DisconnectionException, de:
        Log.error('Maybe the network is unavailable or device is abnormal, details:' + traceback.format_exc())
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg, {})
    except Exception, e:
        resultCode = SCRIPT_RET_CODE.FAIL
        errMsg = resource.getString('dev.conn.failure')
        return (resultCode, errMsg, {})
    except:
        Log.error('Catch exception, details:' + traceback.format_exc())
        resultCode = SCRIPT_RET_CODE.FAIL
        errMsg = resource.getString('err.collect.Standard.info.fail')
        return (resultCode, errMsg, {})


def initCmdExecRetDict(itemName):
    '''
    @summary: initial the global dictionary of command executed result 
    @param itemName: the collect item name
    '''
    G_cmdExecRetDict[itemName] = {G_IS_CMD_SUCC:True, G_IS_RET_VALID:True, G_IS_RET_PARSABLE:True, G_CMDRET:''}


def countSingleItemSum(context, itemDataMap, collectItem):
    '''
    @summary: get single item sum
    @date: 2014-10-09
    @param context: context object
    @param itemDataMap: item data map
    @param collectItem: collect item
    @return: single item sum
    '''
    logger = context.get('logger')
    itemDictList = itemDataMap.get(collectItem)
    componentNumKey = collectItem + 'Num'
    context[componentNumKey] = len(itemDictList)
    logger.info('[collectStandard]' + componentNumKey + ':' + str(context[componentNumKey]))


def updateDimeDictList(stdItemDataMap, context):
    '''
    @summary: update the list of dimension information dictionary 
    @date: 2014-09-29
    @param stdItemDataMap: standard item data map
    @param context: context object
    @return: 
    '''
    collectItem = 'Dimension'
    logger = context.get('logger')
    enclosureDictList = stdItemDataMap.get('Enclosure')
    dimensionDictList = []
    enclTypeNumDict = {}
    
    #获取所有框类型个数字典
    for enclosureDict in enclosureDictList:
        enclosureType = enclosureDict.get('Logic Type')#获取框类型
        enclTypeNumDict[enclosureType] = enclTypeNumDict.get(enclosureType, 0) + 1
    logger.info('[collectStandard] enclTypeNumDict=' + str(enclTypeNumDict))
    
    #按返回格式组装数据
    for enclType in enclTypeNumDict:
        dimensionDict = {}
        dimensionDict['Type'] = enclType
        dimensionDict['Num'] = str(enclTypeNumDict.get(enclType, 'NA'))
        dimensionDictList.append(dimensionDict)
    logger.info('[collectStandard] dimensionDictList=' + str(dimensionDictList))
    stdItemDataMap[collectItem] = dimensionDictList
    countSingleItemSum(context, stdItemDataMap, collectItem)


def updateRackDictList(stdItemDataMap, context):
    '''
    @summary: update the list of rack information dictionary 
    @date: 2014-09-29
    @param stdItemDataMap: standard item data map
    @param context: context object
    @return: 
    '''
    collectItem = 'Rack'
    logger = context.get('logger')
    enclosureDictList = stdItemDataMap.get('Enclosure')
    logger.info('[collectStandard] enclosureDictList=' + str(enclosureDictList))
    rackDictList = []
    
    for enclosureDict in enclosureDictList:
        enclosureLoc = enclosureDict.get('ID')#获取框位置
        rackLoc = enclosureLoc.split(',')[0]#获取柜位置
        rackDict = {'rackLocation': rackLoc}
        if rackDict not in rackDictList:
            rackDictList.append(rackDict)
    stdItemDataMap[collectItem] = rackDictList
    countSingleItemSum(context, stdItemDataMap, collectItem)


def adjustItemLocInfo(collectItem, dataInfoDictList, context, devModel):
    '''
    @summary: adjust item location info
    @date: 2014-08-08
    @param collectItem: collect item
    @param dataInfoDictList: the list of data dictionary
    @param context: context object
    @param devModel: device model
    @return: 
    '''
    global rackInfoDict
    
    logger = context.get('logger')
        
    ctrlOrBBUSlotPosMap = { 'S2600T':'slotpos=1',#1表示后插板,0表示前插板
                    'S5500T':'slotpos=1',
                    'S5600T':'slotpos=0',
                    'S5800T':'slotpos=0',
                    'S6800T':'slotpos=0',
                    'HVS85T':'slotpos=0',
                    'HVS88T':'slotpos=0',
                    '18800F':'slotpos=0',
                    '18500':'slotpos=0',
                    '18800':'slotpos=0'}
    
    tmpDictList = copy.copy(dataInfoDictList)
    for dataInfoDict in tmpDictList:
        for val in dataInfoDict.values():
            if val:#IBMS系统不允许所有字段值为空,其中一次自验发现报告中一块硬盘只有“name”字段.
                break
        else:
            logger.error('This information dictionary is invalid:' + str(dataInfoDict))
            dataInfoDictList.remove(dataInfoDict)
                
    for dataInfoDict in dataInfoDictList:
        tmpDict = copy.copy(dataInfoDict)
        for key in tmpDict.keys():
            if not dataInfoDict.get(key):#IBMS系统不允许字段值为空
                logger.warn('Invalid item value for item:' + str(collectItem) + ',key:' + str(key))
                dataInfoDict.pop(key)
                    
    if collectItem in ['Controller', 'BBU']:#控制器、BBU位置相同（都在前插板或者后插板）
        logger.info('Adjusting item location:' + collectItem)
        for dataInfoDict in dataInfoDictList:
            ctrlOrBBULoc = dataInfoDict.get('Location', '')
            updateKey = 'Location'
            if not ctrlOrBBULoc:
                ctrlOrBBULoc = dataInfoDict.get('ID', '')
                updateKey = 'ID'
                
            if not ctrlOrBBULoc:
                logger.warn('Empty ID or location:' + str(collectItem))
                pass
            else:
                slot = ctrlOrBBULoc
                shelf = ctrlOrBBULoc.split('.')[0]
                rackLocStr = globalRackInfoDict.get(shelf, 'rack=-1')
                slotPos = ctrlOrBBUSlotPosMap.get(devModel, 'slotpos=0')
                dataInfoDict[updateKey] = rackLocStr + ',shelf=' + shelf + ',subshelf=-1,' + slotPos + ',slot=' + slot
    elif collectItem in ['Enclosure']:
        logger.info('Adjusting item location:' + collectItem)
        for dataInfoDict in dataInfoDictList:
            enclosureID = dataInfoDict.get('ID', '')
            location = dataInfoDict.get('Location', '')
            if not enclosureID or not location:
                pass
            else:
                slot = enclosureID
                shelf = enclosureID.split('.')[0]
                if location == '--':
                    globalRackInfoDict[enclosureID] = 'rack=-1'
                else:
                    globalRackInfoDict[enclosureID] = 'rack=' + location.split('.')[0]
               
                rackLocStr =  globalRackInfoDict[enclosureID]
                shelfLocStr = 'shelf=' + shelf
                dataInfoDict['ID'] = rackLocStr + ',' + shelfLocStr
    elif collectItem in ['Disk']:#框、盘在前插板
        logger.info('Adjusting item location:' + collectItem)
        for dataInfoDict in dataInfoDictList:
            diskID = dataInfoDict.get('ID', '')
            if not diskID:
                pass
            else:
                slot = diskID
                shelf = diskID.split('.')[0]
                rackLocStr = globalRackInfoDict.get(shelf, 'rack=-1')
                dataInfoDict['ID'] = rackLocStr + ',shelf=' + shelf + ',subshelf=-1,slotpos=0,slot=' + slot                
    elif collectItem in ['ExpansionModule', 'InterfaceModule']:#部件在后插板
        logger.info('Adjusting item location:' + collectItem)
        for dataInfoDict in dataInfoDictList:
            componetID = dataInfoDict.get('ID', '')
            if not componetID:
                pass
            else:
                slot = componetID
                shelf = componetID.split('.')[0]
                rackLocStr = globalRackInfoDict.get(shelf, 'rack=-1')
                dataInfoDict['ID'] = rackLocStr + ',shelf=' + shelf + ',subshelf=-1,slotpos=1,slot=' + slot
    elif collectItem in ['Fan', 'PowerModule']:#部件在后插板
        logger.info('Adjusting item location:' + collectItem)
        for dataInfoDict in dataInfoDictList:
            componetID = dataInfoDict.get('ID', '')
            if not componetID:
                pass
            else:
                slot = componetID
                shelf = componetID.split('.')[0]
                if 'DSW' in componetID:#只有交换机风扇在前插板
                    slotPos = 'slotpos=0'
                else:
                    slotPos = 'slotpos=1'
                rackLocStr = globalRackInfoDict.get(shelf, 'rack=-1')    
                dataInfoDict['ID'] = rackLocStr + ',shelf=' + shelf + ',subshelf=-1,' + slotPos + ',slot=' + slot                
    else:
        logger.info('No need to adjusting item location:' + collectItem)
        pass    
    
def collectLicense(context, cmdLst, logger):
    '''
    @summary: collect license
    @date: 2014-08-08
    @param context: the context object
    @param cmdLst: command list
    @param logger: a logger provided by tool
    @return: the dictionary list of result
    '''
    global isLiceFileExst
    retDictList = []
    cmdName1 = cmdLst[0]#'show license'
    cmdName2 = cmdLst[1]#'show license_active'
    collectItem = 'License'
    resource = getResource(context)
    sshConnector = getSshConnector(context)
    
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmdName1, logger)
    if not isCmdExecSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.error('[collectStandard] Execute command [' + cmdName1 + '] failed!')
        return []
    #license不存在
    if re.search('The License file does not exist', cmdRet, re.IGNORECASE):
        isLiceFileExst = False
    
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []
    cliRetLine = cmdRet.splitlines()
    tempMap = {}
    for line in cliRetLine:
        if re.search('License SN', line, re.IGNORECASE):
            tempMap['License SN'] = line.split(':')[1].strip()
            break
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmdName2, logger)
    if not isCmdExecSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.error('[collectStandard] Execute command [' + cmdName2 + '] failed!')
        return []
    
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []
    formattedDictList = utils.formatStr(cmdRet)
    formattedDict = formattedDictList[0]
    formattedDict.update(tempMap)
    retDictList.append(formattedDict)
    logger.info('formattedDict=' + str(formattedDict))
    return retDictList


def collectSoftware(context, cmd, logger):
    '''
    @summary: collect software
    @date: 2014-08-08
    @param context: the context object
    @param cmd: command
    @param logger: a logger provided by tool
    @return: the dictionary list of result
    '''
    retDictList = []
    collectItem = 'Software'
    resource = getResource(context)
    sshConnector = getSshConnector(context)
    
    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger)
    if not isSucc:
        errMsg = resource.getString('dev.conn.failure')
        tmpDict = {G_IS_CMD_SUCC:False, G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:errMsg}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.error('[collectStandard] Execute command [' + cmd + '] failed!')
        return []
    isExecSuc, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isExecSuc:
        tmpDict = {G_IS_RET_VALID:False, G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.error('[collectStandard]Failed to execute command and the result is not parsable.')
        return []
    elif not isParsable:
        tmpDict = {G_IS_RET_PARSABLE:False, G_CMDRET:cmdRet}
        G_cmdExecRetDict.get(collectItem).update(tmpDict)
        logger.warn('[collectStandard]Executed command successfully but the result is not parsable.')
        return []
    cmdRetList = cmdRet.splitlines()
    softwareStr = ''
    hotPatchStr = ''
    isSoftware = False
    isHotPatch = False
    for line in cmdRetList:
        if re.search('Software Version', line, re.IGNORECASE):
            isSoftware = True
            isHotPatch = False
        elif re.search('HotPatch Version', line, re.IGNORECASE):
            isSoftware = False
            isHotPatch = True
        elif re.search(':/>', line, re.IGNORECASE):
            isSoftware = False
            isHotPatch = False
        if isSoftware:
            softwareStr += line + os.linesep
        elif isHotPatch:
            hotPatchStr += line + os.linesep
    softwareDictList = utils.formatHDict2List(softwareStr)
    hotPatchDictList = utils.formatHDict2List(hotPatchStr)
    for softwareDict in softwareDictList:
        swVerName = softwareDict.get('Name')
        swVerSn = softwareDict.get('SN')
        swVerCurVer = softwareDict.get('Current Version')
        swVerTpe = softwareDict.get('Type')
        for hotPatchDict in hotPatchDictList:
            hpVerName = hotPatchDict.get('Name')
            if hpVerName == swVerName:
                hpVerCurVer = hotPatchDict.get('Current Version')
                tmpDict = {}
                tmpDict['SN'] = swVerSn
                tmpDict['SoftwareCurrentVersion'] = swVerCurVer
                tmpDict['Type'] = swVerTpe
                tmpDict['HotPatchCurrentVersion'] = hpVerCurVer
                retDictList.append(tmpDict)
    logger.info('retDictList=' + str(retDictList))
    
    newRetDictList = []
    for retDict in retDictList:
        SoftwareCurrentVersion = retDict.get('SoftwareCurrentVersion')
        if '--' != SoftwareCurrentVersion:
            newRetDictList.append(retDict)
            break
    if not newRetDictList:
        newRetDictList = [retDictList[0]]
    return newRetDictList


def getAbnormalPowerCmdList(context):
    sshConnector = getSshConnector(context)
    cliCmd = 'show power_supply |filterRow column=Running\sStatus predict=not predict2=equal_to value=Online'
    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, getLogger(context))
    if not isCmdExecSucc:
        Log.error('Failed to execute command [%s] !' % cliCmd)
        return None
    
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet)
    if not isRetValid:
        Log.error('The result of command [%s] is invalid and not parsable.' % cliCmd)
        return []
    elif not isParsable:
        Log.warn('The result of command [%s] is valid but not parsable.' % cliCmd)
        return []
    
    cmdRetDictList = utils.formatHDict2List(cmdRet)
    cmdList = []
    for cmdRetDict in cmdRetDictList:
        cmdList.append('show power_supply power_supply_id=' + cmdRetDict.get('ID'))
    return cmdList


def getAbnormalPowerCliInfo(context):
    '''
    @summary: get abnormal power CLI information
    @param context: context object
    '''
    sshConnector = getSshConnector(context)
    cliCmdList = getAbnormalPowerCmdList(context)
    resource = getResource(context)
    allCmdRet = ''
    if None == cliCmdList:
        errMsg = resource.getString('dev.conn.failure')
        return errMsg
    for cliCmd in cliCmdList:
        isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cliCmd, getLogger(context))
        if not isCmdExecSucc:
            errMsg = resource.getString('dev.conn.failure')
            Log.error('Failed to execute command [%s] !' % cliCmd)
            allCmdRet += errMsg
            break
        allCmdRet += cmdRet
    Log.info('Abnormal Power allCmdRet=' + allCmdRet)
    return allCmdRet


def judgeCollectResult(context, stdItemDataMap):
    '''
    @summary: Judge collect result.
    @date: 2014-08-12
    @param context: context object
    @param stdItemDataMap: the data map of standard item 
    @return: (return code, error message) as (integer, string)
    '''
        
    errMsgList = []
    resource = getResource(context)
    collectResutList = []
    Log.info('G_cmdExecRetDict=%s' % unicode(G_cmdExecRetDict))
    errMsgDetailDict = {}
    
    for itemName in stdItemDataMap.keys():
        #由于柜和量纲是从框信息里面提取的，所以柜和量纲取框的回显
        if itemName in ['Rack', 'Dimension']:
            cmdRet = G_cmdExecRetDict.get('Enclosure').get(G_CMDRET)
        else:
            cmdRet = G_cmdExecRetDict.get(itemName).get(G_CMDRET)
        
        if 'License' == itemName:
            if stdItemDataMap.get(itemName) or not isLiceFileExst:
                Log.info('Collect %s information successfully!' % itemName)
                errMsg = resource.getString('info.collect.%s.info.success' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = ''
                collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
            else:
                Log.error('Failed to collect %s information!' % itemName)
                errMsg = resource.getString('err.collect.%s.info.fail' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = cmdRet
                collectResutList.append(SCRIPT_RET_CODE.FAIL)
        if itemName in ['System', 'Controller', 'Software', 'Dimension']:
            if not stdItemDataMap.get(itemName):
                Log.error('Failed to collect %s information!' % itemName)
                errMsg = resource.getString('err.collect.%s.info.fail' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = cmdRet
                collectResutList.append(SCRIPT_RET_CODE.FAIL)
            else:
                Log.info('Collect %s information successfully!' % itemName)
                errMsg = resource.getString('info.collect.%s.info.success' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = ''
                collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
        elif itemName in ['Enclosure', 'PowerModule', 'Disk', 'Rack']:
            itemNum = context.get('%sNum' % itemName, 0)
            if not stdItemDataMap.get(itemName):
                Log.error('Failed to collect %s information!' % itemName)
                errMsg = resource.getString('err.collect.%s.info.fail' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = cmdRet
                collectResutList.append(SCRIPT_RET_CODE.FAIL)
            elif len(stdItemDataMap.get(itemName)) != itemNum:
                Log.warn('Collect part %s information successfully!' % itemName)
                errMsg = resource.getString('err.collect.part.%s.info.fail' % itemName)
                errMsgList.append(errMsg)
                if 'PowerModule' == itemName:
                    errMsgDetailDict[errMsg] = cmdRet + getAbnormalPowerCliInfo(context)
                else:
                    errMsgDetailDict[errMsg] = cmdRet
                collectResutList.append(SCRIPT_RET_CODE.PART_SUCCESS)
            else:
                Log.info('Collect all %s information successfully!' % itemName)
                errMsg = resource.getString('info.collect.%s.info.success' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = ''
                collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
        elif itemName in ['ExpansionModule', 'InterfaceModule', 'BBU', 'Fan']:
            itemNum = context.get('%sNum' % itemName, 0)
            if itemNum == 0:
                Log.warn('No %s information need to collect.' % itemName)
                errMsg = resource.getString('info.collect.%s.info.success' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = ''
                collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
            elif len(stdItemDataMap.get(itemName)) == 0:
                Log.error('Failed to collect %s information!' % itemName)
                errMsg = resource.getString('err.collect.%s.info.fail' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = cmdRet
                collectResutList.append(SCRIPT_RET_CODE.FAIL)
            elif len(stdItemDataMap.get(itemName)) != itemNum:
                Log.warn('Collect part %s information successfully!' % itemName)
                errMsg = resource.getString('err.collect.part.%s.info.fail' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = cmdRet
                collectResutList.append(SCRIPT_RET_CODE.PART_SUCCESS)
            else:
                Log.info('Collect all %s information successfully!' % itemName)
                errMsg = resource.getString('info.collect.%s.info.success' % itemName)
                errMsgList.append(errMsg)
                errMsgDetailDict[errMsg] = ''
                collectResutList.append(SCRIPT_RET_CODE.SUCCESS)
    
    Log.info('errMsgDetailDict=' + unicode(errMsgDetailDict))
    
    if collectResutList.count(SCRIPT_RET_CODE.SUCCESS) == len(stdItemDataMap):
        Log.info('Collect all standard item information success.')
        return (SCRIPT_RET_CODE.SUCCESS, '', errMsgDetailDict)
    elif collectResutList.count(SCRIPT_RET_CODE.FAIL) == len(stdItemDataMap):
        Log.error('Collect all standard item information failed!')
        return (SCRIPT_RET_CODE.FAIL, '\n'.join(errMsgList), errMsgDetailDict)
    else:
        Log.warn('Collect part standard item information success.')
        return (SCRIPT_RET_CODE.PART_SUCCESS, '\n'.join(errMsgList), errMsgDetailDict)
