# -*- coding:utf-8 -*-
from defusedxml import ElementTree as ET
import os
import traceback
from common.devNode import DevNode
from common import commonFunction
from common.commonFunction import getResource, isSshConnectionNormal, checkConnAndUserPwdStatus
from common.constant import SCRIPT_RET_CODE
from common import cliCmdManager
from common.cTV2R2 import cHandleTypeList
from common import utils
from common.log import Log
from common.contextUtil import getPath, getSshConnector, getLogger
from common.exception import DisconnectionException

def execute(context):
    '''
    @summary: Initialize device node.
    @date: 2014-07-28
    @param context: context object
    @return: 
    '''
    try:
        initLog(context)
        Log.info('Start to prepare handle...')
        resource = getResource(context)
        
        devNode = context.get('devNode')
        
        if not devNode:
            devNode = DevNode(context)
            
            context['sshConnector'] = devNode.getSshConnector(context)
            context['devNode'] = devNode
            
            Log.info(str(devNode))
            
            #检查连接是否正常、用户密码是否即将过期和是否已过期
            isSucc, errMsg = checkConnAndUserPwdStatus(context, resource)
            if not isSucc:
                return (SCRIPT_RET_CODE.FAIL, errMsg)
            
            if not devNode.isSystemNormal(context):
                errMsg = resource.getString('err.sys.status.abnormal')
                return (SCRIPT_RET_CODE.FAIL, errMsg)  
        
        errCode, errMsg = parseStdCfgFile(context)
        if errCode == SCRIPT_RET_CODE.FAIL:
            return (errCode, errMsg, {})
        extendCfgFile = os.sep.join([getPath(context), 'script', 'config', 'cmd4Excel.xml'])
        errCode, errMsg = parseCmdCfgFile(context, extendCfgFile)
        Log.info('Finish to prepare handle.')
        
        return (errCode, errMsg, {})
    except DisconnectionException, de:
        Log.error('Maybe the network is unavailable or device is abnormal, details:' + traceback.format_exc())
        errMsg = resource.getString('dev.conn.failure')
        return (SCRIPT_RET_CODE.FAIL, errMsg, {})

def initLog(context):
    Log(context)
    Log.updateScriptFlag('prepare')

    
def parseStdCfgFile(context):
    '''
    @summary: load and parse configure file, then load to memory
    @date: 2014-07-28
    @param context: context object
    @param xmlFile: the absolute path of configure file
    @param contextKey: the key of context map object
    @return: 
    '''
    logger = context.get('logger')
    resource = getResource(context)
    
    cfgFile4Standard = context.get('path') + os.sep + 'script' + os.sep + 'config' + os.sep + 'cmd4IBMS.xml'
    root = ET.parse(cfgFile4Standard)
    
    itemList = root.findall('item')
    itemCmdListDict = {}
    errMsg = ''
    errCode = SCRIPT_RET_CODE.SUCCESS
    
    try:
        for item in itemList:
            itemName = item.attrib['name']
            itemCmdList = []
            baseCmd = item.findall('cmd')[0].attrib['command']
            if not item.findall('cmd/param'):
                fullCmd = baseCmd
                itemCmdList.append(fullCmd)
            else:
                paraName = item.findall('cmd/param')[0].attrib['paraName']
                paraValCmd = item.findall('cmd/param')[0].attrib['LoopBaseCmd']
                filedName = item.findall('cmd/param')[0].attrib['loop']
                for paraVal in getParaValByCLI(context, paraValCmd, filedName):
                    fullCmd = baseCmd + ' ' + paraName + '=' + paraVal
                    itemCmdList.append(fullCmd)
            itemCmdListDict[itemName] = itemCmdList
            componentNumKey = itemName + 'Num'
            if 'PowerModule' == itemName:
                context[componentNumKey] = getPowerModuleNum(context, baseCmd)
            else:
                context[componentNumKey] = len(itemCmdList)
            logger.info(componentNumKey + ':' + str(context[componentNumKey]))
    except DisconnectionException, de:
        logger.error('Catch except in parseCmdCfgFile, details:' + traceback.format_exc())
        raise de
    except Exception, e:
        logger.error('Exception:' + str(e))
        errMsg = resource.getString('dev.conn.failure')
        errCode = SCRIPT_RET_CODE.FAIL
        return (errCode, errMsg)
    except:
        logger.error('Parse IBMS configure file caught exception.')
        errMsg = resource.getString('err.parse.IBMS.cfg.failed')
        errCode = SCRIPT_RET_CODE.FAIL
        return (errCode, errMsg)
    else:
        logger.info('Standard item name-cmdList:' + str(itemCmdListDict))
        context['stdItemDict'] = itemCmdListDict
        errCode = SCRIPT_RET_CODE.SUCCESS
        return (errCode, errMsg)


def parseCmdCfgFile(context, cmdCfgFile):
    '''
    @summary: load and parse command configure file, then load to memory
    @param context: context object
    @param cmdCfgFile: command configuration file
    @return: (result code, error message) as (integer, string)
    '''
    
    logger = context.get('logger')
    resource = getResource(context)
    
    root = ET.parse(cmdCfgFile)
    
    itemList = root.findall('item')
    itemCmdListDict = {}
    errMsg = ''
    retCode = SCRIPT_RET_CODE.SUCCESS
    
    try:
        for item in itemList:
            itemName = item.attrib['name']
            itemCmdList = []
            if 'remote lun' == itemName:#该项情况特殊
                itemCmdListDict[itemName] = getRemoteLunCmdList(context)
                continue
            baseCmd = item.findall('cmd')[0].attrib['command']
            if not item.findall('cmd/param'):
                fullCmd = baseCmd
                itemCmdList.append(fullCmd)
            else:
                paraName = item.findall('cmd/param')[0].attrib['paraName']
                paraValCmd = item.findall('cmd/param')[0].attrib['LoopBaseCmd']
                filedName = item.findall('cmd/param')[0].attrib['loop']
                for paraVal in getParaValByCLI(context, paraValCmd, filedName):
                    fullCmd = baseCmd + ' ' + paraName + '=' + paraVal
                    logger.info('full command:' + str(fullCmd))
                    itemCmdList.append(fullCmd)
            itemCmdListDict[itemName] = itemCmdList
    except DisconnectionException, de:
        logger.error('Catch except in parseCmdCfgFile, details:' + traceback.format_exc())
        raise de
    except Exception, e:
        logger.error('Exception caught:' + str(e))
        errMsg = resource.getString('dev.conn.failure')
        retCode = SCRIPT_RET_CODE.FAIL
        return (retCode, errMsg)
    else:
        Log.info('[prepare]itemTag name-cmdList:' + str(itemCmdListDict))
        context['extItemCmdDict'] = itemCmdListDict
        retCode = SCRIPT_RET_CODE.SUCCESS
        return (retCode, errMsg)


def getRemoteLunCmdList(context):
    '''
    @summary: Get the remote LUN command list
    @param context: context object
    @return: the command list of remote LUN
    '''
    try:
        cmdList = []
        sshConnector = getSshConnector(context)
        dependentCmd = 'show remote_device general'
    
        isCmdSucc, cmdRet = cliCmdManager.execCmd(sshConnector, dependentCmd, getLogger(context))
        if not isCmdSucc:
            Log.error('Failed to execute dependent command [%s] !' % dependentCmd)
            return []
        isRetValid, isRetParsable = utils.checkCliInfoValid(cmdRet, True)
        if not isRetValid:
            Log.error('The result of command [%s] is invalid and not parsable.' % dependentCmd)
            return []
        elif not isRetParsable:
            Log.warn('The result of command [%s] is valid but not parsable.' % dependentCmd)
            return []
        
        dataInfoDictList = utils.formatHDict2List(cmdRet)
        if not dataInfoDictList:
            return []
        for dataInfoDict in dataInfoDictList:
            cmd = 'show remote_lun general array_type='
            deviceType = dataInfoDict.get('Device Type')
            id = dataInfoDict.get('ID')
            if 'Current Manufacturer Device' == deviceType:
                cmd += 'current_manufacturer_device remote_device_id=' + id
            elif 'Other Manufacturer Device' == deviceType:
                cmd += 'other_manufacturer_device'
            cmdList.append(cmd)
        Log.info('remoteLun cmdList=' + str(cmdList))
        return cmdList
    except:
        Log.error('Catch except when getRemoteLunCmdList, details:' + traceback.format_exc())
        return []


def getPowerModuleNum(context, cmd):
    '''
    @summary: Get the number of power module
    @date: 2014-08-22
    @param context: context object
    @param cmd: the command of power module
    @return: the number of power module
    '''
    
    sshConnector = context['sshConnector']
    logger = context.get('logger')

    isSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, logger)
    if not isSucc:
        logger.error('[prepare] Execute command [' + cmd + '] failed!')
        return 0
    cmdRetLineList = cmdRet.splitlines()
    isSwitchOn = False
    counter = 0
    for line in cmdRetLineList:
        if -1 != line.find('---'):
            isSwitchOn = True
            continue
        if -1 != line.find(':/>'):
            isSwitchOn = False
        if not line.strip():
            continue
        if isSwitchOn:
            counter += 1
    logger.info('PowerModuleNum=' + str(counter))
    return counter
    
    
def getParaValByCLI(context, cmd, fieldName):
    '''
    @summary: Get the value of parameter.
    @date: 2014-07-28
    @param context: context object
    @param cmd: command
    @param fieldName: 
    @return: the value list of parameter
    '''
     
    sshConnector = getSshConnector(context)
    resource = getResource(context)

    isCmdExecSucc, cmdRet = cliCmdManager.execCmd(sshConnector, cmd, getLogger(context))
    if not isCmdExecSucc:
        Log.error('Execute base loop command [' + cmd + '] failed!')
        raise Exception('dev.conn.failure')
    
    Log.info('filedName:' + str(fieldName))
    isRetValid, isParsable = utils.checkCliInfoValid(cmdRet, True)
    if not isRetValid:
        Log.error('Executed command successfully but the result is not valid.')
        return []
    elif not isParsable:
        Log.warn('Executed command successfully but the result is not parsable.')
        return []
    formatFunction = cHandleTypeList(cmdRet, [fieldName])
    
    try:
        fieldInfoDictList = formatFunction.handle()[1]
        fieldValList = [fieldInfoDict.get(fieldName, '') for fieldInfoDict in fieldInfoDictList]
        if '' in fieldValList:
            fieldValList.remove('')
    except Exception, e:
        Log.error('Parse CLI echo message caught exception:' + str(e))
        return []
    else:
        Log.info('FieldValueList' + str(fieldValList))
        return fieldValList
