# -*- coding: UTF-8 -*-
import re
from frame.cli import cliResource
from common.log import Log

MAX_RETRYS = 5
CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEVELOPER_MODEL_FLAG = "developer:/>"
CLI_RET_END_FLAG = ":/>"
CLI_NOT_EXSIT_FLAG = "^"

SERIES_T = "T_SERIES"
SERIES_V3 = "V3_SERIES"
SERIES_18000 = "18000_SERIES"

STOR_DEV_INFO_DICT = {
"S2600T":SERIES_T,
"S5500T":SERIES_T,
"S5600T":SERIES_T,
"S5800T":SERIES_T,
"S6800T":SERIES_T,
"2100 V3":SERIES_V3,
"2200 V3":SERIES_V3,
"2600 V3":SERIES_V3,
"5100 V3":SERIES_V3,
"5300 V3":SERIES_V3,
"5500 V3":SERIES_V3,
"5600 V3":SERIES_V3,
"5800 V3":SERIES_V3,
"6800 V3":SERIES_V3,
"6900 V3":SERIES_V3,
"HVS85T":SERIES_18000,
"HVS88T":SERIES_18000,
"18500":SERIES_18000,
"18800":SERIES_18000,
"18800F":SERIES_18000,
}

STOR_DEV_ARM = ["2100 V3", "2200 V3", "2600 V3", "5100 V3",
                "5110 V5", "5110F V5", "5210 V5", "5210F V5","5300 V5",
                "5310 V5", "5510 V5", "5610 V5", "6810 V5","18510 V5",
                "5810 V5","5810F V5", "18810 V5", "18810F V5",
                "5310F V5", "5510F V5", "5610F V5", "6810F V5","18510F V5"]

# V5R7新增部分支持VM的型号
V5R7C70_SUPPORT_VM_MODEL = ["5310 V5", "5300K V5", "5300 V5", "2810 V5",
                            "5510 V5", "5500K V5", "5500 V5"]
# V5R7C71及后续版本新增部分支持VM的型号
V5R7C71_SUPPORT_VM_MODEL = ["5310 V5", "5300K V5", "5300 V5", "2810 V5",
                            "5510 V5", "5500K V5", "5500 V5", "2600 V5",
                            "5210 V5 Enhanced"]

def getMsg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"
    
    try:
        if not resource.has_key(msg):
            return errMsg
        
        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg
        
        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg
    
def isChinese(lang):
    '''
    @summary: 判断lang是否为中文
    @param lang: 语言lang
    @return: 
        True: 中文
        False: 非中文
    '''
    if lang == "zh":
        return True
    return False

def isNoneLicense(cliRet):
    '''
    @summary: 判断回显结果里面是否包含需要license
    @param cliRet: cli回显
    @return: 
        True: cli回显包含需要license
        False: cli回显不包含需要license
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and ("error" in lowerLine or "suggestion" in lowerLine):
            return True
    
    return False

def isCmdExsit(cliRet):
    '''
    @summary: 根据回显判断阵列上CLI命令是否存在
    @param cliRet: cli回显
    @return: 
        True: cli存在
        False: cli不存在
    '''
    cliRetList = cliRet.splitlines()
    if len(cliRetList) > 1 and CLI_NOT_EXSIT_FLAG in cliRetList[1]:
        return False
    return True

def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    if re.search(CLI_EXECUTE_CMD_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

def isInternalError(cliRet):
    '''
    @summary: 判断回显结果是否包含内部错误信息
    @param cliRet: cli回显
    @return: 
        True: cli回显包含内部错误信息
        False: cli回显不包含内部错误信息
    '''
    lowerCliRet = cliRet.lower()
    if "internal error" in lowerCliRet:
        return True
    else:
        return False
    
def isNoneObject(cliRet):
    '''
    @summary: 判断回显结果里面是否包含查询的对象
    @param cliRet: cli回显
    @return: 
        True: cli查询的对象为空
        False: cli查询的对象不为空
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "error" in lowerLine and "not exist" in lowerLine:
            return True
    
    return False

def isNotSupport(cliRet):
    '''
    @summary: 判断回显结果里面是否包含 does not support the VM service
    @param cliRet: cli回显
    @return: 
        True: cli回显包含 does not support the VM service
        False: cli回显不包含does not support the VM service
    '''
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "not support" in lowerLine:
            return True
    
    return False

def isVMNotStarted(cliRet):
    '''
    @summary: 判断回显结果里面是否包含 The VM service is not started
    @param cliRet: cli回显
    @return: 
        True: cli回显包含 The VM service is not started
        False: cli回显不包含The VM service is not started
    '''
    
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "the vm service is not started" in lowerLine:
            return True
    
    return False
    
def getUserName(cli, lang):
    '''
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""
    userName = ""

    try:
        cmd = "show system general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if not checkRet[0]: 
            return checkRet
        
        cliRet = checkRet[1]
        lineList = cliRet.splitlines()
        
        if len(lineList) == 0:
            errMsg = getMsg(lang, "cannot.get.sys.info")
            return (False, cliRet, errMsg)
        
        lastLine = lineList[-1]
        if CLI_RET_END_FLAG in lastLine:
            userName = lastLine.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            return (True, userName, errMsg)
    
    except:
        errMsg = getMsg(lang, "cannot.get.sys.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def getUserPrivilege(cli, lang):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang) 
        if not checkRet:
            return checkRet
        
        userName = checkRet[1]
        cmd = "show user user_name=%s" % userName
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if not checkRet[0]: 
            return checkRet
        
        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.user.info")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            name = line["Name"]
            if name == userName:
                userPrivilege = line.get("Level", "Super_admin")
                return (True, userPrivilege, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def hasAdminOrSuperAdminPrivilege(cli, lang):
    '''
    @summary: 判断用户是否具有管理员或超级管理员权限
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时
                True: 具有管理员或超级管理员权限
                False： 不具有管理员或超级管理员权限
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    checkRet = getUserPrivilege(cli, lang)
    if not checkRet[0]:
        return checkRet
    
    userLevel = checkRet[1]
    if userLevel is None or len(userLevel) == 0 or userLevel.lower() not in ["super_admin", "admin"]:
        return (True, False, "")
    else:
        return (True, True, "")
            
def enterDeveloperMode(cli, lang):
    '''
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @param devPwd: 语言devPwd
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    errMsg = ""
    cliRet = ""
    switch = True
    try:
        checkRet = hasAdminOrSuperAdminPrivilege(cli, lang)
        if not checkRet[0]:
            return checkRet
        
        isAdminOrSuperAdminPrivilege = checkRet[1]
        if not isAdminOrSuperAdminPrivilege:
            errMsg = getMsg(lang, "cannot.enter.to.developer.model")
            return (False, cliRet, errMsg)
        
        #打开切换developer模式开关
        checkRet,switch = openDeveloperSwitch(cli,lang)
        
        cmd = "change user_mode current_mode user_mode=developer"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if not checkRet[0]: 
            return checkRet
        
        if isInDeveloperMode(checkRet[1]):
            return (True, checkRet[1], errMsg)
        
        #dorado c00进入developer需要进行交互,最多三次
        cliRet = checkRet[1]
        cliRetAll = cliRet
        for times in range(3):
            if re.search("(y/n)", cliRet, re.IGNORECASE):
                cliRet = cli.execCmd("y")
                cliRetAll += cliRet
                 
            if isInDeveloperMode(cliRet):
                return (True, cliRetAll, "")
        
        #未正常进入
        errMsg = getMsg(lang, "cannot.access.developer.mode") 
        developerMode2CliMode(cli)
        return (False, checkRet[1], errMsg)
    
    except:
        errMsg = getMsg(lang, "cannot.access.developer.mode")
        return (False, checkRet[1], errMsg)
    finally:
       #关闭开关
       if not switch :
           closeDeveloperSwitch(cli,lang)
    
def openDeveloperSwitch(cli,lang):
    '''
            打开切换developer模式开关
    '''
    switch = True
    #兼容V3R6版本，查看切developer模式开关是否打开
    developercmd = "show user_mode enabled"
    checkRet = excuteCmdInCliMode(cli, developercmd, True, lang)
    if checkRet[0] != True:
        return checkRet,switch
    
    cliRetDictList = getVerticalCliRet(checkRet[1])
    developerswitch = ""
    for dict in cliRetDictList:
        if "Developer View" in dict:
            developerswitch = dict.get("Developer View","")
            break
        
    if developerswitch == "Disabled":
        switch = False
        #开关关闭则打开 
        opencmd = "change user_mode enabled user_mode=developer enabled=yes"
        checkRet = excuteCmdInCliMode(cli, opencmd, True, lang)
        if checkRet[0] != True: 
            return checkRet,switch
    
    return checkRet,switch

def closeDeveloperSwitch(cli,lang):
    '''
    @summary: 关闭developer视图开关
    '''
    try:
        closecmd = "change user_mode enabled user_mode=developer enabled=no"
        excuteCmdInCliMode(cli, closecmd, True, lang)
    except:
        pass

def developerMode2CliMode(cli):
    '''
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    '''
    index = 0
    try:
        cliRet = cli.execCmd("show system general")
        while ("Password is wrong" in cliRet or "Password:" in cliRet or isInDeveloperMode(cliRet)):
            index += 1
            cliRet = cli.execCmd("exit")
            if "Are you sure to exit?(y/n):" in cliRet:
                cliRet = cli.execCmd("n")
            if index > MAX_RETRYS:
                break
    except:
        pass
        
    return None
        
def isInDeveloperMode(cliRet):
    '''
    @summary: 判断当前是否在developer模式下
    @param cliRet: cli回显
    @return: 
        True: 当前在developer模式下
        False: 当前不在developer模式下
    '''
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

def checkLineInBlackList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""
    
    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

def excuteCmdInCliMode(cli, cmd, logResult, lang):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param logResult: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""
    
    if cli is None:
        errMsg = getMsg(lang, "dev.conn.failure")
        return (False, cliRet, errMsg)
    
    try:
        if logResult:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)
    except:
        errMsg = getMsg(lang, "dev.conn.failure")
        return (False, cliRet, errMsg)
    
    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)
    
    if not isCmdExsit(cliRet):
        errMsg = getMsg(lang, "cmd.not.exsit")
        return (False, cliRet, errMsg)
    
    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    if isNoneLicense(cliRet):
        return (True, cliRet, errMsg)
    
    if isInternalError(cliRet):
        errMsg = getMsg(lang, "cli.excute.find.internal.error")
        return (False, cliRet, errMsg)
    
    if isNoneObject(cliRet):
        errMsg = getMsg(lang, "object.does.not.exit")
        return (False, cliRet, errMsg)

    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (False, cliRet, errMsg)  
    
    return (True, cliRet, errMsg)

def excuteCmdInDeveloperMode(cli, cmd, isHasLog, lang):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param devPwd: devPwd
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
    if not enterDeveloperCheckRet[0]:
        developerMode2CliMode(cli)
        return enterDeveloperCheckRet

    checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
    cliRet = checkRet[1]
    
    if isInDeveloperMode(cliRet):
        developerMode2CliMode(cli)
    
    return checkRet

def getProductModelVersion(cli, lang):
    '''
    @summary: 获取产品类型和产品版本
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型 产品版本 
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    checkRet = getOEMproductModelAndVersion(cli, lang)
    if checkRet[0] == True:
        return (True, " ".join((checkRet[3], checkRet[4])), "")
    
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    product_version = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
        elif fieldName == "Product Version":
            product_version = fieldValue
            
        if len(product_model) != 0 and len(product_version) != 0 and product_model != "--" and product_version != "--":
            return (True, " ".join((product_model, product_version)), "")
    
    return (False, cliRet, getMsg(lang, "cannot.get.product.version.info"))

def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    checkRet = getOEMproductModel(cli, lang)
    if checkRet[0] == True:
        return (True, checkRet[3], "")
    
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, product_model, "")
        
    return (False, cliRet, getMsg(lang, "cannot.get.product.model.info"))

def getProductVersion(cli, lang, context):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    product_version = ""
    devNode = context["dev"]
    if devNode != None:
        product_version = devNode.getProductVersion()
        del devNode    
    if product_version != "" and product_version != None:
        return (True, product_version, "")
    
    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Version":
            product_version = fieldValue
            
        if len(product_version) != 0 and product_version != "--":
            return (True, product_version, "")
        
    return (False, cliRet, getMsg(lang, "cannot.get.product.model.info"))


def getOEMproductModelAndVersion(cli, lang):
    """
    @summary: 获取深度OEM设备的华为型号和版本
    """
    flag, cliRet, errMsg, infoDict = getOEMsysInfo(cli, lang)
    if flag != True:
        return (flag, cliRet, errMsg, "", "")
    
    productModel = infoDict.get("Internal Product Model")
    productVersion = infoDict.get("Product Version")
    
    if productModel and productVersion:
        return (True, cliRet, "",  productModel.strip(), productVersion.strip())
    
    return (False, cliRet, getMsg(lang, "cannot.get.sys.info"), "", "")


def getOEMproductModel(cli, lang):
    """
    @summary: 获取深度OEM设备的华为型号和版本
    """
    flag, cliRet, errMsg, infoDict = getOEMsysInfo(cli, lang)
    if flag != True:
        return (flag, cliRet, errMsg, "")
    
    productModel = infoDict.get("Internal Product Model")
    if productModel:
        return (True, cliRet, "",  productModel.strip())
    
    return (False, cliRet, getMsg(lang, "cannot.get.sys.info"), "")


def getOEMsysInfo(cli, lang):
    """
    @summary: 获取深度OEM设备的华为型号等基本信息（V2R6C10，V5R7及其之后版本show system general获取的是OEM厂商型号）
    """
    cmd = "show system general|filterColumn exclude columnList=Product\sModel"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return (flag, cliRet, errMsg, {})
    
    dictList = getVerticalCliRet(cliRet)
    if len(dictList) == 0:
        return (False, cliRet, getMsg(lang, "cannot.get.sys.info"), {})
    
    return (True, cliRet, "", dictList[0])


def getStorageType(cli, lang):
    '''
    @summary: 获取存储设备类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，存储设备类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    getProductModelRet = getProductModel(cli, lang)
    if not getProductModelRet[0]:
        return getProductModelRet
    
    pdtModel = getProductModelRet[1]
    
    if not STOR_DEV_INFO_DICT.has_key(pdtModel):
        errMsg = getMsg(lang, "cannot.get.storage.type.info")
        return (False, "", errMsg)
    
    storageType = STOR_DEV_INFO_DICT.get(pdtModel)
    return (True, storageType, "")

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile(r"^\s*-+(\s+-+)*\s*$")
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []

        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0

        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break

        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))

        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break

            # 标题换行的场景
            if re.search(r"^-+(\s+-+)*\s*$", line):
                continue

            if len(line.strip()) == 0:
                continue

            if len(line) <= requiredLineLen:
                continue

            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))

        return dictList
    except:
        return []

def getHorizontalNostandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合,此方法用来解析CLI回显未对其情况
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        lenkeys = len(keys)
        Log.info("title is %s" % keys)
        dictList = []
        for line in field_words:
            if line.find(":/>") >= 0:
                break
            
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            vals = []
            valueList = line.strip().split("  ")
            
            Log.info("valueList is %s" % valueList)
            valueSpaceList =[]
            for value in valueList:
                if value != "":
                    vals.append(value.strip().decode("utf8"))
                    valueSpaceList.append(value)
            lenvalue = len(valueSpaceList)
            if lenvalue == lenkeys:
                dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []

def getVerticalCliRet(cliRet, isParseElcLabel=True):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    elcLabel = ""
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
        
        if re.search("^-+\r*\n*$", line):
            if not isParseElcLabel and lineDict.has_key("Electronic Label"):
                lineDict["Electronic Label"] = lineDict["Electronic Label"] + elcLabel
                elcLabel = ""
                
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            if isParseElcLabel and "=" in line:
                fields = line.split("=")
            elif "=" in line and re.search("^\s{16,}", line):
                elcLabel += line
                continue
            else:
                continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        if not isParseElcLabel and lineDict.has_key("Electronic Label"):
            lineDict["Electronic Label"] = lineDict["Electronic Label"] + elcLabel
        dictList.append(lineDict.copy())
    
    return dictList

def getSplitedCliRet(cliRet, splitMark):
    '''
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cliRet: cli回显
    @param splitMark: 分割标记 
    '''
    cliRetlist = cliRet.split(splitMark)
    if len(cliRetlist) < 2:
        return ""
    
    splitedCliRet = cliRetlist[1]
    match = re.search("^.*:(/>)*\r*\n*$", splitedCliRet, re.MULTILINE)
    if match:
        splitedCliRet = splitedCliRet[0:match.start()]
    
    if len(splitedCliRet.strip()) == 0:
        return ""
    else:
        return splitedCliRet

def isEngineId(value):
    '''
    @summary: 根据id判断是否为引擎ID
    @param value: 回显中的ID
    @return: 
        True: 是引擎ID
        False: 不是引擎ID
    '''
    if "ENG" in value or "CTE" in value:
        return True
    else:
        return False

def isDiskId(value):
    '''
    @summary: 根据id判断是否为硬盘ID
    @param value: 回显中的ID
    @return: 
        True: 是硬盘ID
        False: 不是硬盘ID
    '''
    if "DAE" in value:
        return True
    else:
        return False

def isDataSwitchId(value):
    '''
    @summary: 根据id判断是否为交换机ID
    @param value: 回显中的ID
    @return: 
        True: 是交换机ID
        False: 不是交换机ID
    '''
    if "DSW" in value:
        return True
    else:
        return False

def isHighDensityDisk(enclosureId, cli, lang):
    '''
    @summary: 判断硬盘框是否为高密框
    @param enclosureId: 框ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (flag, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，是高密框
            flag为False时，不是高密框
        errMsg: 错误时的消息
    '''
    if not isDiskId(enclosureId):
        return (True, False, "")
    
    checkRet = getHighDensityDiskEnclosureIdList(cli, lang)
    if not checkRet[0]: 
        return checkRet
    
    highDensityDiskEnclosureIdList = checkRet[1]
    if enclosureId in highDensityDiskEnclosureIdList:
        return (True, True, "")
    else:
        return (True, False, "")

def getEnclosureHeight(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框高度(U)
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的高度(U)
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Height(U)" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Height(U)":
            enclosureHeight = fieldValue
            return (True, enclosureHeight, "")
        
    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)

def getEnclosureType(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框类型
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Type" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Type":
            enclosureType = fieldValue
            return (True, enclosureType, "")
        
    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)

def getControllerIdList(cli, lang):
    '''
    @summary: 获取设备所有控制器ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，设备所有控制器ID列表
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show controller general|filterColumn include columnList=Controller"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = cliRet.splitlines()
    controlList = []
    
    for line in cliRetLinesList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Controller":
            ctrlId = fieldValue
            controlList.append(ctrlId)
            
    if len(controlList) == 0:
        errMsg = getMsg(lang, "cannot.get.controller.info")
        return (False, cliRet, errMsg)
    
    return (True, controlList, "")

def getHighDensityDiskEnclosureIdList(cli, lang):
    '''
    @summary: 获取高密硬盘框ID集合
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，高密硬盘框ID集合
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    
    cliRet = checkRet[1]
    highDensityDiskEnclosureIdList = []
    cliRetLinesList = getHorizontalCliRet(cliRet)
    
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)
    
    for line in cliRetLinesList:
        enclosureType = line["Type"]
        enclosureId = line["ID"]
        if "4U 75" in enclosureType:
            highDensityDiskEnclosureIdList.append(enclosureId)
    
    return (True, highDensityDiskEnclosureIdList, "")



def getHorizontalCliRetFilterElabel(cliRet, getDetail=True):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合（只处理水平表格中过滤显示电子标签场景）
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    
    if not ("Electronic\sLabel" in cliRet or "Electronic Label" in cliRet):
        return []
    
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            reg_headline = re.compile("^\s*-+(\s+-+)*\s*$") 
            match_headline = reg_headline.search(line)
            if match_headline:
                headline = match_headline.group()
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        reg_split = re.compile("\s*-+\s*")
        tuple_idxs = []
        start_pos = 0
        end_pos = 0
        
        while (start_pos <= len(headline)):
            match = reg_split.search(headline, start_pos)
            if match:
                end_pos = match.end()
                tuple_idxs.append((start_pos, end_pos))
                start_pos = end_pos
            else:
                break
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        dictList = []
        for line in field_words:
            #标题换行的场景
            if re.search("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            if CLI_RET_END_FLAG in line:
                break
            
            #处理电子标签属性行
            if not line.startswith(" "):
                if len(dictList) == 0:
                    continue
                lineDict = dictList[-1]
                eLabelValue = lineDict.get("Electronic Label")
                newELabelValue = eLabelValue + "\n" + line
                lineDict["Electronic Label"] = newELabelValue
                dictList[-1] = lineDict.copy()
                continue
                
            #处理新行    
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
        
        #处理电子标签详细属性值
        if getDetail:
            for lineDict in dictList:
                eLabelValue = lineDict.get("Electronic Label")
                eLableLines = eLabelValue.splitlines()
                #没有电子标签场景
                if len(eLableLines) <=1:
                    lineDict["BoardType"] = ""
                    lineDict["BarCode"] = ""
                    lineDict["Item"] = ""
                    lineDict["Description"] = ""
                    lineDict["Manufactured"] = ""
                    lineDict["VendorName"] = ""
                    lineDict["IssueNumber"] = ""
                #存在电子标签属性
                else:
                    propertyList = []
                    for elabelProperty in eLableLines[1:]:
                        if "=" in elabelProperty:
                            propertyList.append(elabelProperty)
                        else:
                            if len(propertyList) == 0:
                                continue
                            propertyList[-1] += elabelProperty
                    for elabelProperty in propertyList:
                        props = elabelProperty.split("=")
                        propKey = props[0].strip().decode("utf8")
                        propValue = "=".join(props[1:]).strip().decode("utf8")
                        lineDict[propKey] = propValue
        return dictList
    except:
        return []
def getVerticalCliRetForLicense(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合(只针对解析license表头回显，版本信息存在换行情况)
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合,处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
        fields = line.split(" : ")
        if len(fields) < 2:
            if  fields:
                mark = line.strip().decode("utf8")
                if lineDict.has_key("CopyRight") and len(mark)>0:
                    lineDict["CopyRight"] = lineDict["CopyRight"] + "\n" + mark
            continue
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    dictList.append(lineDict.copy())
    return dictList