# -*- coding: UTF-8 -*-
from common.log import Log
from common.constant import SCRIPT_RET_CODE
from common.result import getErrMsgFromLanguageFile
from funcFactory.cliParser import getInfoByCliParser
from frame.cli.cliUtil import isCmdExsit, isVMNotStarted, isNotSupport
import os


def AddlistToDict(list, keyName):
    '''
    @summary: 将集合list转换为以keyName为键的字
    @param list: 集合
    @param keyName: 字典的键
    @return: 字典
    '''
    map = {}
    map[keyName] = list
    return map


def getCmdNodeInfo(cmdNode):
    '''
    @summary: 从cmd节点中获取cli命令信息
    @param cmdNode: cmd节点解析后的字典
    @return: (cli命令，cli类型，需要获取的回显字段)
    '''
    command = cmdNode["elementAttrbutes"]["command"]
    cliType = cmdNode["elementAttrbutes"]["type"]
    
    collectItemTitleList = []        
    attrNodeList = cmdNode["elementChildEles"][0]["elementChildEles"] 
    for attrNode in attrNodeList:
        collectItemTitleList.append(attrNode["elementText"])
    return (command, cliType, collectItemTitleList)


def getBaseCmdNodeInfo(baseCmdNode):
    '''
    @summary: 从baseCmd节点中获取cli命令信息
    @param cmdNode: baseCmd节点解析后的字典
    @return: (cli命令，cli类型，需要获取的回显字段(作为其他cli命令的参数)
    '''
    command = baseCmdNode["elementAttrbutes"]["command"]
    cliType = baseCmdNode["elementAttrbutes"]["type"]
    paramName = baseCmdNode["elementAttrbutes"]["paramName"]
    
    return (command, cliType, paramName)


def setDetailedErrMsg(errMsgKey, failMsg, cliRet, context):
    '''
    @summary: 将CLI执行时的错误信息和回显根据中英文环境构造为一个字符串；
            并以errMsg为key，该字符串为value构造一个dict
    '''
    errMsg = getErrMsgFromLanguageFile(context, errMsgKey)
    
    if cliRet == "" or cliRet == None:
        return {errMsg:failMsg}
    else:
        return {errMsg:cliRet}
    
def setDetailedSuccMsg(succMsgKey,context):
    '''
    @summary: 收集成功的项需要将其设置到界面上
    '''
    succMsg = getErrMsgFromLanguageFile(context, succMsgKey)
    return {succMsg:""}


def collectData(context, xmInfo):     
    '''
    @summary: 根据每个收集项的收集类型和使用的cli类型收集数据
    @param xmInfo: 解析xml后的信息(一个嵌套的字典)
    @return: None
    '''
    #收集项失败数
    failNum = 0
    #收集项的数据
    allRetDataMap = {}
    #失败的收集项构成的字符串，用于工具界面显
    allErrMsg = {}
    
    collectItemNodes = xmInfo["elementChildEles"]
    
    collectNum = float(len(collectItemNodes))
    currentCollectNum = 0
    listener = context.get('listener')
    globalDict = globals()
    for node in collectItemNodes:
        #根据cmd4IBMS.xml中配置的类型自动调用dataCollectUtil.py中的同名方法来收集
        collectItemType = node["elementAttrbutes"]["type"]
        collectItemMethod = globalDict[collectItemType]
        (code, retDataMap, cliRet, failMsg) = collectItemMethod(context, node)
        
        collectItemName = node["elementAttrbutes"]["name"]
        #全部收集失败
        if code == SCRIPT_RET_CODE.FAIL:
            if not isCmdExsit(cliRet) or isVMNotStarted(cliRet) or isNotSupport(cliRet):
                retDataMap = None
                continue
                
            failNum += 1
            retDataMap = {collectItemName:[]}
            errMsgKey = "collect.fail.msg.%s" % collectItemName
            detailedErrMsgMap = setDetailedErrMsg(errMsgKey, failMsg, cliRet, context)
            allErrMsg.update(detailedErrMsgMap)
            
        #部分收集成功
        if code == SCRIPT_RET_CODE.PART_SUCCESS:
            failNum += 1
            errMsgKey = "collect.partfail.msg.%s" % collectItemName
            detailedErrMsgMap = setDetailedErrMsg(errMsgKey, failMsg, cliRet, context)
            allErrMsg.update(detailedErrMsgMap)
        #success, but no data
        if code == SCRIPT_RET_CODE.SUCCESS and retDataMap == None:
            retDataMap = {collectItemName:[]} 
        
        #收集成功，将收集项设置到界面上
        if code == SCRIPT_RET_CODE.SUCCESS:
            succMsgKey = "collect.succ.msg.%s" % collectItemName
            detailedSuccMap = setDetailedSuccMsg(succMsgKey,context)
            allErrMsg.update(detailedSuccMap)
        
        allRetDataMap.update(retDataMap)
        
        #收集进度
        currentCollectNum += 1
        progress = int((currentCollectNum / collectNum) * 100)
        listener.refleshProgress(progress)
    
    #全部收集成功   
    if failNum == 0:
        code = SCRIPT_RET_CODE.SUCCESS
    #全部收集失败
    elif failNum == len(collectItemNodes):
        code = SCRIPT_RET_CODE.FAIL
    #部分收集成功
    else:
        code = SCRIPT_RET_CODE.PART_SUCCESS
    
    return (code, allRetDataMap, allErrMsg)
    

def single(context, collectItemNode):
    '''
    @summary: 通过单条cli命令收集收集项
    @param collectItemNode: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    '''
    cmdNode = collectItemNode["elementChildEles"][0]
    (command, cliType, collectItemTitleList) = getCmdNodeInfo(cmdNode)
    
    collectItemName = collectItemNode["elementAttrbutes"]["name"]
    #retDataList:[{},{},...]
    (isSuccess, retDataList, cliRet, errMsg) = getInfoByCliParser(context, command, cliType, collectItemTitleList, collectItemName)
    
    #success, but no data
    if isSuccess and retDataList == None:
        return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
    
    #收集失败
    if not isSuccess:
        return (SCRIPT_RET_CODE.FAIL, None, cliRet, errMsg)
    #收集成功
    else:
        collectItemName = collectItemNode["elementAttrbutes"]["name"]
        return (SCRIPT_RET_CODE.SUCCESS, AddlistToDict(retDataList, collectItemName), cliRet, errMsg)


def dependency(context, collectItemNode):
    '''
    @summary: 通过两条cli命令收集收集项，其中第二条cli的参数来自于第一条cli的回显
    @param collectItemNode: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    '''
    
    
    (baseCommand, baseCliType, paramName) = ("", "", "")
    (command, cliType, collectItemTitleList) = ("", "", [])
    
    collectItemName = collectItemNode["elementAttrbutes"]["name"]
    
    #获取<baseCmd><cmd>中的内容
    cmdNodes = collectItemNode["elementChildEles"]
    for cmdNode in cmdNodes:
        if cmdNode["elementName"] == "baseCmd":
            (baseCommand, baseCliType, paramName) = getBaseCmdNodeInfo(cmdNode)
        else:
            (command, cliType, collectItemTitleList) = getCmdNodeInfo(cmdNode)
            
    #执行<baseCmd>的cli命令，最终获取的数据将作<cmd>的cli命令的参
    baseCollectItemTitleList = []
    baseCollectItemTitleList.append(paramName)
    (isSuccess, paramMapList, cliRet, errMsg) = getInfoByCliParser(context, baseCommand, baseCliType, baseCollectItemTitleList, collectItemName)
    
    #success, but no data
    if isSuccess and paramMapList == None:
        return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
    
    #收集失败
    if not isSuccess:
        return (SCRIPT_RET_CODE.FAIL, None, cliRet, errMsg)
        
    #循环执行<cmd>中的命令，并<baseCmd>获取的结果作为参
    failNum = 0
    retDataMapList = [] 
    alllCliRet = ""
    
    titleListMap = context["titleListMap"]
    
    for paramMap in paramMapList:
        param = paramMap[paramName]
        # 无数据，设置收集成功
        collectItemTitleList = titleListMap.get(collectItemName)
        if not collectItemTitleList:
            return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
        (isSuccess, retDataList, cliRet, errMsg) = getInfoByCliParser(context, command % param, cliType, collectItemTitleList, collectItemName)
        #收集失败
        if not isSuccess:
            failNum += 1
            alllCliRet = alllCliRet + cliRet
        elif retDataList != None:
            retDataMapList.extend(retDataList)
    
    #全部收集成功   
    if failNum == 0:
        return (SCRIPT_RET_CODE.SUCCESS, AddlistToDict(retDataMapList, collectItemName), alllCliRet, errMsg)
    
    elif failNum == len(paramMapList):#全部收集失败
        return (SCRIPT_RET_CODE.FAIL, None, alllCliRet, errMsg)

    else:#部分收集成功
        return (SCRIPT_RET_CODE.PART_SUCCESS, AddlistToDict(retDataMapList, collectItemName), alllCliRet, errMsg)


def many(context, collectItemNode):
    '''
    @summary: 通过多条条cli命令收集收集项，此方法只适用于cli回显单个表格数据的情况
    @param collectItemNode: cli配置文件中的<collectItem>节点
    @return: 收集项的数据（一个字典）
    '''
    cmdNodes = collectItemNode["elementChildEles"]

    retDataMap = {}
    for cmdNode in cmdNodes:
        #执行命令
        (command, cliType, collectItemTitleList) = getCmdNodeInfo(cmdNode)
        (isSuccess, retDataList, cliRet, errMsg) = getInfoByCliParser(context, command, cliType, collectItemTitleList)
        #success, but no data
        if isSuccess and retDataList == None:
            return (SCRIPT_RET_CODE.SUCCESS, None, cliRet, errMsg)
        
        #只要有一条命令执行失败就认为收集失败
        if not isSuccess:
            return (SCRIPT_RET_CODE.FAIL, None, cliRet, errMsg)
        #将获取到的cli回显解析后的map追加到retDataMap
        retDataMap.update(retDataList[0])
          
    retDataMapList = []
    retDataMapList.append(retDataMap)
    
    collectItemName = collectItemNode["elementAttrbutes"]["name"]
    return (SCRIPT_RET_CODE.SUCCESS, AddlistToDict(retDataMapList, collectItemName), cliRet, errMsg)


def more(context, collect_item_node):
    """
    通过多条cli命令收集收集项，此方法用name标签区分不同cli命令，并将收集项的返回数据，以{name：list}的形式存入ret_data_map中
    :param context: 工具提供的上下文对象
    :param collect_item_node: cli配置文件中的<collectItem>节点
    :return: 收集项的数据（一个字典）
    """
    cmd_nodes = collect_item_node.get("elementChildEles")
    ret_data_map = {}
    ret_data_map_list = []
    results_list = []
    all_cli_ret = ""
    all_err_msg = ""

    for cmd_node in cmd_nodes:
        # 执行命令
        command, cli_type, name, collect_item_title_list = get_cmd_node_info(cmd_node)
        is_success, ret_data_list, cli_ret, err_msg = getInfoByCliParser(context, command, cli_type,
                                                                         collect_item_title_list)
        all_cli_ret.join(cli_ret + "\n")
        all_err_msg.join(err_msg + "\n")
        results_list.append(is_success)
        if not is_success:
            Log.warn("{} collect failed. cli_ret is: {}. err_msg is: {}.".format(name, cli_ret, err_msg))
            continue

        # 以cmd4IBMS.xml文件的name标签为区分。name标签为“”的，将{data_key：data[data_key]}
        # 存入ret_data_map，name标签有值的，以{name：list}的形式存入ret_data_map中
        if not name or name == "":
            save_data(ret_data_list, ret_data_map)
        else:
            ret_data_map[name] = ret_data_list

    ret_data_map_list.append(ret_data_map)

    if not results_list:
        return SCRIPT_RET_CODE.FAIL, None, all_cli_ret, all_err_msg

    collect_item_name = collect_item_node.get("elementAttrbutes").get("name")
    status, data_map = check_collect_result_status(results_list, ret_data_map_list, collect_item_name)
    return status, data_map, all_cli_ret, all_err_msg


def check_collect_result_status(results_list, ret_data_map_list, collect_item_name):
    if len(set(results_list)) == 2:
        return SCRIPT_RET_CODE.PART_SUCCESS, AddlistToDict(ret_data_map_list, collect_item_name)
    if list(set(results_list))[0]:
        return SCRIPT_RET_CODE.SUCCESS, AddlistToDict(ret_data_map_list, collect_item_name)
    return SCRIPT_RET_CODE.FAIL, None


def save_data(ret_data_list, ret_data_map):
    for data in ret_data_list:
        for data_key in data:
            ret_data_map[data_key] = data.get(data_key)


def get_cmd_node_info(cmd_node):
    """
    从cmd节点中获取cli命令信息
    :param cmd_node: cmd节点解析后的字典
    :return: cli命令, cli类型, cmd节点对应的硬件名称, 需要获取的回显字段
    """
    command = cmd_node.get("elementAttrbutes").get("command")
    cli_type = cmd_node.get("elementAttrbutes").get("type")
    name = cmd_node.get("elementAttrbutes").get("name")

    collect_item_title_list = []
    attr_node_list = []
    if cmd_node.get("elementChildEles"):
        attr_node_list = cmd_node.get("elementChildEles")[0].get("elementChildEles")
    for attr_node in attr_node_list:
        collect_item_title_list.append(attr_node.get("elementText"))
    return command, cli_type, name, collect_item_title_list
