# -*- coding: UTF-8 -*-
import ast
import java.lang.Exception as JException
import json
import re
import traceback
from com.huawei.ism.tool.obase.utils import ManualConfigItemMgr
from datetime import datetime
from utils import Products

from cbb.business.checkitems import UpdateInnerCheckVersion
from cbb.frame.base import baseUtil
from cbb.frame.base import constants
from cbb.frame.base.baseUtil import UpgradeRes
from cbb.frame.base.exception import UnCheckException
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest.restUtil import Tlv2Rest

# 检查失败的状态。
FAIL_STATUS_TUPLE = ("3", "4")

UPD_CHECK_STATUS_2_STR = {
    # 界面展示 0等待检查, 1正在检查, 2警告, 3通过, 4不通过， 5忽略
    "0": 'check_wait',  # 等待执行
    "1": 'check_run',  # 执行中
    "2": 'check_success',  # 执行成功
    "3": 'check_failed',  # 执行失败
    "4": 'check_failed',  # 执行错误
    "5": 'check_ignore',  # 忽略
}

REST_CAN_NOT_CONNECT_CODE = "1073949185"

# 支持更新内置检查项的check_type检查类型列表
SUPPORT_UPDATE_INNER_CHECK_TYPES = ["FastUpgradeEvaluate", "RollUpgradeEvaluate",
                                    "OfflineEvaluate", "HotpatchPrecheck", "HotpatchEvaluate", "Inspection"]

# 升级后检查的check_type检查类型列表
POST_CHECK_CHECK_TYPES = ["FastPostUpgrade", "RollUpgradePostCheck", "OfflinePostCheck"]

# 容器升级评估的check_type检查类型列表
A8000_EVAL_CHECK_TYPES = ["A8000UpgradeEval", "A8000PatchPreCheck"]

# 补丁安装和补丁评估check_type列表
HOT_PATCH_CHECK_TYPES = ["HotpatchPrecheck", "HotpatchEvaluate"]


class Extras(object):
    not_pass_hosts = "not_pass_hosts"
    pass_hosts = "pass_hosts"
    can_not_check_hosts = "can_not_check_hosts"


HOST_EVAL_RESULT_KEYS = (
    Extras.pass_hosts,
    Extras.not_pass_hosts,
    Extras.can_not_check_hosts,
)


def execute(dataDict):
    return UpgradeCheck(dataDict).execute()


class UpgradeCheck():
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.logger = dataDict.get("logger")
        self.restService = RestService(dataDict)
        self.uiObserver = dataDict.get("observer")
        self.lang = dataDict.get("lang")
        self.devIp = contextUtil.getIp(dataDict)
        dev_obj = contextUtil.getDevObj(self.dataDict)
        self.dev_type = str(contextUtil.get_dev_type(self.dataDict)).replace("_container", "")
        self.dev_version = contextUtil.getCurVersion(dataDict)
        self.dest_version = self.dataDict.get("dest_version", "")
        self.upgrade_res = UpgradeRes(self.dev_type)
        self.is_a800_scene = self.dataDict.get("isA800Scene", False)
        self.is_zone_model = self.dataDict.get("preCheckMode", "")
        self.zone_id = self.dataDict.get("zone_id", "")
        self.execute_ip = self.dataDict.get("execute_ip", "")

        # 原子信息：dict(id="工具自定义数字id", nameId="检查项KEY", name="检查项名称",
        # startTime="开始时间", status="状态", ctrl="控制器", suggession="修复建议", operate="支持的操作")
        self.checkItems = []
        self.ignorePreCheckItemsStr = self.dataDict.get('ignorePreCheckItems', "")
        self.checkType = self.dataDict.get("checkType")
        self.curProcess = 0  # 初始进度

    def execute(self):
        try:
            self.logger.info(
                "is_a800_scene:{},is_zone_model:{},zone_id:{}".format(self.is_a800_scene, self.is_zone_model,
                                                                      self.zone_id))
            flowchoice_num = self.dataDict.get('flowchoice', 0)
            if 1 == flowchoice_num:
                self.uiObserver.ignoreItem(self.ignorePreCheckItemsStr)
                return True, "", "System inner check passed."
            self._setRunningProcess(1)
            self.logger.info("dev_type is {}".format(self.dev_type))
            # 查询的zone节点的每10s轮询等待执行节点执行成功或者失败结束流程。
            if self.is_a800_scene and self.is_zone_model == "query":
                if not self.polling_wait():
                    err_msg = baseUtil.getPyResource(self.lang, "SystemInnerCheck.check.error")
                    return False, err_msg, "System inner check failed,excute exception."
            # 生成一个检查项，检查阵列上的版本是否大于等于工具集成的版本
            if self.supportUpdateInnerCheck():
                self.compareCheckItemVersion()
            self.uiObserver.postMsg(self.checkItems)
            self.execute_check()
            self.queryCheckResult()
            return True, "", "System inner check passed."
        except UnCheckException as e:
            self.logger.error("excute uncheckexception.:%s" % traceback.format_exc())
            return False, e.errorMsg, "System inner check failed,excute uncheckexception."
        except Exception as e:
            return self.handle_exception(e)
        except:
            self.logger.error("excute exception.:%s" % traceback.format_exc())
            return (False, baseUtil.getPyResource(self.lang, "SystemInnerCheck.check.error"),
                    "System inner check failed,excute exception.")

    def polling_wait(self):
        """
            查询的zone节点的每10s轮询等待执行节点执行命令的结果。
            如果结果为失败，或者超时；则修改结果为False结束程序
            如果返回成功；则继续流程，查询执行的结果。
        """
        count = 0
        while True:
            count = count + 1
            zone_status = self.dataDict.get("zoneDataMap").get(self.execute_ip, "")
            self.logger.info("zone_status:{},count:{},execute_ip:{}".format(zone_status, count, self.execute_ip))
            if count > 30:
                return False
            if zone_status == "failed":
                return False
            if zone_status == "success":
                return True
            baseUtil.safeSleep(10)

    def execute_check(self):
        # A800的场景中的查询的节点，不支持执行命令。
        if self.is_zone_model == "query":
            return
        if self.checkType in A8000_EVAL_CHECK_TYPES:
            self.logger.info("check_type is {}".format(self.checkType))
            self.restService.execute_container_check(self.checkType, self.ignorePreCheckItemsStr)
        else:
            self.restService.excuteCheck(self.checkType, self.ignorePreCheckItemsStr)

    def handle_exception(self, exception):
        self.logger.info("excute exception:%s" % traceback.format_exc())
        if REST_CAN_NOT_CONNECT_CODE == str(exception.args[0]):
            return (False, baseUtil.getPyResource(
                self.lang, "SystemInnerCheck.check.conn.error"),
                    "System inner check failed,Communicating with the device failed.")
        return (False, baseUtil.getPyResource(self.lang, "SystemInnerCheck.check.error"),
                "System inner check failed,excute exception.")

    def supportUpdateInnerCheck(self):
        # A800的场景中的查询的节点，不支持执行命令。
        if self.is_zone_model == "query":
            return False
        self.logger.info("check type is {}, dev_version:{}".format(self.checkType, self.dev_version))
        # dorado需要新增升级后检查的内置检查项更新。
        if baseUtil.is_dorado_series_devs(self.dev_type) or Products.isKunpeng(self.dev_version):
            SUPPORT_UPDATE_INNER_CHECK_TYPES.extend(POST_CHECK_CHECK_TYPES)

        if self.checkType in SUPPORT_UPDATE_INNER_CHECK_TYPES or self.checkType in A8000_EVAL_CHECK_TYPES:
            self.logger.info("support update inner check")
            return True
        if self.dev_version and self.is_support_disk_pre_check() and self.checkType in ["OnlineDiskUpgPreCheck",
                                                                                        "OfflineDiskUpgPreCheck"]:
            self.logger.info("dorado diskupgrade support update inner check")
            return True
        return False

    def is_support_disk_pre_check(self):
        return (Products.isDigitalVer(self.dev_version) and Products.compareVersion(self.dev_version, "6.0.0") >= 0) \
               or Products.compareVersion(self.dev_version, "V700R001C00") >= 0

    def compareCheckItemVersion(self):

        startTime = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

        # 先初始化原子为执行中
        itemMsg = {
            "id": 0,
            "nameId": "UpdateInnerCheckVersion",
            ##检查项名称，类型String
            "name": self.upgrade_res.get_res("UpdateInnerCheckVersion"),
            "startTime": startTime,
            "status": self.uiObserver.toUiStatus("1"),
            "ctrl": "ALL",
            ##修复建议
            "suggession": "",
            # 检查项暂停可进行的操作,7代表暂停+忽略+终止（不包含重试）
            "operate": 7,
            "itemType": "innercheck"
        }
        self.checkItems = [itemMsg]
        flag, errMsg = UpdateInnerCheckVersion.excute(self.dataDict)
        status = "2" if flag else "3"
        itemMsg["status"] = self.uiObserver.toUiStatus(status)
        if flag and errMsg == "":
            errMsg = baseUtil.getPyResource(self.lang, "precheck.common.pass")
        itemMsg["suggession"] = errMsg
        self.checkItems = [itemMsg]

    def queryCheckResult(self):
        status = 0
        DEVEICE_UP_FALLD_BASE_VALUE = 0

        try:
            while True:
                DEVEICE_UP_FALLD_BASE_VALUE = DEVEICE_UP_FALLD_BASE_VALUE + 1
                if DEVEICE_UP_FALLD_BASE_VALUE > \
                        constants.UPGRADE_PACKAGE_POLL_TIMES:
                    self.logger.info("Upgrade check is timeout, the time:%s."
                                     % str(DEVEICE_UP_FALLD_BASE_VALUE))
                    raise UnCheckException(baseUtil.getPyResource(
                        self.lang, "SystemInnerCheck.check.timeout"))
                if self.restService.queryFailedCnt >= 8:
                    self.logger.info("query failed count:%s."
                                     % self.restService.queryFailedCnt)
                    raise UnCheckException(baseUtil.getPyResource(
                        self.lang, "SystemInnerCheck.check.conn.error"))
                # 查询升级检查进度
                recs = self.query_check_progress()
                self.logger.info("query check progress result:%s" % recs)
                if recs and recs[0]:
                    rec = recs[0]
                    self.logger.info("upgrade process:" + str(rec))
                    status = Tlv2Rest.getRecordValue(
                        rec,
                        restData.UpgradeCheck.GetCheckProgress.CHECK_STATUS)
                    process = Tlv2Rest.getRecordValue(
                        rec,
                        restData.UpgradeCheck.GetCheckProgress.CHECK_PROGRESS)
                    self._setRunningProcess(process)

                # 刷新检查项
                self.refreshAtomInfos()
                if restData.Enum.CheckStatus.CHECK_STATUS_FINISHED == status:
                    # 设置最终进度
                    self._setFinalProcess()
                    break
                baseUtil.safeSleep(5)
        except (Exception, JException) as e:
            self.logger.error("device upgrade query progress error"
                              + traceback.format_exc() + str(e))
            raise UnCheckException(baseUtil.getPyResource(
                self.lang, "SystemInnerCheck.queryresult.error"))
        finally:
            self.refreshExecutingItemToFailed()

    def query_check_progress(self):
        if self.checkType in A8000_EVAL_CHECK_TYPES:
            self.logger.info("check_type is {}".format(self.checkType))
            return self.restService.query_container_check_progress()
        return self.restService.queryCheckProgress()

    def query_repair_result(self):
        query_count = 0
        try:
            while True:
                query_count = query_count + 1
                if query_count > constants.REPAIR_POLL_TIMES:
                    self.logger.info("repair is timeout, the time:{}.".format(str(query_count)))
                    raise UnCheckException(baseUtil.getPyResource(self.lang, "SystemInnerCheck.check.timeout"))
                if self.restService.queryFailedCnt >= 8:
                    self.logger.info("query failed count:{}.".format(self.restService.queryFailedCnt))
                    raise UnCheckException(baseUtil.getPyResource(self.lang, "SystemInnerCheck.check.conn.error"))
                # 查询升级检查进度
                recs = self.restService.queryCheckProgress()
                status = self.get_repair_progress_status(recs)
                self.refresh_atom_Infos_to_check_items()
                # 刷新检查项
                if restData.Enum.CheckStatus.CHECK_STATUS_FINISHED == status:
                    # 设置最终进度
                    self.logger.info("query check progress result is finished.")
                    break
                baseUtil.safeSleep(5)
        except (Exception, JException) as e:
            self.logger.error("device repair query progress error:{}".format(traceback.format_exc()))
            return False, self.checkItems
        finally:
            self.refresh_excuting_item_to_failed()
        return True, self.checkItems

    def get_repair_progress_status(self, recs):
        self.logger.info("query check progress result:{}".format(recs))
        if recs and recs[0]:
            rec = recs[0]
            self.logger.info("repair process:" + str(rec))
            return Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckProgress.CHECK_STATUS)
        return 0

    def _setRunningProcess(self, process):
        """
        功能说明：设置升级前评估过程中进度（防止进度后退，并且不能达到100%）
        输入：升级进度
        """
        if process >= 100:
            self.curProcess = 99
        elif process > self.curProcess:
            self.curProcess = process
        self.uiObserver.setProgress(self.curProcess)

    def _setFinalProcess(self):
        """
        功能说明：设置升级前评估完成进度100%
        """
        self.curProcess = 100
        self.uiObserver.setProgress(self.curProcess)
        self.uiObserver.postMsg(self.checkItems)

    def checkUpgradeStatusIsUpgrade(self, status):
        """
        功能说明：检查升级检查状态是不是检查中和检查成功
        输入：升级检查状态
        输出：bool检查结果False/True
        """
        # 检查升级状态是否处于升级中和升级成功
        if restData.Enum.CheckStatus.CHECK_STATUS_INIT == status or restData.Enum.CheckStatus.CHECK_STATUS_RUNNING \
                == status or restData.Enum.CheckStatus.CHECK_STATUS_FINISHED == status:
            return True
        return False

    def refreshAtomInfos(self):
        self.refresh_atom_Infos_to_check_items()
        self.uiObserver.postMsg(self.checkItems)

    def refresh_atom_Infos_to_check_items(self):
        new_atom_group_list = self.queryNewAtoms()
        if len(new_atom_group_list) == 0 or None == new_atom_group_list:
            self.logger.info("have no new atoms!")
            return

        for new_group in new_atom_group_list:
            is_exist = False
            newkey = (new_group["name"], new_group["ctrl"])
            upgrade_ui_msg_len = len(self.checkItems)
            for index in range(upgrade_ui_msg_len):
                old_atom = self.checkItems[index]
                old_name = old_atom["name"]
                old_node_id = old_atom["ctrl"]
                old_key = (old_name, old_node_id)
                start_time = old_atom.get('startTime')
                if old_key == newkey:
                    is_exist = True
                    # 开始时间不应该更新
                    new_group['startTime'] = start_time
                    self.checkItems[index] = new_group
                    break
            if not is_exist:
                self.checkItems.append(new_group)

    def queryNewAtoms(self):
        newAtomGroupsDictList = []
        logger = self.dataDict.get("logger")
        recs = self.query_check_details()
        for rec in recs:
            (isSuccess, atomDictList) = self.toAtomInfo(self.dataDict, rec)
            if not isSuccess:
                logger.info("get atom info failed!")
                continue

            # 处理并发原子，已存在列表中则更新信息，不存在则添加到列表
            for atomDict in atomDictList:
                isExist = False
                atomKey = (atomDict["nameId"], atomDict["ctrl"])
                newAtomGroupDictListLen = len(newAtomGroupsDictList)
                for index in range(newAtomGroupDictListLen):
                    newAtomGroupDict = newAtomGroupsDictList[index]
                    atomGroupKey = (newAtomGroupDict["nameId"], newAtomGroupDict["ctrl"])
                    if atomGroupKey == atomKey:
                        isExist = True
                        newAtomGroupDict["status"] = atomDict["status"]
                        newAtomGroupDict["suggession"] = atomDict["suggession"]
                        newAtomGroupsDictList[index] = newAtomGroupDict

                if not isExist:
                    newAtomGroupsDictList.append(atomDict)

        return newAtomGroupsDictList

    def query_check_details(self):
        if self.checkType in A8000_EVAL_CHECK_TYPES:
            self.logger.info("check_type is {}".format(self.checkType))
            return self.restService.query_container_check_details()
        return self.restService.queryCheckDetails()

    def refreshExecutingItemToFailed(self):
        """
        将执行中的检查项刷为失败
        """
        self.refresh_excuting_item_to_failed()
        self.uiObserver.postMsg(self.checkItems)

    def refresh_excuting_item_to_failed(self):
        for item in self.checkItems:
            itemStatus = item.get("status")
            executingStatus = "1"
            failedStatus = "3"
            if itemStatus == self.uiObserver.toUiStatus(executingStatus):
                item["status"] = self.uiObserver.toUiStatus(failedStatus)
                item["suggession"] = baseUtil.getPyResource(
                    self.lang, "query.result.abnormal")

    def toAtomInfo(self, dataDict, rec):
        """ 获取检查项的原子信息 """
        if len(rec) <= 1:
            return (True, [])

        itemCount = Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckResult.ITEM_COUNT)
        checkItem = Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckResult.CHECK_ITEM)
        startTime = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        checkResult = Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckResult.CHECK_RESULT)
        supportSkip = Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckResult.SUPPORT_SKIP)  # 0(支持)，1（不支持）
        nodeDetails = Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckResult.NODE_DETAIL_INFO)
        extras = Tlv2Rest.getRecordValue(rec, restData.UpgradeCheck.GetCheckResult.CHECK_EXTRAS)
        if checkResult is not None and str(checkResult) == "0":
            # 等待中的原子不展示出来
            return (True, [])
        checkItemName = self.upgrade_res.get_res(checkItem)
        if not checkItemName or checkItemName == '--' or checkItemName == '':
            checkItemName = checkItem
        operate = 0
        supportManualIgnore = ManualConfigItemMgr.getInstance().expected(self.devIp, checkItem, "yes")
        if 1 == supportSkip or supportManualIgnore:
            operate = 7
        atomDictList = []
        detail = baseUtil.getPyResource(self.lang, "SystemInnerCheck.checkitem.detail.title")

        for node, nodeDetail in nodeDetails.items():
            # A800场景，有zone_id，节点信息中不包含ZONE则跳过。
            if self.is_a800_scene and self.zone_id and "ZONE{}".format(self.zone_id) not in node:
                continue
            if isinstance(nodeDetail, dict):
                errMsg = self._get_err_msg(nodeDetail)
                detail = self.get_detail_msg(detail, errMsg, node)
            else:
                nodeStatus = self.upgrade_res.get_res(UPD_CHECK_STATUS_2_STR.get(nodeDetail))
                # 控制器0A：通过
                # 控制器0B：不通过
                detail = self.get_detail_msg(detail, nodeStatus, node)
        if self.con_add_suggestion_to_detail() and str(checkResult) in FAIL_STATUS_TUPLE:
            # 除了升级评估，如果检查项不通过，需要将修复建议增加到检查结果中。
            self.logger.info("check not passed. itemId=%s" % checkItem)
            detail += self.get_check_item_detail(checkItem)

        item_msg = self.get_item(checkItem, checkItemName, checkResult, detail, operate, startTime)
        if extras:
            self.logger.info("accept extras={}, type={}".format(extras, type(extras)))
            item_msg["extras"], host_eval_result_info = self._extras_host_check(extras)
            if host_eval_result_info:
                item_msg["suggession"] += "\n" + host_eval_result_info

        atomDictList.append(item_msg)

        return (True, atomDictList)

    def get_item(self, check_item, check_item_name, check_result, detail, operate, start_time):
        return {
            "id": 0,
            "nameId": check_item,
            # 检查项名称，类型String
            "name": check_item_name,
            "startTime": start_time,
            # 0等待检查, 1正在检查, 2警告, 3通过 ,4不通过 # 转换为界面可显示的状态
            "status": self.uiObserver.toUiStatus(str(check_result)),
            "ctrl": "ALL",
            # 修复建议
            "suggession": detail,
            # 检查项暂停可进行的操作,7代表暂停+忽略+终止（不包含重试）
            "operate": operate,
            "itemType": "innercheck"
        }

    def get_check_item_detail(self, check_item):
        method = self.upgrade_res.get_res(check_item + ".method")
        method_info = baseUtil.getPyResource(self.lang, "checkitem.method.title", method)
        criterion = self.upgrade_res.get_res(check_item + ".criterion")
        criterion_info = baseUtil.getPyResource(self.lang, "checkitem.criterion.title", criterion)
        sugesstion = self.upgrade_res.get_res(check_item + ".suggestion")
        sugesstion_info = baseUtil.getPyResource(self.lang, "checkitem.suggestion.title", sugesstion)
        return "{}{}{}".format(method_info, criterion_info, sugesstion_info)

    def con_add_suggestion_to_detail(self):
        return "evaluate" not in self.checkType.lower() and "inspection" not in self.checkType.lower()

    def get_detail_msg(self, detail, errMsg, node):
        """
        提示信息区分ip框的级联板及控制器

        :param detail: 详情
        :param errMsg: 错误信息
        :param node: 节点信息
        :return: 详情提示。
        """
        if "DAE" in node:
            self.logger.info("SystemInnerCheck expboard detail node : {}".format(node))
            detail += baseUtil.getPyResource(self.lang, "SystemInnerCheck.checkitem.expboard.detail", (node, errMsg))
        else:
            detail += baseUtil.getPyResource(self.lang, "SystemInnerCheck.checkitem.detail", (node, errMsg))
        return detail

    def _get_err_msg(self, node_detail):
        """
        获取控制器报错信息
        数据举例：
        {u'errcode': u'0x01B5000b|0x01B50005', u'params': [u'1', u'3|QLE2672|7.03.00|3.0.1.0']}     # noqa

        :param node_detail: 节点详细信息
        :return: 解析后的错误信息
        """
        errcode = (node_detail.get("errcode") or "").lower()
        params = node_detail.get("params") or []

        err_code_list = errcode.split("|")
        if isinstance(params, list):
            param_list = params
        elif isinstance(params, basestring):  # noqa
            param_list = [params]
        else:
            param_list = []

        err_msg_list = []
        for index, err_code in enumerate(err_code_list):
            if not err_code:
                continue
            err_msg = self.upgrade_res.get_res("ErrorCode." + err_code)
            if self.is_json_params(err_msg):
                err_msg_list.append(self.parse_json_params(err_msg, params))
            elif len(err_code_list) == 1 and len(param_list) > 1:
                err_msg_list.append(
                    self.parse_err_msg_with_params(err_msg, param_list)
                )
                break
            elif index < len(param_list):
                err_msg_list.append(
                    self.parse_err_msg_with_params(err_msg, param_list[index])
                )
            else:
                err_msg_list.append(err_msg)

        if err_msg_list:
            self.logger.info("node_detail:{} {}".format(
                type(node_detail), node_detail))
            self.logger.info("err_msg_list:{}".format(err_msg_list))

        if len(err_msg_list) <= 1:
            return "".join(err_msg_list)
        return "\n" + "\n".join(
            ["{}.{}".format(n + 1, i) for n, i in enumerate(err_msg_list)]
        )

    def parse_err_msg_with_params(self, error_msg, params):
        """
        解析错误信息
        :param error_msg:错误消息
        :param params:错误参数列表
        :return:
        """
        self.logger.info(
            "parse_err_msg_with_params params:{}, error_msg:{}".format(
                params, error_msg)
        )
        if isinstance(params, basestring):  # noqa
            param_list = params.split("|")
        elif isinstance(params, list):
            param_list = params
        else:
            param_list = []
        try:
            return error_msg % tuple(param_list)
        except Exception:
            self.logger.error("parse error! params:{},error_msg:{}".format(
                params, error_msg))
            return "{}\n{}".format(error_msg, params)

    def is_json_params(self, error_msg):
        """ 判断返回错误参数是否为json串

        :param error_msg: 错误消息
        :return:
        """
        key_match_obj = re.search(r"{\d+:key}", error_msg)
        value_match_obj = re.search(r"{\d+:value}", error_msg)
        return bool(key_match_obj and value_match_obj)

    def parse_json_params(self, error_msg, params):
        """解析错误参数中的key和value，替换错误消息中的对应字段

        :param error_msg:错误消息
        :param params:错误参数列表
        :return:
        """
        self.logger.info("parse_json_params params: %s" % str(params))
        error_msgs = []
        for param in params:
            param = ast.literal_eval(param)
            param_id = param.get("id")
            param_dict = param.get("value")
            for (key, value) in param_dict.items():
                key_str = "{%s:key}" % str(param_id)
                value_str = "{%s:value}" % str(param_id)
                temp_msg = \
                    error_msg.replace(key_str, key).replace(value_str, value)
                error_msgs.append(temp_msg)
        ret_msg = "\n" + "\n".join(error_msgs)
        return ret_msg

    def _extras_host_check(self, extras):
        """
        主机收集信息检查结果解析
        数据举例：
        {u'0A': u'{"hostCompaEvaluCheck": "True", "not_pass_hosts": "[]", "pass_hosts": "[]", "can_not_check_hosts": "[0, 1, 2, 3, 5]"}'}   # noqa
        :param extras: 获取的评估结果数据
        :return: （解析的数据更新, 报告展示的主机检查结果信息）
        """
        if not isinstance(extras, dict):
            return extras
        host_check_result_info = ""
        for key, value in extras.items():
            self.logger.info("_extras_host_check value:{} type:{}".format(
                value, type(value)))
            if isinstance(value, basestring) and Extras.can_not_check_hosts in value:  # noqa
                host_check_result_info = self._parse_host_check_detail(value)
                checked_value = self._is_disabled_host_check(value)
                extras[key] = checked_value
        return extras, host_check_result_info

    def _parse_host_check_detail(self, value):
        detail_list = []
        try:
            data = json.loads(value)
            key = Extras.pass_hosts
            val = data[key]
            if not isinstance(val, basestring):  # noqa
                val = str(val)
            host_id_list = list(set(ast.literal_eval(val)))
            if host_id_list:
                detail_list.append(
                    baseUtil.getPyResource(
                        self.lang,
                        "host_evalu_result.pass_hosts",
                        ",".join(map(str, host_id_list))
                    )
                )
        except Exception:
            self.logger.error(
                "host_check parse error! value:{}\n{}".format(
                    value,
                    traceback.format_exc()
                )
            )
        return "\n" + "\n".join(detail_list)

    def _is_disabled_host_check(self, value):
        """
        解析数据，判断是否需要灰化主机兼容性检查，并对数据进行标注
        1.数据格式化：去掉"[","]"和" "
        2.添加disabled字段
        :param value:
        :return:
        """
        try:
            data = json.loads(value)
            for key in HOST_EVAL_RESULT_KEYS:
                val = data[key]
                if not isinstance(val, basestring):  # noqa
                    data[key] = str(val)
                data[key] = val.replace(" ", "").replace(
                    "[", "").replace("]", "")
            if not data[Extras.can_not_check_hosts].strip():
                data["disabled"] = True
            return json.dumps(data)
        except Exception:
            self.logger.error("host_check error! value:{}\n{}".format(
                value, traceback.format_exc()))

        return value


class RestService():
    def __init__(self, dataDict):
        self.dataDict = dataDict
        self.lang = dataDict.get("lang")
        self.logger = dataDict.get("logger")
        self.devIp = contextUtil.getIp(dataDict)
        self.dev_version = contextUtil.getCurVersion(dataDict)
        self.dev_type = str(contextUtil.get_dev_type(self.dataDict)).replace("_container", "")
        self.queryFailedCnt = 0

    # 59594907761
    def excuteCheck(self, checkType, ignorePreCheckItemsStr):
        rest = contextUtil.getRest(self.dataDict)
        params = []
        param64 = (restData.UpgradeCheck.ExecCheck.CHECK_TYPE, checkType)
        param65 = (restData.UpgradeCheck.ExecCheck.CHECK_CMD, restData.Enum.ExecCheckCmd.START)

        diskIdStr = self.dataDict.get("selectedDiskIdStr", "")
        if diskIdStr and diskIdStr != "":
            param12 = (restData.UpgradeCheck.ExecCheck.ASSOCIATE_OBJ_ID, diskIdStr)
            params.append(param12)

        flowchoiceNum = self.dataDict.get('flowchoice', 0)
        if flowchoiceNum == 2:
            param65 = (restData.UpgradeCheck.ExecCheck.CHECK_CMD, restData.Enum.ExecCheckCmd.RETRY)
        params.extend([param64, param65])
        if ignorePreCheckItemsStr:
            param66 = (restData.UpgradeCheck.ExecCheck.CHECK_ITEM_NAME, ignorePreCheckItemsStr)
            params.append(param66)
        extras = self.dataDict.get("extras")
        if extras:
            param67 = (restData.UpgradeCheck.ExecCheck.CHECK_EXTRAS, extras)
            params.append(param67)
        # 补丁评估和热补丁安装，下发目标补丁版本。
        if checkType in HOT_PATCH_CHECK_TYPES and self.dataDict.get("pkgVersion"):
            extras = {"targetVersion": str(self.dataDict.get("pkgVersion"))}
            param67 = (restData.UpgradeCheck.ExecCheck.CHECK_EXTRAS, extras)
            params.append(param67)
        zone_ids = self.dataDict.get("zone_ids", [])
        if zone_ids:
            param68 = (restData.UpgradeCheck.ExecCheck.CHECK_ZONES, zone_ids)
            params.append(param68)
        self.logger.info("exec upgrade check param:%s" % params)
        try:
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.EXEC_UPGRADE_CHECK, params)
            self.logger.info("exec upgrade check result:%s" % recs)
            if self.dataDict.get("isA800Scene", False):
                self.dataDict.get("zoneDataMap")[self.devIp] = "success"
        except Exception as e:
            self.logger.error("exec upgrade check except:%s" % str(e))
            if self.dataDict.get("isA800Scene", False):
                self.dataDict.get("zoneDataMap")[self.devIp] = "failed"
            raise UnCheckException(baseUtil.getPyResource(self.lang, "SystemInnerCheck.callcmd.error"))

    # 59594907778
    def queryCheckProgress(self):
        try:
            rest = contextUtil.getRest(self.dataDict)
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.QUERY_UPGRADE_CHECK_PROGRESS, [])
            self.queryFailedCnt = 0
            return recs
        except:
            self.logger.error("query check progress error.")
            self.queryFailedCnt += 1
            return None

    # 59594907765
    def queryCheckDetails(self):
        # 查询检查详细信息
        try:
            rest = contextUtil.getRest(self.dataDict)
            details = []
            if self.dev_type == "OceanStor A800":
                details = self.query_a800_details(rest)
            else:
                details = self.query_details(rest)
            self.queryFailedCnt = 0
            return details
        except:
            self.logger.error("query check details error.")
            self.queryFailedCnt += 1
            return []

    @staticmethod
    def query_details(rest):
        param15 = (restData.UpgradeCheck.GetCheckResult.QUERY_CONDITION, json.dumps({'range': '0-1000'}))
        return Tlv2Rest.execCmd(rest, restData.TlvCmd.QUERY_UPGRADE_CHECK_RESULT, params=[param15])

    @staticmethod
    def query_a800_details(rest):
        details = []
        for batch in range(201):
            start_index = batch * 5
            end_index = start_index + 5
            param15 = (restData.UpgradeCheck.GetCheckResult.QUERY_CONDITION,
                       json.dumps({'range': '{}-{}'.format(start_index, end_index)}))
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.QUERY_UPGRADE_CHECK_RESULT, params=[param15])
            details.extend(recs)
            if len(recs) < 5:
                break
        return details

    # 59594908477
    def execute_container_check(self, check_type, ignore_pre_check_items):
        """
        执行容器应用升级检查
        check_type:检查类型
        ignore_pre_check_items：忽略项
        """
        self.logger.info("execute container check start.")
        rest = contextUtil.getRest(self.dataDict)
        params = []
        # A8000 Tlv命令字传递参数
        param64 = (restData.UpgradeCheck.ContainerUpgradeCheck.CHECK_TYPE, check_type)
        params.append(param64)
        # checkCmd
        param65 = (restData.UpgradeCheck.ContainerUpgradeCheck.CHECK_CMD, restData.Enum.ExecCheckCmd.START)
        params.append(param65)
        if ignore_pre_check_items:
            # skipItem
            param66 = (restData.UpgradeCheck.ContainerUpgradeCheck.SKIP_ITEM, ignore_pre_check_items)
            params.append(param66)
        # 升级评估使用的当前容器版本。
        target_upg_app_version = self.dataDict.get("dev").getContainerVersion()
        # appName
        application_name = "DataBackup"
        if '1.2.1' in target_upg_app_version:
            application_name = "OceanProtect-DataProtect"
        param68 = (restData.UpgradeCheck.ContainerUpgradeCheck.APPLICATION_NAME, application_name)
        params.append(param68)
        self.logger.info("target_upg_app_version:%s" % target_upg_app_version)
        # targetVersion
        param69 = (restData.UpgradeCheck.ContainerUpgradeCheck.TARGET_VERSION, target_upg_app_version)
        params.append(param69)
        # arrayAccount
        param70 = (
            restData.UpgradeCheck.ContainerUpgradeCheck.ARRAY_ACCOUNT, contextUtil.getDevObj(self.dataDict).get("user")
        )
        # arrayPwd
        param71 = (
            restData.UpgradeCheck.ContainerUpgradeCheck.ARRAY_PASSWORD, contextUtil.getDevObj(self.dataDict).get("pawd")
        )
        params.extend([param70, param71])
        try:
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.EXEC_CTNR_APP_UPGRADE_CHECK, params)
            self.logger.info("exec upgrade check result:%s" % recs)
        except Exception as e:
            self.logger.error("exec upgrade check except:%s" % str(e))
            raise UnCheckException(baseUtil.getPyResource(self.lang, "SystemInnerCheck.callcmd.error"))

    # 59594907778
    def query_container_check_progress(self):
        """查询容器应用升级检查进度"""
        try:
            self.logger.info("query container check progress start.")
            rest = contextUtil.getRest(self.dataDict)
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.QUERY_CTNR_APP_UPGRADE_CHECK_DETAIL_PROGRESS, [])
            self.logger.info("exec upgrade check progress result:%s" % recs)
            self.queryFailedCnt = 0
            return recs
        except Exception:
            self.logger.error("query check progress error.")
            self.queryFailedCnt += 1
            return []

    # 59594907765
    def query_container_check_details(self):
        """查询容器应用升级检查结果"""
        try:
            self.logger.info("query container check detail start.")
            rest = contextUtil.getRest(self.dataDict)
            # queryCondition
            param15 = (restData.UpgradeCheck.GetCheckResult.QUERY_CONDITION, json.dumps({'range': '0-1000'}))
            recs = Tlv2Rest.execCmd(rest, restData.TlvCmd.QUERY_CTNR_APP_UPGRADE_CHECK_RESULT, [param15])
            self.logger.info("exec upgrade check details result:%s" % recs)
            self.queryFailedCnt = 0
            return recs
        except Exception:
            self.logger.error("query check details error.")
            self.queryFailedCnt += 1
            return []
