# -*- coding: UTF-8 -*-
from cbb.frame.checkitem.base_dsl_check import BaseCheckItem, CheckStatus
from cbb.frame.adapter import replace_adapter
from cbb.frame.dsl import fault_mode
from cbb.frame.ruleConfig.common import getMsg
from cbb.frame.dsl.adapter import NEED_NEW_CTRL_CONN


class AdDomainConnCheck(BaseCheckItem):
    TEST_FAIL_FLAG = "the server cannot be connected"
    TEST_NO_CONFIG_FLAG = "the dns server are not configured"
    TEST_PASS_FLAG = "command executed successfully."

    def __init__(self, tool_context):
        super(AdDomainConnCheck, self).__init__(tool_context)
        self.dev_version = ""
        self.patch_version = ""
        self.context[NEED_NEW_CTRL_CONN] = False

    def execute(self):
        if not self.is_risk_version():
            return CheckStatus.NO_SUPPORT, "Not a risk version."

        flag, msg = self.check_domain()
        if flag is not True:
            return CheckStatus.NOTPASS, msg

        vstore_ids = self.dsl(
            "exec_cli 'show vstore |filterColumn include "
            "columnList=ID' | horizontal_parser",
        )

        err_msg_list = []
        for vstore_info in vstore_ids:
            vstore_id = vstore_info.get("ID")
            self.dsl(
                "exec_cli 'change vstore view id={}'".format(vstore_id),
            )
            flag, err_msg = self.check_domain(vstore_id)
            if flag is not True:
                err_msg_list.append(err_msg)
            self.dsl(
                "exec_cli 'exit'"
            )
        if err_msg_list:
            return CheckStatus.NOTPASS, "".join(err_msg_list)

        return CheckStatus.PASS, "".join(err_msg_list)

    def is_risk_version(self):
        (
            flag,
            self.dev_version,
            self.patch_version,
        ) = replace_adapter.get_system_version(self.context)
        if any(
                [
                    self.dev_version.startswith("V300R006")
                    and self.dev_version <= "V300R006C61",
                    self.dev_version.startswith("V500R007")
                    and self.dev_version <= "V500R007C70SPC200",
                ]
        ):
            return True

        return False

    def check_domain(self, vstore_id=None):
        ad_info_dict = self.dsl(
            "exec_cli 'show domain ad' | vertical_parser",
        )
        if not ad_info_dict:
            return True, ""

        if ad_info_dict.get("Domain Status") != "Joined":
            return True, ""

        self.dsl(
            "exec_cli 'test domain dns'",
            return_if={fault_mode.FindException: ''}
        )
        if not self.check_ret(self.last_dsl.origin_info):
            err_key_default = "ad.domain.conn.check.default.fail"
            err_key_vstore = "ad.domain.conn.check.dns.fail"
            err_key = err_key_default if vstore_id is None else err_key_vstore
            return False, getMsg(self.lang, err_key, vstore_id)

        self.dsl(
            "exec_cli 'test domain ad'",
            return_if={fault_mode.FindException: ''}
        )

        if not self.check_ret(self.last_dsl.origin_info):
            err_key_default = "ad.domain.conn.check.default.fail"
            err_key_vstore = "ad.domain.conn.check.ad.domain.fail"
            err_key = err_key_default if vstore_id is None else err_key_vstore
            return False, getMsg(self.lang, err_key, vstore_id)

        return True, ""

    @staticmethod
    def check_ret(ret_list):
        ret = ''
        if ret_list:
            ret = ret_list[-1]
        ret = ret.lower()
        if any([
            AdDomainConnCheck.TEST_PASS_FLAG in ret,
            AdDomainConnCheck.TEST_NO_CONFIG_FLAG in ret
        ]):
            return True
        if AdDomainConnCheck.TEST_FAIL_FLAG in ret:
            return False

        raise Exception()
