#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2020~ All rights reserved.
import ast
import traceback
import json

# noinspection PyUnresolvedReferences
from com.huawei.ism.tool.obase.exception import ToolException
# noinspection PyUnresolvedReferences
from com.huawei.ism.tool.obase.connection import SftpTransporter

from cbb.business.collect.hostinfo import const
from cbb.common.query.software.host import get_host_links_of_storage
from cbb.common.query.software.host import (
    get_hosts_can_be_collected_via_storage,
)
from cbb.common.query.software.host import get_hyper_hosts
from cbb.common.query.software.host import (
    is_need_execute_compatibility_evaluation,
)
from cbb.frame.base.baseUtil import getString
from cbb.frame.context import contextUtil


class HostInfoCollectException(Exception):
    def __init__(self, err_code, log_msg, res_msg):
        """

        :param err_code:错误码
        :param log_msg:错误消息，日志记录。
        :param res_msg: 国际化错误消息。
        """
        self.error_code = err_code
        self.log_msg = log_msg
        self.res_msg = res_msg


def get_host_info_collect_request_url(params_dict, request_name):
    """根据请求资源获取请求 URL

    :param params_dict:
    :param request_name:
    :return:
    """
    # com.huawei.ism.tool.obase.entity.DevNode
    dev_node_obj = params_dict.get("devNode")
    dev_ip = params_dict.get("devIp")
    is_ipv6 = ":" in dev_ip
    return (
        "https://{ip}:8088/deviceManager/rest/{dev_sn}"
        "/HOST_INFO/{request_name}".format(
            ip="[{}]".format(dev_ip) if is_ipv6 else dev_ip,
            dev_sn=dev_node_obj.getDeviceSerialNumber(),
            request_name=request_name,
        )
    )


def send_rest_request_to_storage(
    params_dict, request_name, request_method="get", *args, **kwargs
):
    """向存储设备发送 Rest 请求。

    :param params_dict:
    :param request_name: 请求资源名称
    :param request_method: 请求方法，默认 get
    :param args: 请求参数
    :param kwargs: 请求参数
    :return:
    """
    request_func_dict = {"get": "execGet", "put": "execPut", "post": "execPost", "delete": "execDelete"}
    java_map = contextUtil.getContext(params_dict)
    rest_conn_wrapper = contextUtil.getRest(java_map)
    rest_connection = rest_conn_wrapper.getRest()
    rest_url = get_host_info_collect_request_url(params_dict, request_name)
    rest_connection_func = getattr(
        rest_connection, request_func_dict.get(request_method)
    )
    response_info = rest_connection_func(rest_url, json.dumps(kwargs))
    response_dict = ast.literal_eval(response_info.getContent())
    err_code = response_dict.get("error", {}).get("code", 0)
    err_desc = response_dict.get("error", {}).get("description", 0)
    resp_data = response_dict.get("data", {})

    err_msg_key_dict = {
        const.ERR_CODE_MEMORY_INSUFFICIENT: const.ERR_KEY_COLLECT_HOST_INFO_MEMORY_INSUFFICIENT_ERROR,
        const.ERR_CODE_TASK_CONFLICTING: const.ERR_KEY_COLLECT_HOST_INFO_TASK_CONFLICTING_ERROR,
    }
    if err_code != 0:
        err_msg_key = err_msg_key_dict.get(
            err_code, const.ERR_KEY_REST_REQUEST_ERROR
        )
        err_msg = getString(err_msg_key)
        raise HostInfoCollectException(err_code, err_desc, err_msg)
    return resp_data


def query_host_info_collect_script_upload_path(dev_obj, *args, **kwargs):
    """查询主机信息收集上传脚本路径。

    :param dev_obj:
    :param args:
    :param kwargs:
    :return:
    """
    try:
        resp_data = send_rest_request_to_storage(
            dev_obj, const.REST_CMD_GET_UPLOAD_PATH, *args, **kwargs
        )
        upload_path = resp_data.get("uploadDirectory", "")
        if not upload_path:
            raise HostInfoCollectException(
                0,
                "upload path empty",
                getString(const.ERR_KEY_COLLECT_HOST_INFO_GENERIC_ERROR),
            )
        return upload_path
    except HostInfoCollectException as hice:
        raise HostInfoCollectException(
            hice.error_code,
            hice.log_msg,
            getString(const.ERR_KEY_QUERY_SCRIPT_UPLOAD_PATH_FAILED),
        )


def query_host_info_collect_status(dev_obj):
    """查询主机信息收集进度。

    :param dev_obj:
    :return:
    """
    try:
        resp_data = send_rest_request_to_storage(
            dev_obj, const.REST_CMD_GET_HOST_INFO_COLLECTION_PROGRESS
        )
    except HostInfoCollectException as hice:
        raise HostInfoCollectException(
            hice.error_code, hice.log_msg, hice.res_msg
        )

    return resp_data.get("progress", "0"), resp_data.get("status", "0")


def query_host_info_collect_download_path(dev_obj):
    """查询主机信息收集结果下载路径。

    :param dev_obj:
    :return:
    """
    try:
        resp_data = send_rest_request_to_storage(
            dev_obj, const.REST_CMD_GET_DOWNLOAD_PKG
        )
    except HostInfoCollectException as hice:
        raise HostInfoCollectException(
            hice.error_code,
            hice.log_msg,
            getString(const.ERR_KEY_QUERY_PKG_DOWNLOAD_PATH_FAILED),
        )

    return resp_data.get("downLoadPackage", "")


def notify_storage_device_delete_host_info_pkg(
        params_dict, request_method="put", *args, **kwargs):
    """通知存储设备删除主机信息收集结果包。

    :param params_dict:
    :param request_method:
    :param args:
    :param kwargs:
    :return:
    """
    try:
        send_rest_request_to_storage(
            params_dict,
            const.REST_CMD_DELETE_PACKAGE,
            request_method=request_method,
            *args,
            **kwargs
        )
    except HostInfoCollectException as hice:
        raise HostInfoCollectException(
            hice.error_code, hice.log_msg, hice.res_msg
        )


def notify_storage_start_collect_host_info(
        dev_obj, request_method="put", *args, **kwargs):
    """通知存储设备开始收集主机信息。

    :param dev_obj:
    :param request_method:
    :param args:
    :param kwargs:
    :return:
    """

    try:
        send_rest_request_to_storage(
            dev_obj,
            const.REST_CMD_START_HOST_INFO_COLLECTION,
            request_method=request_method,
            *args,
            **kwargs
        )
    except HostInfoCollectException as hice:
        raise HostInfoCollectException(
            hice.error_code, hice.log_msg, hice.res_msg
        )


def is_storage_support_collect_host_info(params_dict):
    """判断存储设备是否支持免主机采集特性。

    :param params_dict:
    :return:
    """
    dev_obj = params_dict.get("devNode")
    sys_ver = dev_obj.getProductVersion()
    hotpatch_ver = dev_obj.getHotPatchVersion()

    logger = params_dict.get("logger")
    logger.info("sys_ver:{}, patch ver: {}".format(sys_ver, hotpatch_ver))
    return sys_ver.split()[
        0
    ].strip() >= const.SUPPORT_COLLECT_HOST_INFO_BY_STORAGE_SYS_START_VER or (
        sys_ver == const.SUPPORT_COLLECT_HOST_INFO_BY_SYS_C60SPC300
        and hotpatch_ver
        and hotpatch_ver
        >= const.SUPPORT_COLLECT_HOST_INFO_BY_STORAGE_C60SPH305
    )


def is_host_initiator_support_collect_info(params_dict):
    """通过查询启动器信息判断主机是否支持免主机采集（协议新接口）：
    存储设备对应的任一主机支持，则支持。

    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    # noinspection: PyBroadException
    try:
        host_sn_to_wwn_or_iqns_dict = get_hosts_can_be_collected_via_storage(
            params_dict
        )
        return len(host_sn_to_wwn_or_iqns_dict) > 0
    except:  # noqa E722
        logger.warn(
            "Query whether storage support host info collect failed,"
            " the system version may not be implemented"
        )
        return False


def is_host_support_collect_info_by_storage(params_dict):
    """判断主机是否支持免主机采集：存储设备对应的任一主机支持，则支持。

    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    all_host_links = get_host_links_of_storage(params_dict)
    logger.info("all_host_links:{}".format(all_host_links))
    ultrapath_vers = []
    for host_link_info in all_host_links:
        ultrapath_vers.append(host_link_info.get("ULTRAPATHVERSION"))

    logger.info("ultrapath_vers:{}".format(ultrapath_vers))

    # noinspection PyBroadException
    try:
        support_ultrapath_ver = list(
            filter(
                lambda ver: ver
                and ver.strip()
                and tuple(map(int, ver.strip().split(".")))
                >= const.SUPPORT_COLLECT_HOSTINFO_BY_STOR_ULTRAPATH_START_VER,
                ultrapath_vers,
            )
        )

        logger.info("support_ultrapath_ver:{}".format(support_ultrapath_ver))
        return len(support_ultrapath_ver) > 0

    except BaseException as be:
        logger.error("judge ultrapath version exception:{}".format(be))
        logger.error(
            "judge ultrapath version exception:{}".format(
                traceback.format_exc()
            )
        )
        return False


def is_support_collect_host_info(params_dict):
    """
    判断是否支持收集主机信息。判断条件如下：
    1. V5R7C70、C61 和6.1版本；直接通过启动器查询接口查询是否支持收集；
    2. V5R6C60SPC300 版本，如果未打补丁，不支持收集；如果已打补丁；
        通过启动器查询接口查询是否支持收集；

    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    dev_obj = params_dict.get("devNode")
    sys_ver = dev_obj.getProductVersion().split()[0]
    is_patch_ver = sys_ver == const.SUPPORT_COLLECT_HOST_INFO_BY_SYS_C60SPC300
    if is_patch_ver and not dev_obj.getHotPatchVersion():
        logger.info(
            "Is V500R007C60SPC300 but not installed patch, " "not support."
        )
        params_dict["not_support"] = True
        params_dict["py_detail"] = getString(
            const.ERR_KEY_NEED_INSTALL_PATCH_TO_COLLECT
        )
        return False

    if not is_support_collect_host_via_storage(params_dict):
        logger.info("Not support collect host info by storage.")
        params_dict["not_support"] = True
        params_dict["py_detail"] = getString(const.ERR_KEY_NO_NEED_COLLECT)
        return False

    return True


def is_need_collect_host_info(params_dict):
    """判断是否应该收集主机信息，满足以下两个条件之一则需要收集：
    1. 需要执行兼容性评估。

    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    try:
        is_need_execute_comp_eval = is_need_execute_compatibility_evaluation(
            params_dict
        )
        return is_need_execute_comp_eval
    except Exception as e:
        logger.error(
            "Query where should display host info collect module "
            "exception:{}".format(e)
        )
        logger.error("Traceback:{}".format(traceback.format_exc()))
        return False


def check_conn_is_alive(params_dict, logger):
    """
    检查连接是否断开，并重连
    :param params_dict:
    :param logger:
    :return:
    """
    cmd = "show system general"
    ssh = params_dict["SSH"]
    try:
        ssh.execCmd(cmd)
    except(ToolException, Exception) as e:
        logger.error("conn is not alive: {}".format(e))
        reconnect_conn(params_dict, logger)


def reconnect_conn(params_dict, logger):
    """
    重连ssh 和 sftp
    :param params_dict:
    :param logger:
    :return:
    """
    try:
        ssh = params_dict.get("SSH")
        ssh.reConnect()
        sftp = params_dict.get("SFTP")
        if sftp is not None:
            sftp.close()
        sftp = SftpTransporter(ssh)
        params_dict["SFTP"] = sftp
    except(ToolException, Exception) as e:
        logger.error("reconnect error: {}".format(e))


def is_support_collect_host_via_storage(params_dict):
    """判断是否可以通过免主机采集主机信息

    :param params_dict:
    :return:
    """
    return is_host_initiator_support_collect_info(params_dict)
