# coding:utf-8
""" 
@version: Toolkit V200R005C00
@time: 2018/06/26 
@file: errorMsgMgr.py 
@function: 
@modify: 
"""
import os
import traceback

# noinspection PyUnresolvedReferences
from java.util.regex import Pattern

from cbb.business.operate.expansion import common

try:
    import defusedxml.cElementTree as ET
except Exception:
    import defusedxml.ElementTree as ET


class ErrorCodeMgr(object):
    """错误码管理类。错误码的含义及对应修复建议从阵列配置文件（本地保存，后续可以考虑实时从阵列获取）中解析。

    注意：1）标准错误码（阵列配置文件中配置的）解析为正整数，非字符串。
          2）错误原因中的参数信息（括号中的内容，包括括号）需要净化掉。
          3）支持自定义错误码：字符串类型，通过调common.getMsg(errMsgKey)获取。

    """

    def __init__(self, rootPath, lang='zh', logger=None):
        self.lang = lang
        self.logger = logger
        self.errMsgFile = os.path.join(rootPath, 'res', 'errorMsg', 'error_%(lang)s.xml' % {'lang': lang})
        self.errMsgDict = {}
        self.duplicateErrCodeSet = set()
        self.errorReasonParaRegPtr = Pattern.compile(r'\(.*##.*\)' if lang == 'en' else u'（.*##.*）')
        self.parseErrMsgRes(logger=logger)

    def get(self, errCode):
        if errCode == 0:
            return '', ''

        if errCode not in self.errMsgDict:  # 错误码文件中无定义，找自定义错误码。
            errMsg, suggest = common.getMsg(self.lang, errCode)
        else:
            errMsg, suggest = self.errMsgDict.get(errCode, ('', ''))

        if not errMsg or not suggest:
            dftErrMsg = u"操作失败。" if self.lang == 'zh' else 'The operation fails.'
            dftSuggest = u'请重试，如果问题仍然存在请联系技术支持工程师。' if self.lang == 'zh' \
                else 'Try again. If the problem persists, contact technical support engineers.'

            errMsg, suggest = dftErrMsg, dftSuggest
        return errMsg, suggest

    def parseErrMsgRes(self, logger=None):
        # noinspection PyUnusedLocal,PyUnusedLocal
        try:
            tree = ET.parse(self.errMsgFile)  # 打开xml文档
            root = tree.getroot()

            for errorsTree in root.findall("errors"):
                erorrTree = errorsTree.findall('error')
                for erorrNode in erorrTree:
                    errorCode = int(erorrNode.attrib["id"])
                    reason, suggestion = '', ''
                    for reasonNode in erorrNode.findall('reason'):
                        reason = reasonNode.attrib['description']
                        reason = self.errorReasonParaRegPtr.matcher(reason).replaceAll("")  # 使用Java正则规避jython正则引擎bug

                    for suggestNode in erorrNode.findall('suggestions/suggestion'):
                        suggestion = suggestNode.attrib['description']

                    if errorCode in self.errMsgDict:
                        self.duplicateErrCodeSet.add(errorCode)

                    self.errMsgDict[errorCode] = (reason, suggestion)
        except Exception as e:
            if logger:
                logger.logInfo('Parse error message file:%s exception:%s' % (self.errMsgFile, e))
                logger.logInfo('traceback:%s' % traceback.format_exc())
            return {}
        else:
            if logger:
                logger.logInfo('Duplicate error code set:%s' % str(self.duplicateErrCodeSet))
                logger.logInfo('Parse error dict:%s' % str(self.errMsgDict))
