# -*- coding: UTF-8 -*-
'''
140327-Question：文件引用问题
Description： 文件之间不能相互引用，否则调用函数时会出现未定义函数的问题。
        如下面，若TestFactory中引用了FuncFactory，此处又引用TestFactory，将会导致在TestFactory中访问FuncFactory中的函数时出现异常。
Solution: 排查引用关系，保证所有文件之间均为单向引用。（包含所有文件中不引用TestFactory）
Status: 已解决
'''

import ast
import os
import re
import subprocess
import time

import cbb.business.operate.fru.common.BaseFactory as BaseFactory
import cbb.business.operate.fru.common.SftpFactory as SftpFactory
import cbb.business.operate.fru.common.resource as resource
from cbb.business.operate.fru.common import config as conf
from cbb.business.operate.fru.common.checkItem import checkEncTemperature
from cbb.common.conf.productConfig import ENC_TEMPERATURE_ALM_IDS
from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil
from cbb.frame.util import ability_map_util
from cbb.common.conf.productConfig import AlarmLevel
from cbb.frame.rest.restUtil import CommonRest

import java.lang.Exception as jException

########################################基础功能定义区########################################
__________TEST_ZONE__________ = None

__________ZONE_PUBLIC_VARIABLE__________ = None

# context中更换指导链接生成所需的FRU类型
PERSIST_GUIDE_FRU_TYPE = "ReplaceFruTypeForGuide"

# context中更换指导链接生成所需的额外参数
PERSIST_GUIDE_EXTRA_PARAM = "ReplaceExtraParamForGuide"

# context中更换指导链接生成所需的系统产品类型
PERSIST_GUIDE_PRODUCT_MODEL = "SysProductModelForGuide"

# context中更换指导链接生成所需的框信息
PERSIST_GUIDE_PARENT_ENC_TLVREC = "ReplaceFruParentEncTlvRecordForGuide"

PERSIST_PRODUCTMODEL_ISNEEDDRAWGRAPH = "is18000V3UnitReplacedForGraph"

# context中更换指导链接生成所需的设备控制框类型
PERSIST_GUIDE_ENGINE_MODEL = "EngineModle"

# context中集群中已连接的SAS端口
PERSIST_LINKED_SAS_PORT = "LINKED_SAS_PORT"

# context中集群中在线的BBU
PERSIST_ONLINE_BBU = "ONLINE_BBU"

# context中集群中运行的风扇
PERSIST_RUNNING_FAN = "RUNNING_FAN"

# context中集群中连接的管理网口
PERSIST_LINKUP_MGT_PORT = "LINKUP_MGT_PORT"

# context中集群中连接的电源
PERSIST_ONLINE_POWER = "ONLINE_POWER"

# context中集群中连接的电源
PERSIST_LINKED_PCIE_PORT = "LINKED_PCIE_PORT"

# context中批量的更换风扇
PERSIST_MULTI_REPLACE_FAN = "MULTI_REPLACE_FAN"

PORT_TYPES = (restData.Enum.ObjEnum.ETH_PORT,
              restData.Enum.ObjEnum.FC_PORT,
              restData.Enum.ObjEnum.SAS_PORT)


# 线缆更换指导生成时需用到的线缆类型
class GUIDE_CABLE_TYPE():
    CABLE_UNKNOWN = "CABLE_UNKNOWN"
    CABLE_FRONT_END = "CABLE_FRONT_END"
    CABLE_MGMT = "CABLE_MGMT"
    CABLE_PCIE = "CABLE_PCIE"
    CABLE_SAS = "CABLE_SAS"
    CABLE_IP_SCALEOUT = "CABLE_IP_SCALEOUT"


# 线缆更换指导生成时需用到的线缆信息
class Guide_Cable_Info():
    cableType = GUIDE_CABLE_TYPE.CABLE_UNKNOWN
    portType = None
    intfModel = None
    otherParam = None
    subPortType = None

    def __init__(self, _cableType, _portType, _intfModel=None, _otherParam=None):
        self.cableType = _cableType
        self.portType = _portType
        self.intfModel = _intfModel
        self.otherParam = _otherParam


'''
前端接口卡模型编号列表
'''
g_frontEndIntfModels = (  # 所有的前端接口model
    1,  # 4x4G FC接口模块
    2,  # 2x4G FC接口模块
    3,  # 2xGE 电接口模块
    5,  # Swapped FC接口模块
    6,  # 2x10GE 光接口模块
    7,  # 1x8G FC光接口模块
    8,  # 2x8G FC光接口模块
    11,  # 2xFC+2xGE 接口模块
    12,  # 4xGE 电接口模块
    13,  # 4x8G FC光接口模块
    21,  # 4x10G FCoE光接口模块
    25,  # 4x10GE接口模块
    27,  # 2x10G FCoE光接口模块
    28,  # 4xGE光接口模块
    29,  # 2x16G FC光接口模块
    31,  # 4x10GE 电接口模块
    32,  # 2x40GE IB接口模块
    33,  # 2x56GE IB接口模块
    34,  # 4xGE 电接口模块
    36,  # 4x10GE 电接口模块
    37,  # 4xSmart 光接口模块
    38,  # 8x8G FC光接口模块
    39,  # 4x8G FC光接口卡
    40,  # 4x16G FC光接口卡
    58,  # 8x16G FC光接口卡
    65,  # 4 端口 SmartIO2.0 I/O模块（光口SFP+）X16",
    66,  # 4 端口 SmartIO2.0 I/O模块（光口SFP+）X8",
    67,  # 2 端口 100Gb ETH I/O模块（QSFP28）X16",
    68,  # 2 端口 100Gb ETH I/O模块（QSFP28）X8",
    71,  # 4端口 10Gb ETH I/O模块",
)
g_strFrontEndIntfModelList = [str(x) for x in g_frontEndIntfModels]

# 所有的后端接口model
g_backEndIntfModels = (  # 所有的后端接口model
    4,  # 4xSAS I 接口模块
    9,  # 2xSAS II电接口模块
    10,  # 4xSAS II电接口模块
    16,  # 4xmini-SAS接口模块
    22,  # 2x6G SAS接口模块
    23,  # 2x6G SAS接口模块
    30,  # 4x12G SAS接口模块
    41,  # 12端口4*12Gb SAS后端全互联IO模块
    55,  # 2接口PCIe接口模块
)
g_strBackEndIntfModelList = [str(x) for x in g_backEndIntfModels]

# 盘控一体控制框产品型号列表
g_diskCntrBundleEncProductModels = (
    restData.Enum.ProductModeEnum.V3_5300,  # V3
    restData.Enum.ProductModeEnum.V3_5500,
    restData.Enum.ProductModeEnum.V3_2600,
    restData.Enum.ProductModeEnum.V3_2600ForVideo,
    restData.Enum.ProductModeEnum.V3_5500F,
    restData.Enum.ProductModeEnum.V3_2600F,
    restData.Enum.ProductModeEnum.V3_2800,
    restData.Enum.ProductModeEnum.V3_2100,
    restData.Enum.ProductModeEnum.V3_2200,
    restData.Enum.ProductModeEnum.V3_5100,
    restData.Enum.ProductModeEnum.Dorado3000_V3,
    restData.Enum.ProductModeEnum.Dorado5000_V3,
    restData.Enum.ProductModeEnum.V5_2800,
    restData.Enum.ProductModeEnum.V5_5110,
    restData.Enum.ProductModeEnum.V5_5110F,
    restData.Enum.ProductModeEnum.V5_5210,
    restData.Enum.ProductModeEnum.V5_5210F,
    restData.Enum.ProductModeEnum.V5_5300,
    restData.Enum.ProductModeEnum.V5_5300F,
    restData.Enum.ProductModeEnum.V5_5500,
    restData.Enum.ProductModeEnum.DoradoNAS,
    restData.Enum.ProductModeEnum.V5_5500F,
    restData.Enum.ProductModeEnum.V5_5500_Elite,
    restData.Enum.ProductModeEnum.V3_2200_Enhanced,
    restData.Enum.ProductModeEnum.V3_2600_Enhanced,
    restData.Enum.ProductModeEnum.V3_2600F_Enhanced,
    restData.Enum.ProductModeEnum.V5_5300_Enhanced,
    restData.Enum.ProductModeEnum.V5_2810,
    restData.Enum.ProductModeEnum.V5_2810_C72,
    restData.Enum.ProductModeEnum.V5_2210,
    restData.Enum.ProductModeEnum.V5_2220,
    restData.Enum.ProductModeEnum.V5_5110,
    restData.Enum.ProductModeEnum.V5_5110F,
    restData.Enum.ProductModeEnum.V5_5210,
    restData.Enum.ProductModeEnum.V5_5210F,
    restData.Enum.ProductModeEnum.V5_5310,
    restData.Enum.ProductModeEnum.V5_5310F,
    restData.Enum.ProductModeEnum.V5_5510,
    restData.Enum.ProductModeEnum.V5_5510F,
    restData.Enum.ProductModeEnum.V5_5610,
    restData.Enum.ProductModeEnum.V5_5610F,
    restData.Enum.ProductModeEnum.V5_5810,
    restData.Enum.ProductModeEnum.V5_5810F,
    restData.Enum.ProductModeEnum.Dorado3000_V6_SAS,
    restData.Enum.ProductModeEnum.Dorado3000_SAS,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_2000,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_2020,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_2100,
    restData.Enum.ProductModeEnum.Dorado3000_V6_IPSAS,
    restData.Enum.ProductModeEnum.Dorado5000_V6_SAS,
    restData.Enum.ProductModeEnum.Dorado5000_V6_IPSAS,
    restData.Enum.ProductModeEnum.Dorado5000_V6_NVMe,
    restData.Enum.ProductModeEnum.Dorado5000_SAS,
    restData.Enum.ProductModeEnum.Dorado5000_NVMe,
    restData.Enum.ProductModeEnum.Dorado5000_IPSAS,
    restData.Enum.ProductModeEnum.Dorado6000_V6_SAS,
    restData.Enum.ProductModeEnum.Dorado6000_V6_IPSAS,
    restData.Enum.ProductModeEnum.Dorado6000_V6_NVMe,
    restData.Enum.ProductModeEnum.Dorado6000_SAS,
    restData.Enum.ProductModeEnum.Dorado6000_NVMe,
    restData.Enum.ProductModeEnum.Dorado6000_IPSAS,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5300_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5300,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5500_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5500,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5600_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5600,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5600K_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5600K,
    restData.Enum.ProductModeEnum.OceanStor_2200,
    restData.Enum.ProductModeEnum.OceanStor_2220,
    restData.Enum.ProductModeEnum.OceanStor_5120,
    restData.Enum.ProductModeEnum.OceanStor_2600,
    restData.Enum.ProductModeEnum.OceanStor_2620,
    restData.Enum.ProductModeEnum.OceanStor_5210,
    restData.Enum.ProductModeEnum.OceanStor_5220,
    restData.Enum.ProductModeEnum.OceanStor_5310,
    restData.Enum.ProductModeEnum.OceanStor_5310_Capacity_Flash,
    restData.Enum.ProductModeEnum.OceanStor_5510_Capacity_Flash,
    restData.Enum.ProductModeEnum.OceanStor_A300,
    restData.Enum.ProductModeEnum.OceanStor_A800,
    restData.Enum.ProductModeEnum.OceanStor_5320,
    restData.Enum.ProductModeEnum.OceanStor_5510,
    restData.Enum.ProductModeEnum.OceanStor_5510S,
    restData.Enum.ProductModeEnum.OceanStor_5610,
    restData.Enum.ProductModeEnum.OceanStor_5810HS,
    restData.Enum.ProductModeEnum.OceanStor_5300K,
    restData.Enum.ProductModeEnum.OceanStor_5500K,
    restData.Enum.ProductModeEnum.V5_5210_Enhanced,
    restData.Enum.ProductModeEnum.V5_5220,
    restData.Enum.ProductModeEnum.V5_5210F_Enhanced,
    restData.Enum.ProductModeEnum.V5_5110F_Enhanced,
    restData.Enum.ProductModeEnum.V5_5110_Enhanced,
    restData.Enum.ProductModeEnum.V5_5120,
    restData.Enum.ProductModeEnum.V5_2600,
    restData.Enum.ProductModeEnum.OceanProtect_A8000,
    restData.Enum.ProductModeEnum.OceanProtect_X3000,
    restData.Enum.ProductModeEnum.OceanProtect_X8000,
    restData.Enum.ProductModeEnum.OceanProtect_E8000,
    restData.Enum.ProductModeEnum.OceanProtect_X6000,
    restData.Enum.ProductModeEnum.OceanProtect_X8000K,
    restData.Enum.ProductModeEnum.OceanStor_Micro_1300,
    restData.Enum.ProductModeEnum.OceanStor_Micro_1500,
    restData.Enum.ProductModeEnum.OceanDisk_1300,
    restData.Enum.ProductModeEnum.OceanDisk_1500,
    restData.Enum.ProductModeEnum.OceanDisk_1600,
    restData.Enum.ProductModeEnum.OceanDisk_1500T,
    restData.Enum.ProductModeEnum.OceanDisk_1600T,
    restData.Enum.ProductModeEnum.OceanDisk_1610,
    restData.Enum.ProductModeEnum.OceanDisk_1610T,
    restData.Enum.ProductModeEnum.OceanStor_2910,
)
g_strDiskCntrBundleEncProductModels = [str(x) for x in g_diskCntrBundleEncProductModels]

# 无盘独立机头控制框产品型号列表
g_noDiskCntrEncProductModels = (
    restData.Enum.ProductModeEnum.V3_5600,  # V3
    restData.Enum.ProductModeEnum.V3_5800,
    restData.Enum.ProductModeEnum.V3_6800,
    restData.Enum.ProductModeEnum.V3_5600F,  # V3
    restData.Enum.ProductModeEnum.V3_5800F,
    restData.Enum.ProductModeEnum.V3_6800F,
    restData.Enum.ProductModeEnum.Dorado6000_V3,
    restData.Enum.ProductModeEnum.Dorado18000_V3,
    restData.Enum.ProductModeEnum.V3_18500,
    restData.Enum.ProductModeEnum.V3_18800,
    restData.Enum.ProductModeEnum.V3_18500F,
    restData.Enum.ProductModeEnum.V3_18800F,
    restData.Enum.ProductModeEnum.V5_5600,
    restData.Enum.ProductModeEnum.V5_5600F,
    restData.Enum.ProductModeEnum.V5_5800,
    restData.Enum.ProductModeEnum.V5_5800F,
    restData.Enum.ProductModeEnum.V5_6800,
    restData.Enum.ProductModeEnum.V5_6800F,
    restData.Enum.ProductModeEnum.V5_18500,
    restData.Enum.ProductModeEnum.V5_18500F,
    restData.Enum.ProductModeEnum.V5_18800,
    restData.Enum.ProductModeEnum.V5_18800F,

    restData.Enum.ProductModeEnum.V5_6810,
    restData.Enum.ProductModeEnum.V5_6810F,
    restData.Enum.ProductModeEnum.V5_18510,
    restData.Enum.ProductModeEnum.V5_18510F,
    restData.Enum.ProductModeEnum.V5_18810,
    restData.Enum.ProductModeEnum.V5_18810F,
    restData.Enum.ProductModeEnum.Dorado8000_V6_IPSAS,
    restData.Enum.ProductModeEnum.Dorado8000_V6_SAS,
    restData.Enum.ProductModeEnum.Dorado8000_V6_NVMe,
    restData.Enum.ProductModeEnum.Dorado8000_SAS,
    restData.Enum.ProductModeEnum.Dorado8000_NVMe,
    restData.Enum.ProductModeEnum.Dorado8000_IPSAS,
    restData.Enum.ProductModeEnum.Dorado18000_V6_IPSAS,
    restData.Enum.ProductModeEnum.Dorado18000_V6_SAS,
    restData.Enum.ProductModeEnum.Dorado18000_V6_NVMe,
    restData.Enum.ProductModeEnum.Dorado18000_SAS,
    restData.Enum.ProductModeEnum.Dorado18000_NVMe,
    restData.Enum.ProductModeEnum.Dorado18000_IPSAS,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_5800_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_6800_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_6800,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_18500_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_18500,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_18800_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_18800,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_18800K_V6,
    restData.Enum.ProductModeEnum.OceanStor_Dorado_18800K,
    restData.Enum.ProductModeEnum.OceanStor_6810,
    restData.Enum.ProductModeEnum.OceanStor_18510,
    restData.Enum.ProductModeEnum.OceanStor_18810,
    restData.Enum.ProductModeEnum.OceanStor_18500K,
    restData.Enum.ProductModeEnum.OceanProtect_X9000,
    restData.Enum.ProductModeEnum.OceanProtect_A9000,
    restData.Enum.ProductModeEnum.OceanProtect_X9000K,
)
g_strNoDiskCntrEncProductModels = [str(x) for x in g_noDiskCntrEncProductModels]


class FRUCONST():
    INVALID_PORT_ID = "4294967295"  # 0xFFFFFFFF

    # IP Scale-out端口连接正确状态值
    IP_SCALE_OUT_PORT_LINK_RIGHT = "1"
    IP_SCALE_OUT_PORT_LINK_WRONG = "0"
    IP_SCALE_OUT_PORT_LINK_INVALID = "4294967295"  # 0xFFFFFFFF

    # 所在端口是否位于交换机
    IP_SCALE_OUT_PORT_IS_SWITCH_YES = 1
    IP_SCALE_OUT_PORT_IS_SWITCH_NO = 0

    # IP Scale-out集群中正常管理路径的最少数量
    IP_SCALE_OUT_MGMT_CABLE_MIN_NUM = 2

    # 板载端口标记，用于线缆更换指导的生成
    PORT_PARENT_IS_CNTR = "PORT_PARENT_IS_CNTR"

    # 虚拟器件类型定义（如IP scale-out数据交换机）
    VIRTUAL_FRU_TYPE = "VIRTUAL_FRU_TYPE"
    VIRTUAL_FRU_NAME_IP_SCALE_OUT_SWITCH = "VIRTUAL_FRU_NAME_IP_SCALE_OUT_SWITCH"

    # 管理端口的逻辑类型
    MGMT_PORT_MGMT_LOGIC_TYPE_LIST = (restData.Enum.PortLogicTypeEnum.MNGT,
                                      restData.Enum.PortLogicTypeEnum.MNGT_SRV)
    # 维护端口的逻辑类型
    MGMT_PORT_MNTC_LOGIC_TYPE_LIST = (restData.Enum.PortLogicTypeEnum.MAINTENANCE,
                                      restData.Enum.PortLogicTypeEnum.MAINTENANCE_SRV)
    # 所有管理相关端口的逻辑类型
    MGMT_PORT_ALL_LOGIC_TYPE_LIST = (restData.Enum.PortLogicTypeEnum.MNGT,
                                     restData.Enum.PortLogicTypeEnum.MNGT_SRV,
                                     restData.Enum.PortLogicTypeEnum.MAINTENANCE,
                                     restData.Enum.PortLogicTypeEnum.MAINTENANCE_SRV,
                                     restData.Enum.PortLogicTypeEnum.IP_SCALE_OUT)

    # 所有V3的产品类型(由getSystemProductType获取)列表
    V3_ALL_PRODUCT_TYPE_LIST = (BaseFactory.const.SYSMODEL['V3_3U_ENGINE'],
                                BaseFactory.const.SYSMODEL['V3_6U_ENGINE'],
                                BaseFactory.const.SYSMODEL['V3_18000'],
                                BaseFactory.const.SYSMODEL['5X00V3'])

    V5_V6_PRODUCT_TYPE_LIST = (BaseFactory.const.SYSMODEL["V5_V6_2U_ENGINE"],
                               BaseFactory.const.SYSMODEL["V5_V6_4U_ENGINE"])

    # V3独立机头的产品类型列表
    V3_ENGIN_PRODUCT_TYPE_LIST = (BaseFactory.const.SYSMODEL['V3_3U_ENGINE'],
                                  BaseFactory.const.SYSMODEL['V3_6U_ENGINE'],
                                  BaseFactory.const.SYSMODEL['V3_18000'])


__________ZONE_INNER_PUBLIC_FUNC__________ = None


def getDevObj(context):
    '''
    @summary: 获取上下文对象中dev对象
    @param context: 上下文对象
    '''
    return context.get("dev")


'''
Function: 离线控制器
Params: 
    context：工具上下文
    cntrId: 控制器ID
Return: 离线是否成功，True=是，False=否
'''


def offLineSpecifiedController(context, cntrId, isLight=True):
    # 参数合法性检查
    if None == cntrId or "" == cntrId:
        BaseFactory.log.error(context, "the cntrId:%s is invalid." % cntrId)
        BaseFactory.result.setResultFailByKeys(context, LangKey.CNTR_OFFLINE_CNTR_FAIL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False

    # 即使为故障的控制器，还是需要离线控制器
    DEFAULT_TIMEOUT = 30 * 60  # 离线轮训检查时间：秒
    showUI(context, 0, DEFAULT_TIMEOUT)

    if isLight:
        # 对离线的控制器点灯
        try:
            param0 = (restData.Hardware.Controller.TYPE, restData.Enum.ObjEnum.CONTROLLER)
            param1 = (restData.Hardware.Controller.ID, cntrId)
            param2 = (restData.Hardware.Controller.LIGHT_STATUS, restData.Enum.SwitchEnum.ON)
            paramlist = restUtil.Tlv2Rest.getParamList(param0, param1, param2)
            rest = contextUtil.getRest(context)
            restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.MODIFY, paramlist)
        except Exception as ex:
            # 若点灯异常，记录错误并继续
            BaseFactory.log.error(context, "failed to light the controller of " + cntrId + ", exception=" + str(ex))

    # 下发控制器离线命令
    param0 = (restData.Hardware.Controller.ID, cntrId)
    paramlist = restUtil.Tlv2Rest.getParamList(param0)
    rest = contextUtil.getRest(context)
    recs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.OFFLINE_CONTROLLER, paramlist)
    # 解析TLV回显信息，以此来判断回显是否正常
    if None == recs:
        BaseFactory.result.setResultFailByKeys(context, LangKey.CNTR_OFFLINE_CNTR_FAIL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False

    status = False
    beginTime = time.time()
    tmpException = None
    while not status:
        # 检查超时时，有异常抛出异常，无异常返回错误
        if time.time() - beginTime > DEFAULT_TIMEOUT:
            showUI(context, 100, 0)
            # 若直到超时都存在异常，则在界面显示异常信息
            if None != tmpException:
                raise tmpException  # @IGNORE:E0702
            BaseFactory.result.setResultFailByKeys(context, LangKey.CNTR_OFFLINE_CNTR_FAIL,
                                                   LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
            return False

        # 检查控制器是否离线成功。离线的控制器状态为offline或查询报错
        try:
            # 若其中包含没有异常的情况，将异常信息设置为None
            tmpException = None
            rs = getFruRunningStatus(context, restData.Enum.ObjEnum.CONTROLLER, cntrId)
            if rs == "":  # 容错：兼容返回空的控制器运行状态
                status = True
            elif rs == restData.Enum.RunningStatusEnum.OFFLINE:  # 正常建议分支
                status = True
        except Exception as ex:
            strErrCode = ex.args[0]
            # Error Code=1077948996, Description=指定的对象不存在。
            if strErrCode == config.ERR_CODE_NOT_EXIST:
                # 容错：兼容查询不到控制器信息
                BaseFactory.log.error(context, "the controller is not existed. errorCode=1077948996")
                break
            tmpException = ex

        useTime = time.time() - beginTime
        curProgressPer = int((useTime / DEFAULT_TIMEOUT) * 100)
        remainTime = int(DEFAULT_TIMEOUT - useTime)
        showUI(context, curProgressPer, remainTime)
        # 轮询周期为5秒
        time.sleep(5)

    showUI(context, 100, 0)
    BaseFactory.result.setResultPass(context)
    return True


def getProductModel(context):
    '''
    @summary: 获取自定义字典中的产品型号
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    productModel = dev.get("type")
    return productModel


def rebootSpecifiedController(context, cntr_id, begin_time, timeout):
    """复位指定控制器

    :param context: 上下文
    :param cntr_id: 控制器ID
    :param begin_time: 轮询开始时间
    :param timeout: 超时时间
    :return:
    """
    # 入口参数检查
    if None is cntr_id or "" == cntr_id:
        BaseFactory.log.error(context, "the cntrId:%s is invalid." % cntr_id)
        BaseFactory.result.setResultFailByKeys(
            context,
            LangKey.CNTR_REBOOT_FAIL_OR_STATUS_ABNORMAL,
            LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False

    # 检查控制器是否恢复正常，如果已经正常，则无需重启
    try:
        hs = getFruHealthStatus(context,
                                restData.Enum.ObjEnum.CONTROLLER,
                                cntr_id)
        rs = getFruRunningStatus(context,
                                 restData.Enum.ObjEnum.CONTROLLER,
                                 cntr_id)
        # 恢复正常时，则返回成功
        if hs == restData.Enum.HealthStatusEnum.NORMAL \
                and rs == restData.Enum.RunningStatusEnum.ONLINE:
            BaseFactory.log.info(
                context,
                "cntr:%s is already NORMAL ONLINE now, skip reboot." % cntr_id)
            BaseFactory.result.setResultPass(context)
            return True
    except Exception as e:
        BaseFactory.log.error(context, "Query controller status exception: %s" % e)

    # 对指定控制器下发reboot指令
    id_param = (restData.PublicAttributesExtend.ID, cntr_id)
    pawd = context.get("dev").get("pawd")
    psw_param = (restData.PublicAttributesExtend.IMPORTANT_PSW, pawd)
    param_list = restUtil.Tlv2Rest.getParamList(id_param, psw_param)
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(rest,
                                    restData.TlvCmd.REBOOT_CONTROLLER,
                                    param_list,
                                    isShowLog=False)
    del pawd
    if rec is None:
        BaseFactory.log.error(
            context,
            "fail to exec cmd REBOOT_CONTROLLER:%s." % cntr_id)
        BaseFactory.result.setResultFailByKeys(
            context, LangKey.CNTR_REBOOT_FAIL_OR_STATUS_ABNORMAL,
            LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False
    BaseFactory.log.info(context, "execed reboot controller.")
    # 轮询检查控制器是否恢复正常
    tmp_exception = None
    while True:
        # 检查是否超时
        if time.time() - begin_time > timeout:
            # 超时时，有异常返回异常，无则返回错误
            if tmp_exception is not None:
                raise tmp_exception  # @IGNORE:E0702
            else:
                BaseFactory.log.error(
                    context,
                    "fail to exec cmd REBOOT_CONTROLLER:%s." % cntr_id)
                BaseFactory.result.setResultFailByKeys(
                    context,
                    LangKey.CNTR_REBOOT_FAIL_OR_STATUS_ABNORMAL,
                    LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
                return False

        # 检查控制器是否恢复正常
        try:
            tmp_exception = None
            hs = getFruHealthStatus(context,
                                    restData.Enum.ObjEnum.CONTROLLER,
                                    cntr_id)
            rs = getFruRunningStatus(context,
                                     restData.Enum.ObjEnum.CONTROLLER,
                                     cntr_id)
            # 恢复正常时，则返回成功
            if hs == restData.Enum.HealthStatusEnum.NORMAL \
                    and rs == restData.Enum.RunningStatusEnum.ONLINE:
                BaseFactory.log.info(
                    context,
                    "cntr:%s is NORMAL ONLINE now." % cntr_id)
                BaseFactory.result.setResultPass(context)
                return True
        except Exception as ex:
            tmp_exception = ex

        # 轮询周期为5秒
        time.sleep(5)

    # 异常点
    BaseFactory.log.error(context,
                          "unexpect error when reboot cntr:%s." % cntr_id)
    return False


# 获取控制器软件版本号（数字和句点组成的字符串）
PERSIST_KEY_CNTR_SOFTWARE_VERSION = "PERSIST_KEY_CNTR_SOFTWARE_VERSION"


def getCntrSoftVerString(context):
    # 若系统类型已存在于持久化数据中，则直接返回已有类型，否则重新获取并在持久化后返回
    cntrVer = BaseFactory.persist.getModule(context, PERSIST_KEY_CNTR_SOFTWARE_VERSION)
    if None != cntrVer:
        cntrVer = str(cntrVer)
    else:
        # 获取正常在线的控制器信息
        allCntrRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
        condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.HEALTH_STATUS,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    restData.Enum.HealthStatusEnum.NORMAL)
        condition2 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.RUNNING_STATUS,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    restData.Enum.RunningStatusEnum.ONLINE)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition1, condition2)
        normalRecs = restUtil.Tlv2Rest.filter(allCntrRecs, conditionList)
        # 打印结果
        restUtil.Tlv2Rest.logRecList(context, normalRecs, "all NORMAL ONLINE controllers as follows:")

        # 以第一个控制器的版本作为系统软件版本返回，没有则抛出异常
        if len(normalRecs) > 0:
            cntrVer = restUtil.Tlv2Rest.getRecordValue(normalRecs[0], restData.Hardware.Controller.SOFT_VER)
            BaseFactory.persist.setModule(context, PERSIST_KEY_CNTR_SOFTWARE_VERSION, cntrVer)
        else:
            BaseFactory.log.error(context, "the system has no normal online controller, raise a BaseFactory.exception.")
            raise BaseFactory.exception.newToolExceptionByKeys(context,
                                                               LangKey.FUNC_CANNOT_GET_SYSVER_FOR_NO_NORMAL_ONLINE_CNTR_EXISTED,
                                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)

    # 返回结果
    BaseFactory.log.info(context, "cntr software version = %s" % cntrVer)
    return cntrVer


# 获取系统中所有管理端口记录
def getAllMgmtPortRecord(context):
    allEthPortRecs = getFruListInfo(context, restData.Enum.ObjEnum.ETH_PORT)

    if True == isSingleEngine(context):
        logicTypeList = FRUCONST.MGMT_PORT_MGMT_LOGIC_TYPE_LIST
    else:  # 多引擎场景，包含管理口和维护口
        sysType = getSystemProductType(context)
        isDorado = contextUtil.getItem(context, 'isDorado', False)
        if sysType in FRUCONST.V5_V6_PRODUCT_TYPE_LIST:
            logicTypeList = FRUCONST.MGMT_PORT_MGMT_LOGIC_TYPE_LIST
        elif isDorado:
            logicTypeList = FRUCONST.MGMT_PORT_ALL_LOGIC_TYPE_LIST
        elif sysType in FRUCONST.V3_ALL_PRODUCT_TYPE_LIST:
            # V3集群为IP Scale-out组网，仅连接管理口，维护口不连接
            logicTypeList = FRUCONST.MGMT_PORT_MGMT_LOGIC_TYPE_LIST
        else:  # T集群为PCIe组网，管理网线为环路
            logicTypeList = FRUCONST.MGMT_PORT_ALL_LOGIC_TYPE_LIST
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.EthPort.LOGIC_TYPE,
                                                restData.Enum.ConditionTypeEnum.EQOR, logicTypeList)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    mgmtPortRecs = restUtil.Tlv2Rest.filter(allEthPortRecs, conditionList)
    # 打印结果
    restUtil.Tlv2Rest.logRecList(context, mgmtPortRecs, "all mgmt ports as follows:")

    return mgmtPortRecs


# 专用于管理线缆更换，获取用户选择端口的location
def getMgmtPortLocDesc(context, selPortRow=None):
    # 有端口时获取对应端口的location描述
    if None != selPortRow:
        isHostPort = ast.literal_eval(selPortRow["isHostPort"])
        if False == isHostPort:
            return selPortRow["location"]
        else:
            lang = context["lan"]
            if "zh" == lang:
                return u"用户设备网口"
            else:
                return "Network Port on User Device"
    else:  # 无端口时，获取管理端口信息
        lang = context["lan"]
        if "zh" == lang:
            return u"用户设备网口"
        else:
            return "Network Port on User Device"


# 专用于IP Scale-out线缆更换，获取用户选择端口的location
def getIpScaleOutPortLocDesc(context, selPortRow=None):
    # 有端口时获取对应端口的location描述
    if None != selPortRow:
        isHostPort = ast.literal_eval(selPortRow["isSwitchPort"])
        if not isHostPort:
            return selPortRow["location"]
        else:
            lang = context["lan"]
            if "zh" == lang:
                return u"交换机端口:"
            else:
                return "Switch Port:"
    else:  # 无端口时，获取管理端口信息
        lang = context["lan"]
        if "zh" == lang:
            return u"交换机端口:"
        else:
            return "Switch Port:"


def checkEncTempOfFru(context, fruType, fruId):
    """检查FRU所在机框是否超温

    :param context: 工具上下文
    :param fruType: 更换的FRU类型。
    :param fruId: 更换的FRU ID。
    :return: FRU所在机框是否超温
    """
    BaseFactory.log.info(context, "check enclosure temperature for the fru: id=%s, type=%s" %
                         (fruId, restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)))

    # 获取FRU所在框的温度
    encRec = getFruParentEncRec(context, fruType, fruId)
    encId = restUtil.Tlv2Rest.getRecordValue(encRec, restData.Hardware.Enclosure.ID)
    encType = restUtil.Tlv2Rest.getRecordValue(encRec, restData.Hardware.Enclosure.LOGIC_TYPE)
    BaseFactory.log.info(context, "encId=" + encId + ", encType=" + restUtil.Tlv2Rest.getStrEn(
        restData.EnumStr.StrEnclosureTypeEnum, encType))

    checkEncTemperature.execute(context, alarmIds=ENC_TEMPERATURE_ALM_IDS, encId=encId)
    return


'''
Function: 初始化更换指导基本信息
Contents: 设置当前更换的FRU类型和所在框信息，供GuideLinker.py中生成指导链接使用
Params: 
    context：工具上下文
    fruType: 更换的FRU类型。
    strSelRowKey: 用户选中FRU的上下文获取KEY
    useParentInfo: 是否使用父类信息来获取框信息。控制器和接口卡下电后，其信息可能查不到，导致获取不到父框信息，故此处使用父类信息规避该问题。
    extraParam: 额外参数。用于某些场合辅助判断时使用。如更换V3的管理模块时，需指定其参数为管理板的model值
    Attention: 若fruType为框类，请务必设置useParentInfo=False
Return: 无
'''


def initReplaceGuideBaseInfo(context, fruType, strSelRowKey,
                             parentIsNonEnc=True, extraParam=None, fruIdName="id", **kwargs):
    try:
        # 设置更换指导链接生成所需的FRU类型信息和额外参数
        BaseFactory.persist.setModule(context, PERSIST_GUIDE_FRU_TYPE, fruType)
        if None != extraParam:
            BaseFactory.log.info(context, "extraParam=" + BaseFactory.formatUtil.getUnicodeStr(extraParam))
            BaseFactory.persist.setModule(context, PERSIST_GUIDE_EXTRA_PARAM, extraParam)
        else:
            BaseFactory.persist.setModule(context, PERSIST_GUIDE_EXTRA_PARAM, None)

        # 设置更换指导链接生成所需的系统产品类型信息
        # 此类型为系统的ProductModel类型，不为控制框的model。
        # 注意：若后续存储集群支持不同ProductModel的控制框互联，则下面的sysProductModel的设置需要重新考虑，否则更换指导会错位。
        sysProductModel = getSystemProductType(context)
        BaseFactory.persist.setModule(context, PERSIST_GUIDE_PRODUCT_MODEL, sysProductModel)

        # 获取引擎型号(控制框的产品型号)，并缓存到context中用于在更换指导链接生成器GuideLinker中获取
        productModel = getSystemProductModel(context)
        BaseFactory.persist.setModule(context, PERSIST_GUIDE_ENGINE_MODEL, productModel)

        # 设置更换指导链接生成所需的父框信息
        # 清除已有框信息，避免下面获取框信息失败时，之前的其他FRU所在框信息的干扰
        BaseFactory.persist.delModule(context, PERSIST_GUIDE_PARENT_ENC_TLVREC)
        if kwargs and kwargs.get("needMultiCheck", False) and kwargs.get("cacheData"):
            cacheData = kwargs.get("cacheData", {})
            parentType = int(cacheData.parent_type)
            parentId = cacheData.parent_id
            fruId = cacheData.get_first_id()
        else:
            selFruRow = BaseFactory.json.toDict(context.get(strSelRowKey))
            parentType = int(selFruRow.get("parentType"))
            parentId = selFruRow.get("parentID")
            fruId = selFruRow.get(fruIdName)
        if FRUCONST.VIRTUAL_FRU_TYPE == fruType:  # 虚拟器件走此分支
            pass
        else:  # 实际能查询到的器件走此分支
            # 获取FRU所在框信息
            parentEnc = None
            if parentIsNonEnc:  # 非框类FRU走此分支
                parentType = int(parentType)
                parentEnc = getFruParentEncRec(context, parentType, parentId)
            else:  # 框类FRU走此分支
                parentEnc = getFruParentEncRec(context, fruType, fruId)
            # 保存FRU所在框信息
            BaseFactory.persist.setModule(context, PERSIST_GUIDE_PARENT_ENC_TLVREC, parentEnc)

    except Exception as ex:
        BaseFactory.exception.handler(context, ex)
        # 设置操作成功，不处理更换指导初始化异常
        BaseFactory.log.error(context, "exception as above occurred when initialize guide. skip it.")
        BaseFactory.result.setResultPass(context)

    return


'''
Function: 远程复制更换前检查，用于影响前端链路的模块
Contents: 获取并保存受阵列间单链路影响的增值业务和端口信息
Params: 
    context：工具上下文
    fruType: 在线更换的FRU类型。当前支持控制框、控制器、前端接口卡、光模块、业务（端口）线缆
    fruId: FRUID(如端口ID=1114625)。注意，不是位置（如ENG0.A0）
Return: 检查是否通过：True=通过； False=失败
'''


def preCheckReplication(context, fruType, fruId):
    # 定义context中存放数据的关键字
    mdlName = "ValueAddServieInfo"
    objNotice = "IsNeedNoticeAfterRepled"
    objSupport = "IsFruSupported"
    objFcPortIdList = "EffectiveFcPortIdList"
    objiScsiPortIdList = "EffectiveiScsiPortIdList"
    objRepIdList = "EffectiveReplicationIdList"

    # 获取潜在的会受影响的前端端口列表，若FRU不支持，则返回不通过
    BaseFactory.persist.setObject(context, mdlName, objSupport, True)
    effectPortList = getFrontPortsOfFru(context, fruType, fruId)
    BaseFactory.log.info(context, "possible effective Port List=" + str(effectPortList))
    if None == effectPortList:
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
        BaseFactory.persist.setObject(context, mdlName, objSupport, False)
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_UNSUPPORTED_FRU_TYPE,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False
    elif 0 == len(effectPortList):  # 当为不支持的接口卡或无前端接口卡时进入
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
        BaseFactory.result.setResultPass(context)
        return True

    # 若系统中无License信息，则设置无License标识供更换后检查使用，并直接返回通过
    funcRecs = getLicenseFuncInfo(context)
    if not funcRecs:
        BaseFactory.log.info(context, "the license is not existed.")
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
        BaseFactory.result.setResultPass(context)
        return True

    # 判断License是否支持远程复制，若不支持则设置无License标识共更换后检查使用，并直接返回通过
    repFeaIdList = restUtil.Tlv2Rest.getMultiConditionValueList(restData.Enum.LicenseFeatureEnum.HYPER_REPLICATION)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Sys.LicenseFileUsage.CMO_LICENSE_INFO_FEATURE_ID,
                                                restData.Enum.ConditionTypeEnum.EQOR, repFeaIdList)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    tmpRecs = restUtil.Tlv2Rest.filter(funcRecs, conditionList)
    if len(tmpRecs) == 0:
        BaseFactory.log.info(context, "the license is not support Replication.")
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
        BaseFactory.result.setResultPass(context)
        return True

    # 检查系统中是否存在远程复制项目，无则返回通过
    allRepRecs = getFruListInfo(context, restData.Enum.ObjEnum.REPLICATIONPAIR)
    if 0 == len(allRepRecs):
        BaseFactory.log.info(context, "the REPLICATIONPAIR is not existed.")
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
        BaseFactory.result.setResultPass(context)
        return True

    # 判断系统中存在的远程复制是否存在恢复策略为手动的项目，无则返回通过
    # 此处将所有恢复策略为手动的项都获取出来，更换后是否启同步，由到时的状态决定
    condition1 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.ReplicationPair.RECOVERY_POLICY,
                                                restData.Enum.ConditionTypeEnum.EQ, 2)  # recoveryPolicy：1=自动，2=手动
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allRepRecs, conditionList)
    if len(tmpRecs) == 0:
        BaseFactory.log.info(context, "the manual recovery REPLICATIONPAIR is not existed.")
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
        BaseFactory.result.setResultPass(context)
        return True
    # 打印结果
    restUtil.Tlv2Rest.logRecList(context, tmpRecs, "the manual recovery REPLICATIONPAIR list as bellow:")

    # 获取更换影响到的远程复制ID列表，若不为空则记录需在更换后提示用户手动激活
    (effectFcPortIdList, effectiScsiPortIdList, effectRepIdList) = getEffectRepIdList(context, effectPortList, tmpRecs)
    if 0 < len(effectRepIdList):
        BaseFactory.persist.setObject(context, mdlName, objNotice, True)
        BaseFactory.persist.setObject(context, mdlName, objFcPortIdList, effectFcPortIdList)
        BaseFactory.persist.setObject(context, mdlName, objiScsiPortIdList, effectiScsiPortIdList)
        BaseFactory.persist.setObject(context, mdlName, objRepIdList, effectRepIdList)

    else:
        BaseFactory.persist.setObject(context, mdlName, objNotice, False)
    BaseFactory.log.info(context, "effective Replication Id List=" + str(effectRepIdList))

    # 返回检查通过
    BaseFactory.result.setResultPass(context)
    return True


'''
Function: 获取前端接口上所有端口的Id
'''


def getPortIdOfIntf(context, intfId):
    retPortList = []

    # FC和ETH口均获取，规避一个接口板上出现两种端口的情况
    # 获取输入该接口板的ETH端口
    allEthPortRecs = getFruListInfo(context, restData.Enum.ObjEnum.ETH_PORT)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.EthPort.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                intfId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    tmpRecs = restUtil.Tlv2Rest.filter(allEthPortRecs, conditionList)
    recNum = len(tmpRecs)
    for index in range(0, recNum):
        portId = restUtil.Tlv2Rest.getRecordValue(tmpRecs[index], restData.Hardware.EthPort.ID)
        retPortList.append(portId)
    # 获取输入该接口板的FC端口
    allFcPortRecs = getFruListInfo(context, restData.Enum.ObjEnum.FC_PORT)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.FcPort.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                intfId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    tmpRecs = restUtil.Tlv2Rest.filter(allFcPortRecs, conditionList)
    recNum = len(tmpRecs)
    for index in range(0, recNum):
        portId = restUtil.Tlv2Rest.getRecordValue(tmpRecs[index], restData.Hardware.FcPort.ID)
        retPortList.append(portId)
    BaseFactory.log.info(context, "retPortList = %s" % str(retPortList))
    return retPortList


'''
Function: 获取FRU上所有前端端口ID列表。若FRU不支持，则返回None
当前支持FRU：控制框、控制器、前端接口卡、光模块、业务（端口）线缆
'''


def getFrontPortsOfFru(context, fruType, fruId):
    # 当前支持控制器、前端接口卡、光模块、业务（端口）线缆（已实现）
    strFruType = restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)
    BaseFactory.log.info(context, "fru info: type=" + strFruType + ", id=" + fruId)
    portIdList = []
    if fruType in (restData.Enum.ObjEnum.ETH_PORT,
                   restData.Enum.ObjEnum.FC_PORT,
                   restData.Enum.ObjEnum.FCoE_PORT):
        portIdList.append(fruId)
    elif restData.Enum.ObjEnum.SFP_OPTICAL_TRANSCEIVER == fruType:
        # 获取光模块位置，与光口或PCIe口一致
        fruRec = getFruInfo(context, fruType, fruId)
        location = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.Hardware.SfpOpticalTransceiver.LOCATION)
        # 获取并匹配相同Location的光口
        port_type = restData.Enum.ObjEnum.FC_PORT
        if baseUtil.is_computing_dev(contextUtil.getProductModel(context)):
            port_type = restData.Enum.ObjEnum.ETH_PORT
        tmpRecs = getFruListInfo(context, port_type)
        condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.FcPort.LOCATION,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    location)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
        tmpRecs = restUtil.Tlv2Rest.filter(tmpRecs, conditionList)
        if len(tmpRecs) > 0:
            fruRec = tmpRecs[0]
            portId = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.Hardware.FcPort.ID)
            portIdList.append(portId)
    elif restData.Enum.ObjEnum.INTF_MODULE == fruType:
        fruRec = getFruInfo(context, fruType, fruId)
        intfIdList = getIntfIdListByLogicType(context, restData.Enum.PortLogicTypeEnum.HOST)
        intfID = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.Hardware.IntfModule.ID)
        BaseFactory.log.info(context, "intfID = %s" % str(intfID))
        if intfID in intfIdList:
            portIdList = getPortIdOfIntf(context, fruId)
        else:  # 不支持的前端接口板不做处理
            pass
    elif restData.Enum.ObjEnum.CONTROLLER == fruType:
        _set_id_list_for_ctrl(context, fruId, portIdList)
    elif restData.Enum.ObjEnum.ENCLOSURE == fruType:
        _set_port_id_list_for_enc(context, fruId, portIdList)
    elif fruType in [restData.Enum.ObjEnum.IB_PORT]:
        portIdList = []

    else:  # 不支持的FRU，则返回None
        portIdList = None

    return portIdList


@ability_map_util.ability_mapping
def _set_id_list_for_ctrl(context, fru_id, port_id_list):
    all_intf_recs = getFruListInfo(context, restData.Enum.ObjEnum.INTF_MODULE)
    # 打印结果
    restUtil.Tlv2Rest.logRecList(context, all_intf_recs, "the manual recovery normal replication id list as bellow:")
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.EthPort.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                fru_id)
    condition_list = restUtil.Tlv2Rest.getConditionList(condition0)
    tmp_recs = restUtil.Tlv2Rest.filter(all_intf_recs, condition_list)
    rec_num = len(tmp_recs)
    for index in range(0, rec_num):
        fru_rec = tmp_recs[index]
        # 若不为前端接口卡，则不处理
        intf_id_list = getIntfIdListByLogicType(context, restData.Enum.PortLogicTypeEnum.HOST)
        intf_id = restUtil.Tlv2Rest.getRecordValue(fru_rec, restData.Hardware.IntfModule.ID)
        BaseFactory.log.info(context, "intfID = %s" % str(intf_id))
        if intf_id not in intf_id_list:
            continue
        # 获取该前端接口卡上的端口信息
        tmp_id_list = getPortIdOfIntf(context, intf_id)
        # 合并端口信息
        for portId in tmp_id_list:
            port_id_list.append(portId)


def _set_id_list_for_compute_ctrl(context, fru_id, port_id_list):
    all_eth_port_recs = getFruListInfo(context, restData.Enum.ObjEnum.ETH_PORT)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.EthPort.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                fru_id)
    condition_list = restUtil.Tlv2Rest.getConditionList(condition0)
    tmp_recs = restUtil.Tlv2Rest.filter(all_eth_port_recs, condition_list)
    rec_num = len(tmp_recs)
    for index in range(0, rec_num):
        port_id = restUtil.Tlv2Rest.getRecordValue(tmp_recs[index], restData.Hardware.EthPort.ID)
        port_id_list.append(port_id)


def _set_port_id_list_for_enc(context, fru_id, port_id_list):
    all_ctrl_recs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                fru_id)
    condition_list = restUtil.Tlv2Rest.getConditionList(condition0)
    tmp_recs = restUtil.Tlv2Rest.filter(all_ctrl_recs, condition_list)
    rec_num = len(tmp_recs)
    for index in range(0, rec_num):
        # 获取控制器信息
        fru_rec = tmp_recs[index]
        ctrl_id = restUtil.Tlv2Rest.getRecordValue(fru_rec, restData.Hardware.Controller.ID)
        # 获取并合并该控制器上的端口信息
        tmp_id_list = getFrontPortsOfFru(context, restData.Enum.ObjEnum.CONTROLLER, ctrl_id)
        for portId in tmp_id_list:
            port_id_list.append(portId)

'''
Function: 获取License特性列表。若无License，则返回None
'''


def getLicenseFuncInfo(context):
    try:
        # 注意：GETBATCH_LICENSE_INFO命令执行结果中的Id为Int，非Str，不能使用execmd执行。
        rest = contextUtil.getRest(context)
        params = []
        record = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GETBATCH_LICENSE_INFO, params)

        if None == record or len(record) == 0:
            BaseFactory.log.error(context, "license file info is empty, return None.")
            return None
        else:
            return record

    except Exception as ex:
        strErrCode = ex.args[0]
        # 若返回的错误信息包含license，则说明该文件不存在；否则继续抛出异常
        if strErrCode == config.ERR_CODE_FILE_NOT_EXIST:
            BaseFactory.log.error(context, "license file is not existed, return None.")
            return None
        elif strErrCode == config.ERR_CODE_EMPTY:
            BaseFactory.log.error(context, "license file info is empty, return None.")
            return None
        else:
            BaseFactory.log.error(context, "fail to get license, keep throw Exception. Info=" + str(ex))
            raise ex


'''
Function: 获取受更换影响远程复制ID
Params: 
    context：工具上下文
    effectPortList: 潜在的会受影响的前端端口列表
    repSrvRecs: 需要手动恢复的增值业务信息集合
'''


def getEffectRepIdList(context, effectPortList, repSrvRecs):
    # 注意：每个增值项均只是两个设备间的业务，故远端设备ID仅有一个。不存在一个项中包含多个远端设备的情况。
    # 返回值定义
    effectRepIdList = []

    # 接口参数校验
    recNum = len(repSrvRecs)
    if 0 == recNum:
        return effectRepIdList
    if 0 == len(effectPortList):
        return effectRepIdList

    # 获取增值对应的远端设备ID（每个ID对应一个远端设备）及其与增值项ID的对应关系
    # 远端设备ID与链路中的远端设备和复制链路中的同名字段含义一致
    remoteDevIdList = []
    allRepIdList = {}
    for index in range(0, recNum):
        # 获取信息
        repRec = repSrvRecs[index]
        repId = restUtil.Tlv2Rest.getRecordValue(repRec, restData.ValueAdd.ReplicationPair.ID)
        remoteDeviceID = restUtil.Tlv2Rest.getRecordValue(repRec, restData.ValueAdd.ReplicationPair.REMOTE_DEVICE_ID)
        # 记录信息（注意：每个远端设备上可创建多个增值业务）
        if remoteDeviceID not in remoteDevIdList:  # 两个阵列间可能存在多个远程复制
            remoteDevIdList.append(remoteDeviceID)
        repList = []  # 当前远端设备上的增值业务列表
        if remoteDeviceID in allRepIdList:
            repList = allRepIdList[remoteDeviceID]
        repList.append(repId)
        allRepIdList[remoteDeviceID] = repList
    BaseFactory.log.info(context, "remoteDevIdList=" + str(remoteDevIdList))  # for test
    BaseFactory.log.info(context, "allRepIdList=" + str(allRepIdList))  # for test

    # 获取所有正常连接的复制链路信息
    # FC Link
    fcLinkRecs = getFruListInfo(context, restData.Enum.ObjEnum.FC_LINK)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.FcLink.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.NORMAL)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.FcLink.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.LINK_UP)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    fcLinkRecs = restUtil.Tlv2Rest.filter(fcLinkRecs, conditionList)
    # iSCSI Link
    iscsiLinkRecs = getFruListInfo(context, restData.Enum.ObjEnum.iSCSI_LINK)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.IscsiLink.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.NORMAL)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.IscsiLink.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.LINK_UP)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    iscsiLinkRecs = restUtil.Tlv2Rest.filter(iscsiLinkRecs, conditionList)

    # 遍历所有远程复制对应的远端设备，判断其使用的端口ID是否全在待更换的FRU上，若在，则记录受影响的增值项Id
    effectFcPortIdList = []
    effectiScsiPortIdList = []
    for devId in remoteDevIdList:
        portIdList = []
        portTypeList = {}
        # 获取远端设备使用的FC链路的本地端口ID
        condition0 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.FcLink.DEVICE_ID,
                                                    restData.Enum.ConditionTypeEnum.EQ, devId)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
        tmpRecs = restUtil.Tlv2Rest.filter(fcLinkRecs, conditionList)
        recNum = len(tmpRecs)
        for index in range(0, recNum):
            portId = restUtil.Tlv2Rest.getRecordValue(tmpRecs[index], restData.ValueAdd.FcLink.LOCAL_PORT_ID)
            portIdList.append(portId)
            portTypeList[portId] = "fc"
        # 获取远端设备使用的iSCSI链路的本地端口ID
        condition0 = restUtil.Tlv2Rest.getCondition(restData.ValueAdd.IscsiLink.DEVICE_ID,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    devId)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
        tmpRecs = restUtil.Tlv2Rest.filter(iscsiLinkRecs, conditionList)
        recNum = len(tmpRecs)
        for index in range(0, recNum):
            portId = restUtil.Tlv2Rest.getRecordValue(tmpRecs[index], restData.ValueAdd.IscsiLink.LOCAL_PORT_ID)
            portIdList.append(portId)
            portTypeList[portId] = "iscsi"
        # 若该远端设备占用的所有本地端口均受更换影响，则记录受影响的增值项
        isAllEffect = True
        for portId in portIdList:
            if portId not in effectPortList:
                isAllEffect = False
            else:
                if "fc" == portTypeList[portId]:
                    effectFcPortIdList.append(portId)
                else:
                    effectiScsiPortIdList.append(portId)
        if True == isAllEffect:
            for repId in allRepIdList[devId]:
                effectRepIdList.append(repId)

    # 记录受影响的复制链路端口信息，并返回查询结果
    BaseFactory.log.info(context, "details of effective FC Port Id List=" + str(effectFcPortIdList))
    BaseFactory.log.info(context, "details of effective iSCSI Port Id List=" + str(effectiScsiPortIdList))
    BaseFactory.log.info(context, "details of effective Replication Id List=" + str(effectRepIdList))
    return (effectFcPortIdList, effectiScsiPortIdList, effectRepIdList)


'''
Function: 远程复制更换后检查，用于影响前端链路的模块
Contents: 获取并保存受阵列间单链路影响的增值业务和端口信息
Params: 
    context：工具上下文
Return: 检查是否通过：True=通过； False=失败
'''


def postCheckReplication(context):
    # 远程复制告警检查
    distanceAlarmList = ["0x000F01C0001",  # 远程复制异常断开
                         "0x000F01C0002",  # 远程复制不可用
                         "0x000F01C0003",  # 远程复制主Lun数据回滚失败
                         "0x000F01C0004",  # 远程复制从Lun数据回滚失败
                         "0x200F0027002C",  # 远程复制同步失败
                         "0x000F00270038",  # 远程复制异常断开
                         "0x000F00270039",  # 远程复制不可用
                         "0x000F0027003A",  # 远程复制自动启同步失败
                         "0x100F0027008B",  # 远程复制从Lun数据回滚失败
                         "0x000F0E10001",  # 复制链路断开
                         "0x000F00E00014",  # 没有备用链路
                         "0x000F00E00014",  # 没有备用链路
                         "0x000F00E00015",  # 阵列间链路连接数达到规格
                         "0x000F00E00016"  # 阵列间链路加密协议失败
                         ]
    distanceAlarmDict = {}
    for almId in distanceAlarmList:
        distanceAlarmDict[str(int(almId, 16))] = almId
    BaseFactory.log.info(context, "distanceAlarmDict : %s" % str(distanceAlarmDict))
    retVal = getSysAlarmsInfo(context)
    if True != retVal["succ"]:
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return

    sysAlarmList = retVal["info"]
    BaseFactory.log.info(context, "sysAlarmList : %s" % str(sysAlarmList))
    if sysAlarmList is not None:
        almNum = len(sysAlarmList)
        resultList = []
        for index in range(0, almNum):
            curAlm = sysAlarmList[index]
            almId = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_ID)
            if str(almId) in distanceAlarmDict:
                resultList.append(distanceAlarmDict[str(almId)])
        BaseFactory.log.info(context, "resultList : %s" % str(resultList))
        # 通过告警ID获取告警事件名称

        if len(resultList) != 0:
            lang = context.get("lan")
            dialogUtil = context['dialogUtil']
            alarmMsgList = []
            for item in resultList:
                alarmMsgList.append(getAlarm(lang, item))
            msg = getRes(lang, "sync_alarm", ",".join(alarmMsgList))
            rec = dialogUtil.showWarningDialog(msg)

            if not rec:
                BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_EXISTED_SYNC_ERROR)
                return

    # 定义context中存放数据的关键字
    mdlName = "ValueAddServieInfo"
    objNotice = "IsNeedNoticeAfterRepled"
    objFcPortIdList = "EffectiveFcPortIdList"
    objiScsiPortIdList = "EffectiveiScsiPortIdList"
    objRepIdList = "EffectiveReplicationIdList"

    # 获取更换前检查的信息，若无需处理，则直接返回通过
    needNotice = BaseFactory.persist.getObject(context, mdlName, objNotice)
    if None == needNotice or False == needNotice:
        BaseFactory.log.info(context, "non replication need to re-start sync.")
        BaseFactory.result.setResultPass(context)
        return True

    # 等待120S，尽可能保证复制链路连接好(FC卡很快，约10S，但ETH卡很慢，约90S)，并将远程复制状态设置为待恢复
    time.sleep(120)

    # 获取影响远程复制且未连接的FC端口
    unLinkFcPortInfo = ""
    effectFcPortIdList = BaseFactory.persist.getObject(context, mdlName, objFcPortIdList)
    for portId in effectFcPortIdList:
        portRec = getFruInfo(context, restData.Enum.ObjEnum.FC_PORT, portId)
        runStat = restUtil.Tlv2Rest.getRecordValue(portRec, restData.Hardware.FcPort.RUNNING_STATUS)
        if runStat != restData.Enum.RunningStatusEnum.LINK_UP:
            location = restUtil.Tlv2Rest.getRecordValue(portRec, restData.Hardware.FcPort.LOCATION)
            unLinkFcPortInfo = unLinkFcPortInfo + location + ", "
    unLinkFcPortInfo = unLinkFcPortInfo.rstrip(", ")  # 去除尾部多余分隔符

    # 获取影响远程复制且未连接的iSCSI端口
    unLinkiScsiPortInfo = ""
    effectiScsiPortIdList = BaseFactory.persist.getObject(context, mdlName, objiScsiPortIdList)
    for portId in effectiScsiPortIdList:
        portRec = getFruInfo(context, restData.Enum.ObjEnum.ETH_PORT, portId)
        runStat = restUtil.Tlv2Rest.getRecordValue(portRec, restData.Hardware.EthPort.RUNNING_STATUS)
        if runStat != restData.Enum.RunningStatusEnum.LINK_UP:
            location = restUtil.Tlv2Rest.getRecordValue(portRec, restData.Hardware.EthPort.LOCATION)
            unLinkiScsiPortInfo = unLinkiScsiPortInfo + location + ", "
    unLinkiScsiPortInfo = unLinkiScsiPortInfo.rstrip(", ")  # 去除尾部多余分隔符
    BaseFactory.log.info(context, "unLinkiScsiPortInfo = %s" % str(unLinkiScsiPortInfo))
    # 若存在未连接的影响端口，则检查不通过
    unLinkPortInfo = ""
    # FC
    if "" != unLinkFcPortInfo:
        unLinkPortInfo = "FC: " + unLinkFcPortInfo
    # iSCSI
    if "" != unLinkiScsiPortInfo:
        if "" == unLinkPortInfo:
            unLinkPortInfo = "iSCSI: " + unLinkiScsiPortInfo
        else:
            unLinkPortInfo = unLinkPortInfo + "; iSCSI: " + unLinkiScsiPortInfo
    # 判断
    if "" != unLinkPortInfo:
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_EXISTED_LINKDOWN_VALUEADD_PORT_Y, None,
                                              unLinkPortInfo)
        return False

    # 获取并遍历受影响的增值业务列表，若状态不正常，则启动同步
    failSyncRepInfo = ""
    effectRepIdList = BaseFactory.persist.getObject(context, mdlName, objRepIdList)
    BaseFactory.log.info(context, "ready to handle replication pair id list: " + str(effectRepIdList))
    for repId in effectRepIdList:
        # 查询增值项运行状态
        repRec = getFruInfo(context, restData.Enum.ObjEnum.REPLICATIONPAIR, repId)
        repRunStat = restUtil.Tlv2Rest.getRecordValue(repRec, restData.ValueAdd.ReplicationPair.RUNNING_STATUS)

        # 若该项运行状态：正常或分裂或同步中，跳过处理；待恢复(链路恢复后，链路断开状态将变为该状态)或故障或其他，启同步
        strRunStatDesc = restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrRunningStatusEnum, repRunStat)
        if restData.Enum.RunningStatusEnum.NORMAL == repRunStat \
                or restData.Enum.RunningStatusEnum.SPLIT == repRunStat \
                or restData.Enum.RunningStatusEnum.SYNCHRONIZING == repRunStat:
            BaseFactory.log.info(context, "skip sync a replication. id=" + repId + ", runningStatus=" + strRunStatDesc)
            continue
        elif restData.Enum.RunningStatusEnum.TO_BE_RECOVERD == repRunStat:
            BaseFactory.log.info(context,
                                 "start sync a replication. id=" + repId + ", runningStatus=" + strRunStatDesc)
            param0 = (restData.ValueAdd.ReplicationPair.TYPE, restData.Enum.ObjEnum.REPLICATIONPAIR)
            param1 = (restData.ValueAdd.ReplicationPair.ID, repId)
            paramlist = restUtil.Tlv2Rest.getParamList(param0, param1)
            rest = contextUtil.getRest(context)
            restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.SYNCRONIZE_HYPERMIRROR, paramlist)
        else:
            BaseFactory.log.info(context, "fail sync a replication. id=" + repId + ", runningStatus=" + strRunStatDesc)
            localResName = restUtil.Tlv2Rest.getRecordValue(repRec, restData.ValueAdd.ReplicationPair.LOCAL_RES_NAME)
            remoteResName = restUtil.Tlv2Rest.getRecordValue(repRec, restData.ValueAdd.ReplicationPair.LOCAL_RES_NAME)
            failSyncRepInfo = failSyncRepInfo + repId + "," + localResName + "," + remoteResName + "; "

    # 若存在无法同步的增值项，则提示用户重试
    if "" != failSyncRepInfo:
        failSyncRepInfo = "ID,localLunId,RemoteLunId: " + failSyncRepInfo.rstrip("; ")
        BaseFactory.log.info(context, "fail sync replication info: " + failSyncRepInfo)
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_EXISTED_CAN_NOT_SYNC_REPLICATION_Y, None,
                                              failSyncRepInfo)
        return False

    # 返回检查完成
    BaseFactory.result.setResultPass(context)
    return True


'''
Function: 获取控制框的类型，供代码按框类型区分硬件规格
Params: 
    context：工具上下文
    productModel: 从SYSTEM TLV下查回的框的productMode 或 EnclosureTLV下查回的框的model
Return: 控制框类型：参见BaseFactory中的const.SYSMODEL字典
'''


def getSystemProductTypeByTlvProductModel(context, productModel):
    # 初始化设备型号列表
    modelList_V3_3U = [restData.Enum.ProductModeEnum.V3_5600,
                       restData.Enum.ProductModeEnum.V3_5800,
                       restData.Enum.ProductModeEnum.V3_5600F,
                       restData.Enum.ProductModeEnum.V3_5800F,
                       restData.Enum.ProductModeEnum.Dorado6000_V3,
                       restData.Enum.ProductModeEnum.V5_5600,
                       restData.Enum.ProductModeEnum.V5_5600F,
                       restData.Enum.ProductModeEnum.V5_5800,
                       restData.Enum.ProductModeEnum.V5_5800F,
                       ]
    modelList_V3_6U = [restData.Enum.ProductModeEnum.V3_6800,
                       restData.Enum.ProductModeEnum.V3_6800F,
                       restData.Enum.ProductModeEnum.V5_6800,
                       restData.Enum.ProductModeEnum.V5_6800F,
                       restData.Enum.ProductModeEnum.Dorado18000_V3,
                       ]
    modelList_V3_18000 = [restData.Enum.ProductModeEnum.V3_18500,
                          restData.Enum.ProductModeEnum.V3_18800,
                          restData.Enum.ProductModeEnum.V3_18500F,
                          restData.Enum.ProductModeEnum.V3_18800F,
                          restData.Enum.ProductModeEnum.V5_18500,
                          restData.Enum.ProductModeEnum.V5_18500F,
                          restData.Enum.ProductModeEnum.V5_18800,
                          restData.Enum.ProductModeEnum.V5_18800F,
                          ]
    modelList_5X00V3 = [restData.Enum.ProductModeEnum.V3_5300,
                        restData.Enum.ProductModeEnum.V3_5500,
                        restData.Enum.ProductModeEnum.V3_5500F,
                        restData.Enum.ProductModeEnum.V3_5100,
                        restData.Enum.ProductModeEnum.V3_2100,
                        restData.Enum.ProductModeEnum.V3_2200,
                        restData.Enum.ProductModeEnum.V3_2600,
                        restData.Enum.ProductModeEnum.V3_2600ForVideo,
                        restData.Enum.ProductModeEnum.V3_2600F,
                        restData.Enum.ProductModeEnum.V3_2800,
                        restData.Enum.ProductModeEnum.Dorado3000_V3,
                        restData.Enum.ProductModeEnum.Dorado5000_V3,
                        restData.Enum.ProductModeEnum.V5_2800,
                        restData.Enum.ProductModeEnum.V5_5110,
                        restData.Enum.ProductModeEnum.V5_5110F,
                        restData.Enum.ProductModeEnum.V5_5210,
                        restData.Enum.ProductModeEnum.V5_5210F,
                        restData.Enum.ProductModeEnum.V5_5300,
                        restData.Enum.ProductModeEnum.V5_5300F,
                        restData.Enum.ProductModeEnum.V5_5500,
                        restData.Enum.ProductModeEnum.V5_5500F,
                        restData.Enum.ProductModeEnum.V5_5500_Elite,
                        restData.Enum.ProductModeEnum.V3_2200_Enhanced,
                        restData.Enum.ProductModeEnum.V3_2600_Enhanced,
                        restData.Enum.ProductModeEnum.V3_2600F_Enhanced,
                        restData.Enum.ProductModeEnum.V5_5300_Enhanced,
                        restData.Enum.ProductModeEnum.DoradoNAS,

                        restData.Enum.ProductModeEnum.V5_5110,
                        restData.Enum.ProductModeEnum.V5_5110F,
                        restData.Enum.ProductModeEnum.V5_5210,
                        restData.Enum.ProductModeEnum.V5_5210F,

                        ]
    modelListV5V6_2U_ENG = [restData.Enum.ProductModeEnum.V5_2810,
                            restData.Enum.ProductModeEnum.V5_2810_C72,
                            restData.Enum.ProductModeEnum.V5_2210,
                            restData.Enum.ProductModeEnum.V5_2220,
                            restData.Enum.ProductModeEnum.V5_5310,
                            restData.Enum.ProductModeEnum.V5_5310F,
                            restData.Enum.ProductModeEnum.V5_5510,
                            restData.Enum.ProductModeEnum.V5_5510F,
                            restData.Enum.ProductModeEnum.V5_5610,
                            restData.Enum.ProductModeEnum.V5_5610F,
                            restData.Enum.ProductModeEnum.V5_5810,
                            restData.Enum.ProductModeEnum.V5_5810F,
                            restData.Enum.ProductModeEnum.Dorado3000_V6_SAS,
                            restData.Enum.ProductModeEnum.Dorado3000_SAS,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_2000,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_2020,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_2100,
                            restData.Enum.ProductModeEnum.Dorado3000_V6_IPSAS,
                            restData.Enum.ProductModeEnum.OceanStor_Micro_1300,
                            restData.Enum.ProductModeEnum.OceanDisk_1300,
                            restData.Enum.ProductModeEnum.Dorado5000_V6_SAS,
                            restData.Enum.ProductModeEnum.Dorado5000_SAS,
                            restData.Enum.ProductModeEnum.Dorado5000_V6_NVMe,
                            restData.Enum.ProductModeEnum.Dorado5000_NVMe,
                            restData.Enum.ProductModeEnum.OceanStor_Micro_1500,
                            restData.Enum.ProductModeEnum.OceanDisk_1500,
                            restData.Enum.ProductModeEnum.OceanDisk_1600,
                            restData.Enum.ProductModeEnum.OceanDisk_1500T,
                            restData.Enum.ProductModeEnum.OceanDisk_1600T,
                            restData.Enum.ProductModeEnum.OceanDisk_1610,
                            restData.Enum.ProductModeEnum.OceanDisk_1610T,
                            restData.Enum.ProductModeEnum.Dorado5000_V6_IPSAS,
                            restData.Enum.ProductModeEnum.Dorado5000_IPSAS,
                            restData.Enum.ProductModeEnum.Dorado6000_V6_SAS,
                            restData.Enum.ProductModeEnum.Dorado6000_V6_NVMe,
                            restData.Enum.ProductModeEnum.Dorado6000_V6_IPSAS,
                            restData.Enum.ProductModeEnum.Dorado6000_SAS,
                            restData.Enum.ProductModeEnum.Dorado6000_NVMe,
                            restData.Enum.ProductModeEnum.Dorado6000_IPSAS,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5300_V6,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5300,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5500_V6,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5500,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5600_V6,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5600,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5600K_V6,
                            restData.Enum.ProductModeEnum.OceanStor_Dorado_5600K,
                            restData.Enum.ProductModeEnum.OceanStor_2200,
                            restData.Enum.ProductModeEnum.OceanStor_2220,
                            restData.Enum.ProductModeEnum.OceanStor_5120,
                            restData.Enum.ProductModeEnum.OceanStor_2600,
                            restData.Enum.ProductModeEnum.OceanStor_2620,
                            restData.Enum.ProductModeEnum.OceanStor_5210,
                            restData.Enum.ProductModeEnum.OceanStor_5220,
                            restData.Enum.ProductModeEnum.OceanStor_5310,
                            restData.Enum.ProductModeEnum.OceanStor_5310_Capacity_Flash,
                            restData.Enum.ProductModeEnum.OceanStor_5510_Capacity_Flash,
                            restData.Enum.ProductModeEnum.OceanStor_A300,
                            restData.Enum.ProductModeEnum.OceanStor_A800,
                            restData.Enum.ProductModeEnum.OceanStor_5320,
                            restData.Enum.ProductModeEnum.OceanStor_5300K,
                            restData.Enum.ProductModeEnum.OceanStor_5500K,
                            restData.Enum.ProductModeEnum.OceanStor_5510,
                            restData.Enum.ProductModeEnum.OceanStor_5510S,
                            restData.Enum.ProductModeEnum.OceanStor_5610,
                            restData.Enum.ProductModeEnum.OceanStor_5810HS,
                            restData.Enum.ProductModeEnum.V5_5210_Enhanced,
                            restData.Enum.ProductModeEnum.V5_5220,
                            restData.Enum.ProductModeEnum.V5_5210F_Enhanced,
                            restData.Enum.ProductModeEnum.V5_5110_Enhanced,
                            restData.Enum.ProductModeEnum.V5_5120,
                            restData.Enum.ProductModeEnum.V5_5110F_Enhanced,
                            restData.Enum.ProductModeEnum.V5_2600,
                            restData.Enum.ProductModeEnum.OceanProtect_A8000,
                            restData.Enum.ProductModeEnum.OceanProtect_X8000,
                            restData.Enum.ProductModeEnum.OceanProtect_E8000,
                            restData.Enum.ProductModeEnum.OceanProtect_X3000,
                            restData.Enum.ProductModeEnum.OceanProtect_X6000,
                            restData.Enum.ProductModeEnum.OceanProtect_X8000K,
                            restData.Enum.ProductModeEnum.OceanStor_2910,
                            ]
    modelListV5V6_4U_ENG = [
        restData.Enum.ProductModeEnum.V5_6810,
        restData.Enum.ProductModeEnum.V5_6810F,
        restData.Enum.ProductModeEnum.V5_18510,
        restData.Enum.ProductModeEnum.V5_18510F,
        restData.Enum.ProductModeEnum.V5_18810,
        restData.Enum.ProductModeEnum.V5_18810F,
        restData.Enum.ProductModeEnum.Dorado8000_V6_SAS,
        restData.Enum.ProductModeEnum.Dorado8000_V6_NVMe,
        restData.Enum.ProductModeEnum.Dorado8000_V6_IPSAS,
        restData.Enum.ProductModeEnum.Dorado8000_SAS,
        restData.Enum.ProductModeEnum.Dorado8000_NVMe,
        restData.Enum.ProductModeEnum.Dorado8000_IPSAS,
        restData.Enum.ProductModeEnum.Dorado18000_V6_SAS,
        restData.Enum.ProductModeEnum.Dorado18000_V6_NVMe,
        restData.Enum.ProductModeEnum.Dorado18000_V6_IPSAS,
        restData.Enum.ProductModeEnum.Dorado18000_SAS,
        restData.Enum.ProductModeEnum.Dorado18000_NVMe,
        restData.Enum.ProductModeEnum.Dorado18000_IPSAS,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_5800_V6,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_6800_V6,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_6800,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_18500_V6,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_18500,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_18800_V6,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_18800,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_18800K_V6,
        restData.Enum.ProductModeEnum.OceanStor_Dorado_18800K,
        restData.Enum.ProductModeEnum.OceanStor_6810,
        restData.Enum.ProductModeEnum.OceanStor_18510,
        restData.Enum.ProductModeEnum.OceanStor_18810,
        restData.Enum.ProductModeEnum.OceanStor_18500K,
        restData.Enum.ProductModeEnum.OceanProtect_A9000,
        restData.Enum.ProductModeEnum.OceanProtect_X9000,
        restData.Enum.ProductModeEnum.OceanProtect_X9000K,
    ]

    # 据设备所属的型号类别，返回相应的数值
    BaseFactory.log.info(context,
                         "system product model is " + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrProductModeEnum,
                                                                                 productModel))
    sysModel = BaseFactory.const.SYSMODEL["ERROR"]
    if productModel in modelList_V3_3U:
        sysModel = BaseFactory.const.SYSMODEL["V3_3U_ENGINE"]
    elif productModel in modelList_V3_6U:
        sysModel = BaseFactory.const.SYSMODEL["V3_6U_ENGINE"]
    elif productModel in modelList_V3_18000:
        sysModel = BaseFactory.const.SYSMODEL["V3_18000"]
    elif productModel in modelList_5X00V3:
        sysModel = BaseFactory.const.SYSMODEL["5X00V3"]
    elif productModel in modelListV5V6_2U_ENG:
        sysModel = BaseFactory.const.SYSMODEL["V5_V6_2U_ENGINE"]
    elif productModel in modelListV5V6_4U_ENG:
        sysModel = BaseFactory.const.SYSMODEL["V5_V6_4U_ENGINE"]
    else:
        BaseFactory.log.error(context, "Undefined control enclosure model: " + str(productModel))
        raise BaseFactory.exception.newToolExceptionByKeys(context, LangKey.FUNC_UNDEFINED_CNTR_ENC_TYPE_X,
                                                           LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP,
                                                           str(productModel))

    # 返回结果
    BaseFactory.log.info(context, "sys modle type = " + str(sysModel))
    return sysModel


def getSystemProductModel(context, sysRec=None):
    """获取控制框的产品型号

    :param context: 工具上下文（当sysRec不为None时，此值可以为None）
    :param sysRec: 系统信息。若为None，则表示查询context中的结果
    :return: BaseFactory中的const.SYSMODEL字典
    """
    # 获取系统信息
    if not sysRec:
        # 三次重试获取系统信息，避免偶发获取失败
        RETRY_NUM = 3
        curNum = 0
        while curNum < RETRY_NUM:
            param0 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.SYSTEM)
            paramlist = restUtil.Tlv2Rest.getParamList(param0)
            rest = contextUtil.getRest(context)
            sysRec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET, paramlist, False)
            if sysRec:
                break
            else:
                curNum += 1
        # 若获取信息失败，则返回错误
        if not sysRec:
            BaseFactory.log.error(context, "fail to get system info after 3 times retry.")
            raise BaseFactory.exception.newToolExceptionByKeys(context, LangKey.FUNC_GET_CNTR_ENC_MODEL_FAIL,
                                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
    # 获取设备型号
    return restUtil.Tlv2Rest.getRecordValue(sysRec, restData.Sys.System.PRODUCT_MODE)


'''
Function: 获取系统的产品类型
Params: 
    context：工具上下文（当sysRec不为None时，此值可以为None）
    sysRec：系统信息。若为None，则表示查询context中的结果
Return: 产品型号类型：-1=获取出错；0=S2600T；1=S5500T；2=独立机头
Resource: 返回值类型值：BaseFactory中的const.SYSMODEL字典
'''
PERSIST_KEY_SYS_PRODUCT_TYPE = "PERSIST_KEY_SYS_PRODUCT_TYPE"


def getSystemProductType(context, systemRec=None):
    # 若系统类型已存在于持久化数据中，则直接返回已有类型，否则重新获取并在持久化后返回
    sysType = BaseFactory.persist.getModule(context, PERSIST_KEY_SYS_PRODUCT_TYPE)
    if sysType:
        sysType = int(sysType)
    else:
        productModel = getSystemProductModel(context, systemRec)
        sysType = getSystemProductTypeByTlvProductModel(context, productModel)
        BaseFactory.persist.setModule(context, PERSIST_KEY_SYS_PRODUCT_TYPE, sysType)

    BaseFactory.log.info(context, "system product type = " + str(sysType))
    return sysType


'''
Function: 获取框中的控制器数量
Params: 
    context：工具上下文
    encId：框ID
Return: 框下的控制器数量
'''


def getCntrNumInEnc(context, encId):
    cntrRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, encId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    cntrRecs = restUtil.Tlv2Rest.filter(cntrRecs, conditionList)
    return len(cntrRecs)


'''
Function: 获取硬盘框的类型，供代码按框类型区分硬件规格
Params: 
    context：工具上下文
    encModel: 从TLV ENCLISURE下查回的框的model
Return: 硬盘框类型：参见BaseFactory中的const.EXPENCMODEL字典
'''


def getExpEncTypeByTlvEncModel(context, encModel):
    # 据硬盘框的model判定判定其高度
    expModelList_T_2U = [restData.Enum.EnclosureModelEnum.EXPSAS2U_12,
                         restData.Enum.EnclosureModelEnum.EXPSAS2U_24]
    expModelList_T_4U = [restData.Enum.EnclosureModelEnum.EXPSAS4U,
                         restData.Enum.EnclosureModelEnum.EXPFC]  # 4U硬盘框未包含高密框EXPSAS4U_75
    expModelList_T_4U75 = [restData.Enum.EnclosureModelEnum.EXPSAS4U_75,
                           restData.Enum.EnclosureModelEnum.EXP_12G_SAS_4U_75]
    expModelList_V3_2U = [restData.Enum.EnclosureModelEnum.EXPSAS2U_25,
                          restData.Enum.EnclosureModelEnum.EXPSAS2U_25_12GLINK,
                          restData.Enum.EnclosureModelEnum.EXP2U_JBOF,
                          restData.Enum.EnclosureModelEnum.EXP_SAS2U_25,
                          restData.Enum.EnclosureModelEnum.CTRL_12GSAS_2U25_EAR_12GLINK,
                          restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_24,
                          restData.Enum.EnclosureModelEnum.EXP2U_JBOF,
                          restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_MID,
                          restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_MID,
                          restData.Enum.EnclosureModelEnum.CTRL_NVMe2U_2C_36,
                          restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_25_LOW,
                          restData.Enum.EnclosureModelEnum.CTRL_SAS2U_2C_12_LOW,
                          restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_25,
                          restData.Enum.EnclosureModelEnum.EXP_IPSAS_2U_12,
                          restData.Enum.EnclosureModelEnum.EXP_IPNVMe_2U_36]
    expModelList_V3_4U = [restData.Enum.EnclosureModelEnum.EXPSAS4U_24_NEW,
                          restData.Enum.EnclosureModelEnum.CTRL_12GSAS_4U24_EAR_12GLINK,
                          restData.Enum.EnclosureModelEnum.EXP_SAS4U_24,
                          restData.Enum.EnclosureModelEnum.EXP_12G_SAS_4U_75,
                          restData.Enum.EnclosureModelEnum.CTRL_4U_4C,
                          ]

    expModel = BaseFactory.const.EXPENCMODEL["ERROR"]
    if encModel in expModelList_T_2U:
        expModel = BaseFactory.const.EXPENCMODEL["T_2U"]
    elif encModel in expModelList_T_4U:
        expModel = BaseFactory.const.EXPENCMODEL["T_4U"]
    elif encModel in expModelList_T_4U75:
        expModel = BaseFactory.const.EXPENCMODEL["T_4U75"]
    elif encModel in expModelList_V3_2U:
        expModel = BaseFactory.const.EXPENCMODEL["V3_2U"]
    elif encModel in expModelList_V3_4U:
        expModel = BaseFactory.const.EXPENCMODEL["V3_4U"]
    else:
        BaseFactory.log.error(context, "Undefined expand enclosure model: " + str(encModel))
        raise BaseFactory.exception.newToolExceptionByKeys(context, LangKey.FUNC_UNDEFINED_DISK_ENC_TYPE_X,
                                                           LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP,
                                                           str(encModel))

    # 返回结果
    BaseFactory.log.info(context,
                         "encModel=" + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrEnclosureModelEnum, encModel)
                         + ", return expModel=" + str(expModel))
    return expModel


'''
Function: 获取FRU所在硬盘框的模型
Params: 
    context：工具上下文
    fruType: 硬盘框上的FRU类型，仅限电源和风扇
    fruId：具体FRU的ID
Return: 硬盘框TLV信息中的model字段
'''


def getExpEncModel(context, fruType, fruId):
    if fruType == restData.Enum.ObjEnum.ENCLOSURE:
        rec = getFruInfo(context, fruType, fruId)
    else:
        # 智能框下风扇所属级联板，此方法可循环获取到所在硬盘框信息
        rec = getFruParentEncRec(context, fruType, fruId)
    encModel = restUtil.Tlv2Rest.getRecordValue(rec, restData.Hardware.Enclosure.MODEL)
    BaseFactory.log.info(context, "the enc model of fruId=" + fruId + ", fruType=" + str(fruType) + " is "
                         + str(encModel) + ", " + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrEnclosureModelEnum,
                                                                             encModel))
    return encModel


'''
Function: 获取FRU所在硬盘框的类型，供代码按框类型区分硬件规格
Params: 
    context：工具上下文
    fruType: 硬盘框上的FRU类型，仅限电源和风扇
    fruId：具体FRU的ID
Return: 硬盘框类型：参见BaseFactory中的const.EXPENCMODEL字典
'''


def getExpEncType(context, fruType, fruId):
    # 从硬盘框的信息中获取框的model
    encModel = getExpEncModel(context, fruType, fruId)

    # 据硬盘框的model判定判定其类型
    return getExpEncTypeByTlvEncModel(context, encModel)


PERSIST_KEY_IS_SNGL_ENGIN = "PERSIST_KEY_IS_SNGL_ENGIN"  # 系统是否为单控制器框环境


def isSingleEngine(context, reQuery=False):
    """判断系统是否为单引擎环境

    :param context:
    :param reQuery:
    :return: 系统是否为单引擎环境。True=是；False=否 or 获取信息失败
    """
    # 若需要重新查询，则清除持久化数据
    if reQuery:
        BaseFactory.persist.delModule(context, PERSIST_KEY_IS_SNGL_ENGIN)

    # 若存在持久化数据，则使用之；否则重新从阵列查询，并更新到持久化数据中
    isSingle = BaseFactory.persist.getModule(context, PERSIST_KEY_IS_SNGL_ENGIN)
    if isSingle is not None:
        isSingle = bool(isSingle)
    else:
        # 默认为单框
        isSingle = True
        # 获取所有控制框信息，若为多框则更新
        allEncRecs = getFruListInfo(context, restData.Enum.ObjEnum.ENCLOSURE)
        condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Enclosure.LOGIC_TYPE,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    restData.Enum.EnclosureTypeEnum.CTRL)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
        tmpRecs = restUtil.Tlv2Rest.filter(allEncRecs, conditionList)
        if len(tmpRecs) > 1:
            isSingle = False
        # 更新持久化数据
        BaseFactory.persist.setModule(context, PERSIST_KEY_IS_SNGL_ENGIN, isSingle)

    # 若存在多个控制框，则非单引擎
    BaseFactory.log.info(context, "the system is single engine? %s" % isSingle)
    return isSingle


def getUsableRedCntr(context, cntrId, falseEngTrueEnc=False, fruType=None):
    """获取可用的冗余控制器

    :param context: 数据上下文
    :param cntrId: 当前控制器ID
    :param falseEngTrueEnc: 冗余方式。False=引擎下冗余；True=框下冗余
    :return: 冗余控制器信息
    """
    # 获取所有控制器信息
    allCntrRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    if not allCntrRecs:
        BaseFactory.log.error(context, "fail to get controller records.")
        return None

    # 获取当前控制器信息
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.ID,
                                                restData.Enum.ConditionTypeEnum.EQ, cntrId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allCntrRecs, conditionList)
    if not tmpRecs:
        BaseFactory.log.error(context, "the controller info of cntrId=" + cntrId + " is not existed.")
        return None
    # 打印结果
    restUtil.Tlv2Rest.logRecList(context, tmpRecs, "current controller info:")

    curCntrRec = tmpRecs[0]
    location = restUtil.Tlv2Rest.getRecordValue(curCntrRec, restData.Hardware.Controller.LOCATION)
    subLoc = location.split(".")[0]  # CTExx

    # 筛选本框上的其他可用控制器信息
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.LOCATION,
                                                restData.Enum.ConditionTypeEnum.LIKE, subLoc)
    condition2 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.ID,
                                                restData.Enum.ConditionTypeEnum.NE, cntrId)
    condition3 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.ONLINE)
    condition4 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.NORMAL)
    if fruType and fruType in [restData.Enum.ObjEnum.CONTROLLER, restData.Enum.ObjEnum.FAN]:
        conditionList = restUtil.Tlv2Rest.getConditionList(condition1, condition2, condition3, condition4)
    else:
        conditionList = restUtil.Tlv2Rest.getConditionList(condition1, condition2)
    tmpRecs = restUtil.Tlv2Rest.filter(allCntrRecs, conditionList)
    if not tmpRecs:
        BaseFactory.log.error(context, "the normal redundant controller info of cntrId="
                              + cntrId + " is not existed.")
        return None

    # 获取冗余控制器信息
    BaseFactory.log.info(context, "falseEngTrueEnc=%s" % falseEngTrueEnc)
    if falseEngTrueEnc:
        # 框下冗余时，直接返回
        # 当前仅18000V3走此分支
        pass
    else:
        # 引擎下冗余时，根据框中的控制器个数进行处理（默认为双控）
        redCntrPairName = {"A": "B",
                           "B": "A",
                           "C": "D",
                           "D": "C",
                           "0": "1",
                           "1": "0",
                           "2": "3",
                           "3": "2",
                           }
        curCntrName = restUtil.Tlv2Rest.getRecordValue(curCntrRec, restData.Hardware.Controller.NAME)  # A, B, C, D
        redCntrName = redCntrPairName.get(curCntrName[-1], "None")
        BaseFactory.log.info(context, "the curCntrName is %s, and the redCntrName is %s"
                             % (curCntrName, redCntrName))
        condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.NAME,
                                                    restData.Enum.ConditionTypeEnum.LIKE, redCntrName)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
        tmpRecs = restUtil.Tlv2Rest.filter(tmpRecs, conditionList)
        if 0 == len(tmpRecs):
            BaseFactory.log.error(context, "the redandent controller(%s) is not existed." % cntrId)
            return None

    # 返回正常的冗余控制器（此时必定存在）
    # 打印结果
    restUtil.Tlv2Rest.logRecList(context, tmpRecs,
                                 "all NORMAL ONLINE redundant controllers of cntr:%s:"
                                 % cntrId)

    return tmpRecs


'''
公用方法：获取指定FRU的指定父FRU信息
Params:
    context=工具上下文
    fruType=待查FRU类型
    fruId=待查FRU ID
    parentFruType=父FRU类型
Return: parentFru信息 或 None(查询不到)
'''


def getFruParentRecord(context, fruType, fruId, parentFruType):
    initFruId = fruId
    fruName = restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)
    parentFruName = restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, parentFruType)

    # 通过parentID和parentType循环获取父FRU信息
    maxLoopNum = 10  # 设置最大循环次数，防呆
    while maxLoopNum > 0:
        # 获取FRU的父FRU信息
        fruRec = getFruInfo(context, fruType, fruId)
        fruType = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.TYPE)

        # 若FRU类型为parentFruType，则返回；否则继续循环或报错
        if parentFruType == fruType:  # FRU类型为控制器
            return fruRec
        else:
            parentType = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.PARENT_TYPE)
            parentId = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.PARENT_ID)

            if not all([parentId, parentType]):# 无效父类型
                BaseFactory.log.error(context,
                                      "Error: parentType=%s, parentId=%s when get parent fru(type:%s) record of fru(type:%s,id:%s)."
                                      % (parentType, parentId, parentFruName, fruName, initFruId))
                return None
            else:  # 刷新FRU信息为父FRU的信息，并继续循环
                fruType = parentType
                fruId = parentId

        # 更新循环变量
        maxLoopNum -= 1

    # 达到最大循环次数，返回无效值
    BaseFactory.log.error(context, "met maxLoopNum:%s when get parent fru(type:%s) record of fru(type:%s,id:%s)."
                          % (maxLoopNum, parentFruName, fruName, initFruId))
    return None


def getFruParentEncRec(context, fruType, fruId):
    """获取FRU所在框TLV记录

    :param context: 数据上下文
    :param fruType: FRU类型
    :param fruId: 具体FRU的ID
    :return: 父框的类型：异常：无法获取到父框信息
    """
    # 获取FRU的父框信息
    encRec = getFruParentRecord(context, fruType, fruId, restData.Enum.ObjEnum.ENCLOSURE)

    # 若无法获取到，则抛出异常
    if not encRec:
        fruName = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, fruType)
        raise BaseFactory.exception.newToolExceptionByKeys(context,
                                                           LangKey.GET_FRU_X_PARENT_ENC_EXCEPTION,
                                                           LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP, fruName)
    else:
        return encRec


'''
公用方法：获取FRU的父框类型
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: 父框的类型：-1=const.RET_ERROR=FRU的Parent不为框；
        若Parent为框，则具体类型参见restData.Enum..CLOSURE_TYPE_E的定义'''


def getFruParentEncType(context, fruType, fruId):
    fruRec = getFruParentEncRec(context, fruType, fruId)
    return restUtil.Tlv2Rest.getRecordValue(fruRec, restData.Hardware.Enclosure.LOGIC_TYPE)


'''
公用方法：获取指定FRU的父框ID
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: None 或  具体FRU记录
'''


def getFruParentEncId(context, fruType, fruId):
    fruRec = getFruParentEncRec(context, fruType, fruId)
    return restUtil.Tlv2Rest.getRecordValue(fruRec, restData.Hardware.Enclosure.ID)


'''
公用方法：获取UI显示的FRU更换状态语言描述
Params: 
    context=数据上下文；
    isReplaced=FRU是否已更换
Return: FRU更换状态的语言描述（待更换，已更换）
'''


def getFruReplStatusDesc(context, isReplaced):
    # 据语言类型和更换状态生成更换描述信息
    desc = ""
    lang = context["lan"]
    if "zh" == lang:
        if isReplaced:
            desc = u"已更换"
        else:
            desc = u"待更换"
    else:
        if isReplaced:
            desc = "Replaced"
        else:
            desc = "Replacing"
    # 返回信息
    return desc


'''
公用方法：展示未连接末端端口的值
Params: 
    context=数据上下文；
Return: FRU更换状态的语言描述
'''


def getSasPortCurPeerPortDesc(context):
    # 据语言类型和更换状态生成更换描述信息
    desc = ""
    lang = context["lan"]
    if "zh" == lang:
        desc = u"末端端口"
    else:
        desc = "End Port"
    # 返回信息
    return desc


'''
判断是否是2U控制框下的风扇
'''


def is2U_Fan(context, fruType, fruId):
    encType = getFruParentEncType(context, fruType, fruId)
    if restData.Enum.EnclosureTypeEnum.CTRL == encType:  # FRU的Parent为控制框
        sysModel = getSystemProductType(context)
    isFan2U = False
    if restData.Enum.ObjEnum.FAN == fruType:
        if restData.Enum.EnclosureTypeEnum.CTRL == encType:  # FRU的Parent为控制框
            if sysModel is BaseFactory.const.SYSMODEL["5X00V3"]:
                BaseFactory.log.info(context, "the FAN of ctrl enc(26T,55T,5XV3) must be bundle.")
                isFan2U = True
            else:
                pass
    return isFan2U


'''
公用子方法：判断FRU是否为一体化FRU
Params: 
    context=数据上下文；
    fruType=FRU类型；
    fruId=FRU的ID
    [Notes]:支持的一体化FRU一览表：
    S2600T/S5500T: FAN + BBU
    S5300V3/S5500V3: FAN + BBU + POWER
    V3 Disk Enc 2U/4U(slot 2&3): FAN + POWER
Return: 指定的FRU是否为一体化FRU。
'''


def isFruBundle(context, fruType, fruId):
    # 初始化信息
    encType = getFruParentEncType(context, fruType, fruId)
    strFruType = restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)
    BaseFactory.log.info(context, "the parent enc type of current %s:%s is %s."
                         % (strFruType, fruId,
                            restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrEnclosureTypeEnum, encType)))
    expEncModel = None
    sysModel = None
    if restData.Enum.EnclosureTypeEnum.EXP == encType:  # FRU的Parent为硬盘框
        expEncModel = getExpEncType(context, fruType, fruId)
    elif restData.Enum.EnclosureTypeEnum.CTRL == encType:  # FRU的Parent为控制框
        sysModel = getSystemProductType(context)

    # 据FRU类型判断其是否为一体化
    isBundleFru = False
    if restData.Enum.ObjEnum.FAN == fruType:
        if restData.Enum.EnclosureTypeEnum.EXP == encType:  # FRU的Parent为硬盘框
            if expEncModel in [BaseFactory.const.EXPENCMODEL["V3_2U"],
                               BaseFactory.const.EXPENCMODEL["V3_4U"]]:
                BaseFactory.log.info(context, "the enc is V3 disk enc, FAN may be bundle.")
                isBundleFru = True
        elif restData.Enum.EnclosureTypeEnum.CTRL == encType:  # FRU的Parent为控制框
            if sysModel in [BaseFactory.const.SYSMODEL["S2600T"],
                            BaseFactory.const.SYSMODEL["S5500T"],
                            BaseFactory.const.SYSMODEL["5X00V3"]]:
                BaseFactory.log.info(context, "the FAN of ctrl enc(26T,55T,5XV3) must be bundle.")
                isBundleFru = True
        else:
            pass
    elif restData.Enum.ObjEnum.POWER == fruType:
        if restData.Enum.EnclosureTypeEnum.EXP == encType:  # FRU的Parent为硬盘框
            if expEncModel in [BaseFactory.const.EXPENCMODEL["V3_2U"],
                               BaseFactory.const.EXPENCMODEL["V3_4U"]]:
                BaseFactory.log.info(context, "the enc is V3 disk enc, POWER may be bundle.")
                isBundleFru = True
        elif restData.Enum.EnclosureTypeEnum.CTRL == encType:  # FRU的Parent为控制框
            if sysModel in [BaseFactory.const.SYSMODEL["5X00V3"]]:
                BaseFactory.log.info(context, "the POWER of ctrl enc(5XV3) must be bundle.")
                isBundleFru = True
        else:
            pass
    elif restData.Enum.ObjEnum.BACKUP_POWER == fruType:
        if restData.Enum.EnclosureTypeEnum.CTRL == encType:  # FRU的Parent为控制框
            if sysModel in [BaseFactory.const.SYSMODEL["5X00V3"]]:
                BaseFactory.log.info(context, "the BACKUP_POWER of ctrl enc(26T,55T,5XV3) must be bundle.")
                isBundleFru = True
        else:
            pass
    else:
        pass

    # 记录并返回判断结果
    if False == isBundleFru:
        BaseFactory.log.info(context, "the current %s:%s is not a bundel fru." % (strFruType, fruId))
    return isBundleFru


'''
公用子方法：判断FRU是否为故障状态，默认仅检查FAULT状态，多用于更换前检查（故障时便不用执行具体的更换前检查）
Params: 
    context=数据上下文；
    fruType=FRU类型；
    fruId=FRU的ID
    otherFaultStatusList=除FAULT外的其他故障状态，可不指定；如指定，可使用Tlv.Record.getMultiConditionValueList()来生成参数
       e.g faultValList = Tlv.Record.getMultiConditionValueList(restData.Enum.HealthStatusEnum.PRE_FAIL, restData.Enum.HealthStatusEnum.UNKNOWN)
Return: 指定的FRU是否故障。 True=故障； False=Fru非故障 或 系统故障
'''


def isFruFault(context, fruType, fruId, otherFaultHealthStatusList=[], otherFaultRunningStatusList=[]):
    # 获取FRU健康和运行状态，并容错和记录日志
    healthStat = None
    runningStatus = None
    try:
        healthStat = getFruHealthStatus(context, fruType, fruId)
        runningStatus = getFruRunningStatus(context, fruType, fruId)
    except Exception as ex:
        return fruFaultConfirm(context, ex)

    # 正常时记录FRU健康状态，方便调试和问题定位
    BaseFactory.log.info(context, "the health status of fru (id=" + str(fruId)
                         + ", type=" + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)
                         + ") is " + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrHealthStatusEnum, healthStat))
    BaseFactory.log.info(context, "the running status of fru (id=" + str(fruId)
                         + ", type=" + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)
                         + ") is " + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrRunningStatusEnum, runningStatus))

    # 光模块、接口卡、线缆只关注运行状态，不关注健康状态
    # 如果运行状态为（LINk DOWN，POWER OFF， POWER_ON_ERROR）其中之一，则FRU故障，否则FRU不为故障状态
    if fruType in (
            restData.Enum.ObjEnum.ETH_PORT, restData.Enum.ObjEnum.FC_PORT,
            restData.Enum.ObjEnum.SFP_OPTICAL_TRANSCEIVER,
            restData.Enum.ObjEnum.INTF_MODULE):
        return runningStatus in (restData.Enum.RunningStatusEnum.LINK_DOWN, restData.Enum.RunningStatusEnum.POWER_OFF,
                                 restData.Enum.RunningStatusEnum.POWER_ON_ERROR)

    # 如果健康状态为FAULT或任一调用者指定的故障状态，则FRU故障，否则FRU不为故障状态
    if healthStat == restData.Enum.HealthStatusEnum.FAULT:
        return True
    elif healthStat in otherFaultHealthStatusList:
        return True

    # 若FRU已下电，则认为FRU故障
    if restData.Enum.RunningStatusEnum.POWER_OFF == runningStatus:
        return True
    elif runningStatus in otherFaultRunningStatusList:
        return True

    # 返回FRU为非故障状态
    return False


def isFRUNotOnline(context, fruType, fruId):
    """判断FRU是否故障，不考虑健康状态，只考虑运行状态是否故障（非ONLINE）

    :param context: 数据上下文
    :param fruType: FRU类型
    :param fruId: FRU的ID
    :return:
    """
    # 获取FRU健康和运行状态，并容错和记录日志
    runningStatus = None
    try:
        runningStatus = getFruRunningStatus(context, fruType, fruId)
    except Exception as ex:
        return fruFaultConfirm(context, ex)

    # 正常时记录FRU健康状态，方便调试和问题定位
    BaseFactory.log.info(context, "the running status of fru (id=" + str(fruId)
                         + ", type=" + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType)
                         + ") is " + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrRunningStatusEnum, runningStatus))

    # 如果健康状态为FAULT或任一调用者指定的故障状态，则FRU故障，否则FRU不为故障状态
    if runningStatus != restData.Enum.RunningStatusEnum.ONLINE:
        return True

    # 返回FRU为非故障状态
    return False


def fruFaultConfirm(context, fruException):
    """Function: FRU故障确认
    Params: 工具上下文；查询FRU信息时抛出的异常
    Return: True=fru为故障状态；fruException=非FRU故障导致的异常，抛出原异常
    原理：
    #当FRU故障，设备或网络异常时，均可能获取不出来TLV信息，导致抛出异常。
    #此时，需再判断下系统状态，若系统状态正常，则说明是FRU本身故障导致
    Attention: 该函数不能用于checkSysRunningStatus和Tlv.execmd，否则将引起无限循环调用

    :param context:
    :param fruException:
    :return:
    """
    try:
        if checkSysRunningStatus(context):  # 系统状态正常
            BaseFactory.log.error(context, "the fru must be fault since the system is normal.")
            return True
        else:
            # 若系统状态也异常，则说明是设备或网络原因导致，此时返回False，让其他检查继续执行并报错。
            # 此分支可能不会执行到，此时将直接抛出异常。这种情况应继续抛出原来的异常
            BaseFactory.log.error(context, "the device or network must be abnormal since the system is abnormal.")
            raise fruException
    except:
        raise fruException


def getFruListInfo(context, fruType, isShowLog=False):
    """获取FRU硬件列表.

    :param context: 上下文
    :param fruType: FRU 类型
    :param isShowLog: 是否打印查询日志。
    :return:
    """
    # 参数检查
    if not context or not fruType:
        BaseFactory.log.error(context, "getFruInfo: exist invalid parameter. fruType=" + str(fruType))
        return None
    # 查询状态
    param0 = (restData.PublicAttributes.TYPE, fruType)
    paramlist = restUtil.Tlv2Rest.getParamList(param0)
    rest = contextUtil.getRest(context)
    if fruType in (restData.Enum.ObjEnum.DISK,
                   restData.Enum.ObjEnum.REPLICATIONPAIR,
                   restData.Enum.ObjEnum.HOST):
        recs = restUtil.Tlv2Rest.execCmdExtreme(rest, restData.TlvCmd.GET_BATCH_NEXT, paramlist, isShowLog=isShowLog)
    else:
        recs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_BATCH_NEXT, paramlist)
    # 设置返回值
    return recs


def get_fru_list_info_with_not_support_type(context, fru_type, is_show_log=False):
    """获取FRU硬件列表.（当不支持该硬件时，返回空列表）

    :param context: 上下文
    :param fru_type: FRU 类型
    :param is_show_log: 是否打印查询日志。
    :return: 硬件列表信息
    """
    # 参数检查
    if not context or not fru_type:
        BaseFactory.log.error(context, "getFruInfo: exist invalid parameter. fruType={}".format(str(fru_type)))
        return None
    # 查询状态
    param0 = (restData.PublicAttributes.TYPE, fru_type)
    param_list = restUtil.Tlv2Rest.getParamList(param0)
    rest = contextUtil.getRest(context)
    try:
        if fru_type in (restData.Enum.ObjEnum.DISK, restData.Enum.ObjEnum.REPLICATIONPAIR, restData.Enum.ObjEnum.HOST):
            recs = restUtil.Tlv2Rest.execCmdExtreme(rest, restData.TlvCmd.GET_BATCH_NEXT, param_list,
                                                    isShowLog=is_show_log)
        else:
            recs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_BATCH_NEXT, param_list)
        # 设置返回值
        return recs
    except Exception as ex:
        error_code = ex.args[0]
        # 指定对象不支持时，错误码为50331651，返回空列表
        if str(error_code) == "50331651":
            return []
        raise ex


'''
公用方法：获取机柜下所有的控制器数目
Params: context=数据上下文；
Return: ctrlTotal：机柜下的控制器数目
'''


def getCtrlTotalInBay(context):
    # get number of controllers under a bay
    fruType = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.BAY)
    rest = contextUtil.getRest(context)
    # TODO
    bayRecs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_BATCH_NEXT, restUtil.Tlv2Rest.getParamList(fruType))

    # 如果取得为空的话，直接返回None
    if bayRecs == None or len(bayRecs) == 0:
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_GET_INFO_Y_FAIL)
        return None
    isIPScaleOut = (restData.Hardware.Bay.IS_IPSCALE_OUT, True)
    ctrlTotals = []
    for bayRec in bayRecs:
        bayId = restUtil.Tlv2Rest.getRecordValue(bayRec, restData.PublicAttributes.ID)
        bayRec = restUtil.Tlv2Rest.getExpansionBayIdRecord(rest, bayId, isIPScaleOut)
        # 如果取得为空的话，后续处理不执行
        if bayRec == None or len(bayRec) == 0:
            continue
        ctrlTotal = restUtil.Tlv2Rest.getRecordValue(bayRec, restData.Hardware.Bay.CTRL_TOTAL)
        ctrlTotals.append(int(str(ctrlTotal)))
    BaseFactory.log.info(context, "ctrlTotals: %s" % ctrlTotals)
    if sum(ctrlTotals) != int(str(ctrlTotal)) * len(ctrlTotals):
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_GET_INFO_Y_FAIL)
        return None

    return int(str(ctrlTotal))


'''
公用方法：获取指定类型的具体FRU信息
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: None 或  具体FRU记录
'''


def getFruInfo(context, fruType, fruId):
    # 参数检查
    if None == context or None == fruType or None == fruId:
        BaseFactory.log.error(context, "getFruInfo: exist invalid parameter. fruType="
                              + str(fruType) + ", fruId=" + str(fruId))
        return None
    # 查询状态
    param0 = (restData.PublicAttributes.TYPE, fruType)
    param1 = (restData.PublicAttributes.ID, fruId)
    paramlist = restUtil.Tlv2Rest.getParamList(param0, param1)
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET, paramlist, False)
    # 设置返回值
    if None == rec:
        fruInfo = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, fruType) + "(ID:" + str(fruId) + ")"
        BaseFactory.log.error(context, u"获取信息失败。请确认" + fruInfo + u"是否存在。")
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_GET_INFO_Y_FAIL, None, fruInfo)
    return rec


'''
公用方法：获取指定类型的具体FRU信息（通过先批量获取，然后再筛选方式实现，用于处理故障FRU的获取）
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: None 或  具体FRU记录
'''


def getFruInfoByBatch(context, fruType, fruId):
    # 参数检查
    if None == fruType or None == fruId:
        BaseFactory.log.error(context, "getFruInfoByBatch: exist invalid parameter. fruType="
                              + str(fruType) + ", fruId=" + str(fruId))
        return None

    # 批量获取并筛选出指定的FRU
    allFruList = getFruListInfo(context, fruType)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                fruId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    tmpRecs = restUtil.Tlv2Rest.filter(allFruList, conditionList)
    BaseFactory.log.info(context, "========getFruInfoByBatch========:\n%s" % str(tmpRecs))
    # 返回结果
    if 0 == len(tmpRecs):
        fruInfo = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, fruType) + "(ID:" + str(fruId) + ")"
        BaseFactory.log.error(context, u"获取信息失败。请确认" + fruInfo + u"是否存在。")
        raise BaseFactory.exception.newToolExceptionByKeys(context, LangKey.FUNC_GET_INFO_Y_FAIL,
                                                           LangKey.FUNC_GET_INFO_Y_FAIL, None, fruInfo)
    return tmpRecs[0]


'''
公用方法：获取指定类型的具体FRU信息（通过先批量获取，然后再筛选方式实现，用于处理故障FRU的获取）
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: None 或  具体FRU记录
'''


def getFruInfoByLocation(context, fruType, location):
    # 参数检查
    if None == fruType or None == location:
        BaseFactory.log.error(context, "getFruInfoByLocation: exist invalid parameter. fruType="
                              + str(fruType) + ", location=" + str(location))
        return None

    # 批量获取并筛选出指定的FRU
    allFruList = getFruListInfo(context, fruType)
    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.LOCATION,
                                                restData.Enum.ConditionTypeEnum.EQ, location)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    tmpRecs = restUtil.Tlv2Rest.filter(allFruList, conditionList)

    # 返回结果
    if 0 == len(tmpRecs):
        fruInfo = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, fruType) + "(" + str(location) + ")"
        BaseFactory.log.error(context, u"获取信息失败。请确认" + fruInfo + u"是否存在。")
        raise BaseFactory.exception.newToolExceptionByKeys(context, LangKey.FUNC_GET_INFO_Y_FAIL,
                                                           LangKey.FUNC_GET_INFO_Y_FAIL, None, fruInfo)
    return tmpRecs[0]


def get_fru_id_by_location_id(context, fru_type, location):
    """
    通过location获取fruId
    :param context: 上下文信息
    :param fru_type: fru类型
    :param location: locationId
    :return: str fruId
    """
    fru_info = getFruInfoByLocation(context, fru_type, location)
    return restUtil.Tlv2Rest.getRecordValue(fru_info, restData.PublicAttributes.ID)


'''
公用方法：获取指定类型的具体FRU的location
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: 
'''


def getFruLocation(context, fruType, fruId):
    # 获取FRU信息
    rec = getFruInfoByBatch(context, fruType, fruId)
    # 设置返回值
    location = None
    if None != rec:
        location = restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.LOCATION)  # 7=healthStatus
    else:
        BaseFactory.log.error(context, "fail to get the location of fru (id=" + str(fruId)
                              + ", type=" + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType) + ").")
    return location


'''
公用方法：获取指定类型的具体FRU的健康状态
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: 
'''


def getFruHealthStatus(context, fruType, fruId):
    # 获取FRU信息
    rec = getFruInfoByBatch(context, fruType, fruId)
    # 设置返回值
    status = None
    if None != rec:
        status = restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.HEALTH_STATUS)
    else:
        BaseFactory.log.error(context, "fail to get the health status of fru (id=" + str(fruId)
                              + ", type=" + restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrObjEnum, fruType) + ").")
    return status


'''
公用方法：获取指定类型的具体FRU的运行状态
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: 
'''


def getFruRunningStatus(context, fruType, fruId):
    rec = getFruInfoByBatch(context, fruType, fruId)
    return restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.RUNNING_STATUS)


'''
公用方法：获取指定类型的具体FRU的运行状态
Params: context=数据上下文；fruType=FRU类型；fruId=具体FRU的ID
Return: 
'''


def getFruRunningStatusByLocation(context, fruType, fruLoc):
    rec = getFruInfoByLocation(context, fruType, fruLoc)
    return restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.RUNNING_STATUS)


'''
公用方法：获取所有控制器的ID列表
'''


def getCntrIdList(context, *filterParams):
    ctrlIdList = []
    # 获取控制器信息
    recs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    if len(filterParams) > 0:  # 执行条件筛选
        recs = restUtil.Tlv2Rest.filter(recs, filterParams)

    # 对本控制框中所有控制器的CPU利用率求和
    recNum = len(recs)
    for index in range(0, recNum):
        rec = recs[index]
        ctrlId = restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.ID)
        BaseFactory.log.info(context, "ctrlId=" + str(ctrlId) + ", rec=" + str(rec))
        ctrlIdList.append(ctrlId)

    BaseFactory.log.info(context, "all cntrId list is %s." % ctrlIdList)
    return ctrlIdList


def getSysAlarmCount(context):
    """获取系统中告警的数量

    :param context:
    :return:
    """
    paramlist = restUtil.Tlv2Rest.getParamList()
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_ALARM_COUNT, paramlist, False)
    if rec is None:
        errInfo = u"获取系统告警数量失败。"
        BaseFactory.log.error(context, errInfo)
        return BaseFactory.returnVal.dict2(False, errInfo)
    else:
        almCount = restUtil.Tlv2Rest.getRecordValue(rec, restData.Sys.Alarm.CMO_ALARM_COUNT)
        return BaseFactory.returnVal.dict2(True, almCount)


def getSysAlarmsInfo(context):
    """获取系统中的所有告警信息

    :param context:
    :return:
    """
    almRecs = None
    curAlmIndex = 0
    retVal = getSysAlarmCount(context)
    if not retVal["succ"]:
        return retVal
    almCount = retVal["info"]
    # 循环获取所有告警信息
    while curAlmIndex < almCount:
        # 获取信息
        beforeEndIndex = curAlmIndex + config.REST_RECORD_MAXNUM
        strConditions = restUtil.Tlv2Rest.getQueryConditionString(None, None,
                                                                  {"start": curAlmIndex, "end": beforeEndIndex})
        param0 = (restData.PublicAttributesExtend.QUERY_CONDITION, strConditions)
        paramlist = restUtil.Tlv2Rest.getParamList(param0)
        rest = contextUtil.getRest(context)
        recs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_CURRENT_ALARM, paramlist)
        if recs is None:
            errInfo = u"获取系统告警失败。范围从" + str(curAlmIndex) + u"到" + str(beforeEndIndex) + u"。"
            BaseFactory.log.error(context, errInfo)
            # 不能忽略没获取到的部分，直接报错
            errMsg = BaseFactory.lang.getDesc(context, LangKey.FUNC_GET_SYS_ALARM_FAIL)
            return BaseFactory.returnVal.dict2(False, errMsg)
        # 合并信息
        if 0 == curAlmIndex:
            almRecs = recs
        else:
            almRecs.extend(recs)
        # 更新循环条件
        curAlmIndex = beforeEndIndex
    # 返回执行结果
    return BaseFactory.returnVal.dict2(True, almRecs)


'''
公用方法：获取系统中告警的数量
'''


def getSysEventCount(context):
    paramlist = restUtil.Tlv2Rest.getParamList()
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_HISTORY_ALARM_COUNT, paramlist, False)
    if None == rec:
        errInfo = u"获取系统事件数量失败。"
        BaseFactory.log.error(context, errInfo)
        return BaseFactory.returnVal.dict2(False, errInfo)
    else:
        almCount = restUtil.Tlv2Rest.getRecordValue(rec, restData.Sys.Alarm.CMO_ALARM_COUNT)
        return BaseFactory.returnVal.dict2(True, almCount)


def getSystime(context):
    paramlist = restUtil.Tlv2Rest.getParamList()
    rest = contextUtil.getRest(context)
    resc = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_SYS_UTC_TIME, paramlist, False)
    sysTime = restUtil.Tlv2Rest.getRecordValue(resc, restData.Sys.SystemUtcTime.CMO_SYS_UTC_TIME)
    return sysTime


'''
公用方法：获取系统中一定条件（对象类型和起始时间）下的事件信息
'''


def getSysEventsInfo(context, eventIdList, conditionList, paramTime=None):
    curAlmIndex = 0
    sysTime = getSystime(context)
    # 过滤筛选条件
    condition0 = restUtil.Tlv2Rest.getCondition(restData.Sys.Alarm.CMO_ALARM_ID,
                                                restData.Enum.ConditionTypeEnum.EQOR, eventIdList)
    conditionList.append(condition0)
    BaseFactory.log.info(context, "conditionList:%s" % str(conditionList))

    while True:
        # 获取信息
        beforeEndIndex = curAlmIndex + config.REST_RECORD_MAXNUM
        queryDictList = [{"type": "EQUAL", "index": restData.Sys.Alarm.ALARM_OBJ_TYPE["index"],
                          "value": restData.Enum.ObjEnum.DiskPool}]
        if paramTime != None:
            queryDictList.append({"type": "RANGE", "index": restData.Sys.Alarm.CMO_ALARM_TIME["index"],
                                  "value": [paramTime - 1, sysTime + 1]})
        strConditions = restUtil.Tlv2Rest.getQueryConditionString(queryDictList, None,
                                                                  {"start": curAlmIndex, "end": beforeEndIndex})
        param0 = (restData.PublicAttributesExtend.QUERY_CONDITION, strConditions)
        paramlist = restUtil.Tlv2Rest.getParamList(param0)
        rest = contextUtil.getRest(context)
        recs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_HISTORY_ALARM, paramlist, True)

        if None == recs:
            errInfo = u"获取系统事件失败。范围从" + str(curAlmIndex) + u"到" + str(beforeEndIndex) + u"。"
            BaseFactory.log.error(context, errInfo)
            # 不能忽略没获取到的部分，直接报错
            return None

        eventRecs = restUtil.Tlv2Rest.filter(recs, conditionList)
        BaseFactory.log.info(context, "get eventRecs by conditons(num:%d):%s" % (len(eventRecs), str(eventRecs)))

        if len(eventRecs) > 0:
            return eventRecs
        if len(recs) < config.REST_RECORD_MAXNUM:
            return None

        # 更新循环条件
        time.sleep(1)
        curAlmIndex = beforeEndIndex


'''
Sub Function: 通过控制框ID获取该框所属的所有控制器信息
Return: 控制器列表（一定存在存在） or None（encId不存在）
'''


def getCntrByEngId(context, encId, allCtrlEncRecs=None, allCntrRecs=None):
    # 获取系统中的所有引擎信息
    if None == allCtrlEncRecs:
        allCtrlEncRecs = getAllCtrlEncInfo(context)

    # 获取engId对应的引擎信息
    condition1 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.ID,
                                                restData.Enum.ConditionTypeEnum.EQ, encId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allCtrlEncRecs, conditionList)
    if 0 == len(tmpRecs):
        BaseFactory.log.error(context, "the controllers of engin " + str(encId) + " is not existed.")
        return None
    # 打印结果
    restUtil.Tlv2Rest.logRecList(context, tmpRecs)  # for test

    # 调用其他方法获得控制器（None 或 一定不为空的记录列表）
    return getCntrByEncRec(context, tmpRecs[0], allCntrRecs)


'''
Sub Function: 通过控制框信息获取该框所属的所有控制器信息
Return: 控制器列表
'''


def getCntrByEncRec(context, ctrlEncRec, allCntrRecs=None):
    # 获取框下控制器信息
    encId = restUtil.Tlv2Rest.getRecordValue(ctrlEncRec, restData.Hardware.Enclosure.ID)
    if None == allCntrRecs:
        allCntrRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)

    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, encId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allCntrRecs, conditionList)

    return tmpRecs


'''
子方法：检查端口状态（SAS、PCIe）
'''


def checkPortStatus(context, portType, portId):
    # 获取端口运行状态
    runStatus = getFruRunningStatus(context, portType, portId)

    # 端口已连接，则返回检查通过
    if runStatus == restData.Enum.RunningStatusEnum.LINK_UP:
        return True
    # 端口正在启动中，则轮询
    elif runStatus == restData.Enum.RunningStatusEnum.STARTING:
        # 以1S为周期轮询端口状态，连接后通过
        waitSec = 0
        while waitSec < 300:
            # 睡1S
            time.sleep(1)
            waitSec += 1
            # 重查运行状态
            runStatus = getFruRunningStatus(context, portType, portId)
            # 若端口状态为已连接，则返回检查通过
            if runStatus == restData.Enum.RunningStatusEnum.LINK_UP:
                return True
        # 300S后端口仍未连接，则返回错误
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_PORT_X_START_TIMEOUT,
                                               LangKey.PUBSUG_CHECK_PORT_NORMAL, str(portId))
        return False
    # 端口未运行，则报错
    elif runStatus == restData.Enum.RunningStatusEnum.NOT_RUNNING:
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_PORT_X_NOT_WORK, LangKey.PUBSUG_CHECK_PORT_NORMAL,
                                               str(portId))
        return False
    # 端口处于其他状态时，报错
    else:
        strRunStatus = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrRunningStatusEnum, runStatus)
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_PORT_X2_WORK_ABNORMAL,
                                               LangKey.PUBSUG_CHECK_PORT_NORMAL, [str(portId), strRunStatus])
        return False


'''
子方法：获取所有控制框信息
'''


def getAllCtrlEncInfo(context):
    allEncRecs = getFruListInfo(context, restData.Enum.ObjEnum.ENCLOSURE)

    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Enclosure.LOGIC_TYPE,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.EnclosureTypeEnum.CTRL)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    allCtrlRecs = restUtil.Tlv2Rest.filter(allEncRecs, conditionList)

    return allCtrlRecs


def checkCpuUsage(context, allCtrlEncRecs):
    """检查所有控制框中所有控制器的CPU利用率

    :param context: 上下文
    :param allCtrlEncRecs: 所有控制框信息
    :return:
    """
    # 每个控制器允许的最大CPU利用率
    MAX_USAGE_PER_CNTR = 40

    # 遍历所有引擎，检查每个引擎是否满足通过条件
    isPass = True
    allCntrRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    for cntrEncRec in allCtrlEncRecs:
        # 获取当前引擎下的所有控制器信息
        cntrEncId = restUtil.Tlv2Rest.getRecordValue(cntrEncRec, restData.PublicAttributes.ID)
        curCntrRecs = getCntrByEncRec(context, cntrEncRec, allCntrRecs)
        cntrNum = len(curCntrRecs)

        # 统计当前框中所有控制器的IO并发情况
        totalUsage = 0
        for cntrRec in curCntrRecs:
            cntrId = restUtil.Tlv2Rest.getRecordValue(cntrRec, restData.PublicAttributes.ID)
            cpuUsage = restUtil.Tlv2Rest.getRecordValue(cntrRec, restData.Hardware.Controller.CPU_USAGE)
            BaseFactory.log.info(context, "cpuUsage=%s, cntrId=%s" % (cpuUsage, cntrId))
            totalUsage = totalUsage + cpuUsage

        # 若当前引擎CPU总利用率小于允许的最大CPU利用率限制时检查通过
        maxUsageLimit = cntrNum * MAX_USAGE_PER_CNTR
        BaseFactory.log.info(context, "totalUsage=%s, maxUsageLimit=%s, cntrEncId=%s"
                             % (totalUsage, maxUsageLimit, cntrEncId))
        if totalUsage >= maxUsageLimit:
            isPass = False
            BaseFactory.log.error(context, "the cpu usage is exceed limitation under cntrEncId:%s" % cntrEncId)

    # 判断并返回
    if True == isPass:
        BaseFactory.log.info(context, "the cpu usage is normal in all ctrl enc.")
        return True
    else:
        return False


def checkBackIoConcurrent(context, allCtrlRecs, cntrRecs):
    """检查每个控制框中后端IO并发数

    :param context: 上下文
    :param allCtrlRecs: 所有控制框信息
    :param cntrRecs: 所有控制器信息
    :return:
    """
    encNum = len(allCtrlRecs)
    # 遍历所有引擎，检查每个引擎是否满足通过条件
    isPass = True
    for index in range(0, encNum):
        # 获取当前引擎下的所有控制器信息
        tmpRecs = getCntrByEncRec(context, allCtrlRecs[index], cntrRecs)

        # 统计当前框中所有控制器的IO并发情况
        totalBackIo = 0
        totalBackLimit = 0
        cntrNum = len(tmpRecs)
        for ic in range(0, cntrNum):
            cntrRec = tmpRecs[ic]
            ctrlId = restUtil.Tlv2Rest.getRecordValue(cntrRec, restData.Hardware.Controller.ID)

            # 如果控制器状态不为online，则跳过该控制器的运行状态检查
            # offline的控制器不再接管业务，但V3中查询后端并发信息时会报对象不存在错误，此处跳过
            runStat = restUtil.Tlv2Rest.getRecordValue(cntrRec, restData.Hardware.Controller.RUNNING_STATUS)
            if restData.Enum.RunningStatusEnum.OFFLINE == runStat:
                BaseFactory.log.info(context, "the controller running status of ctrlId=" + ctrlId + " is " +
                                     restUtil.Tlv2Rest.getStrEn(restData.EnumStr.StrRunningStatusEnum, runStat) +
                                     ", not online, skip to GET_BACK_END_CONCURRENT of it.")
                continue

            # 获取当前控制器的后端IO并发情况
            param0 = (restData.Sys.ControllerIo.ID, ctrlId)
            paramlist = restUtil.Tlv2Rest.getParamList(param0)
            rest = contextUtil.getRest(context)
            rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET_BACK_END_CONCURRENT, paramlist, False)

            # 后端IO并发和并发限制
            if False == restUtil.Tlv2Rest.isItemExisted(rec, restData.Sys.ControllerIo.BACK_END_IO) \
                    or False == restUtil.Tlv2Rest.isItemExisted(rec, restData.Sys.ControllerIo.BACK_END_LIMIT):
                BaseFactory.log.error(context, "the backEndIO or backEndLimit field is not existed in " + str(rec))
                return False
            totalBackIo = totalBackIo + restUtil.Tlv2Rest.getRecordValue(rec, restData.Sys.ControllerIo.BACK_END_IO)
            totalBackLimit = totalBackLimit + restUtil.Tlv2Rest.getRecordValue(rec,
                                                                               restData.Sys.ControllerIo.BACK_END_LIMIT)

        # 若当前引擎总后端IO并发超过总并发限制时，检查不通过
        BaseFactory.log.info(context, "the back io concurrent is " + str(totalBackIo) + ", limitation is "
                             + str(totalBackLimit) + " in " + str(allCtrlRecs[index]))
        if totalBackIo > 0 and totalBackIo >= totalBackLimit:
            isPass = False
            BaseFactory.log.error(context,
                                  "the back io concurrent " + str(totalBackIo) + " is not little than limitation "
                                  + str(totalBackLimit) + " in ctrl enc rec: " + str(allCtrlRecs[index]))

    # 判断并返回
    if isPass:
        BaseFactory.log.info(context, "the io concurrent is normal in all ctrl enc.")
        return True
    else:
        return False


'''
公用子方法：检查FRU基本的健康状态，用于更换后检查，保证更换后的FRU健康状态正常
'''


def isReplacedFruNormal(context, fruType, fruId, bundleFruType=None, bundleFruId=None):
    '''
    140331-Qeuestion: FRU被拔出后的约10S内，脚本将仍能查询到在线的FRU信息。
    Reason: 因为TLV查回来的信息是系统管理缓存的信息。硬件信息的刷新还需约10S。
    Solution: 在用户确认FRU更换后，再进行FRU检查前，先行睡10S，以避免因用户操作过快而导致工具误检。
    Attention: 此处理办法已能满足大多数FRU的需求。若某FRU仍无法满足要求(如硬盘、控制器等)，需在FRU内部再行进行延时处理。
    Status: done
    '''
    # 先行睡10S（=5+1+4），以避免因用户操作过快而导致工具误检
    BaseFactory.log.info(context, "Fru Info: fruType=%s, fruId=%s, bundleFruType=%s, bundleFruId=%s" %
                         (str(fruType), str(fruId), str(bundleFruType), str(bundleFruId)))
    time.sleep(10)

    # 获取FRU的健康状态
    rec = getFruInfo(context, fruType, fruId)
    status = restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.HEALTH_STATUS)
    # 判断健康状态是否正常（当状态为正常，备电不足（电池），认为状态正常）
    # 注意：FRU、网络或设备异常导致status=None时的处理也已包含在下面的逻辑中
    if status != restData.Enum.HealthStatusEnum.NORMAL \
            and status != restData.Enum.HealthStatusEnum.POWER_NOT_ENOUGH:

        # 获取FRU描述信息
        fruInfo = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, fruType)
        location = restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.LOCATION)
        if "" != location and "None" != location:
            fruInfo = fruInfo + "(" + location + ")"

        # 获取一体化FRU描述信息
        bundleFruInfo = None
        if None != bundleFruType:
            bundleFruInfo = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, bundleFruType)
            if None != bundleFruId:
                bundleFruLoc = getFruLocation(context, bundleFruType, bundleFruId)
                if "" != bundleFruLoc and "None" != bundleFruLoc:
                    bundleFruInfo = bundleFruInfo + "(" + bundleFruLoc + ")"
                else:
                    pass
            else:
                pass
        else:
            pass

        # 设置错误信息并返回不通过
        BaseFactory.log.error(context, "fruInfo=%s, bundleFruInfo=%s, health status=%s" %
                              (fruInfo, bundleFruInfo, str(status)))
        if None != bundleFruInfo:  # 一体化FRU
            if restData.Enum.HealthStatusEnum.POWER_NO_INPUT == status:
                # 电源线缆未连接特异处理分支：若为电源线缆未连接，则刷新错误信息
                BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_REPLEACED_POWER_X1_BIND_IN_X2_NO_CABLE,
                                                      [location, bundleFruInfo])
            else:  # 其他FRU处理分支
                BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_REPLEACED_FRU_X1_BIND_IN_X2_ABNORMAL,
                                                      [fruInfo, bundleFruInfo])
        else:  # 独立FRU
            if restData.Enum.HealthStatusEnum.POWER_NO_INPUT == status:
                # 电源线缆未连接特异处理分支：若为电源线缆未连接，则刷新错误信息
                BaseFactory.result.setResultFailByKey(context, LangKey.POWER_Y_CABLE_IS_DISCONNECTED, None, location)
            # 其他FRU处理分支
            elif fruType == restData.Enum.ObjEnum.DISK and \
                    get_disk_domain_status(context, fruId) == \
                    restData.Enum.RunningStatusEnum.RECONSTRUCTION:
                BaseFactory.result.setResultFailByKey(
                    context,
                    LangKey.DISK_DOMAIN_STATUS_RECONSTRUCTION, fruInfo)
            else:
                BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_REPLEACED_FRU_X_ABNORMAL, fruInfo)

        return False

    else:
        BaseFactory.result.setResultPass(context)
        return True


def get_disk_domain_status(context, disk_id):
    """
    获取硬盘所在硬盘域的状态
    :param context:
    :param disk_id:
    :return:
    """
    sel_fru = getFruInfo(context, restData.Enum.ObjEnum.DISK, disk_id)
    BaseFactory.log.info(context, "selFru:{}".format(str(sel_fru)))
    pool_id = restUtil.Tlv2Rest.getRecordValue(sel_fru,
                                               restData.Hardware.Disk.POOL_ID)

    # 经自验，如果硬盘没有所属的硬盘域，则返回空字符串（''）
    if pool_id is None or "" == pool_id:
        return ""

    # 若硬盘域运行状态不为ONLINE，检查不通过
    running_status = getFruRunningStatus(
        context, restData.Enum.ObjEnum.DiskPool, pool_id)
    return running_status


'''
子方法：检查是否存在主机多路径告警
'''


def checkHostMultPathAlarm(context):
    # 检查主机多链路告警的等待时间
    HOST_MULT_PATH_ALARM_CHECK_TIME_LIMIT = 120
    HOST_MULT_PATH_ALARM_CHECK_TIME_INTEVAL = 30
    # 受影响的主机多路径告警列表
    hostAlmIdList = [
        # "ALARM_NOT_LUN_PATH"
        "0x000F00150019",
        # V5R7C70之后版本 0x000F00150019变为0x000F00150034
        "0x000F00150034",
        # "ALARM_LINK_FAULT"
        "0x000F0015001A",
        # "ALARM_UNSTABLE"
        "0x000F0015001B",
        # "ALARM_NOT_INSTALL"
        "0x100F0015001C",
        # V5R7C70之后版本 0x100F0015001C变为0x100F0015003B
        "0x100F0015003B",
        # "ALARM_NOT_COMPATIBLE"
        "0x000F0015001D",
        # V5R7C70之后版本 0x000F0015001D变为0x000F00150032
        "0x000F00150032",
        # 主机上存在LUN路径不满足冗余规格(V6)
        "0xF00150027",
        # 主机上存在LUN路径不满足控制器冗余规格(V6)
        "0xF00150028",
        # 主机上存在LUN路径不满足冗余规格(V5)
        "0xF00150029",
        # 主机上存在LUN路径不满足控制器冗余规格(V5)
        "0xF0015002A",
        # 共享卡
        "0x000F00150024",
        # 非共享卡
        "0x000F00150025",
    ]
    # 将告警ID转换为10进制的字典
    hostAlmIdDict = {}
    for almId in hostAlmIdList:
        hostAlmIdDict[str(int(almId, 16))] = 1

    startTime = time.time()
    while (time.time() - startTime) < HOST_MULT_PATH_ALARM_CHECK_TIME_LIMIT:
        time.sleep(HOST_MULT_PATH_ALARM_CHECK_TIME_INTEVAL)
        BaseFactory.log.info(context, "Time release : %d" % (time.time() - startTime))
        # 获取系统中的所有告警
        retVal = getSysAlarmsInfo(context)
        if True != retVal["succ"]:
            BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                                   LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
            return
        # 判断系统告警中是否存在多路径告警
        isExisted = False
        sysAlarmList = retVal["info"]
        if None != sysAlarmList:
            almNum = len(sysAlarmList)
            for index in range(0, almNum):
                curAlm = sysAlarmList[index]
                almId = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_ID)
                if str(almId) in hostAlmIdDict:
                    isExisted = True
                    BaseFactory.log.error(context, "found host path alarm. decId=" + str(almId)
                                          + "hexId=" + str(hex(almId)))
                    break

        # 处理返回结果
        if True != isExisted:
            BaseFactory.result.setResultPass(context)
            return

    BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_EXISTED_MULTIPATH_ALARM)
    return


'''
子方法：检查是否存在单链路告警
'''


def checkSingleLinkPathAlarm(context):
    # 受影响的主机多路径告警列表
    hostAlmIdList = ["0xF00CE002A",  # DISK_ENCLOSURE_SINGLE_LINK
                     "0xF00A000C",  # DISK_SINGLE_LINK
                     ]
    # 将告警ID转换为10进制的字典
    singleLinkAlmIdDict = {}
    for almId in hostAlmIdList:
        singleLinkAlmIdDict[str(int(almId, 16))] = 1
    # 获取系统中的所有告警
    retVal = getSysAlarmsInfo(context)
    if True != retVal["succ"]:
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return
    # 判断系统告警中是否存在多路径告警
    isExisted = False
    sysAlarmList = retVal["info"]
    if None != sysAlarmList:
        almNum = len(sysAlarmList)
        for index in range(0, almNum):
            curAlm = sysAlarmList[index]
            almId = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_ID)
            if str(almId) in singleLinkAlmIdDict:
                isExisted = True
                BaseFactory.log.error(context, "found single link path alarm. decId=" + str(almId)
                                      + "hexId=" + str(hex(almId)))
    # 处理返回结果
    if True == isExisted:
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_EXISTED_SINGLELINKSTATUS_ALARM)
    else:
        BaseFactory.result.setResultPass(context)
    # 返回
    return


def checkMgtPortLinkErrorAlarm(context, *almArgs):
    """检查冗余管理板是否存在“链路通信异常或者交换机脱管”相关的告警。
    :param context:
    :param almArgs:
    :return:
    """
    # ALM_DEV_CABLE_LINK_INCORRECT 适用于Dorado5000 V3 & Dorado6000 V3 & Dorado18000 V3
    devCableLinkErrorAlmId = int("0xF00060007", 16)
    dswFailMonitorAlmId = int("0xF01050015", 16)  # 交换机脱管告警
    tgtAlmIdList = [devCableLinkErrorAlmId, dswFailMonitorAlmId]
    retVal = getSysAlarmsInfo(context)
    if retVal and not retVal.get("succ", False):
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    sysAlarmList = retVal["info"]
    if not sysAlarmList:
        BaseFactory.result.setResultPass(context)
        return True
    almSeqList = []
    for curAlm in sysAlarmList:
        almId = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_ID)
        BaseFactory.log.info(context, "almId %s" % almId)
        if almId in tgtAlmIdList:
            almSeq = restUtil.Tlv2Rest.getRecordValue(curAlm, restData.Sys.Alarm.CMO_ALARM_SEQUENCE)
            cmd = 'show alarm sequence=%s' % almSeq
            cmdSucc, cliRet, errMsg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
            if not cmdSucc:
                BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_GET_SYS_ALARM_FAIL,
                                                       LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
                return False

            almDesContainsAllArgs = True
            for almArg in almArgs:
                if almArg not in cliRet:
                    almDesContainsAllArgs = False

            if almId == int(dswFailMonitorAlmId) or (almId == int(devCableLinkErrorAlmId) and almDesContainsAllArgs):
                almSeqList.append(almSeq)
    if almSeqList:
        almSeqList.sort()
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_EXISTED_MGT_PORT_LINK_ALARM,
                                              ','.join(map(str, almSeqList)))
        return False
    else:
        BaseFactory.result.setResultPass(context)
        return True


def checkRedundantMgtboardPortStatus(context, redundantMgtPlane):
    """
    :param context:
    :param redundantMgtPlane:
    :return:
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    cmd = 'show port general physical_type=ETH'
    cmdSucc, cliRet, errMsg = cliUtil.execCmdInCliMode(cli, cmd, True, lang)
    if not cmdSucc:
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_QUERY_ETH_PORT_FAILED,
                                               LangKey.PUBSUG_ASSURE_DEV_NETWORK_NORMAL)
        return False
    BaseFactory.log.info(context, "redundantMgtPlane %s" % redundantMgtPlane)
    checkTtgPort1 = '%s.MGMT1' % redundantMgtPlane
    checkTtgPort2 = '%s.MGMT2' % redundantMgtPlane
    linkDownPorts = []
    for line in cliRet.splitlines():
        line = line.strip()
        if line and (checkTtgPort1 in line or checkTtgPort2 in line) and 'Link Down' in line:
            linkDownPorts.append(line.split()[0])
    BaseFactory.log.info(context, "link down port status %s" % linkDownPorts)
    if linkDownPorts:
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_REDUNDANT_MGT_BOARD_PORT_DOWN,
                                              ', '.join(linkDownPorts))
        return False
    else:
        BaseFactory.result.setResultPass(context)
        return True


def getSysConfigModel(context):
    # 查询状态
    param0 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.SYSTEM)
    paramlist = restUtil.Tlv2Rest.getParamList(param0)
    rest = contextUtil.getRest(context)
    recs = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET, paramlist, False)
    # 设置返回值
    if None == recs:
        fruInfo = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, restData.Enum.ObjEnum.SYSTEM)
        BaseFactory.log.error(context, u"获取" + fruInfo + u"信息列表失败。请确认系统和网络是否正常。")
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_GET_INFO_LIST_X_FAIL, fruInfo)

    return restUtil.Tlv2Rest.getRecordValue(recs, restData.Sys.System.CONFIG_MODEL)


def is2800V3SingleCtrl(context):
    SINGLECTRL = "2800V3SINGLECTRL"
    is2800V3SingleCtrl = BaseFactory.persist.getModule(context, SINGLECTRL)

    if None == is2800V3SingleCtrl:
        SINGLE_CTRL = 0
        productModel = getSystemProductModel(context)
        sysConfigModel = getSysConfigModel(context)
        BaseFactory.log.info(context, "the product model is %s, config model enum is %s" % (
            str(productModel), str(sysConfigModel)))

        if SINGLE_CTRL == sysConfigModel:
            BaseFactory.log.info(context, "this system is single mode.")
            BaseFactory.persist.setModule(context, SINGLECTRL, True)
            return True

        else:
            BaseFactory.persist.setModule(context, SINGLECTRL, False)
            return False

    else:
        return is2800V3SingleCtrl


########################################外部接口函数定义区########################################
__________ZONE_OUTER_PUBLIC_FUNC__________ = None

'''
Funciton: 下电接口模块
Params: context=工具上下下文； cardId=待下电的接口模块ID
Return: 下电是否成功（错误信息和成败信息已设置）。 True=成功； False=失败 或 异常， 此时已设置错误信息
'''


def powerOff_IntfModule(context, cardId):
    # 如果接口卡已下电，则操作成功
    runningStatus = getFruRunningStatus(context, restData.Enum.ObjEnum.INTF_MODULE, cardId)
    powerOffStatusList = [restData.Enum.RunningStatusEnum.POWER_OFF, restData.Enum.RunningStatusEnum.POWER_ON_ERROR]
    if runningStatus in powerOffStatusList:
        BaseFactory.log.info(context, "the card " + cardId + " is already power off. check passed.")
        BaseFactory.result.setResultPass(context)
        return True

    # 执行接口卡下电TLV命令
    param1 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.INTF_MODULE)
    param2 = (restData.PublicAttributes.ID, cardId)
    paramList = restUtil.Tlv2Rest.getParamList(param1, param2)
    try:
        rest = contextUtil.getRest(context)
        restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.INTF_MODULE_POWER_OFF, paramList)
    except:
        BaseFactory.log.error(context, "fail to exec the power off cmd of card " + cardId + ". check failed.")
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_INTF_POWER_OFF_FAIL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
        return False

    # 轮询300S，确认接口卡是否下电成功
    startTime = time.time()
    while True:
        # 若轮询300S后仍未下电成功，则返回错误
        if (time.time() - startTime) > 300:
            BaseFactory.log.error(context, "the card " + cardId
                                  + " is still not power off after exec poweroff cmd 300s. check failed.")
            BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_INTF_POWER_OFF_FAIL,
                                                   LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
            return False
        # 查看接口卡运行状态是否为下电
        runningStatus = getFruRunningStatus(context, restData.Enum.ObjEnum.INTF_MODULE, cardId)

        if runningStatus in powerOffStatusList:
            BaseFactory.log.info(context, "the card " + cardId + " is powered off now. check passed.")
            BaseFactory.result.setResultPass(context)
            break
        # 以一秒为轮询的周期
        time.sleep(5)

    # 接口卡下电成功
    return True


'''
Funciton: 上电接口模块
Params: context=工具上下下文； cardId=待上电的接口模块ID
Return: 上电是否成功。 True=成功； False=失败 或 异常， 此时已设置错误信息
'''


def powerOn_IntfModule(context, cardId):
    # 初始化运行状态
    runningStatus = None

    # 轮询45秒，查看FRU的运行状态是否为已上电或正常运行，若是则检查通过。
    startTime = time.time()
    while True:
        if (time.time() - startTime) > 45:
            BaseFactory.log.info(context, "the card " + cardId + " is still not power on after wait 5s.")
            break
        runningStatus = getFruRunningStatus(context, restData.Enum.ObjEnum.INTF_MODULE, cardId)
        if None == runningStatus:
            BaseFactory.log.error(context, "fail to get the running status of card " + cardId + ". check failed.")
            return False
        elif runningStatus in [restData.Enum.RunningStatusEnum.RUNNING,
                               restData.Enum.RunningStatusEnum.NORMAL]:
            BaseFactory.log.info(context, "the card " + cardId + " is already RUNNING or NORMAL. check passed.")
            BaseFactory.result.setResultPass(context)
            return True
        elif restData.Enum.RunningStatusEnum.POWER_ON == runningStatus:
            BaseFactory.log.info(context, "the card " + cardId + " is already POWER_ON. check passed.")
            BaseFactory.result.setResultPass(context)
            return True
        time.sleep(5)

    # 对接口卡模块执行TLV上电命令
    if restData.Enum.RunningStatusEnum.POWER_OFF == runningStatus:
        param1 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.INTF_MODULE)
        param2 = (restData.PublicAttributes.ID, cardId)
        paramList = restUtil.Tlv2Rest.getParamList(param1, param2)
        rest = contextUtil.getRest(context)
        restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.INTF_MODULE_POWER_ON, paramList)
    else:  # 若接口卡既非正常运行，也非下电，说明接口卡故障，报错
        BaseFactory.log.info(context, "the running status of card:%s is %s that not normal or poweroff. check failed."
                             % (cardId, restUtil.Tlv2Rest.getStrEn(restData.Enum.RunningStatusEnum, runningStatus)))
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_INTF_POWER_ON_FAIL)
        return False

    # 下发TLV命令上电后，300秒轮询上电是否成功
    startTime = time.time()
    while True:
        if (time.time() - startTime) > 300:
            BaseFactory.log.info(context, "the card " + cardId
                                 + " is still not RUNNING or NORMAL after exec power on cmd 300s. check failed.")
            BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_INTF_POWER_ON_FAIL)
            return False
        runningStatus = getFruRunningStatus(context, restData.Enum.ObjEnum.INTF_MODULE, cardId)
        if None == runningStatus:
            BaseFactory.log.error(context, "fail to get the running status of card " + cardId + ". check failed.")
            BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_INTF_POWER_ON_FAIL)
            return False
        elif runningStatus in [restData.Enum.RunningStatusEnum.RUNNING,
                               restData.Enum.RunningStatusEnum.NORMAL]:
            BaseFactory.log.info(context, "the card " + cardId + " is RUNNING or NORMAL now. check passed.")
            break
        time.sleep(5)

    # 接口卡上电成功
    BaseFactory.result.setResultPass(context)
    return True


'''
公用方法：获取存储DB的本地保存路径
'''


def getDBSaveDir():
    # 从当前脚本文件的路径中获取磁盘分区名称
    curFilePath = os.path.abspath(__file__)
    partitionName = curFilePath.split(":")[0]

    # 若不为空，则返回该盘符的根路径，否则返回空字符串
    if "" != partitionName:
        return partitionName + ":\\"
    else:
        return ""


'''
公用方法：备份系统DB
'''


def backupDB(context, localDir, isCtrl=False):
    '''
    @summary: 控制器、控制框、管理模块、硬盘框更换前执行该方法
    @param isCtrl:更换的模块是否是控制器，其更换前检查如果检查失败，框架会重复执行一遍该检查项，用于处理检查失败时重复弹框的问题
    '''
    productModel = context.get("dev").get("type")
    # 文件路径预处理：将单斜杠替换为双斜杠
    if None == re.search("\\\\", localDir):
        localDir = localDir.replace('\\', '\\\\')
    # 执行DB导出命令
    param0 = (restData.Sys.ExportConfigData.CMO_EXPORT_DB_TYPE, 0)  # 0=从内存导出，1=从硬盘导出
    paramlist = restUtil.Tlv2Rest.getParamList(param0)
    rest = contextUtil.getRest(context)

    if productModel in config.DEVS_V5:
        rec = exportDB_V5(context, rest, paramlist, productModel, isCtrl)
    else:
        rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.EXPORT_DB_CONFIG, paramlist, False)
    if None == rec:
        # 记录导出DB失败日志
        recordOperationLog(context, 1)
        clearDB(context)
        BaseFactory.log.error(context, "fail to export db as fail to exec export cmd. localDir=" + localDir)
        return  # 错误信息已设置
    # 获取DB导出地址
    fileInfo = restUtil.Tlv2Rest.getRecordValue(rec, restData.Sys.ExportConfigData.CMO_EXPORT_DB_DATA_DIR)

    remoteFilePath = fileInfo.split(":")[1]
    dbFileName = remoteFilePath.split("/")[-1]
    isSucc = SftpFactory.Sftp.download(context, remoteFilePath, localDir)
    try:
        subprocess.check_call('icacls ' + localDir + '\\' + dbFileName + ' /inheritance:d', shell=False)
        subprocess.check_call('icacls ' + localDir + '\\' + dbFileName + ' /remove:g Users', shell=False)
    except Exception as e:
        BaseFactory.log.error(context, "remove user permission error：%s" % str(e))
    if isSucc:
        # 记录导出DB成功日志
        recordOperationLog(context, 0)
        BaseFactory.result.setResultPass(context)
        BaseFactory.log.error(context, "succeed to export db to localDir: " + localDir)
    else:
        # 错误信息已在download函数中设置
        # 记录导出DB失败日志
        recordOperationLog(context, 1)
        BaseFactory.log.error(context, "fail to export db to localDir: " + localDir)

    # 下载后删除阵列上的临时DB信息
    clearDB(context)
    context["dbFilePath"] = localDir + os.path.sep + "db.dat"
    # 返回
    return


def backup_license(context, local_dir):
    """
    备份license数据
    :param context: 上下文
    :param local_dir: 本地备份保存目录
    :return:
    """
    BaseFactory.log.info(context, "begin download license file")
    try:
        download_license_to_local(context, local_dir)
    except Exception as ex:
        str_err_code = ex.args[0]
        # 若license文件不存在，检查项也通过
        if str_err_code == config.ERR_CODE_FILE_NOT_EXIST:
            BaseFactory.log.error(
                context, "license file is not existed")
        else:
            BaseFactory.log.error(context, "download license file failed.")
        return


def download_license_to_local(context, local_dir):
    """
    下载license文件到本地
    :param context: 上下文
    :param local_dir: 本地备份保存目录
    """
    # 文件路径预处理：将单斜杠替换为双斜杠
    if re.search("\\\\", local_dir) is None:
        local_dir = local_dir.replace('\\', '\\\\')
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(
        rest, restData.TlvCmd.GET_LICENSE_EXPORT_PATH,
        params=[], isBatch=False)
    ret_path = restUtil.Tlv2Rest.getRecordValue(
        rec, restData.Upgrade.LicenseExportPath.CMO_LICENSE_DIR_INFO_DIR)
    remote_file_path = ret_path.split(":")[1]
    license_file_name = remote_file_path.split("/")[-1]
    is_download_success = SftpFactory.Sftp.download(
        context, remote_file_path, local_dir)
    remove_users_file_permission(local_dir, license_file_name, context)
    if is_download_success is False:
        BaseFactory.log.error(context, "download license file failed.")
    context["licenseFilePath"] = \
        local_dir + os.path.sep + license_file_name
    return


def remove_users_file_permission(local_dir, file_name, context):
    """
    移除文件的Users权限
    :param local_dir: 文件路径
    :param file_name: 文件名
    :param context: 上下文
    """
    try:
        subprocess.check_call('icacls ' + local_dir + '\\' + file_name
                              + ' /inheritance:d', shell=False)
        subprocess.check_call('icacls ' + local_dir + '\\' + file_name
                              + ' /remove:g Users', shell=False)
    except Exception as ex:
        BaseFactory.log.error(context,
                              "remove user permission error：%s" % str(ex))


def exportDB_V5(context, rest, paramlist, productModel, isCtrl=False):
    """
    @summary: 处理系统盘故障无法导出数据，更换失败的问题。V5设备,从内存导出系统DB，V5特殊处理是由于配置数据的存放目录不一样。
                该场景会返回两种错误码：主控系统盘故障和备控系统盘故障。1.主控系统盘故障：弹框提示可点击“确定”允许跳过且不备份数据，
                2.备控系统盘故障：检查不通过，提示更换控制器登陆（由于备份数据过程和当前登陆的IP有关），特例：高端由于无法切换ip登陆，统一按
                弹框提示方式，并不体现“主控”。
    @param isCtrl: 更换的模块是否是控制器，其更换前检查如果检查失败，框架会重复执行一遍该检查项；
    """
    rec = None
    try:
        rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.EXPORT_DB_CONFIG, paramlist, False)

    except Exception as ex:
        BaseFactory.log.error(context, "catch Exception: %s" % str(ex))
        strErrCode = str(ex.args[0])
        BaseFactory.log.error(context, "the controller is not existed. %s" % str(strErrCode))
        # 1. 主控制器的系统盘故障,特例由于高端无法切换ip登陆且是双系统盘，统一做弹框可忽略处理
        if strErrCode == config.RETURN_EXPORT_CONFIG_MASTER_CONTROL_SYSTEM_DISK_FAULT or \
                ((strErrCode == config.RETURN_EXPORT_CONFIG_MASTER_CONTROL_SYSTEM_DISK_FAULT
                  or strErrCode == config.RETURN_EXPORT_CONFIG_OTHER_CONTROL_SYSTEM_DISK_FAULT)
                 and productModel in config.HIGH_END_DEVS):
            # 控制器更换前检查失败，框架会执行第二次该检查项
            if isCtrl:
                # 1.isRetry，是否第二次执行该脚本的标志。第一次跑该脚本失败不弹框，直接返回False.框架会再次执行该检查项还是该错误码，再提示弹框。
                isRetry = BaseFactory.persist.getModule(context, "isRetry")
                BaseFactory.log.error(context, "isRetry =%s" % str(isRetry))

                if isRetry:
                    dialogUtil = context['dialogUtil']
                    lang = getLang(context)
                    msg = getRes(lang, "master_controller_system_disk_fault")
                    result = dialogUtil.showWarningDialog(msg)
                    BaseFactory.log.error(context, "showInfoDialog result: %s" % str(result))
                    # 弹框点击“取消”
                    if not result:
                        if productModel in config.HIGH_END_DEVS:
                            BaseFactory.result.setResultFailByKeys(context, LangKey.HIGH_END_CONTROL_SYSTEM_DISK_FAULT,
                                                                   LangKey.MASTER_CONTROL_SYSTEM_DISK_FAULT)
                            BaseFactory.persist.setModule(context, "isRetry", False)
                        else:
                            BaseFactory.result.setResultFailByKey(context, LangKey.MASTER_CONTROL_SYSTEM_DISK_FAULT)
                            BaseFactory.persist.setModule(context, "isRetry", False)
                    # 弹框点击“确定”，忽略导出数据检查项
                    else:
                        BaseFactory.result.setResultPass(context)
                        BaseFactory.persist.setModule(context, "isRetry", False)
                    return rec
                else:
                    BaseFactory.result.setResultFailByKey(context, LangKey.MASTER_CONTROL_SYSTEM_DISK_FAULT)
                    BaseFactory.persist.setModule(context, "isRetry", True)
                    return rec
            # 更换管理模块、硬盘框、控制框走该分支
            else:
                dialogUtil = context['dialogUtil']
                lang = getLang(context)
                msg = getRes(lang, "master_controller_system_disk_fault")
                result = dialogUtil.showWarningDialog(msg)
                BaseFactory.log.info(context, "showInfoDialog result: %s" % str(result))
                # 弹框点击“取消”
                if not result:
                    if productModel in config.HIGH_END_DEVS:
                        BaseFactory.result.setResultFailByKeys(context, LangKey.HIGH_END_CONTROL_SYSTEM_DISK_FAULT,
                                                               LangKey.MASTER_CONTROL_SYSTEM_DISK_FAULT)
                    else:
                        BaseFactory.result.setResultFailByKey(context, LangKey.MASTER_CONTROL_SYSTEM_DISK_FAULT)
                # 弹框点击“确定”，忽略导出数据检查项
                else:
                    BaseFactory.result.setResultPass(context)
                return rec
        # 2.当前登录的控制器系统盘故障
        elif strErrCode == config.RETURN_EXPORT_CONFIG_OTHER_CONTROL_SYSTEM_DISK_FAULT:
            BaseFactory.result.setResultFailByKey(context, LangKey.CURRENT_CONTROL_SYSTEM_DISK_FAULT)
            return rec
        else:
            raise ex
    return rec


'''
公用方法：检查系统运行状态
'''


def checkSysRunningStatus(context):
    # 获取信息
    param0 = (restData.PublicAttributes.TYPE, restData.Enum.ObjEnum.SYSTEM)
    paramlist = restUtil.Tlv2Rest.getParamList(param0)
    rest = contextUtil.getRest(context)
    rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.GET, paramlist, False)
    # 判断结果
    isPass = True
    if None == rec:
        isPass = False
    else:
        runStat = restUtil.Tlv2Rest.getRecordValue(rec, restData.PublicAttributes.RUNNING_STATUS)
        if restData.Enum.RunningStatusEnum.NOT_RUNNING == runStat:
            isPass = False
    # 设置返回值
    if not isPass:
        BaseFactory.result.setResultFailByKeys(context, LangKey.FUNC_SYS_RUN_STATUS_ABNORMAL,
                                               LangKey.PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP)
    else:
        BaseFactory.result.setResultPass(context)
    # 返回
    return isPass


'''
公用方法：检查系统业务压力
'''


def checkSysSrvPressure(context):
    # 采用平均业务压力来进行检查:多次检测，若任一检查项的不通过次数大于通过次数，则检查不通过
    chkNum = 30  # 检测次数
    chkPeriod = 1  # 检测周期（秒）
    passFailNum = 0.7 * chkNum  # 最大检测不通过次数，当实际不通过次数大于最大检测不通过次数时，则检查不通过
    failCpuNum = 0
    failIoConNum = 0

    # 获取系统中的所有引擎信息和引擎数量
    allCtrlEncRecs = getAllCtrlEncInfo(context)
    # 获取系统中的所有控制器信息
    allCtrlRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    # 多次采集CPU利用率和后端IO并发检查
    for i in range(0, chkNum):
        BaseFactory.log.info(context, "sys pressure check process...%d/%d" % (i + 1, chkNum))
        # 执行检查
        if not checkCpuUsage(context, allCtrlEncRecs):
            failCpuNum += 1
        if not checkBackIoConcurrent(context, allCtrlEncRecs, allCtrlRecs):
            failIoConNum += 1
        # 等待下次检测
        time.sleep(chkPeriod)

    # 若任一检查项的不通过次数大于通过次数，则检查不通过
    BaseFactory.log.info(context,
                         "sys pressure check result: checkNum=%d, passFailNum=%d, "
                         "checkCPUFailNum=%d, checkIOFailNum=%d"
                         % (chkNum, passFailNum, failCpuNum, failIoConNum))
    if (failCpuNum > passFailNum) or (failIoConNum > passFailNum):
        BaseFactory.result.setResultFailByKey(context, LangKey.FUNC_SERVICE_PRESSURE_OFF_SCALE)
    else:
        BaseFactory.result.setResultPass(context)

    return


'''
公用方法：检查2U新硬件风扇，电源，BBU的冗余：
检查标准：规格：风扇，电源，BBU一体，且都为1+1，
                       更换风扇，电源，BBU中的一种，其实都是更换风扇，电源，BBU一体，
                       所以风扇，电源，BBU必须都存在冗余（数量为1）
Return: 各子FRU是否冗余：(isBbuRed, isPowerRed, isFanRed)
'''


def checkBundleFruRedundancy_for5X00V3(context, beReplacedFru):
    # 获取用户选中FRU的信息
    parentId = beReplacedFru["parentID"]
    location = beReplacedFru["location"]

    # 判断ARM设备，ARM设备不检查BBU
    productModel = contextUtil.getProductModel(context)
    isArmDev = baseUtil.isArmDev(context, productModel)
    BaseFactory.log.info(context, "current product model:%s" % productModel)

    # 获取所有的风扇，BBU，电源
    allBBUs = getFruListInfo(context, restData.Enum.ObjEnum.BACKUP_POWER)
    allPowers = getFruListInfo(context, restData.Enum.ObjEnum.POWER)
    allFans = getFruListInfo(context, restData.Enum.ObjEnum.FAN)

    # 设置待更换FRU的父模块（控制框）下所有的风扇，BBU，电源的过滤条件
    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, parentId)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.NORMAL)
    bbu_condition1 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.HEALTH_STATUS,
                                                    restData.Enum.ConditionTypeEnum.EQOR,
                                                    [restData.Enum.HealthStatusEnum.NORMAL,
                                                     restData.Enum.HealthStatusEnum.PRE_FAIL])
    # 同一一体化模块中的电源，风扇和BBU的location的最后一位相同
    condition2 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.LOCATION,
                                                restData.Enum.ConditionTypeEnum.NLIKE, location[-1] + "$")
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1, condition2)
    bbu_condition_list = restUtil.Tlv2Rest.getConditionList(condition0, bbu_condition1, condition2)

    # 过滤冗余子FRU并判断是否存在冗余
    # 2U新硬件为：风扇，电源，BBU一体，且都为1+1，当这三者的冗余数量为(1,1,1)
    allRedBBUsInFatherFRU = restUtil.Tlv2Rest.filter(allBBUs, bbu_condition_list)
    allRedFansInFatherFRU = restUtil.Tlv2Rest.filter(allFans, conditionList)

    isBbuRed = True if len(allRedBBUsInFatherFRU) > 0 else False
    isPowerRed = True
    isFanRed = True if len(allRedFansInFatherFRU) > 0 else False

    if isArmDev:
        isBbuRed = True
    # 返回结果
    BaseFactory.log.info(context, "5X00V3 BBU,power,FAN  %s redundancy check result is BBU:%s, power:%s, fan:%s"
                         % (location, isBbuRed, isPowerRed, isFanRed))
    return (isBbuRed, isPowerRed, isFanRed)


'''
公用方法：检查2U新硬件风扇，电源，BBU更换后是否正常

Return: 各子FRU是否正常：(isBbuRed, isPowerRed, isFanRed)
'''


def checkBundleFruNormal_for5X00V3(context, beReplacedFru, curFruType):
    needRetry = False
    if curFruType not in [restData.Enum.ObjEnum.BACKUP_POWER, restData.Enum.ObjEnum.POWER, restData.Enum.ObjEnum.FAN]:
        BaseFactory.result.setResultPass(context)
        return needRetry
    # 获取用户选中FRU的信息
    parentId = beReplacedFru["parentID"]
    location = beReplacedFru["location"]

    # 获取所有的风扇，BBU，电源
    allBBUs = getFruListInfo(context, restData.Enum.ObjEnum.BACKUP_POWER)
    allPowers = getFruListInfo(context, restData.Enum.ObjEnum.POWER)
    allFans = getFruListInfo(context, restData.Enum.ObjEnum.FAN)

    # 设置已更换子FRU的过滤条件
    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, parentId)
    # 同一一体化模块中的电源，风扇和BBU的location的最后一位相同
    condition1 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.LOCATION,
                                                restData.Enum.ConditionTypeEnum.LIKE, location[-1] + "$")
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)

    # 过滤已更换子FRU并获取其id
    subBBUsInBundle = restUtil.Tlv2Rest.filter(allBBUs, conditionList)
    subPowersInBundle = restUtil.Tlv2Rest.filter(allPowers, conditionList)
    subFansInBundle = restUtil.Tlv2Rest.filter(allFans, conditionList)

    curBbuId = None
    if len(subBBUsInBundle) > 0:
        curBbuId = restUtil.Tlv2Rest.getRecordValue(subBBUsInBundle[0], restData.PublicAttributes.ID)
    curPowerId = None
    if len(subPowersInBundle) > 0:
        curPowerId = restUtil.Tlv2Rest.getRecordValue(subPowersInBundle[0], restData.PublicAttributes.ID)
    curFanId = None
    if len(subFansInBundle) > 0:
        curFanId = restUtil.Tlv2Rest.getRecordValue(subFansInBundle[0], restData.PublicAttributes.ID)

    # 判断ARM设备，ARM设备不检查BBU
    productModel = contextUtil.getProductModel(context)
    isArmDev = baseUtil.isArmDev(context, productModel)
    isDoradoV6 = baseUtil.isDoradoV6Dev(productModel)
    BaseFactory.log.info(context, "current product model:%s" % productModel)

    # 定义子FRU检查顺序(按当前更换FRU和其他重要程度排序)
    subFruChkList = []
    if isArmDev:
        subFruChkList.append((restData.Enum.ObjEnum.POWER, curPowerId))
        subFruChkList.append((restData.Enum.ObjEnum.FAN, curFanId))
    elif isDoradoV6:
        subFruChkList.append((restData.Enum.ObjEnum.BACKUP_POWER, curBbuId))
        subFruChkList.append((restData.Enum.ObjEnum.FAN, curFanId))
    else:
        subFruChkList.append((restData.Enum.ObjEnum.POWER, curPowerId))
        subFruChkList.append((restData.Enum.ObjEnum.FAN, curFanId))
        subFruChkList.append((restData.Enum.ObjEnum.BACKUP_POWER, curBbuId))

    # 按顺序检查各子FRU是否在位
    for (subFruType, subFruId) in subFruChkList:
        # 若子FRU不存在，则设置错误信息并返回
        if None == subFruId:
            needRetry = True
            fruName = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrObjEnum, subFruType)
            BaseFactory.log.error(context, "the sub fru:%s in bundle fru is not in position." % fruName)
            BaseFactory.result.setResultFailByKey(context, LangKey.PUBERR_SUBFRU_X_IN_BUNDLE_FRU_INEXISTED, fruName)
            return needRetry

    # 按顺序检查各子FRU是否正常
    for (subFruType, subFruId) in subFruChkList:
        # 若子FRU不存在，则设置错误信息并返回
        isReplacedFruNormal(context, subFruType, subFruId, curFruType)
        if True == BaseFactory.result.isResultFail(context):
            BaseFactory.log.error(context, "the sub fru(type:%s,id=%s) bundled with fru(type:%s) is abnormal."
                                  % (subFruType, subFruId, curFruType))
            return needRetry

    # 返回正常
    BaseFactory.result.setResultPass(context)
    return needRetry


# 注意：使用该方法之前，请自行保证端口的logicType为管理或维护口，此处无法判断
def isMgmtOrMainTenance(sysModel, location):
    '''
            根据不同的设备型号和管理网口的location，判断是否是管理网口和维护网口
            参数：sysModel=设备型号，location网口地址
    return:(是否管理网口，是否维护网口)
    '''
    isMgmt = False
    isMainTenagce = False

    if sysModel == BaseFactory.const.SYSMODEL['T_ENGINE']:
        loc = location[-1]
        if loc == "0":
            isMgmt = True
        elif loc == "1":
            isMainTenagce = True

    elif sysModel in [BaseFactory.const.SYSMODEL['S2600T'], BaseFactory.const.SYSMODEL['S5500T']]:
        loc = location[-1]
        if loc == "4":
            isMgmt = True
        elif loc == "5":
            isMainTenagce = True

    elif sysModel in [BaseFactory.const.SYSMODEL['V3_3U_ENGINE'], BaseFactory.const.SYSMODEL['5X00V3']]:
        loc = location.split(".")[-1]
        locEnd = location[-1]

        if loc == "MGMT" or locEnd == "0":
            isMgmt = True
        elif loc == "MAINTENANCE" or locEnd == "1":
            isMainTenagce = True

    elif sysModel == BaseFactory.const.SYSMODEL['V3_6U_ENGINE']:
        loc = location.split(".")[-1]
        locEnd = location[-1]

        if loc in ["MGMT0", "MGMT1", "MGMT2"] or locEnd in ['0', '1', '2']:
            isMgmt = True
        elif loc == "MAINTENANCE" or locEnd == "3":
            isMainTenagce = True

    return (isMgmt, isMainTenagce)


'''
#获取IOC借口卡的端口工作模式
'''


def getSmartIocPortsById(context, smartIocId):
    # 获取所有的端口信息
    portDict = {}
    allEthPorts = getFruListInfo(context, restData.Enum.ObjEnum.ETH_PORT)
    allFcPorts = getFruListInfo(context, restData.Enum.ObjEnum.FC_PORT)
    allFcoePorts = getFruListInfo(context, restData.Enum.ObjEnum.FCoE_PORT)

    allEthPorts.extend(allFcPorts)
    allEthPorts.extend(allFcoePorts)

    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                smartIocId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
    fruList = restUtil.Tlv2Rest.filter(allEthPorts, conditionList)

    for port in fruList:
        portId = restUtil.Tlv2Rest.getRecordValue(port, restData.Hardware.EthPort.ID)
        worType = restUtil.Tlv2Rest.getRecordValue(port, restData.Hardware.EthPort.LOGIC_TYPE)
        portDict[portId] = worType

    return portDict


'''
获取集群中已连接的SAS端口信息
'''


def getLinkedSasPorts(context):
    linkedSasPorts = None
    allSasPorts = getFruListInfo(context, restData.Enum.ObjEnum.SAS_PORT)
    if None == allSasPorts:
        return linkedSasPorts
    else:
        condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.SasPort.RUNNING_STATUS,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    restData.Enum.RunningStatusEnum.LINK_UP)
        conditionList = restUtil.Tlv2Rest.getConditionList(condition0)
        linkedSasPorts = restUtil.Tlv2Rest.filter(allSasPorts, conditionList)

    return linkedSasPorts


'''
错误语言信息Key定义
'''


class LangKey():
    '''
    Function: 语言描述key定义
    '''

    # 测试
    KEY_TEST = "KEY_TEST"

    # 公共提示内容
    ERROR_USER_PERMISSION_DENIED = "ERROR_USER_PERMISSION_DENIED"
    ERROR_USER_SUPER_PERMISSION_DENIED = "ERROR_USER_SUPER_PERMISSION_DENIED"
    ERROR_PWD_EXPIRING = "ERROR_PWD_EXPIRING"
    PUBERR_DEV_TEMP_OVER_THRESHOLD = "PUBERR_DEV_TEMP_OVER_THRESHOLD"
    PUBERR_SUBFRU_X_IN_BUNDLE_FRU_INEXISTED = "PUBERR_SUBFRU_X_IN_BUNDLE_FRU_INEXISTED"
    PUBERR_REDT_SAS_PATH_NOT_EXISTED = "PUBERR_REDT_SAS_PATH_NOT_EXISTED"
    PUBERR_REDT_MGMT_PATH_NOT_EXISTED = "PUBERR_REDT_MGMT_PATH_NOT_EXISTED"
    PUBERR_SWITCH_CNTR_SERVICE_FAIL = "PUBERR_SWITCH_CNTR_SERVICE_FAIL"
    PUBERR_UNEXPECTED_TOOL_ERROR = "PUBERR_UNEXPECTED_TOOL_ERROR"
    EXEC_CLI_CMD_FAIL = "EXEC_CLI_CMD_FAIL"
    FRU_NOT_SUPPORT = "FRU_NOT_SUPPORT"
    EXEC_TLV_CMD_FAIL = "EXEC_TLV_CMD_FAIL"
    CONTROLLER_NOT_SUPPORT = "CONTROLLER_NOT_SUPPORT"

    # 更换确认提示内容
    CONFIRM_REPLACE_DONE_MEMORY = "CONFIRM_REPLACE_DONE_MEMORY"
    CONFIRM_REPLACE_DONE_SYSTEM_DISK = "CONFIRM_REPLACE_DONE_SYSTEM_DISK"
    CONFIRM_REPLACE_DONE_BACKEND_INTF = "CONFIRM_REPLACE_DONE_BACKEND_INTF"
    CONFIRM_REPLACE_DONE_BBU_MDL = "CONFIRM_REPLACE_DONE_BBU_MDL"
    CONFIRM_REPLACE_DONE_CNTR = "CONFIRM_REPLACE_DONE_CNTR"
    CONFIRM_REPLACE_DONE_CNTRENC = "CONFIRM_REPLACE_DONE_CNTRENC"
    CONFIRM_REPLACE_DONE_DISK = "CONFIRM_REPLACE_DONE_DISK"
    CONFIRM_REPLACE_DONE_DISKENC = "CONFIRM_REPLACE_DONE_DISKENC"
    CONFIRM_REPLACE_DONE_EXP_MDL = "CONFIRM_REPLACE_DONE_EXP_MDL"
    CONFIRM_REPLACE_DONE_FAN_MDL = "CONFIRM_REPLACE_DONE_FAN_MDL"
    CONFIRM_REPLACE_DONE_MGMTFANCTRLBOARD_MDL = "CONFIRM_REPLACE_DONE_MGMTFANCTRLBOARD_MDL"
    CONFIRM_REPLACE_DONE_OPT_MDL = "CONFIRM_REPLACE_DONE_OPT_MDL"
    CONFIRM_REPLACE_DONE_OPT_ELE_MDL = "CONFIRM_REPLACE_DONE_OPT_ELE_MDL"
    CONFIRM_REPLACE_DONE_CABLE_FRONTEND = "CONFIRM_REPLACE_DONE_CABLE_FRONTEND"
    CONFIRM_REPLACE_DONE_CABLE_MGMT = "CONFIRM_REPLACE_DONE_CABLE_MGMT"
    CONFIRM_REPLACE_DONE_CABLE_PCIE = "CONFIRM_REPLACE_DONE_CABLE_PCIE"
    CONFIRM_REPLACE_DONE_CABLE_SAS = "CONFIRM_REPLACE_DONE_CABLE_SAS"
    CONFIRM_REPLACE_DONE_FRONTEND_INTF = "CONFIRM_REPLACE_DONE_FRONTEND_INTF"
    CONFIRM_REPLACE_DONE_OTHER_INTF = "CONFIRM_REPLACE_DONE_OTHER_INTF"
    CONFIRM_REPLACE_DONE_EXPRESS_ACC = "CONFIRM_REPLACE_DONE_EXPRESS_ACC"
    CONFIRM_REPLACE_DONE_MGMT_MDL = "CONFIRM_REPLACE_DONE_MGMT_MDL"
    CONFIRM_REPLACE_DONE_PCIE_MDL = "CONFIRM_REPLACE_DONE_PCIE_MDL"
    CONFIRM_REPLACE_DONE_PCIE_DSW = "CONFIRM_REPLACE_DONE_PCIE_DSW"
    CONFIRM_REPLACE_DONE_POWER_MDL = "CONFIRM_REPLACE_DONE_POWER_MDL"
    CONFIRM_REPLACE_DONE_RISK_POWER_MDL = "CONFIRM_REPLACE_DONE_RISK_POWER_MDL"
    CONFIRM_REPLACE_DONE_TIME_LIMIT_DEFAULT_INFO = "CONFIRM_REPLACE_DONE_TIME_LIMIT_DEFAULT_INFO"
    CONFIRM_REPLACE_DONE_TIME_LIMIT_DEFAULT_INFO_FOR_BUNDLE_FRU = "CONFIRM_REPLACE_DONE_TIME_LIMIT_DEFAULT_INFO_FOR_BUNDLE_FRU"
    CONFIRM_REPLACE_DONE_DISK_TIME_LIMIT_INFO = "CONFIRM_REPLACE_DONE_DISK_TIME_LIMIT_INFO"
    CONFIRM_REPLACE_DONE_IPSCALEOUT_DSW = "CONFIRM_REPLACE_DONE_IPSCALEOUT_DSW"
    CONFIRM_REPLACE_DONE_IPSCALEOUT_INTF = "CONFIRM_REPLACE_DONE_IPSCALEOUT_INTF"
    CONFIRM_REPLACE_DONE_IPSCALEOUT_CABLE = "CONFIRM_REPLACE_DONE_IPSCALEOUT_CABLE"
    CONFIRM_REPLACE_DONE_MANAGEBOARD_TIME_LIMIT_INFO = "CONFIRM_REPLACE_DONE_MANAGEBOARD_TIME_LIMIT_INFO"
    CONFIRM_REPLACE_DONE_CONTROLLER_TIME_LIMIT_INFO = "CONFIRM_REPLACE_DONE_CONTROLLER_TIME_LIMIT_INFO"
    CONFIRM_REPLACE_DONE_CONTROLLER_APPEND_TIME_LIMIT_INFO = "CONFIRM_REPLACE_DONE_CONTROLLER_APPEND_TIME_LIMIT_INFO"
    CONFIRM_REPLACE_DONE_DISK_BOX = "CONFIRM_REPLACE_DONE_DISK_BOX"
    CONFIRM_REPLACE_DONE_INTF_BOX = "CONFIRM_REPLACE_DONE_INTF_BOX"

    # 更换结束提示内容
    FINISH_NOTICE_MEMORY = "FINISH_NOTICE_MEMORY"
    FINISH_NOTICE_SYSTEM_DISK = "FINISH_NOTICE_SYSTEM_DISK"
    FINISH_NOTICE_BACKEND_INTF = "FINISH_NOTICE_BACKEND_INTF"
    FINISH_NOTICE_BBU_MDL = "FINISH_NOTICE_BBU_MDL"
    FINISH_NOTICE_CNTR = "FINISH_NOTICE_CNTR"
    FINISH_NOTICE_STORAGE_NODE = "FINISH_NOTICE_STORAGE_NODE"
    FINISH_NOTICE_CNTRENC = "FINISH_NOTICE_CNTRENC"
    FINISH_NOTICE_DISK = "FINISH_NOTICE_DISK"
    FINISH_NOTICE_DISK_BOX = "FINISH_NOTICE_DISK_BOX"
    FINISH_NOTICE_INTF_BOX = "FINISH_NOTICE_INTF_BOX"
    FINISH_NOTICE_DISKENC = "FINISH_NOTICE_DISKENC"
    FINISH_NOTICE_DISK_DRAWER = "FINISH_NOTICE_DISK_DRAWER"
    FINISH_NOTICE_EXP_MDL = "FINISH_NOTICE_EXP_MDL"
    FINISH_NOTICE_FAN_MDL = "FINISH_NOTICE_FAN_MDL"
    FINISH_NOTICE_MGMTFANCTRLBOARD_MDL = "FINISH_NOTICE_MGMTFANCTRLBOARD_MDL"
    FINISH_NOTICE_OPT_MDL = "FINISH_NOTICE_OPT_MDL"
    FINISH_NOTICE_OPT_ELE_MDL = "FINISH_NOTICE_OPT_ELE_MDL"
    FINISH_NOTICE_CABLE_FRONTEND = "FINISH_NOTICE_CABLE_FRONTEND"
    FINISH_NOTICE_CABLE_MGMT = "FINISH_NOTICE_CABLE_MGMT"
    FINISH_NOTICE_CABLE_PCIE = "FINISH_NOTICE_CABLE_PCIE"
    FINISH_NOTICE_CABLE_SAS = "FINISH_NOTICE_CABLE_SAS"
    FINISH_NOTICE_FRONTEND_INTF = "FINISH_NOTICE_FRONTEND_INTF"
    FINISH_NOTICE_OTHER_INTF = "FINISH_NOTICE_OTHER_INTF"
    FINISH_NOTICE_EXPRESS_ACC = "FINISH_NOTICE_EXPRESS_ACC"
    FINISH_NOTICE_MGMT_MDL = "FINISH_NOTICE_MGMT_MDL"
    FINISH_NOTICE_PCIE_MDL = "FINISH_NOTICE_PCIE_MDL"
    FINISH_NOTICE_PCIE_DSW = "FINISH_NOTICE_PCIE_DSW"
    FINISH_NOTICE_POWER_MDL = "FINISH_NOTICE_POWER_MDL"
    FINISH_NOTICE_RISKY_POWER_MDL = "FINISH_NOTICE_RISKY_POWER_MDL"
    FINISH_NOTICE_IPSCALEOUT_DSW = "FINISH_NOTICE_IPSCALEOUT_DSW"
    FINISH_NOTICE_IPSCALEOUT_INTF = "FINISH_NOTICE_IPSCALEOUT_INTF"
    FINISH_NOTICE_IPSCALEOUT_CALBE = "FINISH_NOTICE_IPSCALEOUT_CALBE"
    FINISH_POWER_POSITION_CONTROL_ENCLOSURE = "FINISH_POWER_POSITION_CONTROL_ENCLOSURE"
    FINISH_POWER_POSITION_IP_ENCLOSURE = "FINISH_POWER_POSITION_IP_ENCLOSURE"

    # 公共部分
    GET_FRU_X_PARENT_ENC_EXCEPTION = "GET_FRU_X_PARENT_ENC_EXCEPTION"
    FUNC_EXISTED_SYNC_ERROR = "FUNC_EXISTED_SYNC_ERROR"
    FRU_INTERFACE_MODULE_NEED_UPG_SPH2 = "FRU_INTERFACE_MODULE_NEED_UPG_SPH2"
    FRU_CHECK_CONTAINER_REDUNDANCY_FAILED = "FRU_CHECK_CONTAINER_REDUNDANCY_FAILED"
    FRU_CHECK_MEMORY_STATUS_FAILED = "FRU_CHECK_MEMORY_STATUS_FAILED"
    FRU_CHECK_MEMORY_ALARM_FAILED = "FRU_CHECK_MEMORY_ALARM_FAILED"

    # 后端接口卡
    BACKINTF_EXISTED_NONE_LINKUP_SAS_PORT = "BACKINTF_EXISTED_NONE_LINKUP_SAS_PORT"
    BACKINTF_SAS_CABLE_CHECK_FAIL = "BACKINTF_SAS_CABLE_CHECK_FAIL"
    BACKINTF_SAS_LOOP_CHECK_FAIL = "BACKINTF_SAS_LOOP_CHECK_FAIL"
    BACKINTF_EXISTED_NONE_REDT_SAS_INTF = "BACKINTF_EXISTED_NONE_REDT_SAS_INTF"
    BACKINTF_EXISTED_NONE_NORMAL_REDT_SAS_INTF = "BACKINTF_EXISTED_NONE_NORMAL_REDT_SAS_INTF"
    BACKINTF_REDT_PATH_CHECK_FAIL = "BACKINTF_REDT_PATH_CHECK_FAIL"
    BACKINTF_REDT_PATH_NOT_SYMMETRY = "BACKINTF_REDT_PATH_NOT_SYMMETRY"

    # 前端接口模块
    FRONTINTF_EXISTED_NONE_WORKING_MODEL_PORT = "FRONTINTF_EXISTED_NONE_WORKING_MODEL_PORT"
    FRONTINTF_WORKING_MODEL_NOT_SAME = "FRONTINTF_WORKING_MODEL_NOT_SAME"
    FRONTINTF_PORT_CHECK_CONSISTENT = "FRONTINTF_PORT_CHECK_CONSISTENT"
    SWITCH_INTF_BUSINESS_FAIL = "SWITCH_INTF_BUSINESS_FAIL"

    PORTS_CHECK_INCONSISTENT_AFTER_REP = "PORTS_CHECK_INCONSISTENT_AFTER_REP"

    # 计算型存储计算节点
    COMPUTE_HOST_CHECK_CONSISTENT = "COMPUTE_HOST_CHECK_CONSISTENT"
    COMPUTE_PORT_CHECK_CONSISTENT = "COMPUTE_PORT_CHECK_CONSISTENT"
    COMPUTE_VM_CHECK_CONSISTENT = "COMPUTE_VM_CHECK_CONSISTENT"
    COMPUTE_SERVICE_VM_NOT_OPEN = "COMPUTE_SERVICE_VM_NOT_OPEN"

    # 电池
    BBU_X_EXISTED_NONE_REDT_BBU = "BBU_X_EXISTED_NONE_REDT_BBU"
    BBU_IN_BUNDLE_FAN_X_EXISTED_NONE_REDT_BBU = "BBU_IN_BUNDLE_FAN_X_EXISTED_NONE_REDT_BBU"
    BBU_IN_BUNDLE_FAN_X_INEXISTED = "BBU_IN_BUNDLE_FAN_X_INEXISTED"
    BBU_IN_BUNDLE_POWER_X_EXISTED_NONE_REDT_BBU = "BBU_IN_BUNDLE_POWER_X_EXISTED_NONE_REDT_BBU"
    BBU_IN_BUNDLE_FAN2_X_EXISTED_NONE_REDT_BBU = "BBU_IN_BUNDLE_FAN2_X_EXISTED_NONE_REDT_BBU"
    BBU_CHECK_CONSISTENT = "BBU_CHECK_CONSISTENT"
    BBU_IN_BUNDLE_NOT_SUPPORT = "BBU_IN_BUNDLE_NOT_SUPPORT"

    FAN_CHECK_CONSISTENT = "FAN_CHECK_CONSISTENT"

    POWER_CHECK_CONSISTENT = "POWER_CHECK_CONSISTENT"

    SYS_ALARM_CHECK_CONSISTENT = "SYS_ALARM_CHECK_CONSISTENT"

    SYS_EXIST_SCALE_OUT_ALMS = "SYS_EXIST_SCALE_OUT_ALMS"

    DISK_STAIUS_CHECK_CONSISTENT = "DISK_STATUS_CHECK_CONSISTENT"
    HOST_INITIATOR_CHECK_CONSISTENT = "HOST_INITIATOR_CHECK_CONSISTENT"
    POWER_STATUS_CHECK_CONSISTENT = "POWER_STATUS_CHECK_CONSISTENT"

    QUERY_RESULT_ABNORMAL = "QUERY_RESULT_ABNORMAL"

    HYPER_METRO_PAIR_RUNNING_STATUS_ABNORMAL = "HYPER_METRO_PAIR_RUNNING_STATUS_ABNORMAL"
    FRU_SWITCH_LIGNT_STATUS_FAILED = "FRU_SWITCH_LIGNT_STATUS_FAILED"
    BBU_BATTERY_PACK_FAULT_EXISTED = "BBU_BATTERY_PACK_FAULT_EXISTED"

    # FuncFactory
    FUNC_CANNOT_GET_SYSVER_FOR_NO_NORMAL_ONLINE_CNTR_EXISTED = "FUNC_CANNOT_GET_SYSVER_FOR_NO_NORMAL_ONLINE_CNTR_EXISTED"
    FUNC_UNSUPPORTED_FRU_TYPE = "FUNC_UNSUPPORTED_FRU_TYPE"
    FUNC_EXISTED_LINKDOWN_VALUEADD_PORT_Y = "FUNC_EXISTED_LINKDOWN_VALUEADD_PORT_Y"
    FUNC_EXISTED_CAN_NOT_SYNC_REPLICATION_Y = "FUNC_EXISTED_CAN_NOT_SYNC_REPLICATION_Y"
    FUNC_GET_CNTR_ENC_MODEL_FAIL = "FUNC_GET_CNTR_ENC_MODEL_FAIL"
    FUNC_UNDEFINED_CNTR_ENC_TYPE_X = "FUNC_UNDEFINED_CNTR_ENC_TYPE_X"
    FUNC_UNDEFINED_DISK_ENC_TYPE_X = "FUNC_UNDEFINED_DISK_ENC_TYPE_X"
    FUNC_GET_INFO_LIST_X_FAIL = "FUNC_GET_INFO_LIST_X_FAIL"
    FUNC_GET_INFO_Y_FAIL = "FUNC_GET_INFO_Y_FAIL"
    FUNC_PORT_X_START_TIMEOUT = "FUNC_PORT_X_START_TIMEOUT"
    FUNC_PORT_X_NOT_WORK = "FUNC_PORT_X_NOT_WORK"
    FUNC_PORT_X2_WORK_ABNORMAL = "FUNC_PORT_X2_WORK_ABNORMAL"
    FUNC_REPLEACED_FRU_X_ABNORMAL = "FUNC_REPLEACED_FRU_X_ABNORMAL"
    DISK_DOMAIN_STATUS_RECONSTRUCTION = "DISK_DOMAIN_STATUS_RECONSTRUCTION"
    EXPBOARD_ABNORMAL = "EXPBOARD_ABNORMAL"
    FUNC_REPLEACED_FRU_PCIE_ABNORMAL = "FUNC_REPLEACED_FRU_PCIE_ABNORMAL"
    FUNC_REPLEACED_FRU_X1_BIND_IN_X2_ABNORMAL = "FUNC_REPLEACED_FRU_X1_BIND_IN_X2_ABNORMAL"
    FUNC_REPLEACED_POWER_X1_BIND_IN_X2_NO_CABLE = "FUNC_REPLEACED_POWER_X1_BIND_IN_X2_NO_CABLE"
    FUNC_GET_SYS_ALARM_FAIL = "FUNC_GET_SYS_ALARM_FAIL"
    FUNC_EXISTED_MULTIPATH_ALARM = "FUNC_EXISTED_MULTIPATH_ALARM"
    FUNC_EXISTED_MULTIPATH_ALARM_FOR_IB = "FUNC_EXISTED_MULTIPATH_ALARM_FOR_IB"
    FUNC_EXISTED_SINGLELINKSTATUS_ALARM = "FUNC_EXISTED_SINGLELINKSTATUS_ALARM"
    FUNC_EXISTED_MGT_PORT_LINK_ALARM = "FUNC_EXISTED_MGT_PORT_LINK_ALARM"
    FUNC_ONE_MGT_MODULE_NOT_EXIST = "FUNC_ONE_MGT_MODULE_NOT_EXIST"
    FUNC_QUERY_ETH_PORT_FAILED = "FUNC_QUERY_ETH_PORT_FAILED"
    FUNC_REDUNDANT_MGT_BOARD_PORT_DOWN = "FUNC_REDUNDANT_MGT_BOARD_PORT_DOWN"
    FUNC_INTF_POWER_OFF_FAIL = "FUNC_INTF_POWER_OFF_FAIL"
    SYSTEM_POWER_OFF_FAIL = "SYSTEM_POWER_OFF_FAIL"
    FUNC_INTF_POWER_ON_FAIL = "FUNC_INTF_POWER_ON_FAIL"
    FUNC_INTF_STATUS_ABNORMAL = "FUNC_INTF_STATUS_ABNORMAL"
    FUNC_SYS_RUN_STATUS_ABNORMAL = "FUNC_SYS_RUN_STATUS_ABNORMAL"
    FUNC_SERVICE_PRESSURE_OFF_SCALE = "FUNC_SERVICE_PRESSURE_OFF_SCALE"
    FUNC_DISK_TYPE_NOT_MATCH_ALARM = "FUNC_DISK_TYPE_NOT_MATCH_ALARM"
    FUNC_DISK_TYPE_NOT_MATCH = "FUNC_DISK_TYPE_NOT_MATCH"
    # 控制器
    CNTR_REDT_CNTR_FAN_ABNORMAL = "CNTR_REDT_CNTR_FAN_ABNORMAL"
    CNTR_REDT_CNTR_QRY_INTERNAL_PDT_MODEL_FAILED = "CNTR_REDT_CNTR_QRY_INTERNAL_PDT_MODEL_FAILED"
    CNTR_REPLEACED_CNTR_FAN_ABNORMAL = "CNTR_REPLEACED_CNTR_FAN_ABNORMAL"
    CNTR_ASTT_COOL_UNIT_NOT_IN_POSITION = "CNTR_ASTT_COOL_UNIT_NOT_IN_POSITION"
    CNTR_ASTT_COOL_UNIT_FAN_ABNORMAL = "CNTR_ASTT_COOL_UNIT_FAN_ABNORMAL"
    CNTR_SSHPORT_NOT_DEFAULT_PORT = "CNTR_SSHPORT_NOT_DEFAULT_PORT"

    USER_HAVE_NO_PERM = 'USER_HAVE_NO_PERM'
    CNTR_REBOOT_FAIL_OR_STATUS_ABNORMAL = "CNTR_REBOOT_FAIL_OR_STATUS_ABNORMAL"
    CNTR_GET_MGMT_MODULE_FAIL = "CNTR_GET_MGMT_MODULE_FAIL"
    CNTR_GET_MGMT_MDL_X_ETH_PORT_FAIL = "CNTR_GET_MGMT_MDL_X_ETH_PORT_FAIL"
    CNTR_MGMT_PORT_X_LINKDOWN = "CNTR_MGMT_PORT_X_LINKDOWN"
    CNTR_GET_CNTR_X_ETH_PORT_FAIL = "CNTR_GET_CNTR_X_ETH_PORT_FAIL"
    CNTR_CNTR_ETH_PORT_X_LINKDOWN = "CNTR_CNTR_ETH_PORT_X_LINKDOWN"
    CNTR_EXISTED_NONE_SAS_MODULE = "CNTR_EXISTED_NONE_SAS_MODULE"
    CNTR_EXISTED_NONE_LINKUP_SAS_PORT = "CNTR_EXISTED_NONE_LINKUP_SAS_PORT"
    CNTR_EXISTED_LINKDOWN_SAS_PORT = "CNTR_EXISTED_LINKDOWN_SAS_PORT"
    CNTR_EXISTED_LINKDOWN_FRONT_PORT = "CNTR_EXISTED_LINKDOWN_FRONT_PORT"
    CNTR_EXISTED_NONE_PCIE_INTF = "CNTR_EXISTED_NONE_PCIE_INTF"
    CNTR_EXISTED_NONE_NORMAL_IPSCALEOUT_INTF = "CNTR_EXISTED_NONE_NORMAL_IPSCALEOUT_INTF"
    CNTR_IPSCALEOUT_PORT_LINK_ABNORMAL = "CNTR_IPSCALEOUT_PORT_LINK_ABNORMAL"
    CNTR_GET_PCIE_PORT_FAIL = "CNTR_GET_PCIE_PORT_FAIL"
    CNTR_PCIE_PORT_X_LINKDOWN_OR_ABNORMAL = "CNTR_PCIE_PORT_X_LINKDOWN_OR_ABNORMAL"
    CNTR_REDT_CNTR_INEXISTED_OR_ABNORMAL = "CNTR_REDT_CNTR_INEXISTED_OR_ABNORMAL"
    CNTR_EXISTED_NONE_REDT_MGMT_MODULE = "CNTR_EXISTED_NONE_REDT_MGMT_MODULE"
    CNTR_GET_REDT_CNTR_X_ETH_PORT_FAIL = "CNTR_GET_REDT_CNTR_X_ETH_PORT_FAIL"
    CNTR_REDT_CNTR_EXISTED_NONE_SAS_MODULE = "CNTR_REDT_CNTR_EXISTED_NONE_SAS_MODULE"
    CNTR_REDT_CNTR_EXISTED_LINKDOWN_SAS_PORT = "CNTR_REDT_CNTR_EXISTED_LINKDOWN_SAS_PORT"
    CNTR_EXISTED_NONE_REDT_CNTR = "CNTR_EXISTED_NONE_REDT_CNTR"
    CNTR_EXISTED_NONE_REDT_PCIE_MODULE = "CNTR_EXISTED_NONE_REDT_PCIE_MODULE"
    CNTR_GET_REDT_PCIE_PORT_FAIL = "CNTR_GET_REDT_PCIE_PORT_FAIL"
    CNTR_IBC_OS_HS_NOT_DEFAULT_PASSWORD = "CNTR_IBC_OS_HS_NOT_DEFAULT_PASSWORD"
    CNTR_IBC_OS_HS_NOT_DEFAULT_KEYPAIR = "CNTR_IBC_OS_HS_NOT_DEFAULT_KEYPAIR"
    CNTR_OFFLINE_CNTR_FAIL = "CNTR_OFFLINE_CNTR_FAIL"
    CNTR_OFFLINE_CNTR_FAIL_TO_CONFIG_BACKUP_DATA = "CNTR_OFFLINE_CNTR_FAIL_TO_CONFIG_BACKUP_DATA"
    CNTR_OFFLINE_CNTR_FAIL_PAGES_FAULT = "CNTR_OFFLINE_CNTR_FAIL_PAGES_FAULT"
    CNTR_OFFLINE_CNTR_FAIL_TO_OFFLINE_CLUSTER = "CNTR_OFFLINE_CNTR_FAIL_TO_OFFLINE_CLUSTER"
    CNTR_NOT_SUPPORT_REPLACEMENT_BY_TOOL = "CNTR_NOT_SUPPORT_REPLACEMENT_BY_TOOL"
    CNTR_OFFLINE_CNTR_FAIL_SYSTEM_BUSY = "CNTR_OFFLINE_CNTR_FAIL_SYSTEM_BUSY"

    CTRL_POWER_ON_TRACE_NOTICE = 'CTRL_POWER_ON_TRACE_NOTICE'
    SASLINK_ALARM_CHECK_CONSISTENT = "SASLINK_ALARM_CHECK_CONSISTENT"

    CTRL_CHECK_FILE_SYS_FAIL = "CTRL_CHECK_FILE_SYS_FAIL"
    CTRL_CHECK_NAS_MAP_NDMP = "CTRL_CHECK_NAS_MAP_NDMP"
    CTRL_CHECK_NAS_NOMAP_NDMP = "CTRL_CHECK_NAS_NOMAP_NDMP"
    CTRL_CHECK_NAS_ONLY_NDMP = "CTRL_CHECK_NAS_ONLY_NDMP"
    FRU_CLI_DOMAIN_CONFIG_NEED_UPG_SPH5 = "FRU_CLI_DOMAIN_CONFIG_NEED_UPG_SPH5"

    # 控制框
    CNTRENC_REPLACED_ENC_STATUS_ABNORMAL = "CNTRENC_REPLACED_ENC_STATUS_ABNORMAL"
    CNTRENC_SAS_INTF_NOT_INPOSITON = "CNTRENC_SAS_INTF_NOT_INPOSITON"
    CNTRENC_GET_INTF_X_SAS_PORT_FAIL = "CNTRENC_GET_INTF_X_SAS_PORT_FAIL"
    CNTRENC_PORT_X_IS_LINKDOWN_OR_ABNORMAL = "CNTRENC_PORT_X_IS_LINKDOWN_OR_ABNORMAL"
    CNTRENC_SAS_PORT_IS_INCONSISTENT = "CNTRENC_SAS_PORT_IS_INCONSISTENT"
    CNTRENC_GET_CNTR_INFO_FAIL = "CNTRENC_GET_CNTR_INFO_FAIL"
    CNTRENC_GET_CNTR_X_SAS_PORT_FAIL = "CNTRENC_GET_CNTR_X_SAS_PORT_FAIL"
    CNTRENC_EXISTED_NOT_INPOSITION_POWER = "CNTRENC_EXISTED_NOT_INPOSITION_POWER"
    CNTRENC_EXISTED_ABNORMAL_POWER_X = "CNTRENC_EXISTED_ABNORMAL_POWER_X"
    CNTRENC_INCONSITANT_FRONTEND_PORT_INFO = "CNTRENC_INCONSITANT_FRONTEND_PORT_INFO"
    CNTRENC_GET_PCIE_INTF_FAIL = "CNTRENC_GET_PCIE_INTF_FAIL"
    CNTRENC_PCIE_INTF_NOT_INPOSITION = "CNTRENC_PCIE_INTF_NOT_INPOSITION"
    CNTRENC_PCIE_PORT_LINKDOWN_OR_ABNORMAL = "CNTRENC_PCIE_PORT_LINKDOWN_OR_ABNORMAL"
    CNTRENC_PCIE_PORT_X_LINKDOWN_OR_ABNORMAL = "CNTRENC_PCIE_PORT_X_LINKDOWN_OR_ABNORMAL"
    CNTRENC_GET_MGMT_MDL_FAIL = "CNTRENC_GET_MGMT_MDL_FAIL"
    CNTRENC_MGMT_MDL_NOT_INPOSITION = "CNTRENC_MGMT_MDL_NOT_INPOSITION"
    CNTRENC_GET_MGMT_MDL_X_PORT_FAIL = "CNTRENC_GET_MGMT_MDL_X_PORT_FAIL"
    CNTRENC_MGMT_PORT_X_LINKDOWN_OR_ABNORMAL = "CNTRENC_MGMT_PORT_X_LINKDOWN_OR_ABNORMAL"
    CNTRENC_GET_CNTR_X_ETH_PORT_FAIL = "CNTRENC_GET_CNTR_X_ETH_PORT_FAIL"
    CNTRENC_CNTR_ETH_PORT_X_LINKDOWN_OR_ABNORMAL = "CNTRENC_CNTR_ETH_PORT_X_LINKDOWN_OR_ABNORMAL"
    CNTRENC_NOT_INSERT_ANY_DISK = "CNTRENC_NOT_INSERT_ANY_DISK"
    CNTRENC_DISK_DOMAIN_INEXISTED_OR_ABNORMAL = "CNTRENC_DISK_DOMAIN_INEXISTED_OR_ABNORMAL"
    CNTRENC_DISK_DOMAIN_STATUS_ABNORMAL = "CNTRENC_DISK_DOMAIN_STATUS_ABNORMAL"
    CNTRENC_REPLACED_ENC_EXISTED_ABNORMAL_CNTR = "CNTRENC_REPLACED_ENC_EXISTED_ABNORMAL_CNTR"
    CNTRENC_CNTR_X_EXISTED_NONE_NORMAL_IPSCALEOUT_INTF = "CNTRENC_CNTR_X_EXISTED_NONE_NORMAL_IPSCALEOUT_INTF"
    CNTRENC_CNTR_X_IPSCALEOUT_PORT_LINK_ABNORMAL = "CNTRENC_CNTR_X_IPSCALEOUT_PORT_LINK_ABNORMAL"
    CNTRENC_POWER_SUPPLY_ABNORMAL = "CNTRENC_POWER_SUPPLY_ABNORMAL"
    DISK_DOMAIN_SMO_NOT_PASS = "DISK_DOMAIN_SMO_NOT_PASS"
    LICENSE_FILE_DOWNLOAD_FAILED = "LICENSE_FILE_DOWNLOAD_FAILED"

    # 硬盘
    DISK_INVALID_DISK_DOMAIN = "DISK_INVALID_DISK_DOMAIN"
    DISK_HEALTHSTATUS_DEGRADE_DISK_DOMAIN = "DISK_HEALTHSTATUS_DEGRADE_DISK_DOMAIN"
    DISK_NOT_JIONINPOOL = "DISK_NOT_JIONINPOOL"
    DISK_HEALTHSTATUS_FAULT_DISK_DOMAIN = "DISK_HEALTHSTATUS_FAULT_DISK_DOMAIN"
    DISK_RUNNINGSTATUS_FAULT_DISK_DOMAIN = "DISK_RUNNINGSTATUS_FAULT_DISK_DOMAIN"
    DISK_START_DISK_PRECOPY_FAIL = "DISK_START_DISK_PRECOPY_FAIL"
    DISK_DISK_PRECOPY_TIMEOUT = "DISK_DISK_PRECOPY_TIMEOUT"
    DISK_PRECOPY_DISK_STATUS_ABNORMAL = "DISK_PRECOPY_DISK_STATUS_ABNORMAL"
    DISK_SPARE_CAPACITY_NOT_ENOUGH = "DISK_SPARE_CAPACITY_NOT_ENOUGH"
    DISK_DISK_PRECOPY_FAIL_BY_ALARM = "DISK_DISK_PRECOPY_FAIL_BY_ALARM"
    DISK_DISK_PRECOPY_FAIL = "DISK_DISK_PRECOPY_FAIL"
    DISK_UNKOWN_DISK_PRECOPY_ERROR = "DISK_UNKOWN_DISK_PRECOPY_ERROR"
    DISK_DISK_REFACTOR_FAIL = "DISK_DISK_REFACTOR_FAIL"
    DISK_UNKOWN_DISK_PRECOPY_ERROR = "DISK_UNKOWN_DISK_PRECOPY_ERROR"
    DISK_RUNNINGSTATUS_NOT_ONLINE = "DISK_RUNNINGSTATUS_NOT_ONLINE"
    DISK_TYPE_NOT_INCONSISTENT = "DISK_TYPE_NOT_INCONSISTENT"
    REPLACEMENT_DISK_TYPE_NOT_MATCH = "REPLACEMENT_DISK_TYPE_NOT_MATCH"
    DISK_CAPACITY_INCORRECT = "DISK_CAPACITY_INCORRECT"
    DISK_NOT_SUPPORT_BY_MULTI_STREAM_POOL = "DISK_NOT_SUPPORT_BY_MULTI_STREAM_POOL"
    DISK_ROLE_NOT_INCONSISTENT = "DISK_ROLE_NOT_INCONSISTENT"
    DISK_LOCATION_INVALID = "DISK_LOCATION_INVALID"
    DISK_NO_SUPPORT_INVALID = "DISK_NO_SUPPORT_INVALID"
    FRU_SOFTWARE_PATCH_NOT_MATCH = "FRU_SOFTWARE_PATCH_NOT_MATCH"
    FRU_SED_MSID_CHECK_FAILED = "FRU_SED_MSID_CHECK_FAILED"
    PERFORMANCE_NOT_ASSOCIATE_DISK_POOL = "PERFORMANCE_NOT_ASSOCIATE_DISK_POOL"
    SUGGEST_REPLACE_DISK_RECONSTRUCT_FINISH = "SUGGEST_REPLACE_DISK_RECONSTRUCT_FINISH"

    # 硬盘框
    DISKENC_REPLACED_ENC_STATUS_ABNORMAL = "DISKENC_REPLACED_ENC_STATUS_ABNORMAL"
    REPLACED_DISKDRAWER_STATUS_ABNORMAL = "REPLACED_DISKDRAWER_STATUS_ABNORMAL"
    DISKENC_GET_EXPBOARD_FAIL = "DISKENC_GET_EXPBOARD_FAIL"
    DISKENC_EXISTED_NOT_INPOSITION_EXPBOARD = "DISKENC_EXISTED_NOT_INPOSITION_EXPBOARD"
    DISKENC_GET_EXPBOARD_X_PORT_FAIL = "DISKENC_GET_EXPBOARD_X_PORT_FAIL"
    DISKENC_PORT_X_LINKDOWN_OR_ABNORMAL = "DISKENC_PORT_X_LINKDOWN_OR_ABNORMAL"
    DISKENC_EXPBOARD_SAS_PORT_ABNORMAL = "DISKENC_EXPBOARD_SAS_PORT_ABNORMAL"
    DISKENC_EXISTED_NOT_INPOSITION_POWER = "DISKENC_EXISTED_NOT_INPOSITION_POWER"
    DISKENC_POWER_MDL_X_IS_ABNORMAL = "DISKENC_POWER_MDL_X_IS_ABNORMAL"
    DISKENC_NOT_INSERT_ANY_DISK = "DISKENC_NOT_INSERT_ANY_DISK"
    DISKENC_DISK_DOMAIN_INEXISTED_OR_ABNORMAL = "DISKENC_DISK_DOMAIN_INEXISTED_OR_ABNORMAL"
    DISKENC_DISK_DOMAIN_STATUS_ABNORMAL = "DISKENC_DISK_DOMAIN_STATUS_ABNORMAL"
    BBU_IN_BUNDLE_POWER_X_EXISTED_INEXISTED = "BBU_IN_BUNDLE_POWER_X_EXISTED_INEXISTED"
    DISKENC_ENCLOSURE_REDUNDANT_CABLE_ERR = \
        "DISKENC_ENCLOSURE_REDUNDANT_CABLE_ERR"

    # 级联板
    EXPBOARD_GET_REPLACED_SES_VERSION_FAIL = "EXPBOARD_GET_REPLACED_SES_VERSION_FAIL"
    EXPBOARD_INCONSITANT_SES_VERSION = "EXPBOARD_INCONSITANT_SES_VERSION"
    EXPBOARD_SAS_CABLE_CHECK_FAIL_BY_GET_PORT_FAIL = "EXPBOARD_SAS_CABLE_CHECK_FAIL_BY_GET_PORT_FAIL"
    EXPBOARD_SAS_CABLE_CHECK_FAIL_BY_PORT_LINKDOWN = "EXPBOARD_SAS_CABLE_CHECK_FAIL_BY_PORT_LINKDOWN"
    EXPBOARD_EXISTED_NONE_REDT_SAS_PATH = "EXPBOARD_EXISTED_NONE_REDT_SAS_PATH"
    EXPBOARD_STATUS_NOT_COINCIDENT = "EXPBOARD_STATUS_NOT_COINCIDENT"

    # 风扇
    FAN_X_EXISTED_NONE_REDT_FAN = "FAN_X_EXISTED_NONE_REDT_FAN"
    FAN_X_EXISTED_NUMBER_NOT_MEET = "FAN_X_EXISTED_NUMBER_NOT_MEET"
    FAN_X_EXISTED_NONE_REDT_FAN_CANNOT_REPLACE_CTRL_NORMAL_FAN = "FAN_X_EXISTED_NONE_REDT_FAN_CANNOT_REPLACE_CTRL_NORMAL_FAN"
    FAN_X_EXISTED_NONE_REDT_FAN_CANNOT_REPLACE_FANCTRL_NORMAL_FAN = "FAN_X_EXISTED_NONE_REDT_FAN_CANNOT_REPLACE_FANCTRL_NORMAL_FAN"
    FAN_IN_BUNDLE_BBU_X_EXISTED_NONE_REDT_FAN = "FAN_IN_BUNDLE_BBU_X_EXISTED_NONE_REDT_FAN"
    FAN_IN_BUNDLE_BBU_X_INEXISTED = "FAN_IN_BUNDLE_BBU_X_INEXISTED"
    FAN_X_IN_BUNDLE_POWER_EXISTED_NONE_REDT_FAN = "FAN_X_IN_BUNDLE_POWER_EXISTED_NONE_REDT_FAN"
    FAN_BUNDLE_FAN_IN_POWER_X_INEXISTED = "FAN_BUNDLE_FAN_IN_POWER_X_INEXISTED"
    FAN_BUNDLE_FAN_IN_POWER2_X_INEXISTED = "FAN_BUNDLE_FAN_IN_POWER2_X_INEXISTED"
    FAN_IN_BUNDLE_BBU2_X_INEXISTED = "FAN_IN_BUNDLE_BBU2_X_INEXISTED"

    # 光模块
    FIBREMDL_NOT_INSERT_FIBRE_MDL_X = "FIBREMDL_NOT_INSERT_FIBRE_MDL_X"
    FIBREMDL_FIBRE_MDL_Y_CABLE_NOT_LINK = "FIBREMDL_FIBRE_MDL_Y_CABLE_NOT_LINK"
    IPSCALEOUT_FIBREMODULE_CABLE_NOT_LINK = "IPSCALEOUT_FIBREMODULE_CABLE_NOT_LINK"
    IPSCALEOUT_FIBREMODULE_CABLE_LINK_WRONG = "IPSCALEOUT_FIBREMODULE_CABLE_LINK_WRONG"
    SIWTCH_FM_SERVICE_FAIL = "SIWTCH_FM_SERVICE_FAIL"
    NOT_EXIST_REDUNDANT_FM_PORT = "NOT_EXIST_REDUNDANT_FM_PORT"

    # IP Scale Out线缆
    IPSCALEOUT_CABLE_SAME_START_END_PORT = "IPSCALEOUT_CABLE_SAME_START_END_PORT"
    IPSCALEOUT_CABLE_SEL_PORTS_BREAK_RULES = "IPSCALEOUT_CABLE_SEL_PORTS_BREAK_RULES"
    IPSCALEOUT_CABLE_PORT_X2_INCONSISTENT_LINK_RIGHT = "IPSCALEOUT_CABLE_PORT_X2_INCONSISTENT_LINK_RIGHT"
    IPSCALEOUT_CABLE_X_RED_PATH_INEXIST = "IPSCALEOUT_CABLE_X_RED_PATH_INEXIST"
    IPSCALEOUT_CABLE_NOT_INSERT_OR_ABNORMAL = "IPSCALEOUT_CABLE_NOT_INSERT_OR_ABNORMAL"
    IPSCALEOUT_CABLE_PORT_X_LINK_WRONG = "IPSCALEOUT_CABLE_PORT_X_LINK_WRONG"
    FRU_CHECK_IP_SCALE_OUT_CABLE_FAILED = "FRU_CHECK_IP_SCALE_OUT_CABLE_FAILED"

    # 前端线缆
    CABFREND_CABLE_NOT_LINK_PORT_Y = "CABFREND_CABLE_NOT_LINK_PORT_Y"
    SIWTCH_FC_SERVICE_FAIL = "SIWTCH_FC_SERVICE_FAIL"

    # 管理网线
    CABMGMT_SAME_START_END_PORT = "CABMGMT_SAME_START_END_PORT"
    CABMGMT_V3_MUST_CONTAINS_MGMT_HOST_PORT = "CABMGMT_V3_MUST_CONTAINS_MGMT_HOST_PORT"
    CABMGMT_SNGL_ENG_DONOT_SUPPORT_MGMT_PORT_CONN = "CABMGMT_SNGL_ENG_DONOT_SUPPORT_MGMT_PORT_CONN"
    CABMGMT_DIFF_START_END_PORT_LINK_STATUS = "CABMGMT_DIFF_START_END_PORT_LINK_STATUS"
    CABMGMT_SUG_CHOOSE_CORRECT_PORT = "CABMGMT_SUG_CHOOSE_CORRECT_PORT"
    CABMGMT_RED_MGMT_PATH_INEXIST = "CABMGMT_RED_MGMT_PATH_INEXIST"
    CABMGMT_CABLE_NOT_INSERT_OR_ABNORMAL = "CABMGMT_CABLE_NOT_INSERT_OR_ABNORMAL"
    CABMGMT_MGMT_PORT_IP_IS_NOT_REACHABLE = "CABMGMT_MGMT_PORT_IP_IS_NOT_REACHABLE"
    CABMGMT_MGMT_DSW_PORT_IP_IS_NOT_REACHABLE = "CABMGMT_MGMT_DSW_PORT_IP_IS_NOT_REACHABLE"
    CABMGMT_MGMT_PORT_X_IP_IS_NOT_REACHABLE = "CABMGMT_MGMT_PORT_X_IP_IS_NOT_REACHABLE"
    CABMGMT_EXISTED_X_UNLINK_PORT_IN_LOOP = "CABMGMT_EXISTED_X_UNLINK_PORT_IN_LOOP"
    CABMGMT_IP_CLUSTER_MUST_CONTAINS_TWO_NORMAL_MGMT_CABLE = "CABMGMT_IP_CLUSTER_MUST_CONTAINS_TWO_NORMAL_MGMT_CABLE"
    CABMGMT_EXISTED_CABMGMTALARM = "CABMGMT_EXISTED_CABMGMTALARM"

    # PCIe线缆
    CABPCIE_HAS_NO_RED_PCIE_PATH = "CABPCIE_HAS_NO_RED_PCIE_PATH"
    CABPCIE_PORT_X_STATUS_ABNORMAL = "CABPCIE_PORT_X_STATUS_ABNORMAL"
    CABPCIE_PCIE_CABLE_Y2_LINK_WRONG = "CABPCIE_PCIE_CABLE_Y2_LINK_WRONG"

    # SAS线缆
    CABSAS_CABLE_NOT_STANDARD_LINK = "CABSAS_CABLE_NOT_STANDARD_LINK"
    CABSAS_CABLE_NOT_INSERT_OR_ABNORMAL = "CABSAS_CABLE_NOT_INSERT_OR_ABNORMAL"
    ALM_DEV_FRAME_SINGLE_LINKED = "ALM_DEV_FRAME_SINGLE_LINKED"
    ALM_DEV_EXP_PORT_CABLE_LINK_ERROR = "ALM_DEV_EXP_PORT_CABLE_LINK_ERROR"
    BACK_PORT_CHECK_CONSISTENT = "BACK_PORT_CHECK_CONSISTENT"
    ALM_DEV_EXP_PORT_CABLE_LINK_NOT_PASS = "ALM_DEV_EXP_PORT_CABLE_LINK_NOT_PASS"

    # 管理板
    MGMTBOARD_GET_REPLACED_MGMT_PORT_FAIL = "MGMTBOARD_GET_REPLACED_MGMT_PORT_FAIL"
    MGMTBOARD_REPLACED_SOME_PORT_LINK_WRONG = "MGMTBOARD_REPLACED_SOME_PORT_LINK_WRONG"
    MGMTBOARD_REDT_MGMT_PATH_CHECK_FAIL = "MGMTBOARD_REDT_MGMT_PATH_CHECK_FAIL"
    MGMTBOARD_REDT_MGMT_MDL_INEXISTED_OR_ABNORMAL = "MGMTBOARD_REDT_MGMT_MDL_INEXISTED_OR_ABNORMAL"
    MGMTBOARD_GET_REDT_MGMT_PORT_FAIL = "MGMTBOARD_GET_REDT_MGMT_PORT_FAIL"
    MGMTBOARD_SWITCH_TOOL_CONNECT_IP_FAIL = "MGMTBOARD_SWITCH_TOOL_CONNECT_IP_FAIL"
    MGMTBOARD_CHECK_UPD_STATUES_FAILED = "MGMTBOARD_CHECK_UPD_STATUES_FAILED"
    MGMTBOARD_UPD_STATUES_ABNORMAL = "MGMTBOARD_UPD_STATUES_ABNORMAL"

    # PCIe接口卡
    PCIEINTF_PCIE_PORT_X_LINKDOWN_OR_ABNORMAL = "PCIEINTF_PCIE_PORT_X_LINKDOWN_OR_ABNORMAL"
    PCIEINTF_REDT_CNTR_EXISTED_NONE_SAS_INTF = "PCIEINTF_REDT_CNTR_EXISTED_NONE_SAS_INTF"
    PCIEINTF_REDT_MGMT_MDL_INEXISTED_OR_ABNORMAL = "PCIEINTF_REDT_MGMT_MDL_INEXISTED_OR_ABNORMAL"
    PCIEINTF_GET_REDT_MGMT_PORT_FAIL = "PCIEINTF_GET_REDT_MGMT_PORT_FAIL"
    PCIEINTF_EXISTED_LINKDOWN_PORT = "PCIEINTF_EXISTED_LINKDOWN_PORT"
    PCIEINTF_EXISTED_LINKERROR_PORT = "PCIEINTF_EXISTED_LINKERROR_PORT"
    PCIEINTF_RED_EXISTED_LINKERROR_PORT = "PCIEINTF_RED_EXISTED_LINKERROR_PORT"
    PCIEINTF_RED_EXISTED_LINKDOWN_PORT = "PCIEINTF_RED_EXISTED_LINKDOWN_PORT"
    PCIEINTF_SWITCH_CNTR_SERVICE_FAIL = "PCIEINTF_SWITCH_CNTR_SERVICE_FAIL"  # 20161101新增，原代码中没有

    # PCIe交换机
    PCIESWT_MGMT_CABLE_CHECK_FAIL = "PCIESWT_MGMT_CABLE_CHECK_FAIL"
    PCIESWT_EXISTED_NONE_REDT_SWT = "PCIESWT_EXISTED_NONE_REDT_SWT"
    PCIESWT_PCIE_CABLE_CHECK_FAIL = "PCIESWT_PCIE_CABLE_CHECK_FAIL"
    PCIESWT_EXISTED_NONE_REDT_PCIE_PATH = "PCIESWT_EXISTED_NONE_REDT_PCIE_PATH"
    PCIESWT_EXISTED_NONE_REDT_MGMT_PATH = "PCIESWT_EXISTED_NONE_REDT_MGMT_PATH"
    PCIESWT_HEALTHSTATUS_CHECK_FAIL = "PCIESWT_HEALTHSTATUS_CHECK_FAIL"
    PCIESWT_REDSWT_EXISTED_LINKDOWN_PORT = "PCIESWT_REDSWT_EXISTED_LINKDOWN_PORT"
    PCIESWT_REDSWT_EXISTED_LINKERROR_PORT = "PCIESWT_REDSWT_EXISTED_LINKERROR_PORT"
    PCIESWT_SWT_EXISTED_LINKDOWN_PORT = "PCIESWT_SWT_EXISTED_LINKDOWN_PORT"
    PCIESWT_SWT_EXISTED_LINKERROR_PORT = "PCIESWT_SWT_EXISTED_LINKERROR_PORT"

    # 电源
    POWER_Y_CABLE_IS_DISCONNECTED = "POWER_Y_CABLE_IS_DISCONNECTED"
    POWER_X_EXIST_NONE_REDT_POWER = "POWER_X_EXIST_NONE_REDT_POWER"
    POWER_X_IN_BUNDLE_FAN_EXIST_NONE_REDT_POWER = "POWER_X_IN_BUNDLE_FAN_EXIST_NONE_REDT_POWER"
    POWER_BUNDLE_POWER_IN_FAN_X_INEXISTED = "POWER_BUNDLE_POWER_IN_FAN_X_INEXISTED"
    POWER_BUNDLE_POWER_IN_BBU_X_INEXISTED = "POWER_BUNDLE_POWER_IN_BBU_X_INEXISTED"
    POWER_BUNDLE_POWER_IN_FAN2_X_INEXISTED = "POWER_BUNDLE_POWER_IN_FAN2_X_INEXISTED"

    # IP Scale Out交换机
    IPSCALEOUT_DSW_RED_DSW_INEXISTED = "IPSCALEOUT_DSW_RED_DSW_INEXISTED"
    IPSCALEOUT_DSW_RED_PATH_INEXISTED = "IPSCALEOUT_DSW_RED_PATH_INEXISTED"
    IPSCALEOUT_DSW_REPLACED_DSW_NOT_EXISTED = "IPSCALEOUT_DSW_REPLACED_DSW_NOT_EXISTED"
    IPSCALEOUT_DSW_REPLACED_DSW_CABLE_NOT_INTEGERITY = "IPSCALEOUT_DSW_REPLACED_DSW_CABLE_NOT_INTEGERITY"
    IPSCALEOUT_DSW_IPADDRESS_ERROR = "IPSCALEOUT_DSW_IPADDRESS_ERROR"
    IPSCALEOUT_DSW_USERNAME_ERROR = "IPSCALEOUT_DSW_USERNAME_ERROR"
    IPSCALEOUT_DSW_PASSWORD_ERROR = "IPSCALEOUT_DSW_PASSWORD_ERROR"
    IPSCALEOUT_DSW_PORT_ERROR = "IPSCALEOUT_DSW_PORT_ERROR"
    IPSCALEOUT_DSW_LAYER_ERROR = "IPSCALEOUT_DSW_LAYER_ERROR"
    IPSCALEOUT_DSW_TRUNKPORT_ERROR = "IPSCALEOUT_DSW_TRUNKPORT_ERROR"
    IPSCALEOUT_DSW_CONNECT_ERROR = "IPSCALEOUT_DSW_CONNECT_ERROR"
    IPSCALEOUT_DSW_VER_CHECK_ERROR = "IPSCALEOUT_DSW_VER_CHECK_ERROR"
    IPSCALEOUT_DSW_CONFIG_ERROR = "IPSCALEOUT_DSW_CONFIG_ERROR"

    # IP Scale Out接口卡
    IPSCALEOUT_INTF_REPLACED_INTF_INEXISTED_OR_ABNORMAL = "IPSCALEOUT_INTF_REPLACED_INTF_INEXISTED_OR_ABNORMAL"
    IPSCALEOUT_INTF_REPLACED_INTF_CONTROLLER_NOT_EXISTED = "IPSCALEOUT_INTF_REPLACED_INTF_CONTROLLER_NOT_EXISTED"
    IPSCALEOUT_INTF_REDT_SWITCH_PATH_NOT_EXISTED = "IPSCALEOUT_INTF_REDT_SWITCH_PATH_NOT_EXISTED"
    IPSCALEOUT_INTF_REPLACED_INTF_X_CABLE_LINK_WRONG = "IPSCALEOUT_INTF_REPLACED_INTF_X_CABLE_LINK_WRONG"
    IPSCALEOUT_INTF_INCONSITANT_IPSCALEOUT_PORT_INFO = "IPSCALEOUT_INTF_INCONSITANT_IPSCALEOUT_PORT_INFO"

    # unuse
    IPSCALEOUT_INTF_SWITCH_CNTR_SERVICE_FAIL = "IPSCALEOUT_INTF_SWITCH_CNTR_SERVICE_FAIL"
    IPSCALEOUT_INTF_REDT_SAS_PATH_NOT_EXISTED = "IPSCALEOUT_INTF_REDT_SAS_PATH_NOT_EXISTED"
    IPSCALEOUT_INTF_REDT_MGMT_MDL_INEXISTED_OR_ABNORMAL = "IPSCALEOUT_INTF_REDT_MGMT_MDL_INEXISTED_OR_ABNORMAL"
    IPSCALEOUT_INTF_GET_REDT_MGMT_PORT_FAIL = "IPSCALEOUT_INTF_GET_REDT_MGMT_PORT_FAIL"
    IPSCALEOUT_INTF_MGMG_CABLE_CHECK_FAIL = "IPSCALEOUT_INTF_MGMG_CABLE_CHECK_FAIL"

    # IP Scale Out线缆
    IPSCALEOUT_CABLE_SAME_START_END_PORT = "IPSCALEOUT_CABLE_SAME_START_END_PORT"
    IPSCALEOUT_CABLE_SEL_PORTS_BREAK_RULES = "IPSCALEOUT_CABLE_SEL_PORTS_BREAK_RULES"
    IPSCALEOUT_CABLE_PORT_X2_INCONSISTENT_LINK_RIGHT = "IPSCALEOUT_CABLE_PORT_X2_INCONSISTENT_LINK_RIGHT"
    IPSCALEOUT_CABLE_X_RED_PATH_INEXIST = "IPSCALEOUT_CABLE_X_RED_PATH_INEXIST"
    IPSCALEOUT_CABLE_NOT_INSERT_OR_ABNORMAL = "IPSCALEOUT_CABLE_NOT_INSERT_OR_ABNORMAL"
    IPSCALEOUT_CABLE_PORT_X_LINK_WRONG = "IPSCALEOUT_CABLE_PORT_X_LINK_WRONG"

    # IP Scale Out光模块
    IPSCALEOUT_FIBREMODULE_CABLE_NOT_LINK = "IPSCALEOUT_FIBREMODULE_CABLE_NOT_LINK"
    IPSCALEOUT_FIBREMODULE_CABLE_LINK_WRONG = "IPSCALEOUT_FIBREMODULE_CABLE_LINK_WRONG"

    # 管理风扇控制板
    MGMTFANCTRLBOARD_REPLACED_INEXISTED_OR_HEALTHSTATUSABNORMAL = "MGMTFANCTRLBOARD_REPLACED_INEXISTED_OR_HEALTHSTATUSABNORMAL"
    MGMTFANCTRLBOARD_REPLACED_RUNNINGSTATUSABNORMAL = "MGMTFANCTRLBOARD_REPLACED_RUNNINGSTATUSABNORMAL"

    # 公用处理建议
    PUBSUG_CHECK_SAS_CABLE_NORMAL = "PUBSUG_CHECK_SAS_CABLE_NORMAL"
    PUBSUG_ASSURE_REDT_CNTR_SAS_PORT_NORMAL = "PUBSUG_ASSURE_REDT_CNTR_SAS_PORT_NORMAL"
    PUBSUG_CHECK_PORT_NORMAL = "PUBSUG_CHECK_PORT_NORMAL"
    PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP = "PUBSUG_CONTACT_TECH_SPT_ENGINEER_FOR_HELP"
    PUBSUG_ASSURE_DEV_NETWORK_NORMAL = "PUBSUG_ASSURE_DEV_NETWORK_NORMAL"
    PUBSUG_CHECK_ETH_CABLE_NORMAL = "PUBSUG_CHECK_ETH_CABLE_NORMAL"
    PUBSUG_ASSURE_CNTR_SAS_PORT_NORMAL = "PUBSUG_ASSURE_CNTR_SAS_PORT_NORMAL"
    PUBSUG_ASSURE_CNTR_FRONT_PORT_NORMAL = "PUBSUG_ASSURE_CNTR_FRONT_PORT_NORMAL"
    PUBSUG_ASSURE_REDT_MGMT_MDL_MORMAL = "PUBSUG_ASSURE_REDT_MGMT_MDL_MORMAL"
    PUBSUG_CHECK_PCIE_CABLE_NORMAL = "PUBSUG_CHECK_PCIE_CABLE_NORMAL"
    PUBSUG_CHECK_CNTR_NORMAL = "PUBSUG_CHECK_CNTR_NORMAL"
    PUBSUG_CONNECT_POWER_CABLE = "PUBSUG_CONNECT_POWER_CABLE"
    PUBSUG_CHECK_PCIE_INTF_INPOSITION = "PUBSUG_CHECK_PCIE_INTF_INPOSITION"
    PUBSUG_CHECK_MGMT_MDL_INPOSITION = "PUBSUG_CHECK_MGMT_MDL_INPOSITION"
    PUBSUG_INSERT_DISK = "PUBSUG_INSERT_DISK"
    PUBSUG_HANDLE_ALARM_AND_RETRY = "PUBSUG_HANDLE_ALARM_AND_RETRY"
    PUBSUG_ASSURE_REDT_MGMT_MDL_MORMAL = "PUBSUG_ASSURE_REDT_MGMT_MDL_MORMAL"
    PUBSUG_RETRY_WHEN_DEV_NETWORK_NORMAL = "PUBSUG_RETRY_WHEN_DEV_NETWORK_NORMAL"
    PUBSUG_ASSURE_EXIST_REDT_SWITCH = "PUBSUG_ASSURE_EXIST_REDT_SWITCH"
    PUBSUG_NOT_SUPPORT_HIGH_DENSITY_DISK_ENCLOSURE = "PUBSUG_NOT_SUPPORT_HIGH_DENSITY_DISK_ENCLOSURE"
    PUBSUG_REDT_QUORUM_LINK_NOT_EXIST = "PUBSUG_REDT_QUORUM_LINK_NOT_EXIST"
    PUBSUG_REPL_QUORUM_LINK_UNNORMAL = "PUBSUG_REPL_QUORUM_LINK_UNNORMAL"
    PUBSUG_QUORUM_LINK_NOT_EXIST = "PUBSUG_QUORUM_LINK_NOT_EXIST"
    REPLACE_IP_SCALE_OUT_FAILED = "REPLACE_IP_SCALE_OUT_FAILED"

    # 虚拟机virtualMachineCheck
    VIRTUAL_MACHINE_NOT_CLOSED = "VIRTUAL_MACHINE_NOT_CLOSED"
    VIRTUAL_MACHINE_NOT_CLOSED_OTHER = "VIRTUAL_MACHINE_NOT_CLOSED_OTHER"
    # 更换FRU后有未连接的SAS端口
    SAS_PORT_NOT_LINKDOWN = "SAS_PORT_NOT_LINKDOWN"

    HIGH_END_CONTROL_SYSTEM_DISK_FAULT = "HIGH_END_CONTROL_SYSTEM_DISK_FAULT"
    MASTER_CONTROL_SYSTEM_DISK_FAULT = "MASTER_CONTROL_SYSTEM_DISK_FAULT"
    CURRENT_CONTROL_SYSTEM_DISK_FAULT = "CURRENT_CONTROL_SYSTEM_DISK_FAULT"
    FAN_X_EXIST_NONE_REDT_EXP = "FAN_X_EXIST_NONE_REDT_EXP"
    FAN_X_REDT_EXP_NOT_IN = "FAN_X_REDT_EXP_NOT_IN"
    FAN_X_REDT_EXP_NOT_IN_NORMAL = "FAN_X_REDT_EXP_NOT_IN_NORMAL"
    EXP_BUNDLE_FAN_X_INEXISTED = "EXP_BUNDLE_FAN_X_INEXISTED"

    SYS_EXIST_TMPRETURE_ALMS = "SYS_EXIST_TMPRETURE_ALMS"

def getElabelDict(elabelStr):
    '''
            功能：获取指电子标签字典，无电子标签返回空字典
    Params: elabelStr：电子标签字符串
    Return: elabelDict：电子标签字典
    '''
    ITEM_SPLIT_FLAG = "="
    elabelDict = {}
    if "" == elabelStr or 0 == len(elabelStr):
        return elabelDict

    elabelList = elabelStr.encode("utf8").splitlines()
    for field in elabelList:
        if None == field or 0 == len(field):
            continue

        fields = field.split(ITEM_SPLIT_FLAG)
        if len(fields) > 1:
            key = fields[0].strip()
            elabelDict[key] = fields[-1].strip()

    return elabelDict


def is18000V3(productModel):
    '''
    @summary: 判断是否是18000V3
    '''
    if productModel.upper() in config.HIGH_END_DEVS:
        return True
    return False


'''
判断是否是SAS大卡设备
'''


def isBigSasProduct(context):
    productModel = context.get("dev").get("type")
    version = context.get("dev").get("version", "")
    if "kunpeng" in version.lower():
        return False
    if productModel.upper() in config.HIGH_MID_BIGSAS:
        return True
    return False


# 获取阵列控制框数量
def getCtrlEncNum(context):
    allEncRecs = getFruListInfo(context, restData.Enum.ObjEnum.ENCLOSURE)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Enclosure.LOGIC_TYPE,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.EnclosureTypeEnum.CTRL)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)

    tmpRecs = restUtil.Tlv2Rest.filter(allEncRecs, conditionList)

    ctrlEncNum = len(tmpRecs)
    BaseFactory.log.info(context, "the system has %s ctrlEnclosures." % str(ctrlEncNum))

    return ctrlEncNum


# 获取阵列控制器数量
def getCtrlNum(context):
    allCtrlRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)

    ctrlNum = len(allCtrlRecs)
    BaseFactory.log.info(context, "the system has %s ctrls." % str(ctrlNum))

    return ctrlNum


# 根据端口location判断是否是SVP上的端口
def isSvpPort(portLoc):
    if portLoc.upper().startswith("SVP"):
        return True
    else:
        return False


# 获取SVP上所有指定器件类型的信息
def getSvpFruInfoListRecs(context, fruType):
    '''
   
    '''

    rest = contextUtil.getSvpRest(context)
    recs = restUtil.CommonRest.getLinkedPorts(rest, fruType)
    return recs


# 18000V3画图，端口映射表（用于前端接口的画图）
FRONT_PORT_CONFIG = {
    "213": "ETH_PORT",
    "212": "FC_PORT",
    "252": "FCoE_PORT",
    "16500": "IB_PORT",
}


# 执行DB导出命令
def clearDB(context):
    # 执行DB导出命令
    param0 = (restData.Sys.ClearExportData.EXPORT_DATA_TYPE, 3)
    paramlist = restUtil.Tlv2Rest.getParamList(param0)
    try:
        rest = contextUtil.getRest(context)
        restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.CLEAR_EXPORT_DATA, paramlist)
    except Exception as ex:
        BaseFactory.log.error(context, "clear DB failed, because:" + str(ex))


# 发送命令记录导出DB的日志。
def recordOperationLog(context, resultNum):
    # 记录导出DB导日志
    param0 = (restData.Sys.RecordOperationLog.OPER_TYPE, 1)
    param1 = (restData.Sys.RecordOperationLog.FILE_TYPE, 50)
    param3 = (restData.Sys.RecordOperationLog.RET_VALUE, resultNum)
    paramlist = restUtil.Tlv2Rest.getParamList(param0, param1, param3)
    try:
        rest = contextUtil.getRest(context)
        restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.RECORD_OPERATION_LOG, paramlist)
        BaseFactory.log.error(context, "record operation log %s" + str(resultNum))
    except Exception as ex:
        BaseFactory.log.error(context, "record operation log failed, because:" + str(ex))


def getAlarm(lang, alarmID, resDict=resource.ALARM_DICT):
    '''
    @summary: 资源国际化
    @param lang: 语言lang
    @param res: 资源
    @param args: 资源对应的参数
    @param resDict: 资源字典
    @return: 经过国际化处理后的资源
    '''
    try:
        key = "%s_%s" % (alarmID, lang)
        if not resDict.has_key(key):
            return "--"

        context = "%s(%s)" % (resDict.get(key), alarmID)

        return context

    except:
        return "--"


def getLang(context):
    """
    @summary: 获取环境语言
    """
    return context.get("lan")


def getRes(lang, res, args="", resDict=resource.RESOURCE_DICT):
    '''
    @summary: 资源国际化
    @param lang: 语言lang
    @param res: 资源
    @param args: 资源对应的参数
    @param resDict: 资源字典
    @return: 经过国际化处理后的资源
    '''
    try:
        key = "%s_%s" % (res, lang)
        if not resDict.has_key(key):
            return "--"

        context = resDict.get(key)
        if "%s" in context or "%i" in context:
            context = context % args

        return context

    except:
        return "--"


def getNodeId(ctrlId, productModel):
    '''
    @summary: 获取节点ID（如：0,1,2……）
    @param ctrlId:控制器ID（如：0A，1B）
    @param productModel:产品型号
    '''
    NODE_ID_MAP = {
        "A": 0,
        "B": 1,
        "C": 2,
        "D": 3,
    }
    CTRL_6U_PRODUCTS = [restData.Enum.ProductModeEnum.V3_6800,
                        restData.Enum.ProductModeEnum.V3_6800F,
                        restData.Enum.ProductModeEnum.V5_6800,
                        restData.Enum.ProductModeEnum.V5_6800F,
                        restData.Enum.ProductModeEnum.Dorado18000_V3
                        ]

    ctrlNum = 2  # 默认一个引擎下存在2个节点
    if productModel in CTRL_6U_PRODUCTS:
        ctrlNum = 4

    engId = int(ctrlId[0])
    ctrlName = ctrlId[-1]
    nodeId = engId * ctrlNum + NODE_ID_MAP[ctrlName]
    return nodeId


def getEnclosureModel(context, encName):
    '''
    @summary: 获取框类型
    @param encName:
    '''
    model = ""
    records = getFruListInfo(context, restData.Enum.ObjEnum.ENCLOSURE)
    for rec in records:
        name = restUtil.Tlv2Rest.getRecordValue(rec, restData.Hardware.Enclosure.NAME)
        model = restUtil.Tlv2Rest.getRecordValue(rec, restData.Hardware.Enclosure.MODEL)
        if encName == name:
            return model
    return model


def isHighEndDev(productModel):
    '''
    @summary: 判读是否为高端设备
    @param productModel: 产品型号
    @return: 
        True: 产品型号为高端
        False: 产品型号不为高端
    '''

    return productModel in config.HIGH_END_DEVS


def getEnclosureList(context):
    '''
    @summary:获取框字典列表
    @param rest: rest连接
    '''
    encList = []
    records = getFruListInfo(context, restData.Enum.ObjEnum.ENCLOSURE)
    for record in records:
        encDict = {}
        encId = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.ID)
        name = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.NAME)
        model = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.MODEL)
        location = restUtil.Tlv2Rest.getRecordValue(record, restData.Hardware.Enclosure.LOCATION)
        encDict = {
            'id': encId,
            'name': name,
            'model': model,
            'location': location,
        }
        encList.append(encDict)
    return encList


def getSysNetMode(context):
    '''
    @summary: 获取组网类型
    @param context: 上下文
    '''

    paramlist = restUtil.Tlv2Rest.getParamList()
    rest = contextUtil.getRest(context)
    records = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.READ_IP_CLUST_BASE_INFO, paramlist)
    for record in records:
        netMode = restUtil.Tlv2Rest.getRecordValue(record, restData.Upgrade.IpCluster.NETMODE)
        return netMode


def initFruInfo(context):
    '''
    @summary: 初始化fru工具需要使用的信息
    @param context: 上下文
    '''
    ENCLOSURE_LIST = "ENCLOSURE_LIST"
    SYS_NET_MODEL = "SYS_NET_MODEL"

    enclosureList = getEnclosureList(context)
    BaseFactory.log.info(context, "get enclosure list:%s" % unicode(enclosureList))
    contextUtil.setItem(context, ENCLOSURE_LIST, enclosureList)

    netModel = getSysNetMode(context)
    BaseFactory.log.info(context, "get system net model:%s" % unicode(netModel))
    contextUtil.setItem(context, SYS_NET_MODEL, netModel)

    return


def getTopSort(fruList):
    '''
    @summary: 对FRU列表进行排序
    @param fruList: FRU列表
    '''
    if None == fruList or len(fruList) == 0:
        return fruList
    condition0 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.HealthStatusEnum.FAULT)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.PublicAttributes.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.OFFLINE)
    # 设置排序条件
    conditionList = restUtil.Tlv2Rest.getConditionList(condition0, condition1)
    # 取得排序结果
    fruList = restUtil.Tlv2Rest.topSort(fruList, conditionList)
    # 返回排序结果
    return fruList


def setTableDatil(context, fruRec, uiRow={}):
    '''
    @summary: 对FRU列表进行排序
    @param context: 上下文
    @param fruRec: FRU明细内容
    @param uiRow: 行信息
    '''
    # ID
    uiRow["id"] = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.ID)
    # 名称
    uiRow["name"] = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.NAME)
    # 位置
    uiRow["location"] = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.LOCATION)
    # 父对象类型
    uiRow["parentType"] = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.PARENT_TYPE)
    # 父对象ID
    uiRow["parentID"] = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.PARENT_ID)
    # 健康状态
    healthStatus = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.HEALTH_STATUS)
    uiRow["healthStatus"] = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrHealthStatusEnum, healthStatus)
    # 运行状态
    runningStatus = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.PublicAttributes.RUNNING_STATUS)
    uiRow["runningStatus"] = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrRunningStatusEnum, runningStatus)
    # 返回设置结果
    return uiRow


def switchBusinessForHost(context, fruType, fruId):
    '''
    @summary: 主动通知多路径切换指定接口卡上的业务
    @param context: 上下文
    @param fruType: 更换的备件类型
    @param fruId: 更换的备件的ID
    @param paramList: 执行命令所需的参数列表 
    '''
    param0 = (restData.Hardware.NotifyServiceSwitchover.TYPE, fruType)
    param1 = (restData.Hardware.NotifyServiceSwitchover.ID, fruId)
    paramList = restUtil.Tlv2Rest.getParamList(param0, param1)
    BaseFactory.log.info(context, "params is " + unicode(paramList))
    try:
        rest = contextUtil.getRest(context)
        rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.NOTIFY_SERVICE_SWITCHOVER_FOR_FRU_REPLACE, paramList)
    except Exception as ex:
        BaseFactory.log.error(context, "switch business failed, because:" + str(ex))
        rec = None
    return rec


def getMagBoardUpStatues(context, magBoradLoc, objType, id, model):
    param0 = (restData.MagBoardUpgrade.LOCATION, magBoradLoc)
    param1 = (restData.MagBoardUpgrade.TYPE, objType)
    param2 = (restData.MagBoardUpgrade.ID, id)
    param3 = (restData.MagBoardUpgrade.MODEL, model)
    paramList = restUtil.Tlv2Rest.getParamList(param0, param1, param2, param3)
    BaseFactory.log.info(context, "params is " + unicode(paramList))
    try:
        rest = contextUtil.getRest(context)
        rec = restUtil.Tlv2Rest.execCmd(rest, restData.TlvCmd.QUERY_FRU_UPD_STATUS, paramList, isBatch=False)
    except Exception as ex:
        BaseFactory.log.error(context, "[Exception] : %s" % str(ex))
        rec = None
    return rec


def getBarCodeBBU(context, fruId, beginTime, timeLimit, timeOut):
    '''
    @summary: 轮询获取BBU的barcode,并刷新进度和剩余时间
    @param context: 上下文
    @param fruId: 更换的备件的ID
    @param beginTime: 轮询开始时间
    @param timeLimit: BBU条码轮询总时间
    @param timeOut: bbu健康状态检查轮询总时间
    '''
    timeTotal = timeLimit + timeOut
    postData = {}
    while True:
        if time.time() - beginTime > timeLimit:
            break

        fruRec = getFruInfo(context, restData.Enum.ObjEnum.BACKUP_POWER, fruId)
        BaseFactory.log.info(context, "getBBU in initData fruId:%s, fruRec:[%s]" % (fruId, str(fruRec)))
        postData["bbuType"] = restUtil.Tlv2Rest.getStr(context, restData.EnumStr.StrBackupPowerTypeEnum,
                                                       restUtil.Tlv2Rest.getRecordValue(fruRec,
                                                                                        restData.Hardware.BackupPower.BACKUP_POWER_TYPE),
                                                       '--')
        # 获取BBU电子标签
        elabelStr = restUtil.Tlv2Rest.getRecordValue(fruRec, restData.Hardware.BackupPower.ELABEL)
        BaseFactory.log.info(context, "elabelStr:[%s]" % elabelStr)
        elabelDict = getElabelDict(elabelStr)
        BaseFactory.log.info(context, "elabelDict:[%s]" % elabelDict)
        barCode = elabelDict.get("BarCode", "")
        BaseFactory.log.info(context, "barCode:[%s]" % barCode)

        useTime = time.time() - beginTime + timeOut
        curProgressPer = int((useTime / timeTotal) * 100)
        remainTime = int(timeTotal - useTime)
        showUI(context, curProgressPer, remainTime)

        if barCode not in ["--", ""]:
            postData['BarCode'] = barCode
            BaseFactory.log.info(context, "break barCode=[%s]" % barCode)
            break
        time.sleep(5)
    return postData


def showUI(context, curProgressPer, remainTime):
    '''
    @summary: 刷新进度和剩余时间
    @param context: 上下文
    @param curProgressPer: 进度百分比数值
    @param remainTime: 剩余时间
    '''
    if curProgressPer < 0 or curProgressPer > 100:
        curProgressPer = 100
    if remainTime < 0:
        remainTime = 0
    context["curProgressPer"] = curProgressPer
    context["remainTime"] = remainTime
    return


def getInternalProductModel(context):
    """查询内部产品型号

    :param context:
    :return:
    """
    devObj = contextUtil.getDevObj(context)
    pdtVersion = devObj.get("version")
    pdtType = devObj.get("type")
    BaseFactory.log.error(context, "Product version:{}, and product model:{}".format(pdtVersion, pdtType))
    if not ((pdtType and 'Dorado' in pdtType and pdtVersion > 'V300R002C10')
            or (pdtType and 'V5' in pdtType and pdtVersion >= 'V500R007C60')):  # 5X10 V5
        BaseFactory.log.error(context, "Internal productModel not supported.")
        return ''

    params = {}
    rest = contextUtil.getRest(context)
    uriParamDict = restUtil.CommonRest.getUriParamDict(restData.RestCfg.SpecialUri.INTERNAL_DEVICE_INFO)
    record = restUtil.CommonRest.execCmd(rest, uriParamDict, params, restData.RestCfg.RestMethod.GET)
    if not record:
        BaseFactory.log.error(context, "get internal productModel failed.")
        # 如果取得为空的话，直接返回None
        return None
    else:
        internalProductModel = restUtil.CommonRest.getRecordValue(restUtil.CommonRest.getData(record),
                                                                  restData.InternalDeviceInfo.INTERNAL_PRODUCT_MODEL)
        BaseFactory.log.info(context, "Product internal model:{}".format(internalProductModel))
        return internalProductModel


'''
公用方法：查询VRC版本
'''


def getProductVersion(context):
    '''
    @param context: 上下文
    '''
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    logger = contextUtil.getLogger(context)
    _ ,productVersion, _ = cliUtil.getDevVersion(cli, lang, logger)
    return productVersion


'''
公用方法：删除DB备份
'''


def deleteDB(context):
    try:
        if context["dbFilePath"]:
            dbFilePath = context["dbFilePath"]
            if os.path.isfile(dbFilePath):
                os.remove(dbFilePath)
                BaseFactory.log.info(context, "Backup DB data has been deleted.")
    except Exception as ex:
        BaseFactory.log.error(context, "[Exception] : %s" % str(ex))
    finally:
        return


def delete_license(context):
    """
    删除license文件备份
    """
    try:
        if context.get("licenseFilePath"):
            license_file_path = context.get("licenseFilePath")
            if os.path.isfile(license_file_path):
                os.remove(license_file_path)
                BaseFactory.log.info(context,
                                     "Backup License data has been deleted.")
    except Exception as ex:
        BaseFactory.log.error(context, "[Exception] : %s" % str(ex))
    finally:
        return


def isCtrlModule(context, parentId):
    """检查是否为控制器模块

    Args:
        context(dict): python上下文
        parentId(str): 父ID

    Returns:
        bool: True-控制器模块 False-非控制器模块
    """
    isParentBeEnc = False
    allCntrRecs = getFruListInfo(context, restData.Enum.ObjEnum.CONTROLLER)
    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.Controller.ID,
                                                restData.Enum.ConditionTypeEnum.EQ, parentId)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    tmpRecs = restUtil.Tlv2Rest.filter(allCntrRecs, conditionList)
    if len(tmpRecs) > 0: #如果parentID与某一控制器的ID相同，则父类型为控制器
        isParentBeEnc = True
    BaseFactory.log.info(context, "fru is ctrl mode: %s" % str(isParentBeEnc))
    return isParentBeEnc


def getAllPorts(context, port_types=[]):
    """查询所有端口信息

    :param context:
    :param port_types:
    :return:
    """
    if not port_types:
        port_types = [restData.Enum.ObjEnum.ETH_PORT,
                      restData.Enum.ObjEnum.FC_PORT,
                      restData.Enum.ObjEnum.SAS_PORT]
    all_ports = []
    for port_type in port_types:
        ports = get_fru_list_info_with_not_support_type(context, port_type)
        all_ports.extend(ports)
    return all_ports


def getRedPowerForHighEnd(powerLoc):
    """4U4C独立机头获取冗余电源位置

    :param powerLoc: 电源location
    :return:
    """
    return powerLoc.split(".")[0] + "." + conf.RED_POWER_HIGH_END.get(powerLoc.split(".")[1], "")

def getInterfIdsByLogicType(context, logicType):
    """根据端口逻辑类型获取接口卡ID

    :param context:
    :param logicType:
    :return:
    """
    fruList = getFruListInfo(context, restData.Enum.ObjEnum.INTF_MODULE)
    intfIdList = getIntfIdListByLogicType(context, logicType)
    # 判断下电的卡是否为前后端接口卡
    intf_id_list = get_intf_id_on_powered_off(context, fruList, logicType)
    intfIdList.extend(intf_id_list)

    # 判断是否为容器安全卡，安全卡没有端口。
    intfIdList.extend(get_security_model(fruList, logicType))

    condition1 = restUtil.Tlv2Rest.getCondition(restData.Hardware.IntfModule.ID,
                                                restData.Enum.ConditionTypeEnum.EQOR, intfIdList)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    return restUtil.Tlv2Rest.filter(fruList, conditionList)


def get_intf_id_on_powered_off(context, fru_list, logic_type):
    """
    获取已下电的接口卡
    :param context:
    :param fru_list:
    :return:
    """
    logger = contextUtil.getLogger(context)
    intf_id_list = []
    for fru_record in fru_list:
        running_status = restUtil.Tlv2Rest.getRecordValue(fru_record,
                                                          restData.PublicAttributes.RUNNING_STATUS)
        service_secne = restUtil.Tlv2Rest.getRecordValue(fru_record,
                                                         restData.Hardware.IntfModule.DMI_INTF_SERVICE_SCENE_E)
        intf_id = restUtil.Tlv2Rest.getRecordValue(fru_record,
                                                   restData.PublicAttributes.ID)
        model = restUtil.Tlv2Rest.getRecordValue(
            fru_record, restData.Hardware.IntfModule.MODEL
        )
        if model in restData.CONTAINER_CARD_MODE:
            intf_id_list.append(intf_id)
            continue
        if running_status == 2:
            continue
        if intf_id in intf_id_list:
            continue
        if not service_secne:

            if (logic_type == 0 and model in restData.frontend_intf_models) or \
                    (logic_type == 1 and model in restData.backend_intf_models):
                intf_id_list.append(intf_id)
        # service_secne:1,前端卡；2，后端卡；5, 前端NOF； 6，前端容器卡；7，后端容器卡; 8，前端NOF AA； 9，前端OVS
        elif (logic_type == 0 and service_secne in [1, 5, 6, 8, 9]) or \
                (logic_type == 1 and service_secne in [2, 7]):
            intf_id_list.append(intf_id)
    return intf_id_list


def get_security_model(fru_list, logic_type):
    """
    获取勒索软件检测模块的接口卡，检测模块没有端口
    :param fru_list:备件列表
    :param logic_type:逻辑类型
    :return:
    """
    intf_id_list = []
    if logic_type != restData.Enum.PortLogicTypeEnum.BACK_CONTAINER_CARD:
        return []
    for fru_record in fru_list:
        intf_id = restUtil.Tlv2Rest.getRecordValue(fru_record, restData.PublicAttributes.ID)
        model = restUtil.Tlv2Rest.getRecordValue(fru_record, restData.Hardware.IntfModule.MODEL)
        if model in restData.SECURITY_CARD_MODE:
            intf_id_list.append(intf_id)
    return intf_id_list


def getIntfIdListByLogicType(context, logicType):
    """根据logic type获取前后端接口卡

    :param context:
    :param logicType:
    :return:
    """
    intfLinkPortList = getPortsByLogicType(context, logicType)

    # 获取接口卡ID列表并返回
    intfIdList = []
    for portRec in intfLinkPortList:
        # 获取端口所属交换机ID，若ID有效，则记录
        intfId = restUtil.Tlv2Rest.getRecordValue(portRec, restData.Hardware.EthPort.PARENT_ID)
        if intfId not in intfIdList:
            intfIdList.append(intfId)

    BaseFactory.log.info(context, "the intf id list is %s" % intfIdList)
    return intfIdList

def getPortsByLogicType(context, logicType):
    """根据逻辑类型获取端口列表

    :param context:
    :param logicType:
    :return:
    """
    allPort = []
    for type in PORT_TYPES:
        portList = get_fru_list_info_with_not_support_type(context, type)
        condition0 = restUtil.Tlv2Rest.getCondition(restData.Hardware.EthPort.LOGIC_TYPE,
                                                    restData.Enum.ConditionTypeEnum.EQ,
                                                    logicType)
        conditions = restUtil.Tlv2Rest.getConditionList(condition0)
        portList = restUtil.Tlv2Rest.filter(portList, conditions)
        allPort.extend(portList)
    return allPort

def checkNormalBBUByCtrlId(context, ctrlId, langKey, errAgrs, suggestArgs):
    """根据控制ID获取归属该控制器的正常BBU

    :param context: 上下文
    :param ctrlId: 控制器ID
    :param langKey: 错误提示及修复建议key值
    :param errAgrs: 错误提示参数
    :param suggestArgs: 修复建议参数
    :return:
    """
    allPower = getFruListInfo(context, restData.Enum.ObjEnum.BACKUP_POWER)
    healthCond = restUtil.Tlv2Rest.getCondition(restData.Hardware.BackupPower.HEALTH_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQOR,
                                                [restData.Enum.HealthStatusEnum.NORMAL,
                                                 restData.Enum.HealthStatusEnum.PRE_FAIL])
    runStatCond = restUtil.Tlv2Rest.getCondition(restData.Hardware.BackupPower.RUNNING_STATUS,
                                                restData.Enum.ConditionTypeEnum.EQ,
                                                restData.Enum.RunningStatusEnum.ONLINE)
    ctrlCond = restUtil.Tlv2Rest.getCondition(restData.Hardware.Power.PARENT_ID,
                                                restData.Enum.ConditionTypeEnum.EQ, ctrlId)
    conditions = restUtil.Tlv2Rest.getConditionList(healthCond, runStatCond, ctrlCond)
    redundantPower = restUtil.Tlv2Rest.filter(allPower, conditions)
    BaseFactory.log.info(context, "bbu info: %s" % str(redundantPower))
    if not redundantPower:
        BaseFactory.result.setResultFailByKey(context, langKey, errAgrs, suggestArgs)
        return

    BaseFactory.result.setResultPass(context)
    return


def get_interface_modules_by_models(context, models):
    """根据卡model获取接口卡信息

    :param context:
    :param models:
    :return:
    """
    fru_list = getFruListInfo(context, restData.Enum.ObjEnum.INTF_MODULE)
    condition1 = restUtil.Tlv2Rest.getCondition(
        restData.Hardware.IntfModule.MODEL,
        restData.Enum.ConditionTypeEnum.EQOR,
        models)
    conditionList = restUtil.Tlv2Rest.getConditionList(condition1)
    return restUtil.Tlv2Rest.filter(fru_list, conditionList)


def get_alarms_from_rest_record(alarm_records):
    """从rest返回数据中获取格式化的告警列表

    :param alarm_records: rest返回数据
    :return:
    """
    if not alarm_records:
        return []

    alarms = []
    for record in alarm_records:
        level = CommonRest.getRecordValue(
            record, restData.RestCfg.Alarm.LEVEL)
        if level in [AlarmLevel.major, AlarmLevel.critical]:
            alarm_id = CommonRest.getRecordValue(
                record, restData.RestCfg.Alarm.EVENTID)
            alarm_parmas = CommonRest.getRecordValue(
                record, restData.RestCfg.Alarm.EVENTPARAM)
            alarm = (alarm_id, alarm_parmas)
            alarms.append(alarm)
    return alarms


def alarm_is_about_location(event_params, location):
    """根据告警参数，判断告警是否和当前location的备件相关
    适用于告警参数中一定包含有location的情况

    :param event_params: 告警参数字符串
    例如64438075504告警event_params=0,CTE0,A,B
    例如64438206532告警event_params=0,CTE0,RDMA,IOM.H9 高端/共享 location CTE0.IOM.H9
    例如64438206532告警event_params=0,CTE0,FC,A.IOM0 中低端  location CTE0.A.IOM0
    :param location: fru的location，如CTE0.A
    :return: True 相关，False 不相关

    """
    location_list = location.split(".", 1)
    param_list = event_params.split(",")
    for pos in location_list:
        if not pos:
            continue
        if pos not in param_list:
            return False
    return True


def exist_not_allowed_fru_alarm(context, location):
    """判断是否有和当前备件相关的不允许更换的告警
    如兼用性告警，备件类型不一致告警等

    :param context: 上下文
    :param location: 备件位置
    :return: True:存在，False:不存在或不涉及
    """
    try:
        BaseFactory.log.info(context, "begin check exist_compatible_alarm_id")
        rest = contextUtil.getRest(context)
        alarm_records = CommonRest.get_alarms_by_period(rest)
        alarm_id_params = get_alarms_from_rest_record(alarm_records)
        BaseFactory.log.info(
            context, "alarm_id_params: %s" % str(alarm_id_params))
        for alarm_id, alarm_params in alarm_id_params:
            if alarm_id not in conf.FRU_NOT_ALLOW_ALARM_LIST:
                continue

            if not alarm_is_about_location(alarm_params, location):
                continue

            return True

        return False
    except (Exception, jException) as e:
        # 可能备件控制器和连接的控制器是同一个，所以发生这种异常可能属于正常情况。
        BaseFactory.log.error(context, "[Exception] :{0}".format(str(e)))
        return False


def exist_alarm_not_allowed_controller_in(context, location):
    """
    判断是否有和当前待更换控制器的告警

    :param context: 上下文
    :param location: 备件位置
    :return: True:存在，False:不存在或不涉及
    """
    try:
        cli = contextUtil.getCli(context)
        lang = contextUtil.getLang(context)
        BaseFactory.log.info(context, "begin check exist_specified_alarms.")
        rest = contextUtil.getRest(context)
        begin_time_stamp = get_timestamp(contextUtil.getItem(context, 'start_sys_date'))
        check_time_stamp = get_timestamp(cliUtil.getSystemDate(cli, lang)[0])
        alarm_records = CommonRest.get_alarms_by_period(rest, begin_time_stamp, check_time_stamp)
        alarm_id_params = get_alarms_from_rest_record(alarm_records)
        BaseFactory.log.info(
            context, "controller alarm_id_params: %s" % str(alarm_id_params))
        alarm_ids = []
        for alarm_id, alarm_params in alarm_id_params:
            if alarm_id not in conf.CONTROLLER_NOT_ALLOW_ALARM_LIST:
                continue
            if not alarm_is_about_location(alarm_params, location):
                continue
            alarm_ids.append(alarm_id)
        return alarm_ids
    except (Exception, jException) as e:
        # 可能备件控制器和连接的控制器是同一个，所以发生这种异常可能属于正常情况。
        BaseFactory.log.error(context, "[Exception] :{0}".format(str(e)))
        return []


def get_timestamp(time_str):
    """
    将时间字符串转换为时间戳

    :param time_str: 时间格式为 2023-9-16/12:00:00
    :return: 对应时间戳
    """
    time_array = time.strptime(time_str, "%Y-%m-%d/%H:%M:%S")
    return int(time.mktime(time_array))


def is_exist_controller_alarm(context, logger, location):
    # 检查控制器上电过程中的特定告警
    alarm_ids = exist_alarm_not_allowed_controller_in(context, location)
    if not alarm_ids:
        return False
    # python2 大整数转换为十六进制会加L后缀；hex()方法默认带0x
    hex_alarm_ids = [("0x" + hex(int(hex_alarm_id))[2:].upper()).rstrip('L') for hex_alarm_id in alarm_ids]
    logger.logInfo("the alarm_ids is {}".format(hex_alarm_ids))
    BaseFactory.result.setResultFailByKey(
        context, LangKey.CONTROLLER_NOT_SUPPORT,
        errParamTuple=', '.join(hex_alarm_ids))
    return True


def get_start_sys_date(context, lang, logger):
    try:
        cli = contextUtil.getCli(context)
        # 记录控制器离线前一刻的存储时间，来作为控制器上电过程中查询指定告警的起始时间
        start_sys_date, _ = cliUtil.getSystemDate(cli, lang)
        contextUtil.setItem(context, "start_sys_date", start_sys_date)
        logger.logInfo(
            "The start time of querying specified alarms during controller power-on:{}".format(str(start_sys_date)))
    except Exception as ex:
        logger.logInfo("Exception when getting start sys date:{}".format(ex))


def get_parent_info(context, fru_type, fru_id):
    """获取特定备件的父器件信息

    :param context: 环境变量
    :param fru_type: 备件类型
    :param fru_id: 备件id
    :return:
    """
    fru_info = getFruInfo(context, fru_type, fru_id)
    parent_type = restUtil.Tlv2Rest.getRecordValue(
        fru_info, restData.PublicAttributes.PARENT_TYPE)
    parent_id = restUtil.Tlv2Rest.getRecordValue(
        fru_info, restData.PublicAttributes.PARENT_ID)
    parent_info = getFruInfo(context, parent_type, parent_id)
    return parent_info
