# -*- coding: UTF-8 -*-
from cbb.business.operate.expansion import common
from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common.checkItem import checkBackPortInfo
from cbb.business.operate.fru.common.checkItem import checkSASCableAlarm
from cbb.common.query.hardware.enclosure import get_all_disk_enc_list
from cbb.frame.context import contextUtil

# 后端线缆告警列表
SAS_LINK_ALARMS = [
    "0x000F00CE002A",  # 单链路
    "0xF0006000F",  # 级联端口线缆连接错误
    "0xF00060010",  # SAS线缆连接错误告警
    "0xF00D10046",  # SAS线缆连接错误告警
    "0xF00060011",  # SAS端口连接不正确
    "0xF00060016"  # 线缆规格与端口不兼容
]

PRE_CHECK_SECS = 100
POST_CHECK_SECS = 370

INTF_MODULE = "INTF_MODULE"
ALL_NORMAL_BACK_PORT = "allNormalBackPort"


def execute(context, preCheck=True, **kwagrs):
    """后端链路检查
    检查端口一致性和告警

    :param context:
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 如果没有硬盘框，直接通过
        if not get_all_disk_enc_list(context):
            contextUtil.handleSuccess(context)
            return

        if not preCheck:
            total_time = POST_CHECK_SECS
        else:
            total_time = PRE_CHECK_SECS
        common.threadUpProcess(context, total_time, 1)

        # 检查端口一致性
        checkBackPortInfo.execute(context, preCheck)
        if not context["succ"]:
            logger.logNoPass("check port consistency failed")
            return

        row_key = kwagrs.get("rowKey") if kwagrs else ""
        fruType = kwagrs.get("fruType") if kwagrs else ""
        if check_no_linked_port_on_current_intf(preCheck, context, row_key, logger, fruType):
            return
        # 检查告警
        checkItem = checkSASCableAlarm.SasCableAlarm(
            context, rowKey=row_key, fruType=fruType,
            mask=preCheck, preCheck=preCheck)
        resultInfo = checkItem.execute()
        BaseFactory.log.info(context, "check sas alarm result:{}".format(resultInfo))
        if resultInfo:
            alarmIds = [trans_alarm_id_to_hex(item.get("alarmId")) for item in resultInfo]
            errMsg, suggestion = (
                common.getMsg(
                    contextUtil.getLang(context), "sas.cable.error.V6",
                    suggestionArgs=" ".join(alarmIds))
                if preCheck
                else common.getMsg(contextUtil.getLang(context), "sas.cable.error")
            )
            contextUtil.setResult(context, False, errMsg=errMsg,
                                  suggestion=suggestion)
            return
        contextUtil.handleSuccess(context)
        logger.logPass()
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return
    finally:
        context["remindInfo"] = ""
        common.finishProcess(context)


def check_no_linked_port_on_current_intf(pre_check, context, row_key, logger, fru_type):
    """
    如果无点亮的端口则更换通过
    :param pre_check: 是否更换前检查
    :param context: 上下文
    :param row_key: 备件key
    :param logger: 日志
    :param fru_type: 备件类型
    :return: True 无点亮端口， False 有点亮端口或非接口卡，需要继续检查
    """
    if not pre_check or fru_type != INTF_MODULE:
        return False
    sel_row = BaseFactory.json.toDict(context.get(row_key))
    linked_sas_ports_before = BaseFactory.persist.getModule(
        context, ALL_NORMAL_BACK_PORT
    )
    current_intf_link_sas_port = checkBackPortInfo.get_current_intf_linked_back_port(
        linked_sas_ports_before, sel_row
    )
    logger.logInfo(
        "current intf linked sas port:{}".format(current_intf_link_sas_port))
    # 如果无点亮的端口则更换通过
    if not current_intf_link_sas_port:
        contextUtil.handleSuccess(context)
        logger.logPass()
        return True
    return False


def trans_alarm_id_to_hex(alarm_id):
    """
    将告警ID从十进制转为16进制，并去掉末位的L（与DM保持一致）
    :param alarm_id: 告警ID
    :return: hex_alarm_id
    """
    hex_id = hex(int(alarm_id)).upper() if alarm_id.isdigit() else alarm_id
    return hex_id[:-1] if hex_id.endswith("L") else hex_id
