# -*- coding: UTF-8 -*-
import os
import time
from java.lang import Exception as JException
from string import find

import config
import logger
import resource
import uploadUtil
from cbb.frame.base.exception import UnCheckException
from cbb.frame.base.regex import Regex
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil

FILE_SUFFIX = "."


def getBaseName(file_path):
    """返回文件路径的文件名，不包含后缀

    :param file_path: 文件路径
    :return:
    """
    file_name = os.path.basename(file_path)
    if FILE_SUFFIX in file_name:
        dot_index = file_name.rindex(FILE_SUFFIX)
        return file_name[0:dot_index]
    else:
        return file_name


def getLogger(loggerInstance, pyFilePath):
    """获取日志类

    :param loggerInstance: logger实例
    :param pyFilePath: py文件路径
    :return:
    """
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)


def getUpgEvaluationRs(flag, CliRet, errMsg):
    """将巡检的结果转为升级评估的结果

    :param flag:结果标记
    :param CliRet:回显
    :param errMsg:错误消息
    :return:
    """
    return cliUtil.RESULT_DICT.get(flag, "2"), CliRet, errMsg


def getLang(py_java_env):
    """从上下文中获取lang

    :param py_java_env: 上下文对象
    :return:
    """
    return py_java_env.get("lang")


def getRes(lang, res, args="", resDict=resource.RESOURCE_DICT):
    """资源国际化

    :param lang: 语言lang
    :param res: 资源
    :param args: 资源对应的参数
    :param resDict: 资源字典
    :return: 经过国际化处理后的资源
    """
    # noinspection PyBroadException
    try:
        key = "%s_%s" % (res, lang)
        if key not in resDict:
            key = "%s_%s" % (res, lang)
            if key not in resDict:
                return "--", ""

        context = resDict.get(key)
        if "%s" in context or "%i" in context:
            context = context % args

        return context

    except Exception:
        return "--", ""


def getMsg(lang, msg, args=""):
    """消息国际化

    :param lang: 语言lang
    :param msg: 消息
    :param args:消息参数
    :return: 经过国际化处理后的消息
    """
    return cliUtil.getMsg(lang, msg, args, resource=resource.MESSAGES_DICT)


def safeSleep(seconds):
    """安全睡眠时间

    :param seconds:seconds为睡眠时间，单位：秒；数据类型：整数或小数
    :return:
    """
    # noinspection PyBroadException
    try:
        if type(seconds) not in [int, long, float] or seconds <= 0:
            return

        startTime = time.clock()
        while True:
            if (time.clock() - startTime) >= seconds:
                return

            # 睡眠一下，避免长时间占用cpu，该时间设置过长会影响睡眠时间精度
            # noinspection PyBroadException
            try:
                time.sleep(0.1)
            except Exception:
                pass

            continue
    except Exception:
        return


def getUpgStatus(status):
    """获取框架可识别的升级状态

    :param status: 为config.UPD_STATUS
    :return:
    """
    if status == config.UPD_STATUS.SUCCESS:
        return True
    return False


def convertDiskToDictList(diskList):
    """

    :param diskList:
    :return:
    """
    resultList = []
    for disk in diskList:
        diskId = disk.getDiskLocation()
        diskFw = disk.getFwVersion()
        diskRole = disk.getRole()
        diskDomainId = disk.getDiskDomainId()
        diskSN = disk.getSerialNumber()
        resultList.append({"ID":diskId,
                         "Firmware Version":diskFw,
                         "Role":diskRole,
                         "Disk Domain ID": diskDomainId,
                        "Serial Number": diskSN})
    return resultList


def getSelectedDisks(context):
    """获取选择的硬盘

    :param context:上下文
    :return:
    """
    selectedDisks = contextUtil.getSelectedDisks(context)
    selectedDisks = convertDiskToDictList(selectedDisks)
    return selectedDisks


def getUpgDisks(dataDict):
    """获取待升级的硬盘

    :param dataDict: 上下文字典
    :return:
    """
    dev = dataDict.get("dev")
    cli = contextUtil.getCli(dataDict)
    lang = dataDict.get("language")
    logger = dataDict.get("logger")
    selectDiskType = str(dev.getUpgradeSetInfo().getSelectDiskType())
    logger.info("current select type:%s" % unicode(selectDiskType))
    if selectDiskType == config.UPD_SELE_DISK_TYPE.CUSTOM_DISK:
        # 自定义选择硬盘直接从框架获取
        allDisks = convertDiskToDictList(dev.getUpgradeSetInfo().getSelectedDisks())
        dev.setDiskNum(len(allDisks))
        return allDisks
    else:
        # 系统全部硬盘则查询系统全部硬盘
        queryRet = cliUtil.queryDiskInfo(cli, lang, logger)
        if not queryRet[0]:
            logger.info("get upgrade disk fail:%s" % unicode(queryRet))
            raise UnCheckException("Failed to Get upgrade disk.")

        allDisks = queryRet[1]
        dev.setDiskNum(len(allDisks))
        return allDisks


def isMemberDisk(role):
    """判断是否为成员盘

    :param role: 硬盘role
    :return:
    """
    if "Member Disk" == role:
        return True
    return False


def formatDiskLocation(diskLocation):
    """将硬盘位置信息格式化为事件可识别的字符串，如DAE000.0转化为 enclosure CTE0, slot ID 0

    :param diskLocation:  硬盘位置信息
    :return:
    """
    pattern = "^[A-Z]{3}[0-9]{1,3}.[0-9]+"
    if not Regex.search(pattern, diskLocation):
        return ""

    eventflag = "enclosure %s, slot ID %s"
    diskLocList = diskLocation.split(".")
    return eventflag % (diskLocList[0], diskLocList[1])


def getStartAndCompleteEvent(cli, lang, eventList, diskLocation):
    """获取指定盘所在Disk Domain重构的开始时间和完成时间

    :param cli: cli对象
    :param lang: 语言
    :param eventList: 事件列表
    :param diskLocation: 硬盘位置
    :return:
    """
    POOL_STARTED_EVENT = "The Disk Domain Reconstruction Started"
    POOL_COMPLETED_EVENT = "The Disk Domain Reconstruction Completed"

    startEvents = []
    completeEvents = []
    diskFlag = formatDiskLocation(diskLocation)
    if diskFlag == "":
        return startEvents, completeEvents

    for event in eventList:
        name = event.get("Name")
        sequence = event.get("Sequence")
        if POOL_STARTED_EVENT in name:
            eventDetail = cliUtil.getEventDetail(cli, lang, sequence)
            detail = eventDetail.get("Detail", "")
            if diskFlag in detail:
                startEvents.append(event)
            continue

        if POOL_COMPLETED_EVENT in name:
            eventDetail = cliUtil.getEventDetail(cli, lang, sequence)
            detail = eventDetail.get("Detail", "")
            if diskFlag in detail:
                completeEvents.append(event)
            continue
    return startEvents, completeEvents


def checkPoolEvent(dataDict, objType, sysTime, diskLocation, timeOut):
    """检查pool重构事件

    :param dataDict: 上下文
    :param objType: 对象类型
    :param sysTime: 发生时间
    :param diskLocation: 硬盘位置
    :param timeOut: 超时时间
    :return:
    """
    cliRet = ""
    lang = dataDict.get("language")
    logger = dataDict.get("logger")

    cmd = "show event object_type=%s from_time=%s" % (str(objType), str(sysTime))
    logger.info("execute cmd[%s]" % cmd)
    startTime = time.clock()
    while (time.clock() - startTime) < timeOut:
        cliRet = ""
        cli = contextUtil.getCli(dataDict)
        ret = cliUtil.excuteCmdInCliMode(cli, cmd, False, lang)
        if not cliUtil.hasCliExecPrivilege(ret[1]):
            cmd = "show event from_time=%s" % str(sysTime)
            continue

        if not ret[0]:
            safeSleep(30)
            continue

        cliRet = ret[1]
        eventList = cliUtil.getHorizontalCliRet(cliRet)
        (startEvents, completeEvents) = getStartAndCompleteEvent(cli, lang, eventList, diskLocation)
        logger.info("start reconstruction event[%s]" % str(startEvents))
        logger.info("complete reconstruction event[%s]" % str(completeEvents))
        for startE in startEvents:
            startESeq = startE.get("Occurred On")
            for compE in completeEvents:
                compESeq = compE.get("Occurred On")
                if startESeq <= compESeq:
                    return True, "", cliRet

        safeSleep(10)
    errMsg = getMsg(lang, "domain.reconstruction.fail", diskLocation)
    return False, errMsg, cliRet


def isSupportDiskUpg(cli, lang):
    """检查当前版本是否支持独立包硬盘升级

    :param cli: cli对象
    :param lang: 语言
    :return:
    """
    cmd = "hdm show diagrecord sdid=48"
    ret = cliUtil.executeCmdInDebugMode(cli, cmd, True, lang)
    if not ret[0]:
        return ret

    if "FW upgrade supported parallel" in ret[1]:
        return True, True

    if "FW upgrade supported" in ret[1]:
        return True, False

    errMsg = getMsg(lang, "not.support.upg.disk")
    return False, False


def checkUpgradeResult(cliRet, lang, logger):
    """根据cli回显判断硬盘升级是否成功

    :param cliRet: cli回显
    :param lang: 语言
    :param logger: 日志对象
    :return:
    """
    errMsg = ""
    logger.info("===begin checkUpgradeResult===")
    for updSucFlag in config.DISK_UPGRADE_SUCCESS_RET_LIST:
        if updSucFlag in cliRet:
            return config.UPD_STATUS.SUCCESS, errMsg

    for noneedUpgFlag in config.DISK_UPGRADE_NONEED_RET_LIST:
        if noneedUpgFlag in cliRet:
            return config.UPD_STATUS.NONEED, errMsg
    # 获取硬盘固件包异常
    for noFileFlag in config.DISK_UPGRADE_NOFILE_RET_LIST:
        if noFileFlag in cliRet:
            return config.UPD_STATUS.FAIL, getMsg(lang, "diskUpgrade.noFile")

    return config.UPD_STATUS.FAIL, getMsg(lang, "diskUpgrade.fail")


def RetryDiskUpgrade(timeout=20):
    """硬盘升级重试

    :param timeout: 超时时间
    :return:
    """

    def wrapper(func):
        def needRetryUpgrade(flag, cliRet):
            if flag == config.UPD_STATUS.Retry:
                return True
            if flag == config.UPD_STATUS.FAIL:
                (checkFlag, checkItems) = isPreCheckFail(cliRet)
                if checkFlag:
                    checkItem = getPreCheckFailedItem(checkItems)
                    if checkItem.get("Check Item Name") in config.NEED_RETRY_CHECK_ITEMS:
                        return True
            return False

        def inner(*params):
            flag, errMsg, cliRet = config.UPD_STATUS.ERROR, "", ""
            startTime = time.clock()
            oldParams = params
            while (time.clock() - startTime) < timeout * 60:
                # 重新连接
                contextUtil.getCli(params[0])

                (flag, errMsg, cliRet) = func(*params)
                if not needRetryUpgrade(flag, cliRet):
                    break
                safeSleep(60)
                # 重置参数，增加是否导包的参数，重试时需要重新上传固件包，只限独立导包场景
                params = oldParams
                params = params + (True,)
            if flag == config.UPD_STATUS.Retry:
                flag = config.UPD_STATUS.ERROR
            return flag, errMsg, cliRet

        return inner

    return wrapper


@RetryDiskUpgrade(timeout=20)
def upgradeDisk(dataDict, diskID, importPkg=False):
    """执行硬盘升级

    :param dataDict: 上下文字典
    :param diskID: 硬盘ID，格式为DAE000.0
    :param importPkg: 是否导包
    :return:
    """
    cliRet = ""

    logger = dataDict.get("logger")
    lang = dataDict.get("language")

    cmd = "upgrade disk upgrade_mode=OnLine disk_id=%s" % diskID
    try:
        importPkgMode = contextUtil.getImportPkgMode(dataDict)
        if importPkg and importPkgMode == config.IMPORT_PKG_MODE.MIX_PKG:
            uploadRet = uploadUtil.uploadUpgradePkg(dataDict)
            if not uploadRet[0]:
                logger.error("[upgradeDisk]upload fw package failed for disk(%s)." % diskID)
                return config.UPD_STATUS.FAIL, uploadRet[1], ""

        cli = contextUtil.getCli(dataDict)
        logger.info("[upgradeDisk]start to execute:%s" % str(cmd))
        execRets = cliUtil.execCmdInDeveloperModePrompt(cli, cmd, True, lang)
        cliRet = execRets[1]
        if not execRets[0]:
            logger.info("[upgradeDisk]error:%s" % str(execRets[2]))
            return config.UPD_STATUS.FAIL, getMsg(lang, "diskUpgrade.cmd.exce.failed"), cliRet

        if config.TIME_OUT in cliRet:
            logger.info("[upgradeDisk]execute cmd time out:%s" % str(cliRet))
            return config.UPD_STATUS.ERROR, getMsg(lang, "diskUpgrade.cmd.exce.failed"), ""

        logger.info("[upgradeDisk]cmd return:%s" % str(cliRet))
        checkRet, msg = checkUpgradeResult(cliRet, lang, logger)
        return checkRet, msg, cliRet

    except Exception as e:
        logger.error("[upgradeDisk]execute cmd exception:%s" % unicode(e))
        return config.UPD_STATUS.ERROR, getMsg(lang, "diskUpgrade.cmd.exce.failed"), cliRet
    except JException as e:
        logger.error("[upgradeDisk]execute cmd exception:%s" % unicode(e))
        return config.UPD_STATUS.Retry, getMsg(lang, "diskUpgrade.cmd.exce.failed"), cliRet


def checkDiskFw(dataDict, oldDiskInfo, timeOut=30):
    """检查硬盘FW版本有无变化

    :param dataDict: 上下文
    :param oldDiskInfo: 旧硬盘信息
    :param timeOut: 超时时间
    :return:
    """
    newFw = "--"
    lang = dataDict.get("language")
    diskId = oldDiskInfo.get("ID")
    oldFw = oldDiskInfo.get("Firmware Version")
    startTime = time.clock()
    while (time.clock() - startTime) < timeOut:
        cli = contextUtil.getCli(dataDict)
        newDiskInfo = cliUtil.getDiskById(cli, lang, diskId)
        if len(newDiskInfo) == 0:
            safeSleep(30)
            continue

        newFw = newDiskInfo.get("Firmware Version")
        if newFw and newFw != oldFw:
            return True, newFw
        safeSleep(30)
    return False, newFw


@RetryDiskUpgrade(timeout=20)
def upgradeDiskWithPkg(dataDict, diskID, importPkg=False):
    """执行硬盘升级

    :param dataDict: 上下文字典
    :param diskID: 硬盘ID，格式为DAE000.0
    :param importPkg: 是否导包，该参数暂时未使用，只是为了RetryDiskUpgrade方法的方便
    :return:
    """
    lang = dataDict.get("language")
    logger = dataDict.get("logger")
    cli = contextUtil.getCli(dataDict)
    dev = dataDict.get("dev")

    cliRet = ""
    cmd = ""
    try:
        passwd = specialCharTrans(dev.getDevNode().getLoginUser().getPassword())
        path = specialCharTrans(dataDict.get("path"))
        user = specialCharTrans(dev.getDevNode().getLoginUser().getUserName())
        cmd = "upgrade disk ip=%s user=%s password=%s path=%s upgrade_mode=OnLine disk_id=%s" % (
            dev.getDevNode().getIp(), user, passwd, path, diskID)
        logger.info("[upgradeDisk]start to execute upgrade disk[%s] with pkg." % str(diskID))
        execRets = cliUtil.execCmdInDeveloperModePrompt(cli, cmd, False, lang)
        cliRet = cliUtil.anonymize_ret(execRets[1], passwd, "****")
        if not execRets[0]:
            logger.info("[upgradeDisk]error:%s" % str(execRets[2]))
            return config.UPD_STATUS.FAIL, getMsg(lang, "diskUpgrade.cmd.exce.failed"), cliRet

        if config.TIME_OUT in cliRet:
            logger.info("[upgradeDisk]execute cmd time out:%s" % str(cliRet))
            return config.UPD_STATUS.ERROR, getMsg(lang, "diskUpgrade.cmd.exce.failed"), ""

        logger.info("[upgradeDisk]cmd return:%s" % str(cliRet))
        checkRet, msg = checkUpgradeResult(cliRet, lang, logger)
        return checkRet, msg, cliRet

    except Exception as e:
        logger.error("[upgradeDisk]execute cmd exception:%s" % unicode(e))
        return config.UPD_STATUS.ERROR, getMsg(lang, "diskUpgrade.cmd.exce.failed"), cliRet
    except JException, e:
        logger.error("[upgradeDisk]execute cmd exception:%s" % unicode(e))
        return config.UPD_STATUS.Retry, getMsg(lang, "diskUpgrade.cmd.exce.failed"), cliRet
    finally:
        del cmd


def specialCharTrans(oldString):
    """特殊字符串处理

    :param oldString: 原字符串
    :return:
    """
    newString = oldString.replace("\\", "\\\\").replace(" ", "\\s").replace("?", "\\q").replace("|", "\\|")
    return newString


def getFileNameAndExt(fileName):
    """获取文件名和扩展名

    :param fileName: 文件路径
    :return:
    """
    _, tempfilename = os.path.split(fileName)
    shotname, extension = os.path.splitext(tempfilename)
    return shotname, extension


def isPreCheckFail(cliRet):
    """判断是否为升级前检查失败，如果倒数第二行存在检查项名称则认为升级前检查不通过，返回检查项详情

    :param cliRet: cli回显
    :return:
    """
    checkItems = []
    cliRetList = cliRet.encode("utf8").splitlines()

    # 避免回显格式不正确
    if len(cliRetList) < 1:
        return False, checkItems

    # 取倒数第二行，是否存在检查项名称
    checkLine = cliRetList[-2].strip()

    # 获取检查项名称
    infoList = checkLine.split(config.CHECK_ITEM_SPLIT_FLAG)
    checkName = infoList[0] if len(infoList) > 0 else ""
    if checkName not in config.CHECK_ITEMS:
        return False, checkItems

    # 过滤出所有检查项回显
    start_index = find(cliRet, config.CHECK_START_FLAG)
    checkCliRet = cliRet[start_index:]

    # 解析检查项
    checkItems = cliUtil.getHorizontalCliRet(checkCliRet)
    return True, checkItems


def getPreCheckFailedItem(checkItems):
    """获取不通过的检查项名称，遇不通过就返回检查项信息

    :param checkItems: 检查项
    :return:
    """
    for item in checkItems:
        for itemkey in item.keys():
            # 检查结果失败名称为Result的内容，兼容“Preck Item Result”及“Result”
            if config.CHECK_ITEM_RESULT not in itemkey:
                continue

            result = item.get(itemkey)
            if result != config.CHECK_ITEM_PASS:
                item["Result"] = result  # key统一转换为"Result"
                return item

    return {}


def getPatchVersion(dataDict):
    """获取补丁版本号

    :param dataDict: 上下文字典
    :return:
    """
    lang = dataDict.get('language')
    cli = contextUtil.getCli(dataDict)

    cmd = "show upgrade package"
    flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if "HotPatch Version" not in cliRet:
        raise UnCheckException("Failed to get HotPatch Version.")
    if not flag:
        return False, '', errMsg

    endIndex = cliRet.find('HotPatch Version')
    cliRetDictList = cliUtil.getHorizontalCliRet(cliRet[endIndex:])
    patchVersion = cliRetDictList[0].get('Current Version')

    return True, patchVersion, ''


def getDiskUpgradeMode(context):
    """获取硬盘升级模式

    :param context: 上下文
    :return:
    """
    selectDiskType = contextUtil.getSelectType(context)
    supportParallel = contextUtil.isSupportParallel(context)
    if selectDiskType == config.UPD_SELE_DISK_TYPE.ALL_DISK:
        return config.DISK_UPGRADE_MODE.ALL
    elif supportParallel and selectDiskType == config.UPD_SELE_DISK_TYPE.CUSTOM_DISK:
        return config.DISK_UPGRADE_MODE.MULTI
    else:
        return config.DISK_UPGRADE_MODE.SINGLE


def getUpgradeType(context):
    """获取在线或离线升级模式

    :param context: 上下文
    :return:
    """
    isOnlineUpgrade = contextUtil.isOnlineUpgrade(context)
    return 0 if isOnlineUpgrade else 1


#
def isSupportParallelBySpcVersion(devType, devVersion):
    """是否支持并行升级（在研版本支持并行升级）

    :param devType: 设备类型
    :param devVersion: 设备版本
    :return:
    """
    if "Dorado" in devType and devVersion > "V300R002C00":
        return True
    if "V3" in devVersion and devVersion > "V300R006C30":
        return True
    if "V5" in devVersion and devVersion > "V500R007C20":
        return True
    return False
