﻿# -*-coding:utf-8 -*-

from com.huawei.ism.exception import IsmException
from cbb.frame.tlv.tlvFactory import *
from cbb.frame.base import baseUtil

from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil


def drawGraph(context, isReloadGraph=False, **propDict):
    """绘制设备图

    :param context:上下文对象
    :param propDict: 支持如下key:
            hardwareTypeList: 可选，硬件类型集合（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"），默认支持
                                所有的硬件设备图绘制
            isBack: 可选，值为True表示绘制后视图，值为False表示绘制前视图，默认值为False
    :param isReloadGraph: 是否刷新设备图
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        defaultHardwareList = ["BAY", "ENCLOSURE", "CONTROLLER", "FAN_CTRL_MODULE", "FAN", "INTF_MODULE", "DISK",
                               "BACKUP_POWER", "POWER", "EXPBOARD", "SAS_PORT", "ETH_PORT", "PCIE_PORT", "FCoE_PORT",
                               "FC_PORT", "IB_PORT", "SFP_OPTICAL_TRANSCEIVER", ]
        hardwareTypeList = propDict.get("hardwareTypeList", defaultHardwareList)
        isBack = propDict.get("isBack", False)

        hardwareList = contextUtil.getItem(context, "hardwareList")
        logger.logInfo("hardwareList:%s" % hardwareList)

        if hardwareList is None or isReloadGraph:
            hardwareList = queryArrayHardwareList(context, hardwareTypeList)
            svpHardwareList = querySVPHardawreList(context)

            hardwareList.extend(svpHardwareList)
            logger.logInfo("hardwareList:%s" % hardwareList)

        devGraphUtils = context.get("DevGraphUtils")
        processPath = context.get("importRootDir")

        cfgFilePath = os.path.join(processPath, "frame", "devGraph", "DeviceGraphConfig.xml")
        imgPath = os.path.join(processPath, "frame", "devGraph", "image")
        cfgMap = {"graphConfigPath": cfgFilePath, "deviceImgCfgPath": imgPath}
        devGraphUtils.buildGraph(cfgMap, hardwareList, isBack, False)
        contextUtil.setItem(context, "hardwareList", hardwareList)

    except IsmException as ismException:
        logger.logException(ismException)

    except Exception as exception:
        logger.logException(exception)

    return


def querySVPHardawreList(context):
    """通过SVP tlv 连接获取SVP上相关器件信息

    :param context: 上下文
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    svpHardwareList = []
    try:
        tlv = Tlv.Conn.getTlvConnectionSVP(context)
        svpHardwareTypeList = ["ENCLOSURE", "DISK", "POWER", "ETH_PORT", ]
        for hardwareType in svpHardwareTypeList:
            try:
                logger.logInfo("begin to get svpHardware[%s] info." % hardwareType)
                svpHardwareList.extend(getHardwareGraphList(tlv, hardwareType))
                logger.logInfo("get svpHardware[%s] info successfully." % hardwareType)
            except IsmException as ismException:
                logger.logException(ismException)
                logger.logInfo("get SVP hardware[%s] info failed." % hardwareType)
            except Exception as exception:
                logger.logException(exception)
                logger.logInfo("get SVP hardware[%s] info failed." % hardwareType)

    except IsmException as ismException:
        logger.logException(ismException)
    except Exception as exception:
        logger.logException(exception)

    return svpHardwareList


def queryArrayHardwareList(context, hardwareTypeList):
    """通过阵列 tlv 连接查询阵列硬件信息.

    :param context: 上下文
    :param hardwareTypeList: 要查询的硬件类型列表.
    :return:
    """
    # 获取阵列的相关器件信息
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    hardwareList = []
    try:
        tlv = Tlv.Conn.getConn(context)
    except Exception as e:
        logger.logException(e)
        logger.logNoPass(
            'Query hardware failed because getting tlv connection failed and hardware list empty in context.')
    else:
        if isDCPower(context):
            baylist = getBayNames(context)
            powerEnclist = getPowerEnclosure(baylist)
            hardwareList.extend(powerEnclist)

        for hardwareType in hardwareTypeList:
            try:
                logger.logInfo("begin to get hardware[%s] info." % hardwareType)
                hardwareList.extend(getHardwareGraphList(tlv, hardwareType))
                logger.logInfo("get hardware[%s] info successfully." % hardwareType)
            except IsmException as ismException:
                logger.logException(ismException)
                logger.logInfo("get hardware[%s] info failed." % hardwareType)
            except Exception as exception:
                logger.logException(exception)
                logger.logInfo("get hardware[%s] info failed." % hardwareType)

    return hardwareList


def highLightHardware(context, hardwareLoc, hardwareType):
    '''
    @summary: 高亮硬件
    @param context: 上下文对象
    @param hardwareLoc: 硬件Location
    @param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    '''
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            condition0 = (hardwareLoc == hardwareInfo["location"])

            # 对于光模块，按实际类型（ETH,FC,FCoE）
            if hardwareType == "SFP_OPTICAL_TRANSCEIVER":
                condition1 = (hardwareInfo["typeName"] in ["ETH_PORT", "FCoE_PORT", "FC_PORT", ])
            else:
                condition1 = (hardwareInfo["typeName"] == hardwareType)

            if condition0 and condition1:
                devGraphUtils = context.get("DevGraphUtils")
                clearGraph(context)
                devGraphUtils.highlightModule(hardwareInfo, 'selected')
                return True

    except IsmException as ismException:
        logger.logException(ismException)
    except Exception as exception:
        logger.logException(exception)

    logger.logInfo("highlight hardware failure(hardwareLoc:%s, hardwareType:%s)" % (hardwareLoc, hardwareType))
    return False


def highLightCable(context, port1Id, port2Id, portType):
    '''
    @summary: 高亮线缆
    @param context: 上下文对象
    @param port1Id: 源端口ID
    @param port2Id: 目的端口ID
    @param portType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    '''
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        port1Info = {}
        port2Info = {}
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if portType != hardwareInfo["typeName"]:
                continue

            if len(port1Info) == 0 and port1Id == hardwareInfo["id"]:
                port1Info = hardwareInfo
                continue

            if len(port2Info) == 0 and port2Id == hardwareInfo["id"]:
                port2Info = hardwareInfo
                continue

        if len(port1Info) > 0 and len(port2Info) > 0:
            devGraphUtils = context.get("DevGraphUtils")
            clearGraph(context)
            devGraphUtils.highlightCable(port1Info, port2Info, 'selected')
            return True

    except IsmException as ismException:
        logger.logException(ismException)
    except Exception as exception:
        logger.logException(exception)

    logger.logInfo("highlight cable failure(port1Id:%s, port2Id:%s, portType:%s)" % (port1Id, port2Id, portType))
    return False


def getHardwareGraphList(tlv, hardwareType):
    '''
    @summary: 获取绘制硬件图的信息集合
    @param tlv: tlv对象
    @param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    @return: 
                    所有硬件对象需要包含如下字段：typeName,id,location,modelName,parentId,parentType,healthStatus
                    框对象需要包含如下字段：name,startU,heightU,logicTypeName
                    端口对象需要包含如下字段：runningStatus,currentPeerId,suggestPeerId
    '''
    enclosureHeight = tlvUtil.getCtrlEnclosureHeight(tlv)
    hardwareGraphList = []

    intStartU = 1
    records = tlvUtil.getHardwareRecords(tlv, hardwareType)
    for record in records:
        infoDict = {}
        infoDict["typeName"] = hardwareType
        infoDict["id"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        infoDict["name"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        infoDict["location"] = location
        infoDict["oldLoaction"] = location
        infoDict["modelName"] = getModelName(record, hardwareType, enclosureHeight)
        infoDict["parentId"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["parentID"])
        infoDict["parentType"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["parentType"])
        infoDict["healthStatus"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["healthStatus"])
        infoDict["runningStatus"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["runningStatus"])
        infoDict["startU"] = ""
        infoDict["heightU"] = ""
        infoDict["logicTypeName"] = ""
        infoDict["currentPeerId"] = ""
        infoDict["suggestPeerId"] = ""

        if hardwareType == "ENCLOSURE":
            logicType = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["logicType"])
            logicTypeName = tlvData.ENCLOSURE_TYPE.get(logicType, "")

            height = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["height"])
            if location in ['', '--']:
                startU = str(intStartU)
                intStartU += height
                location = 'SMB0.%sU' % startU
            else:
                startU = location.split(".")[-1].replace("U", "")

            infoDict["location"] = location
            infoDict["startU"] = startU
            infoDict["heightU"] = height
            infoDict["logicTypeName"] = logicTypeName

        elif hardwareType == "SAS_PORT":
            infoDict["currentPeerId"] = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["suggest_peer_port_id"])

        elif hardwareType == "PCIE_PORT":
            infoDict["currentPeerId"] = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT["suggest_peer_port_id"])

        for key in infoDict.keys():
            infoDict[key] = str(infoDict[key])

        hardwareGraphList.append(infoDict.copy())

    return hardwareGraphList


def getModelName(record, hardwareType, enclosureHeight):
    '''
    @summary: 获取设备图显示需要的modelName
    @param record: tlv记录
    @param hardwareType: 硬件类型（以字符串形式传入，值为OM对象的枚举值，如"BAY","ENCLOSURE"）
    @param enclosureHeight: 框高（2U对应2,3U对应3,6U对应6） 
    @return: 对于框对象、接口卡对象、级联板对象，返回model对应的枚举值（2U/3U/6U接口卡特殊处理）
    '''
    modelName = ""
    if hardwareType == "ENCLOSURE":
        model = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["model"])
        modelName = tlvData.ENCLOSURE_MODEL.get(model, "")
    elif hardwareType == "INTF_MODULE":
        suffixDict = {2: "_CTRL_2U", 3: "_CTRL_3U", 6: ""}  # 6U取默认值
        suffix = suffixDict.get(enclosureHeight, "")
        model = tlvUtil.getRecordValue(record, tlvData.INTF_MODULE["model"])
        modelName = "%s%s" % (tlvData.INTF_MODEL.get(model, ""), suffix)
    elif hardwareType == "EXPBOARD":
        model = tlvUtil.getRecordValue(record, tlvData.EXPBOARD["model"])
        modelName = tlvData.EXPBOARD_TYPE.get(model, "")
    return modelName


def getBayNames(context):
    tlv = Tlv.Conn.getConn(context)
    bayRecords = tlvUtil.getBayRecords(tlv)
    return tlvUtil.getBayNames(bayRecords)


def isDCPower(context):
    tlv = Tlv.Conn.getConn(context)
    powerType = tlvUtil.getPowerType(tlv)
    if powerType == tlvData.POWER_TYPE_E["DC"]:
        return True
    return False


def getPowerEnclosure(baylist):
    startLocs = [36, 39]
    encInfoList = []
    count = 0

    for startU in startLocs:
        for bayName in baylist:
            encName = "enc_power_%s" % str(count)
            encType = "POWER_DC_2U_ENCLOSURE"
            encParentId = bayName
            height = 2
            location = '%s.%sU' % (bayName, str(startU))

            count += 1
            infoDict = {}
            infoDict["typeName"] = 'ENCLOSURE'
            infoDict["id"] = str(time.time() + count)
            infoDict["name"] = encName
            infoDict["location"] = location
            infoDict["oldLoaction"] = location
            infoDict["modelName"] = encType
            infoDict["parentId"] = encParentId
            infoDict["parentType"] = '205'
            infoDict["healthStatus"] = '0'
            infoDict["runningStatus"] = '0'
            infoDict["startU"] = str(startU)
            infoDict["heightU"] = str(height)
            infoDict["logicTypeName"] = "POWER_ENCLOSURE"
            infoDict["currentPeerId"] = "--"
            infoDict["suggestPeerId"] = "--"

            encInfoList.append(infoDict.copy())
    return encInfoList


def clearGraph(context):
    """清除设备图高亮信息

    :param context: 上下文对象
    :return:
    """
    devGraphUtils = context.get("DevGraphUtils")
    devGraphUtils.clearGraph()
    return
