# -*- coding: UTF-8 -*-
from com.huawei.ism.exception import IsmException
from com.huawei.ism.tlv import TLVUtils
from com.huawei.ism.tlv.bean import Param
from com.huawei.ism.tlv.lang import UnsignedInt32
from com.huawei.ism.tlv.lang import UnsignedInt64

from cbb.frame.base import baseUtil
from cbb.frame.oldTlv import tlvData
from cbb.frame.oldTlv import tlvDataType

RETRY_TIMES = 5  # 建立连接时重试次数
TLV_CMD_TIMEOUT = 120
TLV_RECORD_END_INDEX = 4095
ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE = "1089101853"
COMMUNICATION_FAIL_ERROR_CODE = [
    "16797697",  # 与设备断开连接
    "1073949185",  # 与设备通信异常，请检查网络连接或设备状态是否正常
    "16797698",  # 与设备通信异常
]


def getParamValue(paramType, paramValue):
    '''
    @summary: 将参数转换为tlv参数
    '''
    if paramType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
            paramType == tlvDataType.DATA_TYPE_SIGN_INT or \
            paramType == tlvDataType.DATA_TYPE_BOOL or \
            paramType == tlvDataType.DATA_TYPE_ENUM or \
            paramType == tlvDataType.DATA_TYPE_TIME:
        return UnsignedInt32(paramValue)

    elif paramType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        return UnsignedInt64(paramValue)

    elif paramType == tlvDataType.DATA_TYPE_STRING or \
            paramType == tlvDataType.DATA_TYPE_ARRAY or \
            paramType == tlvDataType.DATA_TYPE_JSON:
        return str(paramValue)

    return paramValue


def getSfpOpticalTransceiver(tlv):
    '''
    @summary:获取光模块信息
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SFP_OPTICAL_TRANSCEIVER'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getParamList(params):
    '''
    @summary: 获取tlv参数列表
    '''
    paramList = []
    if params is None or len(params) == 0:
        return paramList
    for paramDefine, paramValue in params:
        paramIndex = paramDefine["index"]
        paramType = paramDefine["type"]
        paramValue = getParamValue(paramType, paramValue)
        paramList.append(Param(paramIndex, paramType, paramValue))
    return TLVUtils.paramList(paramList)


def getRecordValue(record, field):
    '''
    @summary: 获取tlv记录中的值
    @param record:
    @param field:
    '''
    fieldIndex = field["index"]
    fieldType = field["type"]

    if fieldType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
            fieldType == tlvDataType.DATA_TYPE_SIGN_INT or \
            fieldType == tlvDataType.DATA_TYPE_ENUM or \
            fieldType == tlvDataType.DATA_TYPE_TIME:

        val = record.getParamIntValue(fieldIndex)
        if val:
            return val.intValue()
        return 0

    elif fieldType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        return record.getParamIntValue(fieldIndex).longValue()

    elif fieldType == tlvDataType.DATA_TYPE_STRING or \
            fieldType == tlvDataType.DATA_TYPE_ARRAY or \
            fieldType == tlvDataType.DATA_TYPE_BOOL or \
            fieldType == tlvDataType.DATA_TYPE_JSON:
        return str(record.getParamStrValue(fieldIndex))

    return record.getParamValue(fieldIndex)


def execCmd(tlv, cmd, params, isBatch=False, getBatch=False, timeOut=TLV_CMD_TIMEOUT):
    '''
    @summary: 执行tlv命令，下发异常可进行重试
    @param tlv: tlv对象
    @param cmd: 命令字
    @param params: 参数列表
    @param isBatch: 是否为批量处理（参考GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录有结束符标记）
    @param getBatch: 是否为获取批量内容（参考非GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录没有结束符标记）
    @param timeOut: 超时时间
    '''
    retryTimes = 3  # 重试次数
    for i in range(0, retryTimes + 1):
        try:
            records = execCmdOnce(tlv, cmd, params, isBatch, getBatch, timeOut)
            return records
        except Exception as exception:
            errCode = exception.args[0]
            if i < retryTimes and (errCode in COMMUNICATION_FAIL_ERROR_CODE):
                baseUtil.safeSleep(30)  # 间隔30s重新下发
                continue
            # 超过重试次数或非通信异常直接抛异常
            raise exception
    return


def execCmdOnce(tlv, cmd, params, isBatch=False, getBatch=False, timeOut=TLV_CMD_TIMEOUT):
    '''
    @summary: 执行tlv命令
    @param tlv: tlv对象
    @param cmd: 命令字
    @param params: 参数列表
    @param isBatch: 是否为批量处理（参考GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录有结束符标记）
    @param getBatch: 是否为获取批量内容（参考非GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录没有结束符标记）
    @param timeOut: 超时时间
    '''
    try:
        records = None
        paramList = getParamList(params)

        if isBatch:
            records = tlv.bacthNext(cmd, paramList, timeOut, RETRY_TIMES)
            if records is not None and len(records) > 1:
                lastRecord = records[-1]
                lastRecordParamList = lastRecord.getParams()
                if lastRecordParamList[0].getIndex() == TLV_RECORD_END_INDEX:
                    records.pop()
            return records

        if getBatch:
            return tlv.getBatch(cmd, paramList, timeOut, RETRY_TIMES)

        return tlv.invoke(cmd, paramList, timeOut, RETRY_TIMES)

    except IsmException, ismException:
        errCode = ismException.getErrorId()
        # 往从节点下发消息失败，可以忽略。
        if ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE in str(errCode):
            return None

        errMsg = ismException.getErrorMessage()
        raise Exception(errCode, errMsg)
        return


def binary2decimalSystem(num):
    '''
    @summary: 二进制转十进制
    '''
    result = 0
    length = len(num)
    for i in range(0, length):
        result = result + (int(num[i]) << length - i - 1)
    return result


def getEnclosureRecords(tlv):
    '''
    @summary: 获取当前集群所有框信息的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getCtrlEnclosureHeight(tlv):
    '''
    @summary: 获取当前集群框高
    @param tlv: tlv对象
    '''
    records = getEnclosureRecords(tlv)
    for record in records:
        logicType = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logicType == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            return getRecordValue(record, tlvData.ENCLOSURE['height'])
    return None


def getHardwareRecords(tlv, hardwareType):
    '''
    @summary: 根据硬件形态获取该硬件的所有记录
    @param tlv: tlv对象
    @param hardwareType: 硬件形态
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E[hardwareType])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)


def getBayNames(bayRecords):
    '''
    @summary: 获取当前集群的所有柜model列表
    @param tlv: tlv对象
    '''
    bayNames = []
    for record in bayRecords:
        bayName = getRecordValue(record, tlvData.BAY['name'])
        bayNames.append(bayName)
    return bayNames
