# -*- coding: UTF-8 -*-
from cbb.business.operate.expansion import common
from cbb.common.conf.productConfig import NEW_SCALEOUT_INTF_SLOT_CONFIG
from cbb.common.conf.productConfig import SLOT_KEY
from cbb.common.conf.productConfig import INTERF_MODEL_KEY

from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest.restUtil import Tlv2Rest
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil
from cbb.frame.tlv.tlvEnumFactory import TlvEnum
from cbb.business.product.product_selector import get_product_adapter


def exeCmd(tlv, origin_obj_params, ad_obj_param_dict,
           isBatch=False, getBatch=False, timeOut=300, reTry=True,
           retry_interval=config.REST_DEFAULT_RETRY_INTERVAL):
    """通过AD转发执行tlv命令

    :param tlv: tlv对象
    :param origin_obj_params: 命令字
    :param ad_obj_param_dict: 参数列表
    :param isBatch: 是否为批量处理
    :param getBatch: 是否为获取批量内容
    :param timeOut: 超时时间
    :param reTry: 是否重试
    :param retry_interval: 重试间隔时间
    :return: 批量查询结果列表
    """
    cmd = tlvData.CMD["EXPANSION_AD_FORWARD"]
    ad_objparams = []
    for key in ad_obj_param_dict.keys():
        param = (tlvData.EXPANSION_AD_FORWARD_DEFINE.get(key),
                 ad_obj_param_dict.get(key))
        ad_objparams.append(param)
    params = origin_obj_params + ad_objparams

    baseUtil.safeSleep(0.1)
    record = tlvUtil.execCmd(tlv, cmd, params, isBatch, getBatch,
                             timeOut, reTry, retry_interval)
    return record


def getRecordValue(record, field):
    """获取record中指定字段的值

    :param record: record记录
    :param field: 指定字段
    :return:
    """
    return Tlv2Rest.getRecordValue(record, field)


def isEnclosureSNValid(enclosure_sn):
    """判断框SN是否有效

    :param enclosure_sn: 框SN
    :return: True: 框SN有效 False: 框SN无效
    """
    return not enclosure_sn.strip().upper() in ["", "INVALID_SN"]


def getPid(cmd, obj):
    """获取tlv消息的发送对象PID

    :param cmd: 命令字
    :param obj: 操作对象
    :return:
    """
    return tlvData.PID.get((cmd, obj))


def dec2hex(string_num):
    """十进制转十六进制

    :param string_num: 十进制数
    :return:
    """
    str_buffer = [str(x) for x in range(10)] + \
                 [chr(x) for x in range(ord('A'), ord('A') + 6)]
    num = int(string_num)
    mid = []
    while True:
        if num == 0:
            break
        num, rem = divmod(num, 16)
        mid.append(str_buffer[rem])

    return "0x" + ''.join([str(x) for x in mid[::-1]])


def notifyAD(tlv):
    """通知AD执行扩容

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.EXPANSION_AD_NOTIFY_DEFINE['status'], 1)
    params.append(param)
    baseUtil.safeSleep(2)
    record = tlvUtil.execCmd(tlv, tlvData.CMD["EXPANSION_AD_NOTIFY"], params)
    return getRecordValue(record,
                          tlvData.EXPANSION_AD_NOTIFY_DEFINE['result'])


def startAD(tlv):
    """启动AD

    :param tlv: tlv对象
    :return:
    """
    params = []
    param = (tlvData.EXPANSION_AD_PROCESS_DEFINE['status'], 0)
    params.append(param)
    baseUtil.safeSleep(2)
    record = tlvUtil.execCmd(tlv, tlvData.CMD["EXPANSION_AD_PROCESS"], params)
    return getRecordValue(record,
                          tlvData.EXPANSION_AD_PROCESS_DEFINE['successNum'])


def stopAD(tlv):
    """停止AD

    :param tlv:
    :return: tlv对象
    """
    params = []
    param = (tlvData.EXPANSION_AD_PROCESS_DEFINE['status'], 1)
    params.append(param)
    record = tlvUtil.execCmd(tlv, tlvData.CMD["EXPANSION_AD_PROCESS"], params)
    return getRecordValue(record,
                          tlvData.EXPANSION_AD_PROCESS_DEFINE['successNum'])


def getBoards(tlv):
    """获取当前集群和待扩集群的框SN和节点ID

    :param tlv: tlv对象
    :return: 框列表
    """
    baseUtil.safeSleep(2)
    records = tlvUtil.execCmd(tlv, tlvData.CMD["EXPANSION_GET_BOARDS"],
                              None, getBatch=True, timeOut=300)
    boards_list = []
    for record in records:
        enclosure_sn = getRecordValue(
            record,
            tlvData.EXPANSION_GET_BOARDS_DEFINE['enclosureSN'])
        if not isEnclosureSNValid(enclosure_sn):
            continue

        boards_dict = {}
        boards_dict["enclosureSN"] = enclosure_sn
        boards_dict["controllerID"] = getRecordValue(
            record,
            tlvData.EXPANSION_GET_BOARDS_DEFINE['controllerID'])
        boards_list.append(boards_dict.copy())

    return boards_list


def setEnclosureSwitchOn(tlv, board):
    """打开框的定位灯

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    return setEnclosureSwitch(tlv, board, 1)


def setEnclosureSwitchOff(tlv, board):
    """关闭框的定位灯

    :param tlv: tlv对象
    :param board: 指定节点
    :return: True: 设置成功 Flase: 设置失败
    """
    return setEnclosureSwitch(tlv, board, 2)


def setEnclosureSwitch(tlv, board, switch):
    """设置框的定位灯状态

    :param tlv: tlv对象
    :param board: 指定节点
    :param switch: 定位灯状态（ON：1，OFF：2）
    :return: True: 设置成功 Flase: 设置失败
    """
    try:
        params = []
        param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
        params.append(param)

        ad_obj_param_dict = {}
        ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
        ad_obj_param_dict["controllerID"] = board["controllerID"]
        cmd = tlvData.CMD["GET_BATCH_NEXT"]
        obj = tlvData.OM_OBJ_E["ENCLOSURE"]
        ad_obj_param_dict["cmd"] = cmd
        ad_obj_param_dict["obj"] = obj
        ad_obj_param_dict["pid"] = getPid(cmd, obj)

        enclosure_records = exeCmd(tlv, params, ad_obj_param_dict,
                                   isBatch=True, timeout=50)
        if enclosure_records is None or len(enclosure_records) == 0:
            return True

        enclosure_id = ""
        for record in enclosure_records:
            enclosure_sn = getRecordValue(record,
                                          tlvData.ENCLOSURE["serialNum"])
            if enclosure_sn == board["enclosureSN"]:
                enclosure_id = getRecordValue(record, tlvData.PUB_ATTR["id"])
                break

        if len(enclosure_id) == 0:
            return False

        params = []
        param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["ENCLOSURE"])
        params.append(param)
        param = (tlvData.PUB_ATTR["id"], enclosure_id)
        params.append(param)
        param = (tlvData.ENCLOSURE["switch"], switch)
        params.append(param)

        ad_obj_param_dict = {}
        ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
        ad_obj_param_dict["controllerID"] = board["controllerID"]
        cmd = tlvData.CMD["MODIFY"]
        obj = tlvData.OM_OBJ_E["ENCLOSURE"]
        ad_obj_param_dict["cmd"] = cmd
        ad_obj_param_dict["obj"] = obj
        ad_obj_param_dict["pid"] = getPid(cmd, obj)

        exeCmd(tlv, params, ad_obj_param_dict, timeOut=50)

    except Exception:
        return False

    return True


def modifyManIP(lang, logger, tlv, board,
                man_ip_info, ip_ver, port_location):
    """修改指定节点的管理IP

    :param lang: 中英文类型
    :param logger: 日志对象
    :param tlv: tlv对象
    :param board: 指定节点
    :param man_ip_info: 管理IP
    :param ip_ver: 版本
    :param port_location: 端口位置
    :return:
    """
    err_msg = ""
    try:
        records = getEthPortRecords(tlv, board)
        if records is None or len(records) == 0:
            return (False, err_msg)

        record_info = None
        for record in records:
            location = getRecordValue(record, tlvData.PUB_ATTR['location'])
            if location == port_location:
                record_info = record
                break

        if record_info is None:
            logger.logNoPass("the recordInfo %s is abnormal" % record_info)
            err_msg, _ = common.getMsg(lang, "config.manip.exception")
            return (False, err_msg)

        if ip_ver == common.IPV4:
            setIpv4(tlv, record_info, man_ip_info, board)
        elif ip_ver == common.IPV6:
            setIpv6(tlv, record_info, man_ip_info, board)

    except Exception as exception:
        logger.logException(exception)
        err_msg, _ = common.getMsg(lang, "config.manip.exception")
        return (False, err_msg)

    return (True, err_msg)


def getRealMacAddr(ad_mac_addr):
    """将路由信息中的Mac地址转为真实的Mac地址

    :param ad_mac_addr: 路由信息中的Mac地址
    :return:
    """
    arr = []
    for sub in ad_mac_addr.split(":"):
        if len(sub) == 1:
            arr.append("0%s" % sub)
        else:
            arr.append(sub)
    return ":".join(arr)


def getRouteInfoList(tlv):
    """获取当前集群与待扩集群之间ScaleOut端口的所有路由信息

    :param tlv: tlv对象
    :return:
    """
    baseUtil.safeSleep(2)
    records = tlvUtil.execCmd(tlv, tlvData.CMD["EXPANSION_GET_ROUTE_INFO"],
                              None, getBatch=True, timeOut=600)
    route_info_list = []
    for record in records:
        src_enclosure_sn = getRecordValue(
            record,
            tlvData.EXPANSION_GET_ROUTE_INFO_DEFINE['srcEnclosureSN'])
        des_enclosure_sn = getRecordValue(
            record,
            tlvData.EXPANSION_GET_ROUTE_INFO_DEFINE['desEnclosureSN'])
        if not isEnclosureSNValid(src_enclosure_sn):
            continue
        if not isEnclosureSNValid(des_enclosure_sn):
            continue

        routeInfoDict = {}
        routeInfoDict["srcEnclosureSN"] = src_enclosure_sn
        routeInfoDict["srcControllerID"] = getRecordValue(
            record,
            tlvData.EXPANSION_GET_ROUTE_INFO_DEFINE['srcControllerID'])
        routeInfoDict["srcPortMacAddr"] = getRealMacAddr(
            getRecordValue(
                record,
                tlvData.EXPANSION_GET_ROUTE_INFO_DEFINE['srcPortMacAddr']))
        routeInfoDict["desEnclosureSN"] = des_enclosure_sn
        routeInfoDict["desControllerID"] = getRecordValue(
            record,
            tlvData.EXPANSION_GET_ROUTE_INFO_DEFINE['desControllerID'])
        routeInfoDict["desPortMacAddr"] = getRealMacAddr(
            getRecordValue(
                record,
                tlvData.EXPANSION_GET_ROUTE_INFO_DEFINE['desPortMacAddr']))
        route_info_list.append(routeInfoDict.copy())

    return route_info_list


def refreshRouteInfo(tlv, logger):
    """刷新每个节点的路由信息表

    :param tlv: tlv对象
    :param logger: 日志对象
    :return:
    """
    flag = True
    try:
        # 刷新每个节点的路由信息表
        logger.logInfo("start refresh routine info")
        getRouteInfoList(tlv)
        logger.logInfo("refresh routine info success.")
    except Exception as e:
        flag = False
        logger.logInfo("refresh routine info exception : %s." % str(e))

    return flag


def getEthPortRecords(tlv, board):
    """获取指定节点上的所有ETH端口记录

    :param tlv: 指定节点
    :param board: tlv对象
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["ETH_PORT"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getEnclosureRecords(tlv, board):
    """获取指定节点上的所有框记录

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["ENCLOSURE"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getDiskRecords(tlv, board):
    """获取指定节点上的所有硬盘记录

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['DISK'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["DISK"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getInterfaceModuleRecords(tlv, board):
    """获取指定节点上的所有接口卡记录

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['INTF_MODULE'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["INTF_MODULE"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def isScaleOutIntfModule(intf_module_id, ctrl_height):
    """判断接口卡是否能用于ScaleOut扩容
      1.2U设备，ScaleOut卡固定在1号槽位；
      2.3U/6U设备，ScaleOut卡固定在3号槽位。

    :param intf_module_id: 接口卡ID
    :param ctrl_height: 框高，用于区分2U\3U\6U等设备
    :return:
    """
    if len(intf_module_id) < 3:
        return False

    intf_module_slot = intf_module_id[-3:]
    if ctrl_height == 2 and intf_module_slot in ["A.1", "B.1", "C.1", "D.1"]:
        return True

    if ctrl_height in [3, 6] and intf_module_slot in \
            ["A.3", "B.3", "C.3", "D.3"]:
        return True

    return False


def is_scaleout_intf_module_for_new(
        location, product_model, model,
        net_working_type,
        interf_slot_conf=NEW_SCALEOUT_INTF_SLOT_CONFIG):
    """判断V5/V6新硬件接口卡是否能用于ScaleOut扩容
    低端(2U) CTEX.IOM1、CTEX.IOM1
    中端(2U) CTEX.IOM2 、CTEX.IOM2
    高端(4U) CTEX.H3、CTEX.L3、CTEX.H10、CTEX.L10

    :param location: 接口卡location
    :param product_model: 产品型号
    :param model: 接口卡model
    :param net_working_type: 组网方式
    :param interf_slot_conf: 接口卡配置字典
    :return:
    """
    if not location:
        return False
    slot = location.split(".")[-1]

    interf_slots = get_interf_slots(product_model, interf_slot_conf)
    interf_model = get_interf_model(product_model, interf_slot_conf,
                                    net_working_type)
    product_adapter = get_product_adapter()
    if product_adapter:
        support_interface = product_adapter.get_support_interface()
        interf_slots = support_interface.get(SLOT_KEY)
        interf_model = support_interface.get(INTERF_MODEL_KEY, {}) \
            .get(net_working_type, [])

    if slot in interf_slots and model in interf_model:
        return True
    return False


def get_interf_slots(product_model, interf_slot_conf):
    """获取槽位

    :param product_model: 版本
    :param interf_slot_conf: 槽位配置
    :return:
    """
    for key, value in interf_slot_conf.items():
        if product_model in key:
            return value.get(SLOT_KEY, [])
    return []


def get_interf_model(product_model, interf_slot_conf, net_working_type):
    """获取接口卡枚举

    :param product_model:
    :param interf_slot_conf:
    :param net_working_type:
    :return:
    """
    for key, value in interf_slot_conf.items():
        if product_model in key:
            return value.get(INTERF_MODEL_KEY, {}).get(net_working_type, [])
    return []


def getScaleOutIntfModuleList(tlv, board, product_model=None,
                              net_working_type="", logger=None):
    """获取指定节点用于ScaleOut扩容的接口卡集合

    :param tlv: tlv对象
    :param board: 指定节点
    :param product_model: 产品型号
    :param net_working_type: 组网方式
    :param logger: 日志对象
    :return:
    """
    records = getInterfaceModuleRecords(tlv, board)
    if not records:
        return []
    scale_out_intf_module_list = []
    added_loc = []
    for record in records:
        location = getRecordValue(record, tlvData.PUB_ATTR['location'])
        model = getRecordValue(record, tlvData.PUB_ATTR['model'])
        if not is_scaleout_intf_module_for_new(location, product_model,
                                               model, net_working_type):
            continue
        if location not in added_loc:
            scale_out_intf_module_info = {}
            scale_out_intf_module_info["location"] = location
            scale_out_intf_module_info["runMode"] = getRecordValue(
                record,
                tlvData.INTF_MODULE['runMode'])
            scale_out_intf_module_info["healthStatus"] = getRecordValue(
                record,
                tlvData.PUB_ATTR['healthStatus'])
            scale_out_intf_module_info["runningStatus"] = getRecordValue(
                record,
                tlvData.PUB_ATTR['runningStatus'])
            scale_out_intf_module_info["enclosureSN"] = board["enclosureSN"]
            scale_out_intf_module_list.append(
                scale_out_intf_module_info.copy())
            added_loc.append(location)
    return scale_out_intf_module_list


def getScaleOutPortsInfo(tlv, board, ctrl_height, new_config_clust_type):
    """获取指定节点用于ScaleOut扩容的端口集合

    :param tlv: tlv对象
    :param board: 指定节点
    :param ctrl_height: 框高，用于区分2U\3U\6U等设备
    :param new_config_clust_type:
    :return:
    """
    records = getEthPortRecords(tlv, board)
    if records is None:
        return []

    eth_port_info = []
    for record in records:
        location = getRecordValue(record, tlvData.PUB_ATTR['location'])

        intf_module_id = getRecordValue(record, tlvData.PUB_ATTR['parentID'])
        if not isScaleOutIntfModule(intf_module_id, ctrl_height):
            continue
        ctrl_name = intf_module_id.split(".")[0]

        # 交换机组网场景，端口仅连接在P0和P1端口
        if new_config_clust_type == common.CLUST_TYPE_SWITCH:
            if location.endswith("P2") or location.endswith("P3"):
                continue

        port_info_dict = {}
        port_info_dict["portID"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['id'])
        port_info_dict["location"] = location
        port_info_dict["healthStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['healthStatus'])
        port_info_dict["runningStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['runningStatus'])
        port_info_dict["speed"] = getRecordValue(
            record,
            tlvData.ETH_PORT['speed'])
        port_info_dict["macAddress"] = getRecordValue(
            record,
            tlvData.ETH_PORT['macAddress'])
        port_info_dict["enclosureSN"] = board["enclosureSN"]
        port_info_dict["bayid"] = int(ctrl_name[0:-1])
        port_info_dict["nodeName"] = ctrl_name[-1]
        eth_port_info.append(port_info_dict.copy())
    eth_port_info.sort(lambda x, y: common.cmpPortDict(x, y))
    return eth_port_info


def getScaleOutPortsInfoForDorado(tlv, board, new_config_clust_type):
    """获取指定节点用于ScaleOut扩容的端口集合

    :param tlv: tlv对象
    :param board: 指定节点
    :param new_config_clust_type: 框高，用于区分2U\3U\6U等设备
    :return:
    """
    records = getEthPortRecords(tlv, board)
    if records is None:
        return []
    eth_port_info = []
    for record in records:
        location = getRecordValue(record, tlvData.PUB_ATTR['location'])

        intf_module_id = getRecordValue(record, tlvData.PUB_ATTR['parentID'])
        ctrl_name = intf_module_id.split(".")[0]

        # 交换机组网场景，端口仅连接在MGMT和MAINTENANCE端口
        if new_config_clust_type == common.CLUST_TYPE_SWITCH:
            if not (location.endswith("MGMT") or
                    location.endswith("MAINTENANCE")):
                continue

        port_info_dict = {}
        port_info_dict["portID"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['id'])
        port_info_dict["location"] = location
        port_info_dict["healthStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['healthStatus'])
        port_info_dict["runningStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['runningStatus'])
        port_info_dict["speed"] = getRecordValue(
            record,
            tlvData.ETH_PORT['speed'])
        port_info_dict["macAddress"] = getRecordValue(
            record,
            tlvData.ETH_PORT['macAddress'])
        port_info_dict["enclosureSN"] = board["enclosureSN"]
        port_info_dict["bayid"] = int(ctrl_name[0:-1])
        port_info_dict["nodeName"] = ctrl_name[-1]
        eth_port_info.append(port_info_dict.copy())
    eth_port_info.sort(lambda x, y: common.cmpPortDict(x, y))
    return eth_port_info


def getSystemRecord(tlv, board):
    """获取指定节点的系统信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SYSTEM'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    cmd = tlvData.CMD["GET"]
    obj = tlvData.OM_OBJ_E["SYSTEM"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def get_product_model_string(tlv, board):
    """获取指定节点的产品类型string

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    record = getSystemRecord(tlv, board)
    if record is None:
        return None
    return getRecordValue(record, tlvData.SYSTEM['product_model_string'])


def get_product_model_enum(tlv, board):
    """获取指定节点的产品类型枚举值

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    record = getSystemRecord(tlv, board)
    if record is None:
        return None
    return getRecordValue(record, tlvData.SYSTEM['productMode'])


def getProductModel(tlv, board):
    """获取指定节点的产品类型信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    product_model_enum = get_product_model_enum(tlv, board)
    if product_model_enum is None:
        return None
    return tlvUtil.getPdtModel(product_model_enum)


def getSystemRecordTestAd(tlv, board, sendCtrlId=True):
    """获取指定节点的系统信息

    :param tlv: tlv对象
    :param board: 指定节点
    :param sendCtrlId: 控制器ID
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SYSTEM'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    if sendCtrlId:
        ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET"]
    obj = tlvData.OM_OBJ_E["SYSTEM"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def getProductModelTestAd(tlv, board, sendCtrlId=True):
    """获取指定节点的产品类型信息

    :param tlv: tlv对象
    :param board: 指定节点
    :param sendCtrlId: 控制器ID
    :return:
    """
    record = getSystemRecordTestAd(tlv, board, sendCtrlId)
    if record is None:
        return None
    product_model_enum = getRecordValue(record,
                                        tlvData.SYSTEM['productMode'])
    return tlvUtil.getPdtModel(product_model_enum)


def getInternalDeviceInfo(tlv, board):
    """获取内部设备信息(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_INTERNAL_DEVICE_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def getInternalPdtModel(dev_info):
    """获取内部产品型号(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param dev_info:
    :return:
    """
    if dev_info is None:
        return None
    return getRecordValue(
        dev_info,
        tlvData.INTERNAL_DEVICE_INFO['internalProductModel'])


def get_overseas(dev_info):
    """获取是否为海外型号（新融合6.1.3及以后）

    :param dev_info:
    :return:
    """
    if dev_info is None:
        return None
    return getRecordValue(
        dev_info,
        tlvData.INTERNAL_DEVICE_INFO['isOverseas'])


def getCpuModel(dev_info):
    """获取CPU型号(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param dev_info:
    :return:
    """
    if dev_info is None:
        return None
    return getRecordValue(dev_info,
                          tlvData.INTERNAL_DEVICE_INFO['cpuModelInfo'])


def getCpuNumber(dev_info):
    """获取CPU个数(Dorado V3R1C21版本开始新增，用于区分普通版和增强版)

    :param dev_info:
    :return:
    """
    if dev_info is None:
        return None
    return getRecordValue(dev_info,
                          tlvData.INTERNAL_DEVICE_INFO['cpuNumber'])


def getProductVersion(tlv, board):
    """获取指定节点的产品版本信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    record = getSystemRecord(tlv, board)
    if record is None:
        return None, None
    patch_ver = getRecordValue(record, tlvData.SYSTEM['patchVersion'])
    software_version = getRecordValue(record, tlvData.SYSTEM.get("software_version"))
    if software_version:
        return software_version, patch_ver
    point_release = getRecordValue(record, tlvData.SYSTEM['pointRelease'])
    product_ver = getRecordValue(record, tlvData.SYSTEM['productVersion'])
    return (point_release, patch_ver) if point_release \
        else (product_ver, patch_ver)


def getAllPatchVersion(tlv, board):
    """获取补丁版本

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.CMO_CR_VER_INFO_S['CMO_VER_PACKAGE_TYPE'],
             tlvData.UPGRADE_PKGTYPE_E['UPGRADE_PKGTYPE_TYPE_SYS'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["OM_MSG_OP_LST_VER"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    rec = exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)
    try:
        patch_version = getRecordValue(
            rec,
            tlvData.CMO_CR_VER_INFO_S['CMO_VER_CUR_VERSION'])
        return patch_version
    except Exception:
        return "--"


def getControllerRecords(tlv, board):
    """获取指定节点的所有控制器信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['CONTROLLER'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["CONTROLLER"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def checkControllerStatus(tlv, board):
    """检查指定节点的控制器状态

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    flag = True
    ctrl_ids = []
    records = getControllerRecords(tlv, board)
    if records is None:
        return (True, ctrl_ids)

    for record in records:
        health_status = getRecordValue(
            record, tlvData.PUB_ATTR['healthStatus'])
        running_status = getRecordValue(
            record, tlvData.PUB_ATTR['runningStatus'])
        if health_status != tlvData.HEALTH_STATUS_E["NORMAL"] or \
                running_status != tlvData.RUNNING_STATUS_E["ONLINE"]:
            ctrl_id = getRecordValue(record, tlvData.PUB_ATTR["id"])
            flag = False
            ctrl_ids.append(ctrl_id)

    ctrl_ids.sort()
    return (flag, ctrl_ids)


def getSystemAlarm(tlv, board):
    """获取指定节点的告警信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_CURRENT_ALARM"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    records = exeCmd(tlv, params, ad_obj_param_dict, getBatch=True)
    if records is None or len(records) == 0:
        return []
    alarm_list = get_alarm_from_records(records)
    return alarm_list


def getCpuInfoRecords(tlv, board):
    """获取当前集群的控制器CPU信息集合（CPU利用率、剩余内存等）

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param0 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_MSGTYPE'], 0)
    params.append(param0)
    param12 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_DIAG_CMD'],
               "upd sysresourcecheck")
    params.append(param12)
    param13 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_NODE_CFG'], 2)
    params.append(param13)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    cmd = tlvData.CMD["OM_MSG_OP_UPD_EXECUTE"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict, getBatch=True)


def getAllCpuUsage(tlv, board):
    """获取控制器CPU利用率

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    cpu_usage_list = []
    records = getCpuInfoRecords(tlv, board)
    if records is None:
        return cpu_usage_list

    for record in records:
        ctrl_id = Tlv2Rest.getRecordValue(
            record,
            restData.System.CpuUsage.RECORD_CONTROLLER_ID)
        cpu_info = Tlv2Rest.getRecordValue(
            record,
            restData.System.CpuUsage.RECORD_CPU_INFO)
        cpu_usage = cpu_info.split(",")[0]
        cpu_usage_list.append(
            {"ctrlId": ctrl_id, "cpuUsage": cpu_usage}.copy())
    return cpu_usage_list


def getAllMemory(tlv, board):
    """获取内存

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    memory_list = []
    records = getControllerRecords(tlv, board)
    if records is None:
        return memory_list

    for record in records:
        memory_dict = {}
        memory_dict["ctrlId"] = getRecordValue(
            record, tlvData.PUB_ATTR['id'])
        memory_dict["memorySize"] = getRecordValue(
            record, tlvData.CONTROLLER['memorySize'])
        memory_list.append(memory_dict.copy())
    return memory_list


def isIBCPasswordChanged(tlv, board):
    """检查指定节点的IBC密码是否为默认密码

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    record = exeCmd(tlv, params, ad_obj_param_dict)
    if record is None:
        return 0
    return getRecordValue(record,
                          tlvData.CHECK_IBC_PASSWORD['passwordChanged'])


def isIBCKeyPairChanged(tlv, board):
    """检查指定节点的IBC公私钥对是否为默认值

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    record = exeCmd(tlv, params, ad_obj_param_dict)
    if record is None:
        return 0
    return getRecordValue(record,
                          tlvData.CHECK_IBC_PASSWORD['keyPairChanged'])


def readXnetInfo(tlv, board):
    """读取指定节点的配置文件中xnet信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["READ_IP_CLUST_BASE_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def getClusterXnetInfo(ports_scan_list, enclosure_sn, bay_id):
    """根据框SN和柜ID，获取集群的xnet部分信息

    :param ports_scan_list: 扫描端口列表
    :param enclosure_sn: 框号
    :param bay_id: 柜ID
    :return:
    """
    '''
    @summary: 根据框SN和柜ID，获取集群的xnet部分信息
    @param tlv: tlv对象
    @param board: 指定节点
    '''
    port_info_list = []
    for port_info in ports_scan_list:
        if port_info["enclosureSN"] == enclosure_sn \
                and port_info["bayid"] == bay_id:
            port_info_list.append(port_info.copy())
    return port_info_list


def modifyXnetInfo(tlv, board, bay_id, xnet_info):
    """修改指定节点的xnet部分信息

    :param tlv: tlv对象
    :param board: 指定节点
    :param bay_id: 柜ID
    :param xnet_info: 所有集群的xnet部分信息集合
    :return:
    """
    base_ip_addr = xnet_info["baseIpAddr"]
    new_config_clust_type = xnet_info["newConfigClustType"]
    port_info_list = getClusterXnetInfo(xnet_info["portsScanList"],
                                        board["enclosureSN"], bay_id)

    config_info_dict = {}
    for ctrl_name in ["A", "B", "C", "D"]:
        config_info_dict.setdefault(ctrl_name,
                                    [restData.REST_INVALID_VALUE for x in
                                     range(0, 8)])

    params = []
    param = (tlvData.IP_CLUST_BASE_INFO['baseIpAddr'], base_ip_addr)
    params.append(param)
    param = (tlvData.IP_CLUST_BASE_INFO['netMode'], new_config_clust_type)
    params.append(param)

    for ctrl_name in ["A", "B", "C", "D"]:
        config_info = config_info_dict.get(ctrl_name)
        for x in range(0, 4):
            port = None
            for port_info in port_info_list:
                if port_info["controllerName"][-1] == ctrl_name and str(x) == \
                        port_info["location"][-1]:
                    port = port_info.copy()

            port_id_name = "ctr%sPortID%s" % (ctrl_name.lower(), x + 1)
            dsw_id_name = "ctr%sDswID%s" % (ctrl_name.lower(), x + 1)

            param0 = (tlvData.IP_CLUST_BASE_INFO[port_id_name],
                      config_info[x * 2])
            param1 = (tlvData.IP_CLUST_BASE_INFO[dsw_id_name],
                      config_info[x * 2 + 1])

            if port is not None:
                param0 = (tlvData.IP_CLUST_BASE_INFO[port_id_name],
                          long(port["portID"]))
                if int(port["plane"]) == -1:
                    param1 = (tlvData.IP_CLUST_BASE_INFO[dsw_id_name],
                              restData.REST_INVALID_VALUE)
                else:
                    param1 = (tlvData.IP_CLUST_BASE_INFO[dsw_id_name],
                              int(port["plane"]))

            params.append(param0)
            params.append(param1)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_IP_CLUST_BASE_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def checkBaseIPConflict(tlv, board, base_ip_addr):
    """查看基地址IP是否冲突（与管理口、维护口、业务口等进行冲突检测）

    :param tlv: tlv对象
    :param board: 指定节点
    :param base_ip_addr: 基地址IP
    :return:
    """
    params = []
    param = (tlvData.IP_CLUST_BASE_INFO['baseIpAddr'], base_ip_addr)
    params.append(param)
    param = (tlvData.IP_CLUST_BASE_INFO['netMode'], 0)
    params.append(param)

    for ctrl_name in ["A", "B", "C", "D"]:
        for x in range(0, 4):
            port_id_name = "ctr%sPortID%s" % (ctrl_name.lower(), x + 1)
            dsw_id_name = "ctr%sDswID%s" % (ctrl_name.lower(), x + 1)

            param0 = (tlvData.IP_CLUST_BASE_INFO[port_id_name], 0)
            param1 = (tlvData.IP_CLUST_BASE_INFO[dsw_id_name], 0)
            params.append(param0)
            params.append(param1)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_IP_CLUST_BASE_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def modifyBayId(tlv, board, new_bay_id):
    """修改指定节点的柜ID

    :param tlv: tlv对象
    :param board: 指定节点
    :param new_bay_id: 修改后的柜ID
    :return:
    """
    new_bay_id = "%s%s" % ("SMB", new_bay_id)
    params = []
    param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["BAY"])
    params.append(param)
    param = (tlvData.PUB_ATTR["id"], new_bay_id)
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_BAY_CONFIG"]
    obj = tlvData.OM_OBJ_E["BAY"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def modifyCtrlNum(tlv, board, ctrl_num):
    """修改指定节点的控制器数量

    :param tlv: tlv对象
    :param board: 指定节点
    :param ctrl_num: 修改后的控制器数量
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["BAY"])
    params.append(param)
    param = (tlvData.BAY["contrlTotal"], int(ctrl_num))
    params.append(param)
    param = (tlvData.BAY["isExpansionController"],
             int(restData.IsExpansionController.IS_NOT_EXPANSION_CONTROLLER))
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_BAY_CONFIG"]
    obj = tlvData.OM_OBJ_E["BAY"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def modifyBayIDCtrlNum(tlv, board, new_bay_id, ctrl_num):
    """修改指定节点的柜ID和控制器数量

    :param tlv: tlv对象
    :param board: 指定节点
    :param new_bay_id: 修改后的柜ID
    :param ctrl_num: 修改后的控制器数量
    :return:
    """
    new_bay_id = "%s%s" % ("SMB", new_bay_id)
    params = []
    param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["BAY"])
    params.append(param)
    param = (tlvData.PUB_ATTR["id"], new_bay_id)
    params.append(param)
    param = (tlvData.BAY["contrlTotal"], int(ctrl_num))
    params.append(param)
    param = (tlvData.BAY["isExpansionController"],
             int(restData.IsExpansionController.IS_EXPANSION_CONTROLLER))
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_BAY_CONFIG"]
    obj = tlvData.OM_OBJ_E["BAY"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def changeBMC(tlv, board):
    """修改指定节点的BMC信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"],
             tlvData.BMC_CHANGE_MODE_E["change"])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHANGE_BMC_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def deleteBMC(tlv, board):
    """清除指定节点的BMC信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"],
             tlvData.BMC_CHANGE_MODE_E["delete"])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHANGE_BMC_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def syncBMC(tlv, board):
    """同步指定节点的BMC信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"],
             tlvData.BMC_CHANGE_MODE_E["sync"])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHANGE_BMC_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def clearConfig(tlv, board):
    """清除待扩容控制器上的配置

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    # Dorado支持强制清除强制清除配置标记，其他版本无影响
    param = (tlvData.IMPORT_CONFIG_DATA["CMO_FORCE_CLEAR_DB_FLAG"],
             tlvData.FORCE_CLEAR_DB_FLAG["FORCE_CLEAR_DB"])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["SYS_CLEAR_CONFIG"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    record = exeCmd(tlv, params, ad_obj_param_dict, timeOut=600)
    if record is None:
        return None

    return record


def stopFeedDog(tlv, board, ctrls):
    """重启待扩容集群

    :param tlv: tlv对象
    :param board: 指定节点
    :param ctrls: 待扩容集群控制器数量
    :return:
    """
    params = []
    param = (tlvData.STOP_FEED_DEFINE['bitMap'], tlvUtil.getBitMap(ctrls))
    params.append(param)
    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["STOP_FEED"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    record = exeCmd(tlv, params, ad_obj_param_dict, timeOut=70, reTry=False)
    if record is None:
        return None

    return record


def modifyXnetInfoToFixedValue(tlv, board, bay_config_info):
    """修改指定节点的xnet部分信息为特定值

    :param tlv: tlv对象
    :param board: 指定节点
    :param bay_config_info: 配置信息
    :return:
    """
    params = []
    param = (tlvData.IP_CLUST_BASE_INFO["baseIpAddr"],
             tlvUtil.getRecordValue(bay_config_info,
                                    tlvData.IP_CLUST_BASE_INFO["baseIpAddr"]))
    params.append(param)
    param = (
        tlvData.IP_CLUST_BASE_INFO["netMode"],
        tlvUtil.getRecordValue(bay_config_info,
                               tlvData.IP_CLUST_BASE_INFO["netMode"]))
    params.append(param)
    for ctrl_id in ["a", "b"]:
        for x in range(1, 5):
            param_port = tlvData.IP_CLUST_BASE_INFO[
                "ctr%sPortID%s" % (ctrl_id, x)]
            param_dsw = tlvData.IP_CLUST_BASE_INFO["ctr%sDswID%s" %
                                                   (ctrl_id, x)]
            param = (param_port,
                     long(tlvUtil.getRecordValue(bay_config_info, param_port)))
            params.append(param)
            param = (param_dsw,
                     int(tlvUtil.getRecordValue(bay_config_info, param_dsw)))
            params.append(param)

    for ctrl_id in ["c", "d"]:
        for x in range(3, 5):
            param_port = tlvData.IP_CLUST_BASE_INFO[
                "ctr%sPortID%s" % (ctrl_id, x)]
            param_dsw = tlvData.IP_CLUST_BASE_INFO["ctr%sDswID%s" %
                                                   (ctrl_id, x)]
            param = (param_port, restData.REST_INVALID_VALUE)
            params.append(param)
            param = (param_dsw, restData.REST_INVALID_VALUE)
            params.append(param)
    param0 = (tlvData.IP_CLUST_BASE_INFO["ctrcPortID1"], long(33686272))
    param1 = (tlvData.IP_CLUST_BASE_INFO["ctrcDswID1"], int(0))
    param2 = (tlvData.IP_CLUST_BASE_INFO["ctrcPortID2"], long(33686273))
    param3 = (tlvData.IP_CLUST_BASE_INFO["ctrcDswID2"], int(1))
    param4 = (tlvData.IP_CLUST_BASE_INFO["ctrdPortID1"], long(50463488))
    param5 = (tlvData.IP_CLUST_BASE_INFO["ctrdDswID1"], int(0))
    param6 = (tlvData.IP_CLUST_BASE_INFO["ctrdPortID2"], long(50463489))
    param7 = (tlvData.IP_CLUST_BASE_INFO["ctrdDswID2"], int(1))
    params.append(param0)
    params.append(param1)
    params.append(param2)
    params.append(param3)
    params.append(param4)
    params.append(param5)
    params.append(param6)
    params.append(param7)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_IP_CLUST_BASE_INFO"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    return exeCmd(tlv, params, ad_obj_param_dict)


def checkInnterIP(logger, tlv, board, innter_ip_info):
    """判断指定节点的内部IP是否冲突

    :param logger: 日志对象
    :param tlv: tlv对象
    :param board: 指定节点
    :param innter_ip_info:
    :return:
    """
    innter_ip_net0_addr = innter_ip_info.get('innterIpNet0Addr')
    innter_ip_net1_addr = innter_ip_info.get('innterIpNet1Addr')
    subnet_mask = innter_ip_info.get('subnetMask')
    params = []
    param0 = (
        tlvData.CONTROLLER_INNER_IP['innterIpNet0Addr'], innter_ip_net0_addr)
    param1 = (
        tlvData.CONTROLLER_INNER_IP['innterIpNet1Addr'], innter_ip_net1_addr)
    param2 = (tlvData.CONTROLLER_INNER_IP['mask_0'], subnet_mask)
    param3 = (tlvData.CONTROLLER_INNER_IP['mask_1'], subnet_mask)

    params.append(param0)
    params.append(param1)
    params.append(param2)
    params.append(param3)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHECK_INNER_IP"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict)


def modifyInnterIP(tlv, board, innter_ip_info):
    """修改指定节点的内部IP为特定值

    :param tlv: tlv对象
    :param board: 指定节点
    :param innter_ip_info: 内部IP信息
    :return:
    """
    innter_ip_net0_addr = innter_ip_info.get('innterIpNet0Addr')
    innter_ip_net1_addr = innter_ip_info.get('innterIpNet1Addr')
    params = []
    param0 = (tlvData.CONTROLLER_INNER_IP['innterIpNet0Addr'],
              innter_ip_net0_addr)
    param1 = (tlvData.CONTROLLER_INNER_IP['innterIpNet1Addr'],
              innter_ip_net1_addr)

    params.append(param0)
    params.append(param1)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY_INNER_IP"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict)


def getPCIePortsOfIntfCardRecords(tlv, board):
    """获取指定节点上的所有PCIE接口卡上的端口信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param0 = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
    param1 = (tlvData.PUB_ATTR['parentType'], tlvData.OM_OBJ_E['INTF_MODULE'])
    params.append(param0)
    params.append(param1)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["PCIE_PORT"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getPCIePortRecords(tlv, board):
    """获取指定节点上的所有ETH端口记录

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["PCIE_PORT"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getBayIdInfo(tlv, new_boards_list):
    """从PCI端口信息中获取指定节点BayID信息

    :param tlv: tlv对象
    :param new_boards_list: 指定节点列表
    :return:
    """
    bay_id_info = {}
    intf_id_list = []

    for board in new_boards_list:
        intf_records = getScaleOutPCIeIntfRecords(tlv, board)
        if not intf_records:
            continue
        for intf_record in intf_records:
            intf_id = tlvUtil.getRecordValue(intf_record,
                                             tlvData.PUB_ATTR['id'])
            intf_id_list.append(intf_id)

    for board in new_boards_list:
        port_records = getPCIePortsOfIntfCardRecords(tlv, board)
        if not port_records:
            continue
        for record in port_records:
            bay_id = tlvUtil.getRecordValue(record, tlvData.PCIE_PORT[
                "suggestEngineId"])
            intf_id = tlvUtil.getRecordValue(record,
                                             tlvData.PUB_ATTR['parentID'])

            if intf_id in intf_id_list:
                bay_id_info[board["enclosureSN"]] = str(bay_id)

    return bay_id_info


def getScaleOutPCIeIntfRecords(tlv, board):
    """过滤父类型为pcie3.0的接口卡

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param0 = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['INTF_MODULE'])
    param1 = (tlvData.INTF_MODULE['model'], tlvData.INTF_MODEL_E['PCIe_2X'])

    params.append(param0)
    params.append(param1)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]

    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["INTF_MODULE"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getPCIePortInfoForDorado(tlv, board, ctrl_height, new_config_clust_type):
    """获取指定节点PCIe端口信息

    :param tlv: tlv对象
    :param board: 指定节点
    :param ctrl_height: 框高
    :param new_config_clust_type:集群扩容方式
    :return:
    """
    records = getPCIePortRecords(tlv, board)
    if records is None:
        return []
    eth_port_info = []
    for record in records:
        location = getRecordValue(record, tlvData.PUB_ATTR['location'])

        intf_module_id = getRecordValue(record, tlvData.PUB_ATTR['parentID'])
        if not isScaleOutIntfModule(intf_module_id, ctrl_height):
            continue
        ctrl_name = intf_module_id.split(".")[0]

        # 交换机组网场景，端口仅连接在MGMT和MAINTENANCE端口
        if new_config_clust_type == common.CLUST_TYPE_SWITCH:
            if not (location.endswith("P0") or location.endswith("P1")):
                continue

        port_info_dict = {}
        port_info_dict["portID"] = getRecordValue(record,
                                                  tlvData.PUB_ATTR['id'])
        port_info_dict["location"] = location
        port_info_dict["healthStatus"] = getRecordValue(record,
                                                        tlvData.PUB_ATTR[
                                                              'healthStatus'])
        port_info_dict["runningStatus"] = getRecordValue(
            record,
            tlvData.PUB_ATTR['runningStatus'])
        port_info_dict["dswId"] = getRecordValue(record,
                                                 tlvData.PCIE_PORT['dswId'])
        port_info_dict["dswPortIndex"] = getRecordValue(record,
                                                        tlvData.PCIE_PORT[
                                                              'dswPortIndex'])
        port_info_dict["ctrlId"] = getRecordValue(
            record,
            tlvData.PCIE_PORT['ctrlId'])
        port_info_dict["suggestEngineId"] = getRecordValue(
            record,
            tlvData.PCIE_PORT['suggestEngineId'])
        port_info_dict["enclosureSN"] = board["enclosureSN"]
        port_info_dict["bayid"] = int(ctrl_name[0:-1])
        port_info_dict["nodeName"] = ctrl_name[-1]
        port_info_dict["portMark"] = location[-1]

        eth_port_info.append(port_info_dict.copy())
    eth_port_info.sort(lambda x, y: common.cmpPortDict(x, y))
    return eth_port_info


def isExistHighDensityEnc(context, board):
    """检查是否存在4U SAS 75盘级联框

    :param context: 上下文
    :param board: 指定节点
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    tlv = contextUtil.getTlv(context)

    is_exist = False
    high_density_enc_list = []

    recs = getEnclosureRecords(tlv, board)
    if recs is None or len(recs) == 0:
        return is_exist, high_density_enc_list

    for rec in recs:
        model = tlvUtil.getRecordValue(rec, tlvData.ENCLOSURE['model'])
        name = tlvUtil.getRecordValue(rec, tlvData.PUB_ATTR["name"])
        if model == tlvData.ENCLOSURE_MODEL_E["EXPSAS4U_75"]:
            is_exist = True
            high_density_enc_list.append(name)

    logger.logInfo(
        "The High-Density Disk Enc list is %s." % str(high_density_enc_list))
    return is_exist, high_density_enc_list


def getFilterAlarms(tlv, board, filter_alarm_list):
    """获取指定节点的特定告警信息

    :param tlv: tlv对象
    :param board: 指定节点
    :param filter_alarm_list: 指定的告警ID列表
    :return:
    """
    params = []

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_CURRENT_ALARM"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    records = exeCmd(tlv, params, ad_obj_param_dict, getBatch=True)
    if records is None or len(records) == 0:
        return []
    alarm_list = []
    for record in records:
        if len(record) == 0:
            continue
        if tlvUtil.TLV_RECORD_END_INDEX in record.keys():
            return alarm_list
        alarm_dict = {}
        alarm_dict["CMO_ALARM_ID"] = dec2hex(
            getRecordValue(record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_ID']))
        if alarm_dict["CMO_ALARM_ID"] not in filter_alarm_list:
            continue
        alarm_dict["CMO_ALARM_PARMETER"] = getRecordValue(
            record,
            tlvData.CMO_ALARM_INFO_E['CMO_ALARM_PARMETER'])
        alarm_list.append(alarm_dict.copy())
    return alarm_list


def checkManIPConflictAlarm(tlv, board,
                            mgmtPorts=common.MGMT_PORT_CONFIGMANIP):
    """获取指定节点，指定端口上的IP冲突端口信息

    :param tlv: tlv对象
    :param board: 指定节点
    :param mgmt_ports: 管理口列表
    :return:
    """
    conflict_alarm_list = []
    alarm_list = get_important_and_urgent_alarms(tlv, board)
    for line in alarm_list:
        if line["CMO_ALARM_CLEAR_TIME"] != 0:
            continue
        if line["CMO_ALARM_RECOVERY_TIME"] != 0:
            continue
        if line["CMO_ALARM_ID"] not in common.CHECK_MANIP_CONFLICT:
            continue

        alarm_info = getConflictAlarmInfo(line)
        port_location = alarm_info.get("portLocation", None)
        if port_location in mgmtPorts:
            conflict_alarm_list.append(alarm_info)

    if len(conflict_alarm_list) > 0:
        return (False, conflict_alarm_list)
    return (True, [])


def checkConflictRecovery(tlv, board, port_info):
    """检查指定端口的IP冲突告警是否恢复

    :param tlv: tlv对象
    :param board: 指定节点
    :param port_info: 端口信息
    :return:
    """
    alarm_list = getSystemAlarm(tlv, board)
    port_id = port_info["portLocation"]
    for line in alarm_list:
        if line["CMO_ALARM_CLEAR_TIME"] != 0:
            continue
        if line["CMO_ALARM_RECOVERY_TIME"] != 0:
            continue
        if line["CMO_ALARM_ID"] not in common.CHECK_MANIP_CONFLICT:
            continue

        alarm_info = getConflictAlarmInfo(line)
        port_location = alarm_info.get("portLocation", None)
        port_ip_type = common.IPV4 if common.isIpV4(
            alarm_info.get("portIp")) else common.IPV6
        ip_ver = common.IPV4 if common.isIpV4(
            port_info.get("portIp")) else common.IPV6
        if port_location == port_id and port_ip_type == ip_ver:
            return (False, line)

    return (True, {})


def setIpv4(tlv, record_info, man_ip_info, board):
    """设置IPV4地址

    :param tlv:
    :param record_info:
    :param man_ip_info:
    :param board:
    :return:
    """
    id = getRecordValue(record_info, tlvData.PUB_ATTR['id'])
    params = []
    param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["ETH_PORT"])
    params.append(param)
    param = (tlvData.PUB_ATTR["id"], id)
    params.append(param)
    param = (tlvData.ETH_PORT["ipv4Addr"], man_ip_info["manIp"])
    params.append(param)
    param = (tlvData.ETH_PORT["ipv4Mask"], man_ip_info["mask"])
    params.append(param)
    gateway = man_ip_info["gateway"]
    if gateway is not None and gateway != "":
        param = (tlvData.ETH_PORT["ipv4Gateway"], gateway)
        params.append(param)
    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY"]
    obj = tlvData.OM_OBJ_E["ETH_PORT"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    exeCmd(tlv, params, ad_obj_param_dict)
    return


def setIpv6(tlv, record_info, man_ip_info, board):
    """设置IPV6地址

    :param tlv:
    :param record_info:
    :param man_ip_info:
    :param board:
    :return:
    """
    id = getRecordValue(record_info, tlvData.PUB_ATTR['id'])
    params = []
    param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["ETH_PORT"])
    params.append(param)
    param = (tlvData.PUB_ATTR["id"], id)
    params.append(param)
    param = (tlvData.ETH_PORT["ipv6Addr"], man_ip_info["manIp"])
    params.append(param)
    param = (tlvData.ETH_PORT["ipv6Mask"], man_ip_info["mask"])
    params.append(param)
    gateway = man_ip_info["gateway"]
    if gateway is not None and gateway != "":
        param = (tlvData.ETH_PORT["ipv6Gateway"], gateway)
        params.append(param)
    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["MODIFY"]
    obj = tlvData.OM_OBJ_E["ETH_PORT"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    exeCmd(tlv, params, ad_obj_param_dict)
    return


def getConflictAlarmInfo(alarm_dict):
    """获取管理ip冲突告警的端口信息，ip地址

    :param alarm_dict: 告警字典
    :return:
    """
    alarm_info = {}

    param_info = alarm_dict['CMO_ALARM_PARMETER']
    params = param_info.split(",")
    if len(params) > 2:
        alarm_info["portLocation"] = params[1]
        alarm_info["portIp"] = params[2]
        return alarm_info

    return {}


def getCtrlEncModel(tlv, board, logger):
    """获取指定框的框model

    :param tlv: tlv对象
    :param board: 指定节点
    :param logger: 日志对象
    :return:
    """
    enclosure_records = getEnclosureRecords(tlv, board)
    if enclosure_records is None:
        logger.logInfo("record is none")
        return None

    for record in enclosure_records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logic_type == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            model = getRecordValue(record, tlvData.ENCLOSURE['model'])
            return model

    logger.logInfo("ctrl enclosure is not existed")
    return None


def restartAD(context, tlv, logger, ctrl_num):
    """ 重新下发AD命令

    :param context: 上下文
    :param tlv: tlv连接
    :param logger: 日志
    :param ctrl_num: 原集群控制器数量
    :return: param1: 执行结果：True-执行成功，Flase-执行失败
             param2：错误码
    """
    err_code = 0
    is_ad_started = contextUtil.getItem(context, "isADStarted", False)
    if not is_ad_started:
        try:
            # 启动AD
            started_ad_num = startAD(tlv)
            if started_ad_num < ctrl_num:
                logger.logNoPass(
                    "start AD failure(startedADNum:%s, ctrlNum:%s)" % (
                        started_ad_num, ctrl_num))
                return False, err_code
            else:
                contextUtil.setItem(context, "isADStarted", True)
                logger.logInfo("AD started")
        except Exception as ex:
            logger.logInfo("AD started exception.")
            if len(ex.args) > 1:
                try:
                    err_code = int(ex.args[0])
                except Exception as e:
                    logger.logInfo("no error code: %s." % str(e))
            return False, err_code

        # 通知AD执行扩容
        try:
            notify_ad_ret = notifyAD(tlv)
            if notify_ad_ret == 0:
                logger.logNoPass("notify AD failure")
                return False, err_code
            else:
                logger.logInfo("notify AD success")
        except Exception as ex:
            logger.logInfo("Notify AD exception.")
            if len(ex.args) > 1:
                try:
                    err_code = int(ex.args[0])
                except Exception as e:
                    logger.logInfo("no error code: %s." % str(e))
            return False, err_code

        # 扫描节点信息
        try:
            orgin_boards_list = getBoards(tlv)
            logger.logInfo("get boards result:%s" % orgin_boards_list)
            new_config_ctrl_num = contextUtil.getItem(context,
                                                      "newConfigCtrlNum")
            if len(orgin_boards_list) < new_config_ctrl_num:
                return False, err_code
        except Exception as ex:
            logger.logInfo("get boards exception.")
            if len(ex.args) > 1:
                try:
                    err_code = int(ex.args[0])
                except Exception as e:
                    logger.logInfo("no error code: %s." % str(e))
            return False, err_code

        # 刷新每个节点的路由信息表
        try:
            # 刷新每个节点的路由信息表
            logger.logInfo("start refresh routine info")
            getRouteInfoList(tlv)
            logger.logInfo("refresh routine info success.")
        except Exception as ex:
            logger.logInfo("refresh routine info exception.")
            if len(ex.args) > 1:
                try:
                    err_code = int(ex.args[0])
                except Exception as e:
                    logger.logInfo("no error code: %s." % str(e))
            return False, err_code

    return True, err_code


def preExpansion(tlv, new_ctrl_num, logger):
    """扩容前预处理

    :param tlv: tlv对象
    :param new_ctrl_num: 新扩控制器数量
    :param logger:日志对象
    :return:
    """
    try:
        # 打开原集群AD模块
        started_ad_num = startAD(tlv)
        logger.logInfo("start ad process result: %s" % unicode(started_ad_num))

        # 下发是否扩容配置
        notify_ad_ret = notifyAD(tlv)
        logger.logInfo("notify expansion result: %s" % unicode(notify_ad_ret))

        # 查询框和节点，更新ad通信路由
        orgin_boards_list = getBoards(tlv)
        logger.logInfo("update route result: %s" % unicode(orgin_boards_list))

        # 刷新每个节点的路由信息表
        refreshRouteInfo(tlv, logger)
    except Exception as e:
        logger.logInfo("prepare expansion exception: %s." % str(e))


def disableMgmtForward(tlv, board, bay_id, new_ctrl_num):
    """禁头尾转发

    :param tlv: tlv对象
    :param board: 指定节点
    :param bay_id: 柜ID
    :param new_ctrl_num:新扩控制器数目
    :return:
    """
    params = []
    param = (tlvData.SCALE_OUT["BAY_ID"], bay_id)
    params.append(param)

    param = (tlvData.SCALE_OUT["NEW_NODE_NUM"], new_ctrl_num)
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["DISABLE_MGMT_PORT_FORWARD"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, 0)

    records = exeCmd(tlv, params, ad_obj_param_dict,
                     isBatch=True, timeOut=50)
    return records


def addInnerIp(tlv, board, bay_id, ip_add1, ip_add2):
    """禁头尾转发

    :param tlv: tlv对象
    :param board: 指定节点
    :param bay_id: 柜ID
    :param ip_add1: IP基地址1
    :param ip_add2: IP基地址2
    :return:
    """
    params = []
    param = (tlvData.SCALE_OUT["BAY_ID"], bay_id)
    params.append(param)

    param = (tlvData.SCALE_OUT["BASE_IP_ADD_1"], ip_add1)
    params.append(param)

    param = (tlvData.SCALE_OUT["BASE_IP_ADD_2"], ip_add2)
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["ADD_INNER_MGMT_IP"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, 0)

    records = exeCmd(tlv, params, ad_obj_param_dict, isBatch=True,
                     timeOut=50)
    return records


def testMgmtCable(tlv, board, bay_id, old_ctrl_num, new_ctrl_num,
                  net_work_mode, is_dorado=False):
    """扩控前线缆检测

    :param tlv: tlv对象
    :param board: 节点
    :param bay_id: 引擎ID
    :param old_ctrl_num: 扩容前控制器数量
    :param new_ctrl_num: 扩容后控制器数量
    :param net_work_mode: 扩容组网方式
    :param is_dorado: 是否是Dorado
    :return:
    """
    params = []
    param = (tlvData.SCALE_OUT["BAY_ID"], bay_id)
    params.append(param)

    param = (tlvData.SCALE_OUT["OLD_NODE_NUM"], old_ctrl_num)
    params.append(param)

    param = (tlvData.SCALE_OUT["NEW_NODE_NUM"], new_ctrl_num)
    params.append(param)

    param = (tlvData.SCALE_OUT["NET_WORK_MODE"], net_work_mode)
    params.append(param)

    adObjParamDict = {}
    adObjParamDict["enclosureSN"] = board["enclosureSN"]
    adObjParamDict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["TEST_MGMT_NETWORK_CABLE"]
    adObjParamDict["cmd"] = cmd
    obj_key = "V6" if is_dorado else 0
    adObjParamDict["pid"] = getPid(cmd, obj_key)

    records = exeCmd(tlv, params, adObjParamDict, isBatch=True, timeOut=50,
                     retry_interval=150)
    return records


def getDiskPoolRecords(tlv, board):
    """获取指定节点上的所有存储池记录

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['DISKPOOL'])
    params.append(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["DISKPOOL"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)
    return exeCmd(tlv, params, ad_obj_param_dict, isBatch=True)


def getDisEnclosureList(tlv, board):
    """获取硬盘框信息

    :param tlv: tlv对象
    :param board: 指定节点
    :return:
    """
    params = []
    param0 = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param0)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    cmd = tlvData.CMD["GET_BATCH_NEXT"]
    obj = tlvData.OM_OBJ_E["ENCLOSURE"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["obj"] = obj
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    enclosure_records = exeCmd(tlv, params, ad_obj_param_dict, isBatch=True,
                               timeOut=50)

    enclosure_list = []
    if enclosure_records is None or len(enclosure_records) == 0:
        return enclosure_list

    for record in enclosure_records:
        logic_type = getRecordValue(record, tlvData.ENCLOSURE["logicType"])
        if logic_type == tlvData.ENCLOSURE_TYPE_E['EXP']:
            enclosure_info = {
                "enclosureSN": board["enclosureSN"],
                "location": getRecordValue(record,
                                           tlvData.PUB_ATTR["location"]),
                "name": getRecordValue(record, tlvData.PUB_ATTR["name"]),
                "macAddress": getRecordValue(record, tlvData.ENCLOSURE[
                    "macAddress"]),
                "serialNum": getRecordValue(record,
                                            tlvData.ENCLOSURE["serialNum"])}
            enclosure_list.append(enclosure_info)

    return enclosure_list


def get_disk_certificate(tlv, board):
    """获取新扩控上的硬盘认证证书信息

    :param tlv: tlv对象
    :param board: 节点
    :return:
    """
    params = []
    ad_obj_param_dict = dict()
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_DISK_CERTIFICATE"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, 0)

    records = exeCmd(tlv, params, ad_obj_param_dict, isBatch=True, timeOut=50)
    return records


def check_disk_certificate(tlv, board, certificate_label, certificate_content):
    """检查原集群是否包含指定的硬盘认证证书

    :param tlv: tlv对象
    :param board: 节点
    :param certificate_label: 硬盘证书标签
    :param certificate_content: 硬盘证书内容
    :return:
    """
    params = []

    param0 = (tlvData.DISK_CERTIFICATE["LABEL_OF_CERTIFICATE"],
              certificate_label)
    params.append(param0)

    param1 = (tlvData.DISK_CERTIFICATE["CONTENT_OF_CERTIFICATE"],
              certificate_content)
    params.append(param1)

    ad_obj_param_dict = dict()
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["CHECK_DISK_CERTIFICATE"]
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, 0)

    records = exeCmd(tlv, params, ad_obj_param_dict, isBatch=False, timeOut=50)
    return records


def get_system_alarm_by_levels(tlv, board, levels=[]):
    """根据告警级别获取指定节点的告警

    :param tlv: tlv连接
    :param board: 指定节点
    :param levels: 告警级别列表
    :return:
    """
    alarm_list = []
    if not levels:
        return alarm_list
    for level in levels:
        alarm_list.extend(get_system_alarm_by_level(tlv, board, level))
    return alarm_list


def get_system_alarm_by_level(tlv, board, level):
    """根据告警级别获取指定节点的告警

    :param tlv: tlv连接
    :param board: 指定节点
    :param level: 告警级别
    :return:
    """
    condition = [{
        "type": "EQUAL",
        "index": TlvEnum.SysMgmt.CMO_ALARM_INFO_E['CMO_ALARM_LEVEL'],
        "value": level
    }]
    condition_str = Tlv2Rest.getQueryConditionString(condition, None, None)
    param = (restData.PublicAttributesExtend.QUERY_CONDITION, condition_str)
    params = Tlv2Rest.getParamList(param)

    ad_obj_param_dict = {}
    ad_obj_param_dict["enclosureSN"] = board["enclosureSN"]
    ad_obj_param_dict["controllerID"] = board["controllerID"]
    cmd = tlvData.CMD["GET_CURRENT_ALARM"]
    obj = 0
    ad_obj_param_dict["cmd"] = cmd
    ad_obj_param_dict["pid"] = getPid(cmd, obj)

    records = exeCmd(tlv, params, ad_obj_param_dict, getBatch=True)
    return get_alarm_from_records(records)


def get_alarm_from_records(records):
    """从tlv结果中解析alarm

    :param records: tlv结果
    :return:
    """
    alarm_list = []
    if not records:
        return alarm_list
    for record in records:
        if len(record) == 0:
            continue
        if tlvUtil.TLV_RECORD_END_INDEX in record.keys():
            return alarm_list
        alarm_dict = {}
        alarm_dict["CMO_ALARM_TYPE"] = getRecordValue(
            record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_TYPE'])
        alarm_dict["CMO_ALARM_ID"] = dec2hex(getRecordValue(
            record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_ID']))
        alarm_dict["CMO_ALARM_LEVEL"] = getRecordValue(
            record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_LEVEL'])
        alarm_dict["CMO_ALARM_CLEAR_TIME"] = getRecordValue(
            record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_CLEAR_TIME'])
        alarm_dict["CMO_ALARM_RECOVERY_TIME"] = getRecordValue(
            record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_RECOVERY_TIME'])
        alarm_dict["CMO_ALARM_PARMETER"] = getRecordValue(
            record, tlvData.CMO_ALARM_INFO_E['CMO_ALARM_PARMETER'])
        alarm_list.append(alarm_dict.copy())
    return alarm_list


def get_important_and_urgent_alarms(tlv, board):
    """获取指定节点的重要和紧急告警

    :param tlv: tlv连接
    :param board: 指定节点
    :return:
    """
    levels = [
        tlvData.CMO_ALARM_LEVEL_E.get("important"),
        tlvData.CMO_ALARM_LEVEL_E.get("urgent")
    ]
    return get_system_alarm_by_levels(tlv, board, levels)


def get_scm_disk(tlv, board):
    """
    获取指定节点上的SCM盘信息
    :param tlv:
    :param board:
    :return:
    """
    records = getDiskRecords(tlv, board)
    if not records:
        return []
    scm_list = []
    for record in records:
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        disk_type = tlvUtil.getRecordValue(record, tlvData.DISK["diskType"])
        if location.startswith("CTE") and str(disk_type) == "17":
            scm_list.append(location)
    return scm_list
