# coding:utf-8
"""
@version: Toolkit V200R006C00
@time: 2019/10/28
@file: sqlite_util.py
@function:
@modify:
"""
import os
import time
import codecs

import com.huawei.ism.tool.obase.db.SqliteManager as SqliteManager
import java.lang.management.ManagementFactory as ManagementFactory
from cbb.frame.cli import sqlite_config
from cbb.frame.base.funcUtils import wrap_all_exception_logged


def create_table(conn, sql=sqlite_config.DEFAULT_CREATE_TABLE_SQL):
    """
    创建表
    :param conn:
    :param sql:
    :return:
    """
    SqliteManager().update(conn, sql)


def drop_table(conn, sql=sqlite_config.DEFAULT_DROP_TABLE_SQL):
    """
    删除表
    :param conn:
    :param sql:
    :return:
    """
    SqliteManager().update(conn, sql)


def create_new_conn(db_path):
    """
    创建连接
    :param db_path:
    :return:
    """
    sqlite_manager = SqliteManager()
    sqlite_conn = sqlite_manager.getConnection(db_path)
    return sqlite_conn


def ini_sqlite_db_for_dev(context, dev_sn, logger=None):
    """
    创建数据库文件、创建连接、存入缓存等动作
    :param context:
    :param dev_sn:
    :param logger:
    :return:
    """
    tmp_dict = context.get(sqlite_config.SQLITE_CONN_DICT.format(dev_sn))
    _log_info(logger, "SQLITE_CONN_DICT %s!" % str(tmp_dict))
    if not tmp_dict:
        tmp_dict = {}

    # 是否已创建数据库，用于区分双活场景requirecheck.py中已创建，
    # preinspect.py中重复创建,会覆盖掉已缓存数据。
    dev_dict = tmp_dict.get(dev_sn, {})
    _log_info(logger, "dev_dict in context is :%s!" % str(dev_dict))
    if dev_dict.get(sqlite_config.SQLITE_DB_CREATED_FLAG, False) is True:
        return dev_dict.get(sqlite_config.SQLITE_CONN)

    db_name = _get_db_name(dev_sn)
    db_path = create_sqlite_db_file(db_name)
    if not db_path:
        _log_info(logger, "create db file failed, db_path:%s" % db_path)
        return None

    _log_info(logger, "create db file success, db_path:%s" % db_path)

    conn = create_new_conn(db_path)
    _log_info(logger, "create conn success, dev_sn:%s" % dev_sn)

    create_table(conn)

    create_index_for_db(conn)

    _log_info(logger, "create table success, dev_sn:%s" % dev_sn)

    tmp_dict[dev_sn] = {
        sqlite_config.SQLITE_CONN: conn,
        sqlite_config.SQLITE_CONN_DB_PATH: db_path,
        sqlite_config.SQLITE_DB_CREATED_FLAG: True
    }
    context.put(sqlite_config.SQLITE_CONN_DICT.format(dev_sn), tmp_dict)
    return conn


def create_index_for_db(conn):
    """
    创建数据库索引
    :param conn:数据库连接
    :return:
    """
    sql_str = "CREATE INDEX [2] ON [table_array_info]([cmd],[sn]);"
    SqliteManager().update(conn, sql_str)


def tear_down_for_dev(context, sn, logger):
    """
    收拾残局，清除数据库连接、删除数据库db等动作。
    :param context:
    :param sn:
    :param logger:
    :return:
    """
    sqlite_conn_dict = context.get(sqlite_config.SQLITE_CONN_DICT.format(sn))
    if not sqlite_conn_dict:
        logger.logInfo("not use db. no need to close.")
        return

    db_conn = sqlite_conn_dict.get(sn, {}).get(sqlite_config.SQLITE_CONN)

    # 清空数据库
    drop_table(db_conn)

    # 关闭连接
    SqliteManager().closeConnectionSafely(db_conn)
    logger.logInfo("close conn successfully. sn[%s]" % sn)

    # 删除数据库文件
    db_path = sqlite_conn_dict.get(sn, {}).get(
        sqlite_config.SQLITE_CONN_DB_PATH)

    sqlite_conn_dict[sn] = {}

    context.put(sqlite_config.SQLITE_CONN_DICT.format(sn), sqlite_conn_dict)

    for i in range(1, 4):
        flag = delete_db_file(db_path)
        if flag is True:
            logger.logInfo(
                "delete db file success in times:%s.sn[%s],path:%s" % (
                    str(i), sn, db_path))
            break

        time.sleep(1)
        logger.logInfo(
            "delete db file except in times:%s.sn[%s],path:%s" % (
                str(i), sn, db_path))


@wrap_all_exception_logged(logger=None)
def delete_db_file(db_path):
    """
    删除数据库文件
    :param db_path:
    :return:
    """
    if not db_path:
        return True

    if not os.path.isfile(db_path):
        return True

    os.remove(db_path)
    return True


def _get_db_name(dev_sn=None):
    """
    私有方法，如果传入sn是以设备维度，否则使用单数据库。
    :param dev_sn:
    :return:
    """
    if not dev_sn:
        return sqlite_config.SQLITE_DEFAULT_DB_NAME

    return "array_info_data_" + dev_sn + ".db"


@wrap_all_exception_logged(logger=None)
def create_sqlite_db_file(db_name=sqlite_config.SQLITE_DEFAULT_DB_NAME):
    """
    根据名称创建数据库文件，并返回数据库路径。
    :param db_name: 默认使用array_info_data.db
    :return: 数据库绝对路径，用于创建连接
    """
    base_path = sqlite_config.SQLITE_DB_ABS_PATH

    # 在公共目录下增加DB管理目录
    base_path = os.path.join(base_path,
                             sqlite_config.SQLITE_DB_ABS_PATH_DEV_DIR)

    # 多个工具同时使用时，使用进程ID区分。
    base_path = os.path.join(base_path, get_current_pid())
    if not os.path.exists(base_path):
        os.makedirs(base_path)

    file_name = os.path.join(base_path, db_name)
    if not os.path.exists(file_name):
        codecs.open(file_name, 'w', encoding='utf-8').close()

    return file_name


def _log_info(logger, info):
    """
    日志记录
    :param logger:日志对象
    :param info: 日志
    :return:无
    """
    if logger:
        logger.logInfo(info)


def get_current_pid():
    """
    获取当前PID，参考InspectUtils.getCurrentPid()实现。
    pid name is:24032@xx
    :return:当前进程ID
    """
    name = ManagementFactory.getRuntimeMXBean().getName()
    pid = name.split("@")[0]
    return str(pid)
