# -*- coding:utf-8 -*-
import re

from cbb.frame.cli import cliUtil
from cbb.frame.base.exception import UnCheckException

RISK_VERSION = re.compile("6.0.*")
SAFETY_HOT_PATCH = 6
RISK_ZOMBIE_NUMBER = 20


class CheckZombieProcess:
    """
    系统进程状态检查
    """

    def __init__(self, cli, lang, env, logger, all_ret_list):
        self.cli = cli
        self.lang = lang
        self.env = env
        self.logger = logger
        self.all_ret_list = all_ret_list
        self.software_version = ""
        self.hot_patch_version = ""

    def execute_check(self):
        """
        当前节点僵尸进程统计，大于等于20则判断有风险
        :return:
        """
        pid_list = self.get_pid_list()
        self.logger.logInfo(
            "pid_list:{}".format(",".join(pid_list)))
        if not pid_list:
            return True
        defunct_ppid_list = self.get_defunct_ppid_list()
        self.logger.logInfo(
            "defunct_ppid_list:{}".format(",".join(defunct_ppid_list)))
        if not defunct_ppid_list:
            return True
        zombie_number = 0
        for ppid in defunct_ppid_list:
            if ppid in pid_list:
                zombie_number += 1
        if zombie_number >= RISK_ZOMBIE_NUMBER:
            return False
        return True

    def get_pid_list(self):
        """
        执行ps -C perf 获取PID
        :return:
        """
        cmd = "ps -C perf"
        flag, ret, msg = cliUtil.excuteCmdInMinisystemModel(
            self.cli, cmd, self.lang
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise UnCheckException(msg)
        pid_list = []
        for pid_info in ret.splitlines():
            pid = pid_info.strip().split(" ")[0]
            if pid.isdigit():
                pid_list.append(pid)
        return pid_list

    def get_defunct_ppid_list(self):
        """
        执行ps -ef获取所有进程并过滤出含有<defunct>的PPID信息
        :return:
        """
        cmd = "ps -ef"
        flag, ret, msg = cliUtil.excute_cmd_in_minisystem_with_end_sign(
            self.cli, cmd, self.lang, [cliUtil.MINISYSTEM_MODEL_FLAG]
        )
        self.all_ret_list.append(ret)
        if flag is not True:
            raise UnCheckException(msg)
        ret = ret.splitlines()
        ppid_error_info = list(filter(lambda x: "<defunct>" in x, ret))
        if not ppid_error_info:
            return []
        ppid_error_list = []
        for ppid_info in ppid_error_info:
            temp_list = ppid_info.strip().split(" ")
            temp_list = [i for i in temp_list if i != ""]
            if len(temp_list) < 3:
                continue
            error_ppid = temp_list[2]
            ppid_error_list.append(error_ppid)
        return ppid_error_list


def is_risk_version(cli, lang, all_ret_list):
    (
        flag,
        software_version,
        hot_patch_version,
        ret,
    ) = cliUtil.get_system_version_with_ret(cli, lang)
    all_ret_list.append(ret)
    if software_version == "6.0.1":
        pattern = re.compile(r"SPH(\d+)", flags=re.IGNORECASE)
        match_hot_path = pattern.search(hot_patch_version)
        if not match_hot_path:
            return True
        if int(match_hot_path.group(1)) < SAFETY_HOT_PATCH:
            return True
    else:
        if RISK_VERSION.search(software_version):
            return True
    return False
