# -*- coding: UTF-8 -*-
import time
from java.lang import Exception as JException
from cbb.frame.adapter.restService import RestAdapter
from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.cli.execute_on_all_controllers import (
    ExecuteOnAllControllers,
    ExeOnAllCtrlContext,
    ResultType,
    FuncResult
)
import re

CMD_QUERY_KV_USAGE = 'iod cpuusage show 10'
CMD_QUERY_CONTEXT_USAGE = 'cmm show prlist mid 142'
# 单控制器上最长执行时间
TIMEOUT = 60 * 2
# 执行次数，然后取平均
EXEC_MAX_TIME = 10
# kv分区最大利用率
KV_USAGE_LIMIT = 25
# context最大使用量
CONTEXT_USAGE_LIMIT = 20000

key_abnormal = 'query.result.abnormal'

PATCH_VERSION = {
    "V300R001C01SPC100": {"start_patch": "V300R001C01SPH106",
                          "emergency": ["V300R001C01SPH170"]}
}


class KvUsageChecker:
    def __init__(self, context):
        self.logger = contextUtil.getLogger(context)
        self.lang = contextUtil.getLang(context)
        self.cli = contextUtil.getCli(context)
        self.exec_context = ExeOnAllCtrlContext(context)
        self.ctrl_num = self.get_ctrl_num_one_engine()
        self.rest = RestAdapter(context)
        self.device_ret = None
        self.error_msg = None
        self.start_patch = None

    def need_check(self, sys_ver, patch_ver):
        required_version = PATCH_VERSION.get(sys_ver)
        if not required_version:
            return True
        self.start_patch = required_version.get("start_patch")
        self.exec_context.original_context["start_patch"] = self.start_patch
        emergency = required_version.get("emergency")
        if patch_ver and\
                (patch_ver in emergency or patch_ver >= self.start_patch):
            return False
        return True

    def execute_check(self):
        result, sys_ver, patch_ver = \
            cliUtil.getSystemVersion(self.cli, self.lang)
        if not result:
            msg = baseUtil.getPyResource(self.lang, "query.result.abnormal")
            return False, "", msg

        if not self.need_check(sys_ver, patch_ver):
            return True, "", ""

        res_type, origin_msg, err_msg = self.check_cpu()
        res_type_context, origin_msg_context, err_msg_context = \
            self.check_context()
        target_res_type = get_highest_res_type(res_type, res_type_context)
        return target_res_type == ResultType.SUCCESS, (
                origin_msg + '\n' + origin_msg_context
        ), err_msg + '\n' + err_msg_context

    def check_cpu(self):
        self.device_ret = []
        self.error_msg = []
        flag = self.check_cpu_usage(self.rest)
        return flag, '\n'.join(self.device_ret), '\n'.join(self.error_msg)

    def check_context(self):
        self.device_ret = []
        self.error_msg = []
        result = check_context_usage(self.exec_context)
        origin_res = {}
        error_msg = {}
        for key in result.other_result:
            self.build_result(
                result.other_result.get(key), origin_res, error_msg)
        return result.result_type, '\n'.join(origin_res.values()), '\n'.join(
            error_msg.values())

    def get_ctrl_num_one_engine(self):
        """
        获取单引擎控制器数量
        :return:
        """
        flag, cli_ret, err_msg, node_tuples \
            = cliUtil.getControllerEngineTopography(self.cli, self.lang)
        ctrl_num = len(node_tuples[2].get("0"))
        return ctrl_num

    def build_result(self, source, out_res, out_msg):
        rs_key = 'checkitem.controller.cpu.ctrl.title'
        for index in source:
            entry = source.get(index)
            ctrl_id = baseUtil.get_ctrl_id_by_node_id(
                int(index), self.ctrl_num)
            out_res[index] = baseUtil.getPyResource(
                self.lang, rs_key, (ctrl_id, '\n' + entry.cli_ret))
            out_msg[index] = '[' + ctrl_id + ']:' + entry.err_msg

    def check_cpu_usage(self, rest_service):
        value_map = {}
        actual_check_time = 0
        start_time = time.time()
        for _ in range(EXEC_MAX_TIME):
            if time.time() - start_time > TIMEOUT:
                break
            try:
                recs = rest_service.excuteDiagnoseCmd(CMD_QUERY_KV_USAGE, 2)
                self.device_ret.append(str(recs))
                flag = parse_cpu_usage(format_res2dict(recs), value_map)
                if not flag:
                    self.error_msg.append(
                        baseUtil.getPyResource(self.lang, key_abnormal))
                    continue
                actual_check_time += 1
            except (Exception, JException):
                baseUtil.getPyResource(self.lang, key_abnormal)
        check_fail = False
        k_normal = 'checkitem.controller.cpu.usage.normal'
        k_abnormal = 'checkitem.controller.cpu.usage.abnormal'
        k_patch_abnormal = 'checkitem.controller.cpu.usage.abnormal.patch'
        for key in value_map:
            val = int(value_map.get(key) / actual_check_time)
            if val >= KV_USAGE_LIMIT:
                check_fail = True
                if not self.start_patch:
                    self.error_msg.append(
                        baseUtil.getPyResource(self.lang, k_abnormal,
                                               (key, str(val))))
                else:
                    self.error_msg.append(
                        baseUtil.getPyResource(self.lang, k_patch_abnormal,
                                               (key, str(val),
                                                self.start_patch)))
            else:
                self.error_msg.append(
                    baseUtil.getPyResource(self.lang, k_normal,
                                           (key, str(val))))
        if actual_check_time == 0:
            return ResultType.NOT_FINISHED
        if check_fail:
            return ResultType.FAILED
        return ResultType.SUCCESS


def get_highest_res_type(res_a, res_b):
    if res_a == ResultType.FAILED or res_b == ResultType.FAILED:
        return ResultType.FAILED
    elif res_a == ResultType.NOT_FINISHED or res_b == ResultType.NOT_FINISHED:
        return ResultType.NOT_FINISHED
    else:
        return res_a


# 从dict中取到每个控制器的usage值，加到dict中
def parse_cpu_usage(dic, value_map):
    for key in dic:
        if key not in value_map:
            value_map[key] = 0
        flag, val = parse_cpu_usage_detail(dic.get(key))
        if not flag:
            return False
        value_map[key] = value_map.get(key) + val
    return True


# 将tlv回显对象转换为ctrl_id到回显的dict
def format_res2dict(recs):
    k_2_v_dict = {}
    control_num = len(recs)
    for index in range(0, control_num):
        rec = recs[index]
        node_id = rec.get('0').get('value')
        result = rec.get('1').get('value')
        k_2_v_dict[node_id] = result
    return k_2_v_dict


# 从回显中解析usage值
def parse_cpu_usage_detail(res):
    match_obj = re.match(r'.+Usage:(\d+)', res)
    if match_obj:
        return True, int(match_obj.group(1))
    else:
        return False, 0


@ExecuteOnAllControllers
def check_context_usage(exec_context):
    current_ctrl_id = exec_context.cur_ctrl_id
    cli = exec_context.dev_info.cli
    lang = exec_context.lang
    start_patch = exec_context.original_context.get("start_patch")
    flag, cli_ret, _ = cliUtil.enterDebugModeFromCliMode(cli, lang)
    if not flag:
        return FuncResult(ResultType.NOT_FINISHED, '', '', False)
    usage_sum = 0
    origin_res = []
    actual_check_time = 0
    start_time = time.time()
    try:
        tmp_err_msg = None
        for _ in range(EXEC_MAX_TIME):
            if time.time() - start_time > TIMEOUT:
                break
            try:
                cmd_ret = cliUtil.execCliCmd(
                    cli, CMD_QUERY_CONTEXT_USAGE, True)
                flag, val, optimized_ret = \
                    parse_context_result(cmd_ret, exec_context.lang)
                origin_res.append(optimized_ret)
                if not flag:
                    # 单此查询结果异常，则跳过此次，并记录下原因
                    tmp_err_msg = val
                    continue
                usage_sum += val
                actual_check_time += 1
            except (Exception, JException):
                # 如果执行命令过程发生异常，则此次查询忽略
                tmp_err_msg = baseUtil.getPyResource(lang, key_abnormal)
        if actual_check_time == 0:
            return FuncResult(ResultType.NOT_FINISHED, '\n'.join(origin_res),
                              tmp_err_msg, other_result=current_ctrl_id)
        k_normal = 'checkitem.controller.cpu.context.normal'
        k_abnormal = 'checkitem.controller.cpu.context.abnormal'
        k_patch_abnormal = 'checkitem.controller.cpu.context.abnormal.patch'
        avg_val = int(usage_sum / actual_check_time)
        if CONTEXT_USAGE_LIMIT <= avg_val:
            msg = baseUtil.getPyResource(lang, k_abnormal, avg_val)
            if start_patch:
                msg = baseUtil.\
                    getPyResource(lang, k_patch_abnormal,
                                  (avg_val, start_patch))
            return FuncResult(
                ResultType.FAILED, '\n'.join(origin_res),
                msg,
                other_result=current_ctrl_id)
        return FuncResult(
            ResultType.SUCCESS, '\n'.join(origin_res),
            baseUtil.getPyResource(lang, k_normal, avg_val),
            other_result=current_ctrl_id)
    finally:
        cliUtil.enterCliModeFromSomeModel(cli, lang)


# 从回显中解析到CONTEXT区的使用量
def parse_context_result(result, lang):
    blocks = result.split('PartId')
    for block in blocks:
        block = 'PartId' + block
        flag, val = parse_context_block(block)
        if flag:
            return True, val, block
    return False, baseUtil.getPyResource(lang, key_abnormal), result


# 对单一block进行解析
def parse_context_block(block):
    if 'PartName' not in block:
        return False, 0
    rows = cliUtil.getHorizontalCliRet(block)
    for entry in rows:
        if entry.get('PartName') == 'CONTEXT':
            return True, int(entry.get('UsedCnt'))
    return False, 0
