#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2019. All rights reserved.

"""
@time: 2019/08/26 
@file: eth_toe_switch_check.py
@function: TOE功能开关检查
"""

from java.lang import Throwable
from cbb.frame.base.constants import CheckStatus
from cbb.frame.cli import cliUtil
from cbb.frame.base import baseUtil
from cbb.frame.base import logger


class ShortDevType:
    DoradoV3 = 1  # Dorado V3设备
    V5 = 2  # 融合存储V5设备


class ToeCheckScene:
    UPGRADE_EVALUATION = 1  # 升级评估场景
    INSPECTION = 2  # 巡检场景
    REPLACE_SPARE_PART = 3  # 备件更换场景


INVOLVE_DEV_TYPE_DICT = {
    ShortDevType.V5: ("5110 V5", "5110F V5", "5210 V5", "5210F V5", "5300 V5", "5300F V5", "5500 V5", "5500F V5",
                      "5500 V5 Elite", "5600 V5", "5600F V5", "5800 V5", "5800F V5", "6800 V5", "6800F V5", "18500 V5",
                      "18500F V5", "18800F V5", "18800 V5"),
    ShortDevType.DoradoV3: ("Dorado5000 V3", "Dorado6000 V3", "Dorado18000 V3")
}
INVOLVE_NO_PATCH_V5_DEV_TYPE_TUPLE = ("5110 V5", "5110F V5", "5210 V5", "5210F V5")
INVOLVE_VERSION_DICT = {
    ShortDevType.V5: ("V500R007C30SPC100"),
    ShortDevType.DoradoV3: ("V300R002C00", "V300R002C00SPC100")
}
INVOLVE_PATCH_VERSION_DICT = {
    "V500R007C30SPC100": "V500R007C30SPH105"
}


def execute(scene_msg):
    return EthToeSwitchCheck(scene_msg).run()


class ToeCheckSceneMsgAdapter:

    def __init__(self, scene, lang, cli, log):
        self._lang = lang
        self._cli = cli
        self._logger = logger.Logger(log, __file__)
        self._scene = scene

    def get_lang(self):
        return self._lang

    def get_cli(self):
        return self._cli

    def get_logger(self):
        return self._logger

    def get_scene(self):
        return self._scene


class EthToeSwitchCheck:

    def __init__(self, scene_msg):
        # 入参scene_msg为ToeCheckSceneMsgAdapter对象
        self._lang = scene_msg.get_lang()
        self._cli = scene_msg.get_cli()
        self._logger = scene_msg.get_logger()
        self._scene = scene_msg.get_scene()
        self._cli_ret = ""
        self._err_msg = ""

    def run(self):
        """
        执行TOE功能开关检查
        :return:(检查结果 CheckStatus，原始信息 str，错误信息 str）
        """
        try:
            # step 1 检查版本和补丁
            is_exe_successful, software_version, patch_version = cliUtil.getSystemVersion(self._cli, self._lang)
            self._cli_ret = "current version:%s\npatch version:%s" % (software_version, patch_version)
            if not is_exe_successful:
                self._err_msg = baseUtil.getPyResource(self._lang, "query.result.abnormal")
                self._logger.error("Version query result abnormal.")
                return CheckStatus.NOCHECK, self._cli_ret, self._err_msg

            is_exe_successful, unsure_value, self._err_msg = cliUtil.getProductModel(self._cli, self._lang)
            if not is_exe_successful:
                self._cli_ret += "\n" + unsure_value
                self._logger.error("Device type query result abnormal.")
                return CheckStatus.NOCHECK, self._cli_ret, self._err_msg
            dev_type = unsure_value
            self._cli_ret += "\ndevice type:%s" % dev_type
            short_dev_Type = self._trans_involve_type(dev_type)

            # 版本以及补丁判断
            if not self._is_involve_version(short_dev_Type, software_version):
                return CheckStatus.PASS, self._cli_ret, self._err_msg

            if self._is_have_patch_version(dev_type, software_version) and \
                    self._have_setup_correct_patch(software_version, patch_version):
                return CheckStatus.PASS, self._cli_ret, self._err_msg

            # step 2 DoradoV3漂移开关检查
            if short_dev_Type is ShortDevType.DoradoV3:
                rift_switch_check_flag = self._rift_switch_check()
                if rift_switch_check_flag is not CheckStatus.EXECUTING:
                    return rift_switch_check_flag, self._cli_ret, self._err_msg

            # step 3 TOE功能开关状态检查
            return self._check_toe_switch(dev_type, software_version), self._cli_ret, self._err_msg
        except Exception as exception:
            self._logger.logException(exception)
            return CheckStatus.NOCHECK, self._cli_ret, baseUtil.getPyResource(self._lang, 'query.result.abnormal')
        except Throwable as throwable:
            self._logger.logException(throwable)
            return CheckStatus.NOCHECK, self._cli_ret, baseUtil.getPyResource(self._lang, 'query.result.abnormal')

    def _rift_switch_check(self):
        """
        漂移开关检查
        :return: 检查结果 CheckStatus
        """
        if self._scene is ToeCheckScene.UPGRADE_EVALUATION:
            cmd = "show upgrade port_failover_switch"
        elif self._scene is ToeCheckScene.INSPECTION or \
                self._scene is ToeCheckScene.REPLACE_SPARE_PART:
            cmd = "show logical_port failover_switch service_type=SAN"
        else:
            self._logger.info("The % scene is not covered." % self._scene)
            return CheckStatus.PASS
        return self._exe_drift_switch_check(cmd)

    def _exe_drift_switch_check(self, cmd):
        """
        执行漂移开关检查
        :param cmd: 检查命令 str
        :return: 检查结果 CheckStatus
        """
        is_cmd_successful, cli_ret, self._err_msg = cliUtil.execCmdInDeveloperModePrompt(self._cli, cmd, True,
                                                                                         self._lang)
        self._cli_ret += "\n" + cli_ret

        # 命令是否可用
        if not is_cmd_successful:
            self._logger.error("Abnormal execution of commands")
            return CheckStatus.NOCHECK

        # 是否有执行cli命令的权限
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            self._logger.error("No permission to execute the cli command")
            return CheckStatus.NOCHECK

        # 判断回显是否为Command executed successfully
        if cliUtil.queryResultWithNoRecord(cli_ret):
            self._logger.info("CliRet is 'Command executed successfully'")
            return CheckStatus.PASS

        if "off" in cli_ret.lower():
            return CheckStatus.PASS
        if "on" in cli_ret.lower():
            return CheckStatus.EXECUTING
        self._logger.error("The drift switch status message is not off or on, but as:%s" % cli_ret)
        return CheckStatus.NOCHECK

    def _check_toe_switch(self, dev_type, software_version):
        """
        TOE功能开关检查
        :param dev_type: 设备型号 str
        :param software_version: 系统版本 str
        :return: 检查结果 CheckStatus
        """
        # 查找TOE功能开关端口
        find_flag, toe_switch_port_list = self._find_switch_port()

        # 查找到的端口
        if len(toe_switch_port_list) == 0:
            return find_flag

        # 检查所有端口状态是否为关闭
        return self._check_switch_port_status(toe_switch_port_list, dev_type, software_version)

    def _find_switch_port(self):
        """
        查找TOE功能开关端口
        :return: (查找结果 CheckStatus，端口ID list）
        """
        cmd = "show port general logic_type=Host_Port physical_type=ETH"
        is_cmd_success, cli_ret, self._err_msg = cliUtil.execCmdInDeveloperModePrompt(self._cli, cmd, True, self._lang)
        self._cli_ret += "\n" + cli_ret

        toe_switch_port_list = list()
        # 命令是否可用
        if not is_cmd_success:
            self._logger.error("Abnormal execution of commands")
            return CheckStatus.NOCHECK, toe_switch_port_list

        # 是否有执行cli命令的权限
        if not cliUtil.hasCliExecPrivilege(cli_ret):
            self._logger.error("No permission to execute the cli command")
            return CheckStatus.NOCHECK, toe_switch_port_list

        # 判断回显是否为Command executed successfully
        if cliUtil.queryResultWithNoRecord(cli_ret):
            self._logger.info("CliRet is 'Command executed successfully'")
            return CheckStatus.PASS, toe_switch_port_list

        cli_ret_lines_list = cliUtil.getHorizontalCliRet(cli_ret)

        if len(cli_ret_lines_list) == 0:
            self._err_msg = baseUtil.getPyResource(self._lang, "query.result.abnormal")
            self._logger.error("Port echo table parse exception.")
            return CheckStatus.NOCHECK, toe_switch_port_list

        for cli_ret_line in cli_ret_lines_list:
            toe_switch_port_list.append(cli_ret_line.get("ID"))
        # 长度为0，不存在端口，检查通过
        if len(toe_switch_port_list) == 0:
            return CheckStatus.PASS, toe_switch_port_list
        return CheckStatus.EXECUTING, toe_switch_port_list

    def _check_switch_port_status(self, toe_switch_port_list, dev_type, software_version):
        """
        检查TOE功能开关状态
        :param toe_switch_port_list: TOE功能开关端口ID list
        :param dev_type: 设备型号 str
        :param software_version: 系统版本 str
        :return: 检查结果 CheckStatus
        """
        cmd = "show port eth_toe_switch eth_port_id="
        is_all_switch_no = True
        yes_switch = ""
        # 进入开发者模式
        cliUtil.enterDeveloperMode(self._cli, self._lang)
        for toe_switch_port in toe_switch_port_list:
            is_cmd_successful, cli_ret, self._err_msg = cliUtil.excuteCmdInCliMode(self._cli, cmd + toe_switch_port,
                                                                                   True, self._lang)
            self._cli_ret += "\n" + cli_ret

            # 命令是否可用
            if not is_cmd_successful:
                self._logger.error("Abnormal execution of commands")
                cliUtil.developerMode2CliMode(self._cli)
                return CheckStatus.NOCHECK

            # 是否有执行cli命令的权限
            if not cliUtil.hasCliExecPrivilege(cli_ret):
                self._logger.error("No permission to execute the cli command")
                cliUtil.developerMode2CliMode(self._cli)
                return CheckStatus.NOCHECK

            if "yes" in cli_ret.lower():
                if is_all_switch_no:
                    yes_switch += toe_switch_port
                else:
                    yes_switch += ", " + toe_switch_port
                is_all_switch_no = False
                continue
            if "no" in cli_ret.lower():
                continue

            self._logger.error("Toe switch port check exception")
            cliUtil.developerMode2CliMode(self._cli)
            return CheckStatus.NOCHECK
        # 退出开发者模式
        cliUtil.developerMode2CliMode(self._cli)
        if not is_all_switch_no:
            if self._scene is ToeCheckScene.REPLACE_SPARE_PART:
                # 备件更换没有修复建议展示，提示信息需独特，单独处理
                if self._is_have_patch_version(dev_type, software_version):
                    self._err_msg = baseUtil.getPyResource(self._lang,
                                                           "EthTOESwitchCheck.both.replaceSparePart.notpass",
                                                           (INVOLVE_PATCH_VERSION_DICT.get(software_version),
                                                            yes_switch))
                else:
                    self._err_msg = baseUtil.getPyResource(self._lang,
                                                           "EthTOESwitchCheck.off.TOESwitch.replaceSparePart.notpass",
                                                           yes_switch)
            else:
                if self._is_have_patch_version(dev_type, software_version):
                    self._err_msg = baseUtil.getPyResource(self._lang,
                                                           "EthTOESwitchCheck.put.patch.notpass",
                                                           (INVOLVE_PATCH_VERSION_DICT.get(software_version),
                                                            yes_switch))
                else:
                    self._err_msg = baseUtil.getPyResource(self._lang, "EthTOESwitchCheck.off.TOESwitch.notpass",
                                                           yes_switch)
            return CheckStatus.NOTPASS
        return CheckStatus.PASS

    @staticmethod
    def _trans_involve_type(dev_type):
        """
        转换设备类型为简称，便于匹配涉及版本
        :param dev_type: 设备类型 str
        :return: 设备类型简称 str
        """
        dev_type = dev_type.strip()
        for short_dev_type in INVOLVE_DEV_TYPE_DICT.keys():
            dev_type_list = INVOLVE_DEV_TYPE_DICT.get(short_dev_type)
            if dev_type in dev_type_list:
                return short_dev_type
        return ""

    @staticmethod
    def _is_involve_version(short_dev_type, software_version):
        """
        是否是涉及的版本
        :param short_dev_type:设备型号简称 str
        :param software_version:当前版本 str
        :return:True/False
        """
        return short_dev_type in INVOLVE_VERSION_DICT and software_version in INVOLVE_VERSION_DICT.get(short_dev_type)

    @staticmethod
    def _is_have_patch_version(dev_type, software_version):
        """
        是否是有补丁的版本
        :param dev_type:设备型号 str
        :param software_version:系统版本 str
        :return:True/False
        """
        return dev_type not in INVOLVE_NO_PATCH_V5_DEV_TYPE_TUPLE and software_version in INVOLVE_PATCH_VERSION_DICT

    @staticmethod
    def _have_setup_correct_patch(software_version, patch_version):
        """
        是否是有补丁的风险版本且安装了补丁
        :param software_version:系统版本 str
        :param patch_version:补丁版本 str
        :return:True/False
        """
        return software_version in INVOLVE_PATCH_VERSION_DICT and patch_version >= INVOLVE_PATCH_VERSION_DICT.get(
            software_version)


