# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
from cbb.frame.base import baseUtil
from cbb.frame.base.constants import CheckStatus
from cbb.frame.cli import cliUtil


class CheckScanStatus:
    """ 元数据扫描状态检查

    """
    def __init__(self, lang, cli, logger):
        self._lang = lang
        self._cli = cli
        self._logger = logger
        self._all_ret = []

    def run(self):
        try:
            # 查询系统中所有控制器ID
            query_ret = cliUtil.getControllerIdList(self._cli, self._lang)
            if query_ret[0] is not True:
                return CheckStatus.NOTPASS, query_ret[1], query_ret[2]

            final_result = True
            # 查询每个控制器上的元数据扫描状态
            for ctrl_id in query_ret[1]:
                flag, show_ret = self.query_scan_info(ctrl_id)
                # 命令执行异常或不支持，跳过
                if flag is not True:
                    continue
                # 存在running的扫描进程，不通过
                if self.check_status_running(show_ret):
                    final_result = False

            if not final_result:
                err_msg = baseUtil.getPyResource(
                    self._lang, "check.scan.status.fail")
                return (CheckStatus.NOTPASS, "\n".join(self._all_ret),
                        err_msg)

            return CheckStatus.PASS, "\n".join(self._all_ret), ""

        except Exception as ex:
            self._logger.logException(ex)
            return (CheckStatus.NOCHECK, "\n".join(self._all_ret),
                    baseUtil.getPyResource(self._lang,
                                           "query.result.abnormal"))

    def query_scan_info(self, ctrl_id):
        """ 查询控制器扫描任务状态

        :param ctrl_id: 控制器ID
        :return:
        """
        # noinspection PyBroadException
        try:
            cmd = "show metadata info type=flow flow_type=0 " \
                  "controller={}".format(ctrl_id)
            flag, show_ret, _ = cliUtil.excuteCmdInDeveloperMode(
                self._cli, cmd, True, self._lang, preExec=False)
            self._all_ret.append(show_ret)
            return flag, show_ret
        except Exception as ex:
            self._logger.logException(ex)
            self._logger.logInfo("Failed to check scan status on "
                                 "controller {}.".format(ctrl_id))
            return False, ""

    @staticmethod
    def check_status_running(scan_ret):
        """ 检查扫描状态是否为running

        :param scan_ret: 扫描状态回显
        :return:
        """
        ret_lines = scan_ret.splitlines()
        temp_ret = "\n".join(ret_lines[4:]) if len(ret_lines) >= 4 \
            else scan_ret
        flow_info_list = cliUtil.getHorizontalCliRet(temp_ret)
        for flow_info in flow_info_list:
            # 存在扫描进程为running
            if flow_info.get("Status", "Default").lower() == "running":
                return True
        return False
