# -*- coding: UTF-8 -*-
from cbb.frame.base import config
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil


class QuorumCommon:

    @staticmethod
    def getQuorumServLinkRecords(rest):
        """批量查询仲裁服务器链路

        :param rest: rest连接
        :return: rest回显
        """
        params = {}
        uriParamDict = restUtil.CommonRest.getUriParamDict(restData.RestCfg.OBJ.QUORUM_SERVER_LINK)
        try:
            record = restUtil.CommonRest.execCmd(rest, uriParamDict, params, restData.RestCfg.RestMethod.GET)
            return restUtil.CommonRest.getData(record)
        except Exception as e:
            errCode = str(e[0])
            # 不支持该命令字，则不需检查仲裁链路，直接通过
            if errCode == config.ERR_CODE_CMD_NOT_SUPPORTED:
                return []
            raise Exception(errCode, e)

    @staticmethod
    def getQuorumServLinkUpRecords(rest):
        """批量查询仲裁服务器链路，过滤选出LinkUp状态的数据

        :param rest: rest连接
        :return: rest回显
        """

        linkData = QuorumCommon.getQuorumServLinkRecords(rest)
        resList = []
        for aData in linkData:
            runningStatus = restUtil.CommonRest.getRecordValue(aData, restData.RestCfg.QuorumServerLink.RUNNINGSTATUS)

            # 筛选出状态为LinkUp的数据
            if runningStatus == restData.RestCfg.RunningStatusEnum.LINK_UP:
                resList.append(aData)

        return resList

    @staticmethod
    def getQuorumServNotLinkUpIDs(rest):
        """批量查询仲裁服务器链路，过滤选出非LinkUp状态的数据

        :param rest: rest连接
        :return: rest回显
        """

        linkData = QuorumCommon.getQuorumServLinkRecords(rest)
        resList = []
        for aData in linkData:
            runningStatus = restUtil.CommonRest.getRecordValue(aData, restData.RestCfg.QuorumServerLink.RUNNINGSTATUS)
            # 筛选出状态为LinkUp的数据
            if runningStatus != restData.RestCfg.RunningStatusEnum.LINK_UP:
                linkId = restUtil.CommonRest.getRecordValue(aData, restData.RestCfg.QuorumServerLink.ID)
                resList.append(linkId)

        return resList

    @staticmethod
    def getQuorumServLinkByCtrl(records, ctrlId):
        """查询指定控制器的仲裁链路

        :param records: 指定控制器的仲裁链路Data数组
        :param ctrlId: 控制器ID
        :return: 所有仲裁链路数据
        """

        resList = []
        if records is None:
            return resList

        for record in records:
            localCtrlId = restUtil.CommonRest.getRecordValue(record,
                                                             restData.RestCfg.QuorumServerLink.LOCALCONTROLLERNAME)
            # 匹配控制器上所有接口卡上的所有端口的数据
            if localCtrlId == ctrlId:
                resList.append(record)
        return resList

    @staticmethod
    def getQuorumServLinkByPort(records, loc):
        """查询指定端口的仲裁链路

        :param records: 所有仲裁链路数据
        :param loc: 端口位置
        :return: 指定端口的仲裁链路Data数组，只一条记录
        """

        resList = []
        if records is None:
            return resList

        for record in records:
            localPortName = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.QuorumServerLink.LOCALPORTNAME)
            # 端口位置只可能匹配一条记录
            if localPortName == loc:
                resList.append(record)
                break
        return resList

    @staticmethod
    def getQuorumServLinkByIntf(records, loc):
        """查询指定端口的仲裁链路

        :param records: 所有仲裁链路数据
        :param loc: 接口模块位置
        :return: 指定端口的仲裁链路Data数组
        """

        resList = []
        if records is None:
            return resList

        for record in records:
            localPortName = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.QuorumServerLink.LOCALPORTNAME)

            # 接口卡上的所有端口都可能有仲裁链路
            if loc in localPortName:
                resList.append(record)
        return resList

    @staticmethod
    def getRedundQuorumLinkByCtrl(records, currLinkData, ctrlId):
        """查询当前仲裁链路的冗余仲裁链路

        :param records: 所有仲裁链路数据
        :param currLinkData: 当前控制器的仲裁链路数据
        :param ctrlId:  更换的控制器ID
        :return: 无冗余仲裁链路的链路ID数组
        """

        resList = []
        if records is None or currLinkData is None:
            return resList

        for aLinkData in currLinkData:
            linkId = restUtil.CommonRest.getRecordValue(aLinkData, restData.RestCfg.QuorumServerLink.ID)
            # 当前仲裁链路的仲裁服务器Id
            quorumServerId = restUtil.CommonRest.getRecordValue(aLinkData, restData.RestCfg.QuorumServerLink.PARENTID)

            redundList = []
            for record in records:
                localCtrlId = restUtil.CommonRest.getRecordValue(record,
                                                                 restData.RestCfg.QuorumServerLink.LOCALCONTROLLERNAME)
                parentId = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.QuorumServerLink.PARENTID)

                # 冗余条件：仲裁服务器相同且非待更换控制器
                if parentId == quorumServerId and localCtrlId != ctrlId:
                    redundList.append(record)
                    break

            # 如果未找到冗余链路，则将当前仲裁链路ID返回
            if len(redundList) == 0:
                resList.append(linkId)

        return resList

    @staticmethod
    def getRedundQuorumLinkByIntf(records, currLinkData, loc):
        """查询当前仲裁链路的冗余仲裁链路

        :param records: 所有仲裁链路数据
        :param currLinkData: 当前仲裁链路数据
        :param loc: 需要排除待更换Loc的链路
        :return: 冗余仲裁链路Data
        """

        resList = []
        if records is None or currLinkData is None:
            return resList

        for aLinkData in currLinkData:
            linkId = restUtil.CommonRest.getRecordValue(aLinkData, restData.RestCfg.QuorumServerLink.ID)
            # 当前仲裁链路的仲裁服务器Id
            quorumServerId = restUtil.CommonRest.getRecordValue(aLinkData, restData.RestCfg.QuorumServerLink.PARENTID)

            redundList = []
            for record in records:
                localPortName = restUtil.CommonRest.getRecordValue(record,
                                                                   restData.RestCfg.QuorumServerLink.LOCALPORTNAME)
                parentId = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.QuorumServerLink.PARENTID)

                # 冗余条件：仲裁服务器相同且非待更换接口模块
                if parentId == quorumServerId and loc not in localPortName:
                    redundList.append(record)
                    break

            # 如果未找到冗余链路，则将当前仲裁链路ID返回
            if len(redundList) == 0:
                resList.append(linkId)

        return resList

    @staticmethod
    def getRedundQuorumLinkByPort(records, currLinkData, loc):
        """查询当前仲裁链路的冗余仲裁链路

        :param records: 所有仲裁链路数据
        :param currLinkData: 当前仲裁链路数据
        :param loc: 需要排除待更换Loc的链路
        :return: 冗余仲裁链路Data
        """

        resList = []
        if records is None or currLinkData is None or len(currLinkData) == 0:
            return resList

        # 当前端口的仲裁链路只可能有一条
        linkId = restUtil.CommonRest.getRecordValue(currLinkData[0], restData.RestCfg.QuorumServerLink.ID)
        # 当前仲裁链路的仲裁服务器Id
        quorumServerId = restUtil.CommonRest.getRecordValue(currLinkData[0], restData.RestCfg.QuorumServerLink.PARENTID)

        redundList = []
        for record in records:
            localPortName = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.QuorumServerLink.LOCALPORTNAME)
            parentId = restUtil.CommonRest.getRecordValue(record, restData.RestCfg.QuorumServerLink.PARENTID)

            # 冗余条件：仲裁服务器相同且非待更换端口（线缆或光模块）
            if parentId == quorumServerId and localPortName != loc:
                redundList.append(record)
                break

        # 如果未找到冗余链路，则将当前仲裁链路ID返回
        if len(redundList) == 0:
            resList.append(linkId)
        return resList

    @staticmethod
    def checkRedundQuorumLinkByCtrl(records, ctrlId):
        """通过控制器检查冗余仲裁链路

        :param records: 所有仲裁链路数据
        :param ctrlId: 控制器id
        :return: 检查是否通过True/False
        """

        quorumLinkList = QuorumCommon.getQuorumServLinkByCtrl(records, ctrlId)

        # 当前控制器上无仲裁链路，检查通过
        if len(quorumLinkList) == 0:
            return (True, [])

        # 查询冗余仲裁链路
        unexistRedtQuorumLink = QuorumCommon.getRedundQuorumLinkByCtrl(records, quorumLinkList, ctrlId)
        # 存在没有冗余的仲裁链路，将没有冗余的仲裁链路返回
        if len(unexistRedtQuorumLink) != 0:
            return (False, unexistRedtQuorumLink)

        return (True, [])

    @staticmethod
    def checkRedundQuorumLinkByIntf(records, loc):
        """通过接口位置检查冗余仲裁链路

        :param records: 所有仲裁链路数据
        :param loc: 接口模块位置
        :return: 检查是否通过True/False
        """

        quorumLinkList = QuorumCommon.getQuorumServLinkByIntf(records, loc)

        # 当前接口上无仲裁链路, 检查通过
        if len(quorumLinkList) == 0:
            return (True, [])

        # 存在仲裁链路，查询冗余仲裁链路
        unexistRedtQuorumLink = QuorumCommon.getRedundQuorumLinkByIntf(records, quorumLinkList, loc)
        # 存在没有冗余的仲裁链路，将没有冗余的仲裁链路返回
        if len(unexistRedtQuorumLink) != 0:
            return (False, unexistRedtQuorumLink)

        return (True, [])

    @staticmethod
    def checkRedundQuorumLinkByPort(records, loc):
        """通过端口检查冗余仲裁链路

        :param records: 所有仲裁链路数据
        :param loc: 端口位置
        :return: 检查是否通过True/False
        """
        quorumLinkList = QuorumCommon.getQuorumServLinkByPort(records, loc)

        # 当前接口上无仲裁链路, 检查通过
        if len(quorumLinkList) == 0:
            return (True, [])

        # 存在仲裁链路，查询冗余仲裁链路
        unexistRedtQuorumLink = QuorumCommon.getRedundQuorumLinkByPort(records, quorumLinkList, loc)
        if len(unexistRedtQuorumLink) != 0:
            return (False, unexistRedtQuorumLink)

        return (True, [])

    @staticmethod
    def checkRedundantQuorumServLink(rest, fruModule, fruId):
        """检查冗余仲裁链路，为总入口方法

        :param rest: rest连接
        :param fruModule:更换的FRU模块（当前涉及控制器、IPScaleOut接口卡、PCIe接口卡、前端接口卡、
                            管理模块、光模块、前端线缆，对应QuorumLinkCheckEnum）
        :param fruId:  fru标识（控制器、IPScaleOut接口卡、PCIe接口卡为控制器ID，
                                前端接口卡、管理模块、光模块、前端线缆为位置location
        :return: 检查是否通过True/False，并且不存在冗余链路的仲裁链路列表
        """
        redundantQuorumData = QuorumCommon.getQuorumServLinkUpRecords(rest)

        # 无仲裁链路，直接通过
        if redundantQuorumData is None or len(redundantQuorumData) == 0:
            return (True, [])

        # 控制器、IPScaleOut 接口卡、PCIe 接口卡检查控制器上有无仲裁链路冗余
        if fruModule in [restData.RestCfg.QuorumLinkCheckEnum.CONTROLLER,
                         restData.RestCfg.QuorumLinkCheckEnum.IP_SCALEOUT_INTF,
                         restData.RestCfg.QuorumLinkCheckEnum.PCIE_INTF]:
            return QuorumCommon.checkRedundQuorumLinkByCtrl(redundantQuorumData, fruId)

        # 前端接口卡、管理模块检查其他接口模块是否存在冗余仲裁链路
        if fruModule in [restData.RestCfg.QuorumLinkCheckEnum.FRONTEND_INTF,
                         restData.RestCfg.QuorumLinkCheckEnum.MANAGE_BOARD]:
            return QuorumCommon.checkRedundQuorumLinkByIntf(redundantQuorumData, fruId)

        # 光模块、前端线缆、管理网线检查关联的端口是否存在冗余仲裁链路
        if fruModule in [restData.RestCfg.QuorumLinkCheckEnum.FIBRE_MODULE,
                         restData.RestCfg.QuorumLinkCheckEnum.CABLE_FRONTEND,
                         restData.RestCfg.QuorumLinkCheckEnum.MANAGE_CABLE]:
            return QuorumCommon.checkRedundQuorumLinkByPort(redundantQuorumData, fruId)

        # 不涉及的模块，直接通过
        return (True, [])

    @staticmethod
    def checkQuorumServLinkStatus(rest, fruModule, fruId):
        """检查当前仲裁链路运行状态

        :param rest: rest连接
        :param fruModule: fru模块
        :param fruId: 模块标识——ID或Location
        :return: 检查是否通过True/False, 并返回状态不正常的仲裁链路列表
        """

        allQuorumLinkData = QuorumCommon.getQuorumServLinkRecords(rest)
        currModuleLinkList = []
        # 获取当前模块的冗余链路，只关注当前模块上的仲裁链路非LINKUP
        if fruModule in [restData.RestCfg.QuorumLinkCheckEnum.CONTROLLER,
                         restData.RestCfg.QuorumLinkCheckEnum.IP_SCALEOUT_INTF,
                         restData.RestCfg.QuorumLinkCheckEnum.PCIE_INTF]:
            currModuleLinkList = QuorumCommon.getQuorumServLinkByCtrl(allQuorumLinkData, fruId)
        elif fruModule in [restData.RestCfg.QuorumLinkCheckEnum.FRONTEND_INTF,
                           restData.RestCfg.QuorumLinkCheckEnum.MANAGE_BOARD]:
            currModuleLinkList = QuorumCommon.getQuorumServLinkByIntf(allQuorumLinkData, fruId)
        elif fruModule in [restData.RestCfg.QuorumLinkCheckEnum.FIBRE_MODULE,
                           restData.RestCfg.QuorumLinkCheckEnum.CABLE_FRONTEND]:
            currModuleLinkList = QuorumCommon.getQuorumServLinkByPort(allQuorumLinkData, fruId)

        unlinkupList = []
        for aLink in currModuleLinkList:
            runningStatus = restUtil.CommonRest.getRecordValue(aLink, restData.RestCfg.QuorumServerLink.RUNNINGSTATUS)
            # 返回冗余链路不正常的数据
            if runningStatus != restData.RestCfg.RunningStatusEnum.LINK_UP:
                linkId = restUtil.CommonRest.getRecordValue(aLink, restData.RestCfg.QuorumServerLink.ID)
                unlinkupList.append(linkId)
        if len(unlinkupList) == 0:
            return (True, [])
        return (False, unlinkupList)
