# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
from cbb.business.operate.fru.common import common
from cbb.common.query.hardware import engine
from cbb.frame.base import baseUtil
from cbb.frame.base import config
from cbb.frame.context import contextUtil
from cbb.frame.cli import cliUtil


def execute(context, is_ctrl_enc=False, ctrl_id=None):
    """

    :param context: 上下文
    :param is_ctrl_enc: 是否为更换控制框场景
    :param ctrl_id: 备件所属控制器ID，is_ctrl_enc=False时，此参数必需
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        # Dorado不检查
        if contextUtil.getItem(context, "isDorado", False):
            contextUtil.handleSuccess(context)
            logger.logPass()
            return

        result_dict = {"flag": False, "errMsg": "", "suggestion": ""}
        cli = contextUtil.getCli(context)
        lang = contextUtil.getLang(context)
        product_model = contextUtil.getProductModel(context)
        # 是否需要区分逻辑引擎
        according_to_logic = product_model in config.SMALL_SAS_6U_PRODUCTS
        # 获取所有引擎ID列表
        flag, eng_id_list = engine.get_all_eng_id_list(cli, lang,
                                                       according_to_logic)
        if not flag:
            result_dict["errMsg"], result_dict["suggestion"] = common.getMsg(
                lang, "common.exception")
            contextUtil.handleFailure(context, result_dict)
            logger.logNoPass("Failed to get engine information.")
            return
        # 创建引擎号与normal保险箱盘列表对应的字典
        list_to_change = [(eng_id, []) for eng_id in eng_id_list]
        engine_disk_dict = dict(list_to_change)

        # 查询状态正常的保险箱盘
        cmd = r"show disk general|filterRow column=Coffer\sDisk " \
              r"predict=equal_to value=Yes logicOp=and column=Health\sStatus" \
              r" predict=equal_to value=Normal logicOp=and " \
              r"column=Running\sStatus predict=equal_to value=Online"
        disk_flag, ret, _ = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
        # 命令不支持，返回成功
        if disk_flag == cliUtil.RESULT_NOSUPPORT:
            contextUtil.handleSuccess(context)
            logger.logPass()
            return
        # 命令查询失败
        if disk_flag is not True:
            result_dict["errMsg"], result_dict["suggestion"] = common.getMsg(
                lang, "common.exception")
            contextUtil.handleFailure(context, result_dict)
            logger.logNoPass("Failed to get coffer disk information.")
            return

        ret_dict = cliUtil.getHorizontalCliRet(ret)
        # 统计各引擎上normal的保险箱盘，保存到字典
        for coffer_disk in ret_dict:
            disk_id = coffer_disk.get("ID")
            # 硬盘所属的引擎ID
            eng_id = engine.get_eng_id_by_disk_id(disk_id, according_to_logic)
            if eng_id in engine_disk_dict:
                engine_disk_dict.get(eng_id).append(disk_id)
        logger.info("engine and coffer disk dict:%s" % str(engine_disk_dict))

        # 检查normal保险箱盘数量是否满足条件
        is_pass, res_key = check_disk_num(context, is_ctrl_enc,
                                          engine_disk_dict, ctrl_id,
                                          according_to_logic)
        if not is_pass:
            result_dict["errMsg"], result_dict["suggestion"] = \
                common.getMsg(lang, res_key)
            contextUtil.handleFailure(context, result_dict)
            logger.logNoPass("the number of normal coffer disk is "
                             "insufficient.")
            return

        contextUtil.handleSuccess(context)
        logger.logPass()
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return


def check_disk_num(context, is_ctrl_enc, engine_disk_dict, ctrl_id=None,
                   according_to_logic=False):
    """检查引擎上normal的保险箱盘数量是否满足条件

    :param context: 上下文
    :param is_ctrl_enc: 是否为更换控制框场景
    :param engine_disk_dict: 引擎与对应normal保险箱盘的字典
    :param ctrl_id: 备件所在控制器ID
    :param according_to_logic: 是否区分逻辑引擎
    :return:
    """
    is_pass = True
    res_key = ""
    product_model = contextUtil.getProductModel(context)
    is_arm = baseUtil.isArmDev(context, product_model)
    # 更换非控制框场景，备件所在引擎至少有一块normal的保险箱盘
    if not is_ctrl_enc:
        # 需要检查的引擎
        engine_to_check = engine.get_eng_id_by_ctrl_id(ctrl_id,
                                                       according_to_logic)
        if len(engine_disk_dict.get(engine_to_check, [])) < 1:
            is_pass = False
            res_key = "check.coffer.disk.controller"
    # 更换控制框场景
    # X10 ARM设备，每个引擎上至少有一个正常的保险箱盘
    elif is_arm:
        for disks in engine_disk_dict.values():
            if len(disks) < 1:
                is_pass = False
                res_key = "check.coffer.disk.enclosure.arm"
                break
    # X00 purely设备，每个引擎上前3盘中，至少有两个正常的保险箱盘
    else:
        for disks in engine_disk_dict.values():
            first_three = [disk for disk in disks
                           if int(disk.split(".")[1]) < 3]
            if len(first_three) < 2:
                is_pass = False
                res_key = "check.coffer.disk.enclosure"
                break

    return is_pass, res_key
