# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2019-2020. All rights reserved.
import re

from cbb.business.operate.fru.common import common
from cbb.business.operate.fru.common import FuncFactory
from cbb.frame.base import funcUtils
from cbb.frame.base import baseUtil
from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil

# 主机系统不一致告警
HOST_SYS_ALARM = ["0xF0015001E", "0xF00150031"]
# 需要检查的临界版本
CHECK_START_VERSION = "V300R006C10"
# 进度总时间
TOTAL_TIME_SECS = 3 * 60


@funcUtils.fakeProgress(TOTAL_TIME_SECS)
def execute(context):
    """检查主机操作系统

    :param context:
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        result_dict = {"flag": False, "errMsg": "", "suggestion": ""}
        lang = contextUtil.getLang(context)
        cli = contextUtil.getCli(context)
        pdt_version = FuncFactory.getProductVersion(context)
        # Dorado不检查, V3R6C10之前的版本不检查
        if contextUtil.getItem(context, "isDorado", False) \
                or pdt_version < CHECK_START_VERSION:
            contextUtil.handleSuccess(context)
            logger.logPass()
            return

        # 查询主机相关warning级别事件
        event_cmd = "show alarm object_type=21 level=warning"
        event_ret = cliUtil.excuteCmdInCliMode(cli, event_cmd, True, lang)
        if event_ret[0] is not True:
            result_dict["errMsg"], result_dict["suggestion"] = common.getMsg(
                lang, "common.exception")
            contextUtil.handleFailure(context, result_dict)
            logger.logNoPass("Failed to get event information.")
            return

        # 获取事件中对应告警的序号列表
        event_seq_list = list()
        event_list = cliUtil.getHorizontalCliRet(event_ret[1])
        for event in event_list:
            if event.get("ID") in HOST_SYS_ALARM:
                event_seq_list.append(event.get("Sequence"))

        # 告警不涉及，则通过
        if not event_seq_list:
            contextUtil.handleSuccess(context)
            logger.logPass()
            return

        # 逐条检查告警时间，取得一致和不一致的主机ID列表
        query_flag, consistent_host, inconsistent_host = \
            check_event_host(context, event_seq_list, logger)

        # 命令查询异常
        if not query_flag:
            result_dict["errMsg"], result_dict["suggestion"] = common.getMsg(
                lang, "common.exception")
            contextUtil.handleFailure(context, result_dict)
            return
        error_msg = ""
        if inconsistent_host:
            error_msg += common.getMsg(
                lang, "check.host.system.inconsistent",
                errMsgArgs=", ".join(inconsistent_host))[0]
        if consistent_host:
            error_msg += common.getMsg(
                lang, "check.host.system.consistent",
                errMsgArgs=", ".join(consistent_host))[0]
        suggestion = common.getMsg(lang, "check.host.system.sug")[1]
        result_dict["errMsg"] = error_msg
        result_dict["suggestion"] = suggestion
        contextUtil.handleFailure(context, result_dict)
        logger.logNoPass("Host alarms exist.")
        return

    except Exception as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        return


def check_event_host(context, event_seq_list, logger):
    """检查每一条事件涉及的主机ID及对应操作系统，返回一致和不一致的主机ID列表

    :param context: 上下文
    :param event_seq_list: 事件序号列表
    :param logger: 日志对象
    :return:
    """
    lang = contextUtil.getLang(context)
    cli = contextUtil.getCli(context)
    query_flag = True
    consistent_host = list()
    inconsistent_host = list()
    for sequence in event_seq_list:
        seq_cmd = "show event sequence={}".format(sequence)
        query_flag, detail = get_vertical_cmd_info(cli, seq_cmd,
                                                   lang, "Detail")
        if not query_flag:
            logger.logNoPass("Failed to get detail event[{}] "
                             "information.".format(sequence))
            break
        host_id, push_sys = get_host_id_and_push_sys(detail)
        host_cmd = "show host general host_id={}".format(host_id)
        query_flag, fact_sys = get_vertical_cmd_info(cli, host_cmd, lang,
                                                     "Operating System")
        logger.logInfo("host[{}] push system:{}, fact system:"
                       "{}".format(host_id, push_sys, fact_sys))
        if not query_flag:
            logger.logNoPass("Failed to get host(ID:{}) "
                             "information.".format(host_id))
            break
        if push_sys == fact_sys:
            consistent_host.append(host_id)
        else:
            inconsistent_host.append(host_id)
    return query_flag, consistent_host, inconsistent_host


def get_vertical_cmd_info(cli, cmd, lang, filed):
    """获取垂直回显的cli命令的指定字段值

    :param cli: cli连接
    :param cmd: cli命令
    :param lang: 语言
    :param filed: 指定字段
    :return:
    """
    cli_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if cli_ret[0] is not True:
        return False, ""
    info_list = cliUtil.getVerticalCliRet(cli_ret[1])
    if not info_list:
        return False, ""
    return True, info_list[0].get(filed, "")


def get_host_id_and_push_sys(detail):
    """截取告警详细信息中的主机ID和推送的主机系统

    :param detail: 告警详细信息
    :return:
    """
    host_id = re.findall(r"\(ID \d+", detail)[0].split(" ")[1]
    push_sys = detail.split(" ")[-1][:-1]
    return host_id, push_sys
