# -*- coding: UTF-8 -*-
import re

from cbb.business.operate.fru.common import BaseFactory
from cbb.business.operate.fru.common import FuncFactory
from cbb.business.operate.fru.common import commonFunction

from cbb.frame.cli import cliUtil
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest import restUtil
from cbb.frame.util import public_ip_address_utils

HOT_PATCH_VERSION_DICT = {
    "V300R006C00SPC100": ["V300R006C00SPH105*"],
    "V300R003C20": ["V300R003C20SPH001*"],
    "V300R003C20SPC200": ["V300R003C20SPH203*"],
    "V300R005C00SPC300": ["V300R005C00SPH303*"],
}

# 补丁集下载地址
SPECIAL_HOT_PATCH_DOWNLOAD_PATH = {
    "zh": {
        "Default": public_ip_address_utils.get_hot_patch_url("DEFAULT_PATCH_SET_URL_ZH"),
        "V300R003C20": public_ip_address_utils.get_hot_patch_url("V300R003C20_PATCH_SET_URL_ZH")
    },
    "en": {
        "Default": public_ip_address_utils.get_hot_patch_url("DEFAULT_PATCH_SET_URL_EN"),
        "V300R003C20": public_ip_address_utils.get_hot_patch_url("V300R003C20_PATCH_SET_URL_EN")
    },
}

# 某些版本的某些设备类型不需要检查补丁
HOT_PATCH_DEVTYPE_WHITE_DICT = {
    "V300R003C20": ["2800 V3"],
}

SMART_IO_MODULE_4PORT = 37
SMART_IO_MODULE_12PORT = 42


def execute(context, intfType, intfId):
    """备件更换前检查热补丁版本

    :param context(dict): python执行上下文
    :param intfType(str): 接口卡类型
    :param intfId(str): 接口卡ID
    :return: dict: 检查结果 ,格式：{"succ":True,"errMsg":"","suggestion":""}
    """
    cli = contextUtil.getCli(context)
    lang = contextUtil.getLang(context)
    (ok, softwareVersion, hotPatchVersion) = cliUtil.getSystemVersion(cli, lang)
    if not ok:
        BaseFactory.log.info(context, "hot patch version check error.")
        BaseFactory.result.setResultFailByKey(context, "PATCH_VERSION_CHECK_ERROR")
        return

    # 某些版本的某些设备类型不需要检查补丁
    devType = str(context.get("dev").get("type")).strip()
    for key in HOT_PATCH_DEVTYPE_WHITE_DICT.keys():
        if key == softwareVersion and devType in HOT_PATCH_DEVTYPE_WHITE_DICT.get(key):
            BaseFactory.log.info(context, "the version %s no need check patch." % key)
            BaseFactory.result.setResultPass(context)
            return

    # 获取更换卡类型--只有smartIO卡才检查
    rec = FuncFactory.getFruInfo(context, intfType, intfId)
    model = restUtil.Tlv2Rest.getRecordValue(rec, restData.Hardware.IntfModule.MODEL)
    BaseFactory.log.info(context, "model=%s" % model)

    if model != SMART_IO_MODULE_4PORT and model != SMART_IO_MODULE_12PORT:
        BaseFactory.log.info(context, "not SmartIO intf model.")
        BaseFactory.result.setResultPass(context)
        return

    # 检查
    BaseFactory.log.info(context, "softwareVersion=%s, hotPatchVersion=%s" % (softwareVersion, hotPatchVersion))
    (status, errKey, params) = check(context, softwareVersion, hotPatchVersion)
    if status == "PASS":
        BaseFactory.log.info(context, "hot patch version check pass.")
        BaseFactory.result.setResultPass(context)
    elif status == "NOPASS":
        # 支持忽略，弹出高危提示确认
        BaseFactory.log.info(context, "hot patch version check not pass.%s" % type(params))
        (errMsg, sugMsg) = BaseFactory.lang.getErrAndSugDesc(context, errKey, errKey, (), params)
        BaseFactory.log.info(context, "sugMsg=%s" % sugMsg)
        warnigMsg = commonFunction.getRes(lang, "hotPatchVersionCheck_ignore_warning_tips", sugMsg)
        dialogUtil = context['dialogUtil']
        recs = dialogUtil.showWarningDialog(warnigMsg)
        BaseFactory.log.info(context, "confirm the next operation:%s" % str(recs))
        if not recs:
            BaseFactory.result.setResultFailByKey(context, errKey, (), params)
        else:
            BaseFactory.result.setResultPass(context)
    else:
        BaseFactory.log.info(context, "hot patch version check not pass.%s" % type(params))
        BaseFactory.result.setResultFailByKey(context, errKey, (), params)


# 检查补丁版本是否满足条件
def check(context, devVersion, currentPatchVersion):
    lang = contextUtil.getLang(context)
    try:
        if devVersion not in HOT_PATCH_VERSION_DICT.keys():
            return _getResult("PASS", "")
        expectedPatchVersions = HOT_PATCH_VERSION_DICT.get(devVersion)
        BaseFactory.log.info(context, "expected patch version%s" % expectedPatchVersions)
        # 判断标准以最后安装补丁为准，设备的补丁版本必须大于等于补丁白名单列表中的最后一个版本
        lastExptectedPatchVersion = expectedPatchVersions[-1]
        if lastExptectedPatchVersion.endswith('+'):
            if currentPatchVersion < lastExptectedPatchVersion[:-1]:
                (errMsg, params) = _getErrMsg(context, devVersion, expectedPatchVersions)
                return _getResult("NOPASS", errMsg, params)
        elif lastExptectedPatchVersion.endswith('*'):
            if re.search("SPH.99$", currentPatchVersion, re.IGNORECASE):
                (errMsg, params) = _getErrMsg(context, devVersion, expectedPatchVersions, True)
                return _getResult("NOPASS", errMsg, params)
            if currentPatchVersion < lastExptectedPatchVersion[:-1]:
                (errMsg, params) = _getErrMsg(context, devVersion, expectedPatchVersions)
                return _getResult("NOPASS", errMsg, params)
        else:
            if currentPatchVersion != lastExptectedPatchVersion:
                (errMsg, params) = _getErrMsg(context, devVersion, expectedPatchVersions)
                return _getResult("NOPASS", errMsg, params)
        return _getResult("PASS", "")
    except Exception as e:
        BaseFactory.log.error(context, "check patch version exception[%s]" % unicode(e))
        return _getResult("ERROR", "PATCH_VERSION_CHECK_ERROR")


# status可能为“PASS” "NOPASSS", "ERROR"三种
def _getResult(status, errorMsg, params=""):
    return (status, errorMsg, params)


def _getErrMsg(context, devVersion, expectedPatchVersions, isX99=False):
    # rollback--是否需要先回退再安装
    lang = contextUtil.getLang(context)
    expectedVersionStr = ", ".join(expectedPatchVersions)
    versionLaterStr = commonFunction.getRes(lang, "hotpatch_version_later")
    expectedVersionStr = expectedVersionStr.replace('*', versionLaterStr)
    expectedVersionStr = expectedVersionStr.replace('+', versionLaterStr)

    # oem_out FRU场景提示信息
    isOem = contextUtil.isOem(context)
    if isOem:
        return ("PATCH_VERSION_NOPASS_OEM_PATCH", [expectedVersionStr])

    # 从配置中获取补丁集有特有的下载地址
    downPathDict = SPECIAL_HOT_PATCH_DOWNLOAD_PATH.get(lang)
    if devVersion in downPathDict.keys():
        isPatchSet = True
        patchDownPath = downPathDict.get(devVersion)
    else:
        isPatchSet = False
        patchDownPath = downPathDict.get("Default")
    if len(expectedPatchVersions) == 1:
        if not isPatchSet and not isX99:
            return ("PATCH_VERSION_NOPASS_NOTPATCHSET_NOTX99", (patchDownPath, expectedVersionStr))
        elif isPatchSet and not isX99:
            return ("PATCH_VERSION_NOPASS_ISPATCHSET_NOTX99",
                    [patchDownPath, expectedVersionStr, expectedVersionStr])
        elif not isPatchSet and isX99:
            return ("PATCH_VERSION_NOPASS_NOTPATCHSET_ISX99", [patchDownPath, expectedVersionStr])
        else:
            return ("PATCH_VERSION_NOPASS_ISPATCHSET_ISX99",
                    [patchDownPath, expectedVersionStr, expectedVersionStr])
    else:
        # 目前没有需要安装多个补丁的场景
        return ("PATCH_VERSION_NOPASS_MANY_PATCH", [expectedVersionStr])
