#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2020~2022 All rights reserved.
import glob
import os
import json
import shutil
import traceback
from collections import defaultdict
from collections import namedtuple

# noinspection PyUnresolvedReferences
from java.lang import System
# noinspection PyUnresolvedReferences
from java.io import File

from cbb.business.collect.hostinfo import const
from cbb.frame.adapter.restService import RestService
from cbb.frame.base.baseUtil import isArmHighEndDev
from cbb.frame.base.baseUtil import isDoradoDev
from cbb.frame.cli.cliUtil import execCliCmd
from cbb.frame.context import contextUtil
from cbb.frame.rest import restData
from cbb.frame.rest.restUtil import CommonRestService
from cbb.frame.rest.restUtil import Tlv2Rest
from cbb.frame.cli.cliUtil import executeCmdInDebugMode
from cbb.frame.base import baseUtil

InitiatorTypes = dict(FC="223", ISCSI="222", IB="16499")
CMD_TIME_OUT_SECS = 20 * 60


Initiator = namedtuple(
    "Initiator",
    field_names=[
        "ini_wwn_or_iqn",
        "ini_type",
        "host_sn",
        "host_id",
        "ultrapath_ver",
        "is_support_hostinfo_collect",
    ],
)


InitiatorV6 = namedtuple(
    "Initiator",
    field_names=[
        "ini_wwn_or_iqn",
        "ini_type",
        "host_ip",
        "host_id",
        "ultrapath_ver",
        "is_support_hostinfo_collect",
    ],
)


def is_version_6_1(dev_obj):
    """
    是否支持Dorado 6.1 及之上处理信息采集的接口。
    :param dev_obj: 节点
    :return: 结果
    """
    sys_ver = dev_obj.getProductVersion()
    dev_type = str(dev_obj.getDeviceType())
    return any([sys_ver.startswith('6.1.'), baseUtil.is_ocean_protect(dev_type), baseUtil.is_new_oceanstor(dev_type),
                baseUtil.is_micro_dev(dev_type)])


class HostQueryException:
    def __init__(self, err_code, log_msg, res_msg):
        """

        :param err_code:错误码
        :param log_msg:错误消息，日志记录。
        :param res_msg: 国际化错误消息。
        """
        self.error_code = err_code
        self.log_msg = log_msg
        self.res_msg = res_msg


def get_base_url(params_dict):
    """

    :param params_dict:
    :return:
    """
    dev_obj = params_dict.get("devNode")
    base_url = r"https://%s:%s/deviceManager/rest/%s/" % (
        params_dict.get("devIp"),
        str(8088),
        str(dev_obj.getDeviceSerialNumber()),
    )
    return base_url


def get_hosts_of_storage(params_dict):
    """查询存储设备的所有主机
    :param params_dict:
    :return:
    """
    # Typo: `restMamager` Java 中拼写错误，导致此处拼写错误。
    java_map = contextUtil.getContext(params_dict)
    rest_conn_wrapper = contextUtil.getRest(java_map)
    host_data_list = CommonRestService.get4Page(rest_conn_wrapper, "host")
    return host_data_list


def get_host_links_of_storage(params_dict):
    """查询存储设备的所有主机链路信息
    Host Link info example:
    [{'CTRL_ID': '0A',
  'HEALTHSTATUS': '1',
  'ID': '21000024ff55c7e0-0000000000112001',
  'INITIATOR_ID': '21000024ff55c7e0',
  'INITIATOR_NODE_WWN': '0000000000000000',
  'INITIATOR_PORT_WWN': '21000024ff55c7e0',
  'INITIATOR_TYPE': '223',
  'PARENTID': '1',
  'PARENTNAME': 'hl_test',
  'PARENTTYPE': 21,
  'RUNNINGSTATUS': '27',
  'TARGET_ID': '0000000000000001',
  'TARGET_NODE_WWN': '0000000000000000',
  'TARGET_PORT_WWN': '2081ac751d260cc1',
  'TARGET_TYPE': '212',
  'TYPE': 255,
  'ULTRAPATHVERSION': '21.06.056'},
  {'CTRL_ID': '0B',
  'HEALTHSTATUS': '1',
  'ID': '21000024ff55c7e1-0000008001112001',
  'INITIATOR_ID': '21000024ff55c7e1',
  'INITIATOR_NODE_WWN': '0000000000000000',
  'INITIATOR_PORT_WWN': '21000024ff55c7e1',
  'INITIATOR_TYPE': '223',
  'PARENTID': '1',
  'PARENTNAME': 'hl_test',
  'PARENTTYPE': 21,
  'RUNNINGSTATUS': '27',
  'TARGET_ID': '0000000000000001',
  'TARGET_NODE_WWN': '0000000000000000',
  'TARGET_PORT_WWN': '2091ac751d260cc1',
  'TARGET_TYPE': '212',
  'TYPE': 255,
  'ULTRAPATHVERSION': '21.06.056'}]

    :param params_dict:
    :return:
    """
    java_map = contextUtil.getContext(params_dict)
    rest_conn_wrapper = contextUtil.getRest(java_map)
    all_host_data = get_hosts_of_storage(params_dict)
    all_host_link_list = []

    url_pattern = "host_link?INITIATOR_TYPE={ini_type}&PARENTID={host_id}"
    for host_info_dict in all_host_data:
        host_id = host_info_dict.get("ID")
        if host_id is None:
            continue

        for initiator_type in InitiatorTypes.values():
            uri = url_pattern.format(ini_type=initiator_type, host_id=host_id)
            host_links = CommonRestService.get4Big(rest_conn_wrapper, uri)
            all_host_link_list.extend(host_links)
    return all_host_link_list


def is_link_support_collect(params_dict, ultra_path_ver):
    """通过存储设备软件版本、热补丁版本、主机多路径版本信息判断链路是否支持
    主机多路径信息收集。

    :param params_dict:
    :param ultra_path_ver:
    :return:
    """
    dev_obj = params_dict.get("devNode")
    sys_ver = dev_obj.getProductVersion()
    hotpatch_ver = dev_obj.getHotPatchVersion()

    logger = params_dict.get("logger")
    logger.info("sys_ver:{}, patch ver: {}".format(sys_ver, hotpatch_ver))

    # 大于V5R7C70 或者 V5R7C60SPH305及之后的
    is_storage_support_collect = sys_ver.split()[
        0
    ].strip() >= const.SUPPORT_COLLECT_HOST_INFO_BY_STORAGE_SYS_START_VER or (
        sys_ver == const.SUPPORT_COLLECT_HOST_INFO_BY_SYS_C60SPC300
        and hotpatch_ver
        and hotpatch_ver
        >= const.SUPPORT_COLLECT_HOST_INFO_BY_STORAGE_C60SPH305
    )
    return (
        is_storage_support_collect
        and ultra_path_ver
        and ultra_path_ver.strip()
        and tuple(map(int, ultra_path_ver.strip().split(".")))
        >= const.SUPPORT_COLLECT_HOSTINFO_BY_STOR_ULTRAPATH_START_VER
    )


def get_hosts_can_not_be_collected_by_storage(params_dict):
    """获取存储设备需要添加存储设备才能收集主机信息的主机。

    判断标准：主机没有链路，或者不支持免主机采集特性。

    主机名称    启动器wwn/主机iqn   设备SN

    :param params_dict:
    :return: {host_name:[initiator_or_iqn_list],
              host_name1: [initiator_or_iqn_list]
             }
    """

    initiator_list = get_storage_all_initiators(params_dict)
    logger = params_dict.get("logger")
    logger.info("all initiators:{}".format(initiator_list))

    all_initiators_ids = {ini.get("ID") for ini in initiator_list}
    logger.info("all_initiators_ids:{}".format(all_initiators_ids))

    all_host_link_info = get_host_links_of_storage(params_dict)
    logger.info("all host links info:{}".format(all_host_link_info))

    all_host_link_port_wwns_support_collect = {
        link.get("INITIATOR_PORT_WWN")
        for link in all_host_link_info
        if is_link_support_collect(params_dict, link.get("ULTRAPATHVERSION"))
    }
    logger.info(
        "all_host_link_port_wwns_support_collect:{}".format(
            all_host_link_port_wwns_support_collect
        )
    )
    no_link_or_not_support_collect_by_storage = (
        all_initiators_ids - all_host_link_port_wwns_support_collect
    )
    if not no_link_or_not_support_collect_by_storage:
        return {}

    hosts_can_not_be_collected = defaultdict(list)
    for no_link_wwn in no_link_or_not_support_collect_by_storage:
        for initiator in initiator_list:
            if initiator.get("ID") == no_link_wwn:
                host_name = initiator.get("PARENTNAME")
                hosts_can_not_be_collected[host_name].append(no_link_wwn)
                break

    logger.info(
        "hosts_can_not_be_collected:{}".format(hosts_can_not_be_collected)
    )
    return hosts_can_not_be_collected


def get_hosts_can_be_collected_by_storage(params_dict):
    """获取存储设备可以通过免主机采集特性收集主机信息的主机。

    判断标准：主机有链路且支持免主机采集特性。

    主机名称    启动器wwn/主机iqn   设备SN

    :param params_dict:
    :return: {host_name:[initiator_or_iqn_list],
              host_name1: [initiator_or_iqn_list]
             }
    """

    initiator_list = get_storage_all_initiators(params_dict)
    logger = params_dict.get("logger")
    logger.info("all initiators:{}".format(initiator_list))

    all_initiators_ids = {ini.get("ID") for ini in initiator_list}
    logger.info("all_initiators_ids:{}".format(all_initiators_ids))

    all_host_link_info = get_host_links_of_storage(params_dict)
    logger.info("all host links info:{}".format(all_host_link_info))

    all_host_link_port_wwns_support_collect = {
        link.get("INITIATOR_PORT_WWN")
        for link in all_host_link_info
        if is_link_support_collect(params_dict, link.get("ULTRAPATHVERSION"))
    }
    logger.info(
        "all_host_link_port_wwns_support_collect:{}".format(
            all_host_link_port_wwns_support_collect
        )
    )
    if not all_host_link_port_wwns_support_collect:
        return {}

    hosts_can_be_collected = defaultdict(list)
    for initiator_port_wwn in all_host_link_port_wwns_support_collect:
        for initiator in initiator_list:
            if initiator.get("ID") == initiator_port_wwn:
                host_name = initiator.get("PARENTNAME")
                hosts_can_be_collected[host_name].append(initiator_port_wwn)
                break

    logger.info("hosts_can_be_collected:{}".format(hosts_can_be_collected))
    return hosts_can_be_collected


def get_hosts_need_to_be_added_to_inspect(params_dict):
    """获取巡检需要添加的双活主机信息（无法免主机采集或免主机采集失败的双活主机）

    :param params_dict:
    :return: {host_name:[initiator_or_iqn_list],
              host_name1: [initiator_or_iqn_list]
             }
    """
    all_hyper_hosts = get_hyper_hosts(params_dict)
    if not all_hyper_hosts:
        return {}

    logger = params_dict.get("logger")
    all_hyper_host_names = [
        hyper_host.get("NAME") for hyper_host in all_hyper_hosts
    ]
    can_not_be_collected_hosts = get_hosts_can_not_be_collected_by_storage(
        params_dict
    )

    can_not_be_collected_hyper_hosts = dict(
        filter(
            lambda item: item[0] in all_hyper_host_names,
            can_not_be_collected_hosts.items(),
        )
    )
    logger.info(
        "can_not_be_collected_hyper_hosts:{}".format(
            can_not_be_collected_hyper_hosts
        )
    )
    return can_not_be_collected_hyper_hosts


def get_storage_all_initiators(params_dict):
    """获取存储设备的所有启动器
    启动器信息实例：
    [{'FAILOVERMODE': '255',
  'HEALTHSTATUS': '1',
  'ID': '21000024ff7f88c2',
  'ISFREE': 'false',
  'MULTIPATHTYPE': '0',
  'OPERATIONSYSTEM': '255',
  'PARENTID': '0',
  'PARENTNAME': 'Host001',
  'PARENTTYPE': 21,
  'PATHTYPE': '255',
  'RUNNINGSTATUS': '28',
  'SPECIALMODETYPE': '255',
  'TYPE': 223},
 {'FAILOVERMODE': '255',
  'HEALTHSTATUS': '1',
  'ID': '21000024ff55c7e1',
  'ISFREE': 'false',
  'MULTIPATHTYPE': '0',
  'OPERATIONSYSTEM': '255',
  'PARENTID': '1',
  'PARENTNAME': 'hl_test',
  'PARENTTYPE': 21,
  'PATHTYPE': '255',
  'RUNNINGSTATUS': '27',
  'SPECIALMODETYPE': '255',
  'TYPE': 223}]

    :param params_dict:
    :return:
    """
    all_hosts_info = get_hosts_of_storage(params_dict)
    initiator_list = []
    java_map = contextUtil.getContext(params_dict)

    rest_service = RestService(java_map)
    for host_info_dict in all_hosts_info:
        host_id = host_info_dict.get("ID")
        if host_id is None:
            continue
        fc_ini_data = rest_service.getFcInitiator(host_id)
        initiator_list.extend(fc_ini_data)
        iscsi_ini_data = rest_service.getIscsiInitiator(host_id)
        initiator_list.extend(iscsi_ini_data)
    return initiator_list


def get_hyper_hosts(params_dict):
    """获取到双活主机信息。
    cbb.frame.rest.restUtil.CommonRest#getHostRecord
    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    dev_obj = params_dict.get("devNode")
    if not dev_obj.isHyperMetro():
        logger.info("Not hyper metro storage.")
        return []

    hyper_metro_luns = get_hyper_metro_luns(params_dict)
    if hyper_metro_luns is None:
        raise HostQueryException(
            -1,
            "query hyper metro paris failed.",
            const.ERR_KEY_QUERY_HYPER_METRO_PAIR_FAILED,
        )

    if not hyper_metro_luns:
        logger.info("No hyper metro pairs, so no hyper hosts.")
        return []

    host_data_list = get_hosts_of_storage(params_dict)
    host_id_list = [host.get("ID") for host in host_data_list]
    hyper_metro_host_ids = []
    for host_id in host_id_list:
        host_lun_ids = get_host_luns(params_dict, host_id)
        if host_lun_ids is None:
            raise HostQueryException(
                -1,
                "query host lun failed.",
                const.ERR_KEY_QUERY_HOST_LUN_FAILED,
            )
        if not host_lun_ids:
            continue

        if set(host_lun_ids) & set(hyper_metro_luns):
            hyper_metro_host_ids.append(host_id)

    logger.info("All hyper hosts id:{}".format(hyper_metro_host_ids))
    hyper_metro_host_list = list(
        filter(
            lambda host_data: host_data.get("ID") in hyper_metro_host_ids,
            host_data_list,
        )
    )
    logger.info("hyper_metro_host_list:{}".format(hyper_metro_host_list))
    return hyper_metro_host_list


def get_hyper_metro_luns(params_dict):
    """

    :param params_dict:
    :return:
    """
    ssh = params_dict.get("SSH")
    logger = params_dict.get("logger")

    # noinspection PyBroadException
    try:
        hyper_metro_cli_lines = execCliCmd(
            ssh, const.SHOW_HYPER_METRO_PAIR_CMD, logResult=True
        )
    except BaseException:
        logger.error(
            "exec cli cmd exception:{}".format(traceback.format_exc())
        )
        return None

    hyper_metro_lun_ids = []

    for line in hyper_metro_cli_lines.splitlines():
        line = line.strip()
        if not line:
            continue

        fields = line.split()
        if len(fields) != 2 or fields[0].startswith("---"):
            continue

        _, lun_id = fields
        hyper_metro_lun_ids.append(lun_id)
    return hyper_metro_lun_ids


def get_host_luns(params_dict, host_id):
    """查询主机的LUN

    :param params_dict:
    :param host_id:
    :return:
    """
    ssh = params_dict.get("SSH")
    logger = params_dict.get("logger")

    # noinspection PyBroadException
    try:
        host_lun_cli_lines = execCliCmd(
            ssh, const.SHOW_HOST_LUN_CMD.format(host_id), logResult=True
        )
    except BaseException:
        logger.error(
            "exec cli cmd exception:{}".format(traceback.format_exc())
        )
        return None

    host_lun_ids = []
    for line in host_lun_cli_lines.splitlines():
        line = line.strip()
        if not line:
            continue

        fields = line.split()
        if len(fields) < 3 or not fields[0].isdigit():
            continue

        lun_id = fields[0]
        host_lun_ids.append(lun_id)
    return host_lun_ids


def is_need_execute_compatibility_evaluation_with_c60spc300_version(
    params_dict
):
    """查询前端接口卡是否支持热升级, 如果不支持，则需要执行兼容性评估。

    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    # noinspection PyBroadException
    try:
        dev_obj = params_dict.get("devNode")
        sys_ver = dev_obj.getProductVersion().split()[0]

        # 非高端 ARM 不支持热升级，需要兼容性评估。
        product_model = dev_obj.getProductModel()
        if not isArmHighEndDev(product_model, sys_ver):
            logger.info("Non high end arm device, need execute comp eval.")
            return True

        return not (
            is_front_intf_support_hot_upgrade_by_upgrade_path(params_dict)
            and is_purely_fc_net_mode(params_dict)
        )
    except:  # noqa E722
        # 查询失败，认为不需要执行兼容性评估。
        logger.error(
            "Query is need compatibility evaluation exception, sys ver may "
            "not support this interface."
        )
        return False


def is_need_execute_compatibility_with_non_c60spc300_version(params_dict):
    """查询 C70/C61/C62 版本前端接口卡是否支持热升级, 如果不支持，

    则需要执行兼容性评估。

    :param params_dict:
    :return:
    """
    logger = params_dict.get("logger")
    # noinspection PyBroadException
    try:
        dev_obj = params_dict.get("devNode")
        sys_ver = dev_obj.getProductVersion().split()[0]

        # X86 版本一定需要兼容性评估
        if sys_ver.startswith(
            const.NEED_UPG_EVAL_AND_SUPPORTING_HOSTINFO_COLLECT_VERS
        ):
            return True

        java_map = contextUtil.getContext(params_dict)
        rest_conn_wrapper = contextUtil.getRest(java_map)
        upg_target_ver = params_dict.get("upgradeTargetVersion", "").rstrip(" Kunpeng")
        logger.info("upg_target_ver:{}".format(upg_target_ver))
        record = Tlv2Rest.get_intf_upgrade_mode_record(
            rest_conn_wrapper, upg_target_ver
        )
        front_intf_upgrade_mode = Tlv2Rest.getRecordValue(
            record, restData.Upgrade.UpgradeMode.CMO_UPD_FRONT_INTF_UPG_MODE
        )
        is_front_intf_support_hot_upgrade = front_intf_upgrade_mode == 0
        logger.info(
            "is_front_intf_support_hot_upgrade:{}".format(
                is_front_intf_support_hot_upgrade
            )
        )
        return not is_front_intf_support_hot_upgrade
    except:  # noqa E722
        # 查询失败，认为不需要执行兼容性评估。
        logger.error(
            "Query is need compatibility evaluation exception, sys ver may "
            "not support this interface."
        )
        return False


def is_need_execute_compatibility_evaluation(params_dict):
    """
    是否需要执行兼容性评估。
    :param params_dict: 参数字典
    :return:
    """
    if not is_upgrade_mode_need_compatibility_evaluation(params_dict):
        return False

    dev_obj = params_dict.get("devNode")
    sys_ver = dev_obj.getProductVersion().split()[0]
    is_patch_ver = sys_ver == const.SUPPORT_COLLECT_HOST_INFO_BY_SYS_C60SPC300
    if not is_patch_ver:
        return is_need_execute_compatibility_with_non_c60spc300_version(
            params_dict
        )

    return is_need_execute_compatibility_evaluation_with_c60spc300_version(
        params_dict
    )


def is_upgrade_mode_need_compatibility_evaluation(params_dict):
    """
    判断升级模式是否需要执行兼容性评估
    :param params_dict: 参数字典
    :return: 是(True)或否(False)
    """
    upgrade_extend = params_dict.get("upgradeExtend") or {}
    upgrade_mode = upgrade_extend.get("upgradeMode")
    selected_stop_business = upgrade_extend.get("stopBusiness") == "true"
    params_dict.get("logger").info(
        "upgrade_mode:{}, selected_stop_business:{}".format(
            upgrade_mode, selected_stop_business)
    )
    if not upgrade_mode:
        return True
    if upgrade_mode == const.UP_MODE_APOLLO:
        return True
    if upgrade_mode == const.UP_MODE_ROLL and not selected_stop_business:
        return True
    return False


def _parse_yml(yml_pth_name, logger):
    """

    :param yml_pth_name:
    :return:
    """
    upg_paths = []
    try:
        with open(yml_pth_name) as f:
            pair_found = False
            src_ver, dst_ver = None, None
            for line in f:
                if "src_version" in line:
                    src_ver = line.split(":")[1].strip().strip("'")
                    continue

                if "dst_version" in line:
                    dst_ver = line.split(":")[1].strip().strip("'")
                    pair_found = True

                if pair_found and src_ver and dst_ver:
                    upg_paths.append((src_ver, dst_ver))
                    pair_found = False
                    src_ver, dst_ver = None, None
    except IOError as ioe:
        logger.error("parse file: {} error:{}".format(yml_pth_name, ioe))

    return upg_paths


def parse_front_intf_upgrade_path(params_dict):
    """解析前端接口卡支持热升级的路径。

    :param params_dict:
    :return:
    """
    zipdirs = []
    logger = params_dict.get("logger")
    try:
        zipdirs = parse_front_intf_hot_upg_path_core(params_dict)
    except Exception as e:
        logger.error(
            "Parse front intf hot upgrade path exception:{}".format(e)
        )
    finally:
        for tmp_dir in zipdirs:
            shutil.rmtree(tmp_dir, ignore_errors=True)


def parse_front_intf_hot_upg_path_core(params_dict):
    """解析前端接口卡支持热升级的路径。

    :param params_dict:
    :return: 解压压缩包的临时目录
    """
    tool_dir = params_dict.get("toolDir")
    upg_component_pkg_path = os.path.join(
        tool_dir, const.UPGRADE_CHECK_COMPONENT_PKG_PATH
    )
    logger = params_dict.get("logger")
    upg_chk_comp_pkg_pattern = os.path.join(
        upg_component_pkg_path, const.UPGRADE_CHECK_COMPONENT_PKG_PATTERN
    )
    upg_check_component_pkgs = glob.glob(upg_chk_comp_pkg_pattern)
    if not upg_check_component_pkgs:
        logger.error("Upgrade check component packages not found.")
        return []

    tar_class = params_dict.get("PYENGINE.PY_ZIP")
    main_yml_pth_dict = dict()
    zip_dirs = []
    for upg_chk_comp_pkg in upg_check_component_pkgs:
        zip_dir = os.path.join(
            upg_component_pkg_path, upg_chk_comp_pkg.strip(".tar.gz")
        )

        tar_class.decompressTarGzFileAndGetFileNames(upg_chk_comp_pkg, zip_dir)

        main_yml_ptrn = os.path.join(
            zip_dir, const.UPGRADE_CHECK_COMPONENT_MAIN_YML_FILE_PATTERN
        )
        zip_dirs.append(zip_dir)
        main_yml = glob.glob(main_yml_ptrn)
        if main_yml:
            key = "dorado" if "dorado" in upg_chk_comp_pkg else "hybrid"
            main_yml_pth_dict[key] = main_yml[0]

    if not main_yml_pth_dict:
        logger.warn("No main.yml found.")
        return zip_dirs

    support_hot_upg_path_dict = dict()
    for key in main_yml_pth_dict:
        yml_pth_name = main_yml_pth_dict.get(key)
        upg_paths = _parse_yml(yml_pth_name, logger)
        if upg_paths:
            support_hot_upg_path_dict[key] = upg_paths

    if support_hot_upg_path_dict:
        logger.info(
            "Cache front hot upgrade path dict:{}".format(
                support_hot_upg_path_dict
            )
        )
        params_dict["FrontHotUpgradePath"] = support_hot_upg_path_dict
    return zip_dirs


def is_front_intf_support_hot_upgrade_by_upgrade_path(params_dict):
    """根据升级评估包中热升级配置路径判断前端接口卡是否支持热升级。

    :param params_dict:
    :return:
    """
    support_hot_upg_path_dict = params_dict.get("FrontHotUpgradePath", {})
    if not support_hot_upg_path_dict:
        parse_front_intf_upgrade_path(params_dict)
        support_hot_upg_path_dict = params_dict.get("FrontHotUpgradePath", {})

    dev_obj = params_dict.get("devNode")
    logger = params_dict.get("logger")
    upg_src_ver = dev_obj.getProductVersion().split()[0]
    upg_dst_ver = params_dict.get("upgradeTargetVersion")
    if not upg_dst_ver:
        logger.error("Upgrade dest version not found.")
        return True

    upg_dst_ver = upg_dst_ver.strip().split()[0]
    product_model = dev_obj.getProductModel()
    is_dorado = isDoradoDev(product_model)
    path_key = "dorado" if is_dorado else "hybrid"
    paths = support_hot_upg_path_dict.get(path_key, [])
    logger = params_dict.get("logger")
    if not paths:
        logger.warn("No path found.")
        return True

    return (upg_src_ver, upg_dst_ver) in paths


def is_purely_fc_net_mode(params_dict):
    """判断是否纯 FC 组网类型。

    :param params_dict:
    :return:
    """
    host_iqn_or_wwn = batch_query_initiators(params_dict)
    for host_ini_objs in host_iqn_or_wwn.values():
        for ini_obj in host_ini_objs:
            if str(ini_obj.ini_type) != InitiatorTypes.get("FC"):
                return False
    return True


def batch_query_initiators(params_dict):
    """批量查询设备的启动器信息

    :param params_dict:
    :return: {host_sn:("ini_wwn_or_iqn",
             "ini_type",
             "host_sn",
             "host_id",
             "ultrapath_ver",
             "is_support_hostinfo_collect"),
             }
             或(V6 6.1版本)
             {host_ip:("ini_wwn_or_iqn",
             "ini_type",
             "host_ip",
             "host_id",
             "ultrapath_ver",
             "is_support_hostinfo_collect"),
             }
    """
    dev_obj = params_dict.get("devNode")
    if is_version_6_1(dev_obj):
        return batch_query_host_initiators_with_6_1_version(params_dict)

    sys_ver = dev_obj.getProductVersion().split()[0]
    is_patch_ver = sys_ver == const.SUPPORT_COLLECT_HOST_INFO_BY_SYS_C60SPC300

    if not is_patch_ver:
        return batch_query_host_initiators_with_c70_or_later_version(
            params_dict
        )

    return batch_query_host_initiators_with_c60spc300_version(params_dict)


def batch_query_host_initiators_with_c60spc300_version(params_dict):
    """V5R7C60SPH305 补丁版本批量查询主机多路径信息：下发diagnose命令，
    导出主机启动器信息文件，并通过 SFTP 下载文件，并解析主机启动器信息。
    :param params_dict:参数字典
    :return:主机SN 及其对应的主机启动器对下列表
    """
    # send diagnose cmd, download file with SFTP, parse file.
    cli = params_dict.get("SSH")
    lang = params_dict.get("lang")
    logger = params_dict.get("logger")

    tmp_file_path = get_host_ini_in_debug_mode(cli, lang, params_dict)
    if not tmp_file_path:
        return dict()

    res_dict = load_files(tmp_file_path, logger)
    host_sn_iqn_or_wwn = {}
    for host_sn, res_info_list in res_dict.items():
        ini_list = get_host_ini_from_json_data(res_info_list)
        if not ini_list:
            continue
        tmp_ini_list = host_sn_iqn_or_wwn.get(host_sn, [])
        tmp_ini_list.extend(ini_list)
        host_sn_iqn_or_wwn[host_sn] = tmp_ini_list

    logger.info("all_huawei_initiators:{}".format(host_sn_iqn_or_wwn))
    return host_sn_iqn_or_wwn


def get_host_ini_in_debug_mode(cli, lang, params_dict):
    """
    下发diagnose命令，
    导出主机启动器信息文件，并通过 SFTP 下载文件，并解析主机启动器信息
    :param cli: ssh连接
    :param lang: 语言
    :param params_dict: 上下文
    :return: 路径
    """
    dev_node = params_dict.get("devNode")
    logger = params_dict.get("logger")
    sftp = params_dict.get("SFTP")

    cmd = const.HOST_INFO_QUERY_INI_DEBUG
    flag, ret, msg = executeCmdInDebugMode(cli, cmd, True, lang)
    if "SUCCESS" not in ret:
        return ""

    java_tmp_dir = System.getProperty("java.io.tmpdir")
    # noinspection PyUnresolvedReferences
    host_info_local_save_dir = os.path.join(
        java_tmp_dir, const.HOST_INFO_COLLECT_SAVE_DIR
    )
    if not os.path.exists(host_info_local_save_dir):
        os.makedirs(host_info_local_save_dir)

    # noinspection PyUnresolvedReferences
    tmp_file_path = os.path.join(
        host_info_local_save_dir,
        dev_node.getDeviceSerialNumber()
        + "_"
        + "ini_list.json",
    )
    logger.info("get ini in debug:tmp_file_name{}".format(tmp_file_path))
    # noinspection PyUnresolvedReferences
    sftp.getFile(const.HOST_COLLECT_C60_FILE_PATH,
                 File(final_result_pkg_full_name), None)

    return tmp_file_path

def get_host_ini_from_json_data(res_info_list):
    """
    从数据中取出启动器信息
    :param res_info_list: json数据
    :return: 启动器列表
    """
    ini_list = []
    if len(res_info_list) == 0:
        return ini_list
    for ini_info_list in res_info_list:
        if len(ini_info_list) < 6:  # 返回的json数据长度为6.
            continue
        # 不支持的
        if ini_info_list[5] != '1':  # 最后一位表示是否支持收集
            continue
        ini_list.append(Initiator(
            ini_info_list[0],
            ini_info_list[1],
            ini_info_list[2],
            ini_info_list[3],
            ini_info_list[4],
            ini_info_list[5]
        ))
    return ini_list


def load_files(final_result_pkg_full_name, logger):
    """
    使用json解析字符串结果
    :param final_result_pkg_full_name: 文件路径
    :return: 结果列表
    """
    try:
        res_list = []
        with open(final_result_pkg_full_name, 'r') as f:
            res_list = f.readlines()
        os.remove(final_result_pkg_full_name)
        res = "".join(res_list)
        return json.loads(res)
    except Exception as e:
        logger.error(str(e))
        return {}


def batch_query_host_initiators_with_c70_or_later_version(params_dict):
    """批量查询主机多路径信息。

    :param params_dict:参数字典
    :return:主机SN 及其对应的主机启动器对下列表
    """
    java_map = contextUtil.getContext(params_dict)
    rest_conn_wrapper = contextUtil.getRest(java_map)
    logger = params_dict.get("logger")
    initiator_list = CommonRestService.get4Page(
        rest_conn_wrapper, "initiator_info", batch_size=1000
    )
    host_sn_iqn_or_wwn = defaultdict(list)
    for ini_info in initiator_list:
        host_sn = ini_info.get("hostSn")
        ini_wwn_or_iqn = ini_info.get("ID")
        host_id = ini_info.get("hostId")
        ultrapath_ver = ini_info.get("ultraPathVersion")
        is_support_hostinfo_collect = ini_info.get(
            "isSupportHostInfoCollection"
        )
        ini_type = ini_info.get("initiatorType")
        ini_obj = Initiator(
            ini_wwn_or_iqn,
            ini_type,
            host_sn,
            host_id,
            ultrapath_ver,
            is_support_hostinfo_collect,
        )
        host_sn_iqn_or_wwn[host_sn].append(ini_obj)
    logger.info("all_huawei_initiators:{}".format(host_sn_iqn_or_wwn))
    return host_sn_iqn_or_wwn


def batch_query_host_initiators_with_6_1_version(params_dict):
    """批量查询主机多路径信息。

    :param params_dict:参数字典
    :param batch_size:一次获取数量
    :return:主机SN 及其对应的主机启动器对下列表
    """
    java_map = contextUtil.getContext(params_dict)
    rest_conn_wrapper = contextUtil.getRest(java_map)
    logger = params_dict.get("logger")
    initiator_list = CommonRestService.get4Page(
        rest_conn_wrapper, "initiator_info", batch_size=100
    )
    host_ip_iqn_or_wwn = defaultdict(list)
    for ini_info in initiator_list:
        host_ip = ini_info.get("hostIp")
        ini_wwn_or_iqn = ini_info.get("ID")
        host_id = ini_info.get("hostId")
        ultrapath_ver = ini_info.get("ultraPathVersion")
        is_support_hostinfo_collect = ini_info.get(
            "isSupportHostInfoCollection"
        )
        ini_type = ini_info.get("initiatorType")
        ini_obj = InitiatorV6(
            ini_wwn_or_iqn,
            ini_type,
            host_ip,
            host_id,
            ultrapath_ver,
            is_support_hostinfo_collect,
        )
        host_ip_iqn_or_wwn[host_ip].append(ini_obj)
    logger.info("all_huawei_initiators:{}".format(host_ip_iqn_or_wwn))
    return host_ip_iqn_or_wwn


def get_hosts_can_be_collected_via_storage(params_dict):
    """查询可以被免主机收集的主机信息及其所有启动器信息，调用协议新接口实现。

    :param params_dict:
    :return:
    """
    host_iqn_or_wwn = batch_query_initiators(params_dict)
    params_dict["all_huawei_initiators"] = host_iqn_or_wwn
    host_to_wwn_or_iqns_dict = defaultdict(list)
    for host in host_iqn_or_wwn:
        ini_obj_list = host_iqn_or_wwn.get(host)
        support_collect_ini_list = list(
            filter(
                lambda ini: ini.is_support_hostinfo_collect == "1",
                ini_obj_list,
            )
        )
        if not support_collect_ini_list:
            continue

        wwn_or_iqn_list = [
            ini.ini_wwn_or_iqn for ini in support_collect_ini_list
        ]
        host_to_wwn_or_iqns_dict[host] = list(set(wwn_or_iqn_list))
    return host_to_wwn_or_iqns_dict
