# coding:utf-8
"""
说明：首先调用本文件中execute_analyze方法获取解析后的config对象。
然后调用get_xx_info_from_running_data从config中获取对象值。
@version: Toolkit V200R006C00
@time: 2020/03/03
@file: running_data_analyze_util.py
@function: config解析
@modify:
"""
import defusedxml.ElementTree as ET
import codecs
import json
import traceback

import com.huawei.ism.tool.obase.db.SqliteManager as SqliteManager

from cbb.frame.util import sqlite_util

# 纵向表格分隔符
SPLIT_FLAG = ":"
# 横向表格结束符
STANDARD_FLAG = "#"

# 纵向表格分隔符
SPLIT_FLAG = ":"
# 横向表格结束符
STANDARD_FLAG = "#"

# 创建数据库DDL
CREATE_TABLE_SQL_STR = """CREATE TABLE if not exists [{table_name}] 
({column_list_sql_str})"""

# 查询数据通用接口
QUERY_SQL = "select * from '{}'"


class RunningDataAnalyze:
    def __init__(self, path, rule_dict, conn, logger):
        self.path = path
        self.rule_dict = rule_dict
        # 数据库连接
        self.conn = conn
        self.logger = logger
        self.res_data = {}
        # 横向数据开始
        self.hor_data_start = False
        # 横向记录解析开始
        self.hor_data_record_start = False
        # 表头找到
        self.hor_header_find = False
        # 处理特殊key开始结束的标识
        self.special_data_start_flag = False
        # 处理特殊key下面存在多组数据对象组时，每一组的结束符标识
        self.special_cur_data_end_flag = False
        # 标记位，垂直分布对象开始
        self.vertical_data_start_flag = False
        # 发现垂直KEY
        self.vertical_start_flag = False
        # 对象计数
        self.count_obj_num = 0
        # 对象总数
        self.obj_num = 0
        # 特殊处理的KEY
        self.special_obj_key = ""
        # 特殊对象属性个数
        self.special_obj_attribute_len = 0
        # 横向表头
        self.hor_header_dict = {}
        # 横向表临时列表
        self.hor_tmp_list = []
        # 特殊对象
        self.special_dict = {}
        # 特殊对象列表
        self.special_list = []
        # host group 级别临时对象
        self.tmp_data = {}
        # 多个host group 级别的临时对象列表
        self.tmp_data_list = []
        # 垂直分布部分对象可能存在总的数量不正确，导致读取对象错位，
        # 比如LUN正在销毁，就可能读取下一个对象的数据
        self.ver_obj_end_line_num = 0

    def analyze_running_data(self):
        """
        解析方法入口
        :return:
        """
        line_num = 0
        item_keys = self.rule_dict.keys()

        tmp_obj_key = ""
        try:
            # 当电子标签有乱码时使用普通open和with open 就会异常截止导致数据丢失。
            f = codecs.open(self.path, "r")
            empty_line = 0
            last_line = ""
            while empty_line < 10:
                line = f.readline()
                line_num += 1
                if not line:
                    empty_line += 1
                else:
                    empty_line = 0

                is_find_keys, obj_key = find_key(line, item_keys)
                # 是否找到对象
                if is_find_keys:
                    # 标记横向或纵向开始
                    tmp_obj_key = self.set_obj_start_flag(obj_key, line)
                    last_line = line
                    continue

                if self.vertical_start_flag:
                    # 防止干扰项导致计算错误。
                    exclude_line_list = self.rule_dict.get(
                        tmp_obj_key, {}
                    ).get("excludeLine", [])
                    if is_exclude_line(exclude_line_list, line):
                        continue

                    obj_key_dict = self.rule_dict.get(tmp_obj_key)
                    self.get_vertical_obj(
                        line, tmp_obj_key, obj_key_dict, last_line
                    )

                elif self.hor_data_start:
                    obj_key_dict = self.rule_dict.get(tmp_obj_key)
                    self.get_horizontal_obj(line, tmp_obj_key, obj_key_dict)

                last_line = line
            self.logger.logInfo(
                "res_data:{0}".format(str(self.res_data.keys()))
            )
            f.close()
        except Exception as e:
            self.logger.logInfo("error dick:{}".format(self.rule_dict.get(tmp_obj_key)))
            self.logger.logInfo("analyze except:{}, \n{}".format(
                str(e), traceback.format_exc()))
            raise
        return self.res_data

    def set_obj_start_flag(self, obj_key, line):
        """
        标记横向或纵向开始
        :param obj_key:
        :param line:
        :return:
        """
        analyze_type = self.rule_dict.get(obj_key, {}).get("analyzeType")
        self.logger.logInfo(
            "find key:{0},type:{1},"
            "line:{2}".format(obj_key, analyze_type, line)
        )
        tmp_list = line.split(SPLIT_FLAG)
        # 有多少个对象
        if len(tmp_list) >= 2 and tmp_list[1].strip().isdigit():
            obj_num = int(tmp_list[1].strip())
            self.obj_num = obj_num

        if analyze_type == "horizontal":
            self.hor_data_start = True
            tmp_obj_key = obj_key
            return tmp_obj_key

        if analyze_type == "vertical":
            self.vertical_start_flag = True
            tmp_obj_key = obj_key
            return tmp_obj_key

    def get_horizontal_obj(self, line, tmp_obj_key, rule_dict):
        """
        获取横向表格的对象
        1. 找到表头
        2. 按列头key加空格的正则分隔找到列头开始结束位置，只获取配置了的属性列
        3. 对数据行按开始结束位置切割，装填
        :param line: 当前行
        :param tmp_obj_key: 已找到关键对象开始的key
        :param rule_dict: 规则文件
        :param res_data: 结果容器
        :return:
        """
        import re

        head_line = rule_dict.get("headline")
        split_flag = rule_dict.get("splitLineFlag", "")
        attribute_list = rule_dict.get("attribute")
        # 找到配置的关键信息，定义横向表格开始，获取并跳过当前表头行
        if head_line in line:
            self.hor_data_record_start = True
            return

        # 如果横向数据还未开始，继续循环
        if not self.hor_data_record_start:
            return

        # 找到列头，并切割记录列的开始结束。
        if not self.hor_header_find:
            for key in attribute_list:
                reg_split = re.compile(r"%s\s*" % key)
                start_pos = line.find(key)
                match = reg_split.search(line)
                if match:
                    end_pos = match.end()
                    self.hor_header_dict[key] = {
                        "start_pos": start_pos,
                        "end_pos": end_pos,
                    }
            if self.hor_header_dict:
                self.hor_header_find = True
                return

        # 判断当前对象整体数据是否结束
        if len(self.hor_tmp_list) == self.obj_num or line.strip().count(
            STANDARD_FLAG
        ) == len(line.strip()):
            self.hor_data_start = False

            self.hor_data_record_start = False

            self.hor_header_find = False

            # 记录对象key的数据。
            self.save_to_db(tmp_obj_key, self.hor_tmp_list)
            self.hor_tmp_list = []
            self.hor_header_dict = {}
            self.obj_num = None
            return

        # 切割列，装填行数据
        if self.hor_header_find:
            res = {}
            tmp_res_list = line.split()
            if split_flag is True and len(tmp_res_list) == len(attribute_list):
                for i in range(len(attribute_list)):
                    res[attribute_list[i]] = tmp_res_list[i]
            else:
                for key in self.hor_header_dict:
                    start_pos = self.hor_header_dict.get(key, {}).get(
                        "start_pos", 0
                    )
                    end_pos = self.hor_header_dict.get(key, {}).get(
                        "end_pos", 0
                    )
                    value = line[start_pos:end_pos].strip()
                    res[key] = value
            if res:
                self.hor_tmp_list.append(res)

    def mark_special_obj_start(self, special_obj_attr_dict, last_line):
        """
        需要特殊处理的 obj key
        :param special_obj_attr_dict:
        :param last_line:
        :return:
        """
        special_last_lines_keys = special_obj_attr_dict.keys()
        tmp_s_key = [
            s_key for s_key in special_last_lines_keys if s_key in last_line
        ]
        # Number of Host 这一层中 Host Group INFO这个特殊KEY找到了
        if tmp_s_key:
            self.special_obj_key = tmp_s_key[0]
            self.special_obj_attribute_len = len(
                special_obj_attr_dict.get(self.special_obj_key, []))
            self.special_data_start_flag = True

    def deal_special_obj(self, special_obj_attr_dict, line):
        """
        处理特殊属性 比如host group 下的
        HOST INFO属性：有多个host存放在special_list中。
        :param special_obj_attr_dict:
        :param line:
        :return:
        """
        sub_obj_keys = special_obj_attr_dict.get(self.special_obj_key, [])
        self.special_list = self.tmp_data.get(self.special_obj_key, [])
        # 可能赋初值是空
        if not self.special_list:
            self.special_list = []

        key_value = line.split(SPLIT_FLAG)

        # 当前数据行的key
        special_tmp_key = key_value[0].strip()
        # 可能IQN值中带:
        special_tmp_value = "".join(key_value[1:]).strip()

        # 如果当前数据行的key在特殊对象属性中，则开始记录数据
        if special_tmp_key in sub_obj_keys:
            special_cur_data_end_flag = True
        else:
            special_cur_data_end_flag = False

        # 判定一组数据结束，如果key在special_dict中存在重复的key，则表明已经是第二组数据开始
        # Host Group INFO第二组对象开始进来。
        if (
            special_tmp_key in self.special_dict.keys()
            and special_cur_data_end_flag
        ):
            flag = check_obj_in_list(self.special_dict, self.special_list)
            # 找个判断属性的方法判断是否在里面。要优化
            if not flag:
                self.special_list.append(self.special_dict)
            self.special_dict = dict()
            self.special_dict[special_tmp_key] = special_tmp_value
            self.tmp_data[self.special_obj_key] = self.special_list

        # Host Group INFO 每个对象都进来赋值
        if special_cur_data_end_flag:
            self.special_dict[special_tmp_key] = special_tmp_value
            self.tmp_data[self.special_obj_key] = self.special_list

        # Host Group INFO 结束。下一个字段Host Port开始
        if not special_cur_data_end_flag:
            flag = check_obj_in_list(self.special_dict, self.special_list)
            # 找个判断属性的方法判断是否在里面。要优化
            if not flag:
                self.special_list.append(self.special_dict)
            self.special_dict = {}
            self.tmp_data[self.special_obj_key] = self.special_list
            self.tmp_data[special_tmp_key] = special_tmp_value
            self.special_data_start_flag = False
            self.special_list = []

        # 最后一个对象时：
        if self.special_obj_attribute_len == len(self.special_dict):
            flag = check_obj_in_list(self.special_dict, self.special_list)
            # 找个判断属性的方法判断是否在里面
            if not flag:
                self.special_list.append(self.special_dict)
                self.tmp_data[self.special_obj_key] = self.special_list

        if not special_cur_data_end_flag:
            self.special_data_start_flag = False

    def get_vertical_obj(self, line, tmp_obj_key, obj_key_dict, last_line):
        """
        纵向表格解析方法
        主要方法：
        1. 发现头部，并找到总数。
        2. 使用:分隔找key和vlaue。
        3. 判断结束，主要是获取的对象和总数相等。
        :param line: 当前行
        :param tmp_obj_key: 已找到关键对象开始的key
        :param obj_key_dict: 已找到关键对象属性字段
        :param last_line: 上一行数据
        :return:
        """
        if not self.vertical_start_flag:
            return

        # 过滤空行，
        if SPLIT_FLAG in line:
            self.vertical_data_start_flag = True
            self.ver_obj_end_line_num = 0
        else:
            if not line.strip():
                self.ver_obj_end_line_num += 1
            self.vertical_data_start_flag = False

        if self.vertical_data_start_flag:
            # 特殊OBJ对象处理
            # Number of Host 这一层中查找Host Group INFO这个特殊KEY
            special_obj_attr_dict = obj_key_dict.get("special_attr")
            if special_obj_attr_dict:
                # 标记特殊处理开始
                self.mark_special_obj_start(special_obj_attr_dict, last_line)

            if self.special_data_start_flag:
                # 处理特殊属性 比如host group 下的HOST INFO属性：
                # 有多个host存放在special_list中。
                self.deal_special_obj(special_obj_attr_dict, line)
            else:
                key_value = line.split(SPLIT_FLAG)
                # 可能IQN值中带:
                tmp_key = key_value[0].strip()
                self.tmp_data[tmp_key] = "".join(key_value[1:]).strip()

        # 每个普通对象存入list; 当且仅当对象之间的空行换行时，对象的描述信息换行不能结束该对象。
        if SPLIT_FLAG not in line and not line.strip() and self.tmp_data:
            min_attr_num = obj_key_dict.get("minAttrNumber")
            # 排除空行干扰
            if not min_attr_num or (
                    min_attr_num and len(self.tmp_data) >= min_attr_num):
                self.vertical_data_start_flag = False
                self.count_obj_num += 1
                self.tmp_data_list.append(self.tmp_data)
                self.tmp_data = {}

        # 所有对象都补齐,赋值给大key，如果对象计数等于总数时退出当前对象。
        # 双空行也表示对象结束，解决：当LUN被销毁时，可能存在会解析错位的问题。
        if (
            self.obj_num == 0 or self.count_obj_num == self.obj_num
            or self.ver_obj_end_line_num >= 2
        ):
            self.vertical_start_flag = False
            self.save_to_db(tmp_obj_key, self.tmp_data_list)
            self.count_obj_num = 0
            self.special_data_start_flag = False
            self.ver_obj_end_line_num = 0
            self.tmp_data_list = []
            self.special_dict = {}
            self.special_list = []
            self.tmp_data = {}

    def save_to_db(self, table_name, value_list):
        """
        将解析后的数据保存到数据库
        当不使用批量提交时，效率及低。
        :param table_name: 数据库名称
        :param value_list: 需要插入的数据
        :return:
        """
        if not value_list:
            return

        table_name = table_name.rstrip(":")

        # 创建数据库，先删除旧的。
        drop_sql = "drop table if exists '{}'".format(table_name)
        sqlite_util.drop_table(self.conn, drop_sql)

        insert_sql = "insert into '{0}'({1}) values({2})"

        index = 0
        # 设置每1000条数据提交一次。
        self.conn.setAutoCommit(False)
        column_list = []
        # 存在少属性的对象，导致创建表的时候少字段。
        for e in value_list:
            tmp_list = e.keys()
            for e_key in tmp_list:
                if e_key not in column_list:
                    column_list.append(e_key)

        # 创建数据库
        insert_column_sql = ",".join(
            ["[{0}] VARCHAR2".format(column) for column in column_list]
        )
        ddl_sql_str = CREATE_TABLE_SQL_STR.format(
            table_name=table_name,
            column_list_sql_str=insert_column_sql,
        )
        self.logger.logInfo("create_table [%s]" % ddl_sql_str)
        sqlite_util.create_table(self.conn, ddl_sql_str)
        self.conn.commit()

        for e in value_list:
            index += 1
            if index % 1000 == 0:
                self.conn.commit()

            value_str_list = []
            value_column_list = []
            for e_key in column_list:
                value_column_list.append("'{0}'".format(e_key))
                value_str_list.append("'{0}'".format(json.dumps(e.get(e_key, ''))))

            insert_sql_str = insert_sql.format(
                table_name,
                ",".join(value_column_list),
                ",".join(value_str_list),
            )
            SqliteManager().update(self.conn, insert_sql_str)
        self.conn.commit()
        self.conn.setAutoCommit(True)


def deal_horizontal_attr(item_node, version_dict, analyze_type, obj_key):
    """
    处理横向表格
    :param item_node:
    :param version_dict:
    :param analyze_type:
    :param obj_key:
    :return:
    """
    attr_node_list = item_node.getiterator("attr")
    if not attr_node_list:
        return
    attr_list = attr_node_list[0].text.strip().split(",")

    headline_node_list = item_node.getiterator("headline")
    if not headline_node_list:
        return
    headline_str = headline_node_list[0].text.strip()

    split_flag_list = item_node.getiterator("splitLineFlag")
    split_flag = False
    if split_flag_list:
        split_flag = bool(split_flag_list[0].text.strip())
    version_dict[obj_key] = {
        "attribute": attr_list,
        "headline": headline_str,
        "analyzeType": analyze_type,
        "splitLineFlag": split_flag,
    }


def deal_vertical_attr(item_node, version_dict, analyze_type, obj_key):
    """
    处理纵向对象
    :param item_node:
    :param version_dict:
    :param analyze_type:
    :param obj_key:
    :return:
    """
    tmp_dict = {"analyzeType": analyze_type}

    exclude_line_node_list = item_node.getiterator("excludeLine")
    if exclude_line_node_list:
        tmp_dict["excludeLine"] = (
            exclude_line_node_list[0].text.strip().split(",")
        )
        version_dict[obj_key] = tmp_dict

    min_attr_num = item_node.getiterator("minAttrNumber")
    if min_attr_num:
        tmp_dict["minAttrNumber"] = int(min_attr_num[0].text.strip())
        version_dict[obj_key] = tmp_dict

    special_node_list = item_node.getiterator("special_inner_attr")
    if not special_node_list:
        version_dict[obj_key] = tmp_dict
        return

    special_attr_node_list = special_node_list[0].getiterator("attr")
    if not special_attr_node_list:
        version_dict[obj_key] = tmp_dict
        return

    special_dict = {}
    for special_attr_node in special_attr_node_list:
        attr_key_list = special_attr_node.text.strip().split(",")
        special_attr_key = special_attr_node.attrib.get("key")
        special_dict[special_attr_key] = attr_key_list
    tmp_dict["special_attr"] = special_dict

    version_dict[obj_key] = tmp_dict


def parse_xml(rule_path):
    """
    解析running data的规则配置文件
    :param rule_path: 规则文件绝对路径
    :return: 解析后的字典
    """
    xmlElementTree = ET.parse(rule_path)
    checkItems = xmlElementTree.getiterator("productVersion")
    rule_dict = {}
    for versionItem in checkItems:
        version_dict = {}
        version = versionItem.attrib.get("version")
        # 处理风险版本检查
        item_node_list = versionItem.getiterator("item")
        for item_node in item_node_list:
            obj_key = item_node.attrib.get("key")
            analyze_type_list = item_node.getiterator("analyzeType")
            if not analyze_type_list:
                continue

            analyze_type = analyze_type_list[0].text.strip()

            if analyze_type == "horizontal":
                deal_horizontal_attr(
                    item_node, version_dict, analyze_type, obj_key
                )

            elif analyze_type == "vertical":
                deal_vertical_attr(
                    item_node, version_dict, analyze_type, obj_key
                )
        rule_dict[version] = version_dict

    return rule_dict


def execute_analyze(
    running_data_file_path, p_version, rule_path, conn, logger
):
    """
    解析running data文件，融合存储和Dorado V6解析方法差异较大，
    需要按版本区分
    注意：解析失败会抛异常出来，需自己处理
    :param running_data_file_path: config文件绝对路径
    :param p_version: 当前版本（可不带SPC）
    :param rule_path: 规格文件绝对路径
    :param conn: 数据库连接
    :param logger: 日志
    :return: 返回解析后的字典{obj_key:[{key:value}...]}
    """
    import re

    current_version_rule_dict = {}
    # 解析xml
    rule_dict = parse_xml(rule_path)
    logger.logInfo("rule dict is: {0}".format(str(rule_dict)))
    reg_versions = rule_dict.keys()
    for reg_version in reg_versions:
        reg = re.compile(reg_version)
        res = reg.match(p_version)
        if res:
            logger.logInfo(
                "version is:{0},match version: {1}".format(
                    p_version, str(reg_version)
                )
            )
            current_version_rule_dict = rule_dict.get(reg_version)
            break

    if current_version_rule_dict:
        # 通过类来解决多线程并发问题
        RunningDataAnalyze(
            running_data_file_path, current_version_rule_dict, conn, logger
        ).analyze_running_data()
    return True


def get_mapping_info_from_running_data(conn):
    """
    获取config中mapping对象信息
    :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of Mapping"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_mapping_view_info_from_running_data(conn):
    """
    获取config中mapping view对象信息
    :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of Mapping View"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_lun_group_info_from_running_data(conn):
    """
    获取config中LUN GROUP对象信息
     :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of Lun Group"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_host_group_info_from_running_data(conn):
    """
    获取config中HOST GROUP对象信息
     :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of Host Group"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_host_info_from_running_data(conn):
    """
    获取config中HOST对象信息
    :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of Host"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_port_group_info_from_running_data(conn):
    """
    获取config中PORT GROUP对象信息
     :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of Port Group"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_fc_initiator_info_from_running_data(conn):
    """
    获取config中FC启动器对象信息，
    重点特殊说明：1. 融合存储config中没有fc启动器，必须使用命令查询
    2. ISCSI启动器不能从config中获取，被加密。
     :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number of FC Inititaor"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_host_group_host_info_from_running_data(conn):
    """
    获取config中主机组下的主机对象信息
    特殊说明：只有dorado v6才有此key。融合存储只能通过主机组-》主机
     :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number Of Relation For Host Group And Host"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_host_lun_info_from_running_data(conn):
    """
    获取config中主机和LUN的关系信息
    特殊说明：只有dorado v6才有此key。
    融合存储只能通过MAPPING VIEW ->LUN组、LUN,MAPPING VIEW ->主机组->主机
    :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number Of Relation For Host, LUN And Port"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_lun_group_lun_info_from_running_data(conn):
    """
        获取config中LUN组下的LUN对象信息
        特殊说明：只有dorado v6才有此key。
        融合存储通过查询LUN组获取
        :param conn: 数据库连接
        :return: 对象字典列表[{obj_key:value}...]
        """
    table_name = "Number Of Relation For Lun Group And Lun"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_port_group_port_info_from_running_data(conn):
    """
    获取config中端口组下的端口对象信息
    特殊说明：只有dorado v6才有此key。
    融合存储通过查询LUN组获取
    :param conn: 数据库连接
    :return: 对象字典列表[{obj_key:value}...]
    """
    table_name = "Number Of Relation For Port Group And Port"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )


def get_lun_info_from_running_data(conn):
    """
    融合存储和Dorado v6存在差异。of一个大写一个小写。
    :param conn:数据库连接
    :return:对象字典列表[{obj_key:value}...]
    """
    table_name_v6 = "Number of Lun"
    table_name = "Number Of Lun"
    res = deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(table_name))
    )
    if not res:
        res = deal_dumps_value(
            SqliteManager().query(conn, QUERY_SQL.format(table_name_v6))
        )
    return res


def get_kubernetes_data_info_from_running_data(conn):
    """
        获取Kubernetes Running Data对象信息
        特殊说明：只有dorado v6才有此key。
        :param conn: 数据库连接
        :return: 对象字典列表[{obj_key:value}...]
        """
    key_name = "#####################################Kubernetes Running Data"
    return deal_dumps_value(
        SqliteManager().query(conn, QUERY_SQL.format(key_name))
    )


def deal_dumps_value(data_list):
    """
    处理被dumps的数据
    :param data_list:
    :return:
    """
    for e_data in data_list:
        for e_key, e_value in e_data.items():
            e_data[e_key] = loads_data(e_value)
    return data_list


def loads_data(data):
    try:
        return json.loads(data)
    except TypeError:
        return data

def find_key(line, keys):
    """
    查找对应行是否存在对象关键字
    :param line:
    :param keys:
    :return:
    """
    for key in keys:
        if line.strip().startswith(key):
            return True, key
    return False, ""


def check_obj_in_list(obj, obj_list):
    """
    对象不能重复添加
    :param obj:
    :param obj_list:
    :return:
    """
    for tmp_obj in obj_list:
        exist_flag = True
        for key in obj:
            if obj.get(key) != tmp_obj.get(key):
                exist_flag = False
                break
        if exist_flag:
            return True

    return False


def is_exclude_line(exclude_list, line):
    """
    排除一些干扰行
    :param exclude_list:
    :param line:
    :return:
    """
    return any([line for key in exclude_list if key in line])
