﻿# coding:utf-8
# noinspection PyUnresolvedReferences
import time

import graphUtilForDorado18000
# noinspection PyUnresolvedReferences
from java.lang import Exception as JException

from cbb.business.operate.expansion import common
from cbb.frame.base import baseUtil
from cbb.frame.context import contextUtil
from cbb.frame.tlv import tlvData
from cbb.frame.tlv import tlvUtil
from cbb.frame.devGraph import devGraphCfg

# 机架的宽度 162
RACK_WIDTH = 162
# 机架的高度 462
RACK_HEIGHT = 462
# 机架高度空白
RACK_TOP_BLANK = 22
# 占位CELL的宽度  29
PLACE_HOLDER_CELL_WIDTH = 29
# 具有标题的panel高度
TITLE_PANEL_HEIGHT = 30
# 常量2，用于求某个数值的一半
HALF_TWO = 2


class DeviceStatus:
    # 默认状态
    DEFAULTS = 0
    # 正常状态
    NORMAL = 1
    # 异常状态
    ABNORMAL = 2
    # 选中状态
    SELECTED = 3
    # 点灯状态
    LIGHTING = 4


def drawGraph(context, propDict={}, isReloadGraph=False):
    """绘制设备图(接口层，根据是否是Dorado18000设备走不同分支), 用于FRU

    :param context:上下文对象
    :param propDict: 支持如下key:
            hardwareTypeList: 可选，硬件类型集合（以字符串形式传入，
                值为OM对象的枚举值，如"BAY","ENCLOSURE"），
                默认支持所有的硬件设备图绘制
            isBack: 可选，值为True表示绘制后视图，值为False表示绘制前视图，
                默认值为False
    :param isReloadGraph: 是否刷新设备图
    :return:
    """
    graphUtilForDorado18000.drawGraph(context, isReloadGraph, **propDict)


def drawGraphForExp(context, propDict={}):
    """绘制设备图，用于扩容

    :param context: 上下文对象
    :param propDict: 支持如下key:
            hardwareTypeList: 可选，硬件类型集合（以字符串形式传入，
            值为OM对象的枚举值，如"BAY","ENCLOSURE"），默认支持所有的硬件设备图绘制
            isBack: 可选，值为True表示绘制后视图，值为False表示绘制前视图，
                    默认值为False
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        hardwareTypeList = propDict.get("hardwareTypeList",
                                        devGraphCfg.DEFAULT_HARD_WARE_LIST)
        isBack = propDict.get("isBack", False)
        tlv = contextUtil.getTlv(context)

        hardwareList = []
        tlvHardwareList = []

        for hardwareType in hardwareTypeList:
            try:
                grapList = getHardwareGraphList(tlv, hardwareType)
                hardwareList.extend(grapList)
                tlvHardwareList.extend(grapList)
                logger.logInfo(
                    "get hardware[%s] info successfully." % hardwareType)
            except (JException, Exception) as exception:
                logger.logInfo("get hardware[%s] info fail." % hardwareType)
                logger.logException(exception)

        logger.logInfo("hardwareList:%s" % hardwareList)

        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {"graphConfigPath": devGraphCfg.DEV_GRAPH_CFG_FILE,
                  "deviceImgCfgPath": devGraphCfg.DEV_GRAPH_IMG_PATH}
        devGraphUtils.buildGraph(cfgMap, hardwareList, isBack, False)
        contextUtil.setItem(context, "hardwareList", hardwareList)
        contextUtil.setItem(context, "tlvHardwareList", tlvHardwareList)
        return

    except (JException, Exception) as exception:
        logger.logException(exception)

    return


def highLightHardwares(context, modules):
    """批量高亮硬件

    :param context: 上下文对象
    :param modules: 硬件对象 [{hardwareId:hardwareType}]
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)
        for module in modules:
            for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
                if module.get("hardwareId", "") == hardwareInfo["id"] \
                        and module.get("hardwareType", "") == \
                        hardwareInfo["typeName"]:
                    devGraphUtils = context.get("DevGraphUtils")
                    devGraphUtils.highlightModule(hardwareInfo, 'selected')

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo("highlight hardware failure(%s)" % str(modules))
    return False


def highLightHardware(context, hardwareId, hardwareType):
    """高亮硬件

    :param context: 上下文对象
    :param hardwareId: 硬件ID
    :param hardwareType: 硬件类型
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if hardwareId == hardwareInfo["id"] \
                    and hardwareType == hardwareInfo["typeName"]:
                devGraphUtils = context.get("DevGraphUtils")
                clearGraph(context)
                devGraphUtils.highlightModule(hardwareInfo, 'selected')
                return True

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo("highlight hardware failure(hardwareId:%s, "
                   "hardwareType:%s)" % (hardwareId, hardwareType))
    return False


def highLightHardwareByLocation(context, hardwareLocation, hardwareType):
    """通过location高亮硬件

    :param context: 上下文对象
    :param hardwareLocation: 硬件location
    :param hardwareType: 硬件类型（以字符串形式传入
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        clearGraph(context)
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if hardwareLocation == hardwareInfo["location"] \
                    and hardwareType == hardwareInfo["typeName"]:
                devGraphUtils = context.get("DevGraphUtils")
                devGraphUtils.highlightModule(hardwareInfo, 'selected')
                return True
    except (JException, Exception) as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        pass

    logger.logInfo(
        "highlight hardware failure(hardwareLocation:%s, "
        "hardwareType:%s)" % (hardwareLocation, hardwareType))
    return False


def highLightHardwareByName(context, hardwareName, hardwareType):
    """通过name高亮硬件

    :param context: 上下文对象
    :param hardwareName: 硬件name
    :param hardwareType: 硬件类型（以字符串形式传入
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        clearGraph(context)
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if hardwareName == hardwareInfo["name"] \
                    and hardwareType == hardwareInfo["typeName"]:
                devGraphUtils = context.get("DevGraphUtils")

                devGraphUtils.highlightModule(hardwareInfo, 'selected')
                return True
    except (JException, Exception) as exception:
        contextUtil.handleException(context, exception)
        logger.logException(exception)
        pass

    logger.logInfo("highlight hardware failure(hardwareName:%s, "
                   "hardwareType:%s)" % (hardwareName, hardwareType))
    return False


def highLightCable(context, port1Id, port2Id, portType):
    """高亮线缆

    :param context: 上下文对象
    :param port1Id: 源端口ID
    :param port2Id: 目的端口ID
    :param portType: 端口类型
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)

        port1Info = {}
        port2Info = {}
        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if portType != hardwareInfo["typeName"]:
                continue

            if len(port1Info) == 0 and port1Id == hardwareInfo["id"]:
                port1Info = hardwareInfo
                continue

            if len(port2Info) == 0 and port2Id == hardwareInfo["id"]:
                port2Info = hardwareInfo
                continue

        if len(port1Info) > 0 and len(port2Info) > 0:
            devGraphUtils = context.get("DevGraphUtils")
            clearGraph(context)
            devGraphUtils.highlightCable(port1Info, port2Info, 'selected')
            return True

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo("highlight cable failure(port1Id:%s, port2Id:%s, "
                   "portType:%s)" % (port1Id, port2Id, portType))
    return False


def highLightCableByLocation(context, port1Location, port2Location, portType):
    """通过端口location高亮线缆

    :param context:上下文对象
    :param port1Location:源端口location
    :param port2Location:目的端口location
    :param portType:端口类型（以字符串形式传入）
    :return:
    """
    logger = baseUtil.getLogger(context.get("logger"), __file__)
    try:
        clearGraph(context)
        port1Info = {}
        port2Info = {}

        for hardwareInfo in contextUtil.getItem(context, "hardwareList"):
            if portType != hardwareInfo["typeName"]:
                continue

            if not port1Info and port1Location == hardwareInfo["location"]:
                port1Info = hardwareInfo
                continue

            if not port2Info and port2Location == hardwareInfo["location"]:
                port2Info = hardwareInfo
                continue
        logger.logInfo("port1Info:%s,port2Info:%s" % (str(port1Info),
                                                      str(port2Info)))
        if len(port1Info) > 0 and len(port2Info) > 0:
            devGraphUtils = context.get("DevGraphUtils")
            logger.logInfo("highLightCableByLocation:%s <--> %s" % (
                port1Info, port2Info))
            devGraphUtils.highlightCable(port1Info, port2Info, 'selected')
            return True

    except (JException, Exception) as exception:
        logger.logException(exception)

    logger.logInfo(
        "highlight cable failure(port1Id:%s, port2Id:%s, portType:%s)" % (
            port1Location, port2Location, portType))
    return False


def clearGraph(context):
    """清除设备图高亮信息

    :param context: 上下文对象
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        devGraphUtils = context.get("DevGraphUtils")
        devGraphUtils.clearGraph()
    except (JException, Exception) as exception:
        logger.logException(exception)
    return


def getHardwareGraphList(tlv, hardwareType):
    """获取绘制硬件图的信息集合

    :param tlv: tlv对象
    :param hardwareType: 硬件类型
    :return:
    """
    enclosureHeight = tlvUtil.getCtrlEnclosureHeight(tlv)
    hardwareGraphList = []

    intStartU = 1
    records = tlvUtil.getHardwareRecords(tlv, hardwareType)
    for record in records:
        infoDict = {}
        infoDict["typeName"] = hardwareType
        infoDict["id"] = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        infoDict["name"] = tlvUtil.getRecordValue(
            record, tlvData.PUB_ATTR["name"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        infoDict["location"] = location
        infoDict["oldLoaction"] = location
        infoDict["modelName"] = getModelName(
            record, hardwareType, enclosureHeight)
        infoDict["parentId"] = tlvUtil.getRecordValue(
            record, tlvData.PUB_ATTR["parentID"])
        infoDict["parentType"] = tlvUtil.getRecordValue(
            record, tlvData.PUB_ATTR["parentType"])
        infoDict["healthStatus"] = tlvUtil.getRecordValue(
            record, tlvData.PUB_ATTR["healthStatus"])
        infoDict["runningStatus"] = tlvUtil.getRecordValue(
            record, tlvData.PUB_ATTR["runningStatus"])
        infoDict["startU"] = ""
        infoDict["heightU"] = ""
        infoDict["logicTypeName"] = ""
        infoDict["currentPeerId"] = ""
        infoDict["suggestPeerId"] = ""

        if hardwareType == "ENCLOSURE":
            if "0.0U" in location:
                continue
            logicType = tlvUtil.getRecordValue(record,
                                               tlvData.ENCLOSURE["logicType"])
            logicTypeName = tlvData.ENCLOSURE_TYPE.get(logicType, "")
            height = tlvUtil.getRecordValue(record,
                                            tlvData.ENCLOSURE["height"])
            if location in ['', '--']:
                startU = str(intStartU)
                intStartU += height
                location = 'SMB0.%sU' % startU
            else:
                startU = location.split(".")[-1].replace("U", "")

            infoDict["location"] = location
            infoDict["startU"] = startU
            infoDict["heightU"] = height
            infoDict["logicTypeName"] = logicTypeName

        elif hardwareType == "SAS_PORT":
            infoDict["currentPeerId"] = tlvUtil.getRecordValue(
                record, tlvData.SAS_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = tlvUtil.getRecordValue(
                record, tlvData.SAS_PORT["suggest_peer_port_id"])

        elif hardwareType == "PCIE_PORT":
            infoDict["currentPeerId"] = tlvUtil.getRecordValue(
                record, tlvData.PCIE_PORT["current_peer_port_id"])
            infoDict["suggestPeerId"] = tlvUtil.getRecordValue(
                record, tlvData.PCIE_PORT["suggest_peer_port_id"])

        for key in infoDict.keys():
            infoDict[key] = str(infoDict[key])

        hardwareGraphList.append(infoDict.copy())

    return hardwareGraphList


def getModelName(record, hardwareType, enclosureHeight):
    """获取设备图显示需要的modelName

    :param record: tlv记录
    :param hardwareType: 硬件类型
    :param enclosureHeight: 框高
    :return:
    """
    modelName = ""
    if hardwareType == "ENCLOSURE":
        model = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["model"])
        modelName = tlvData.ENCLOSURE_MODEL.get(model, "")
    elif hardwareType == "INTF_MODULE":
        suffixDict = {2: "_CTRL_2U", 3: "_CTRL_3U", 6: ""}  # 6U取默认值
        suffix = suffixDict.get(enclosureHeight, "")
        model = tlvUtil.getRecordValue(record, tlvData.INTF_MODULE["model"])
        modelName = "%s%s" % (tlvData.INTF_MODEL.get(model, ""), suffix)
    elif hardwareType == "EXPBOARD":
        model = tlvUtil.getRecordValue(record, tlvData.EXPBOARD["model"])
        modelName = tlvData.EXPBOARD_TYPE.get(model, "")
    return modelName


def createBaysGraph(bayNameList):
    """生成指定机柜绘图信息字典

    :param bayNameList: 机柜名列表
    :return:
    """
    if not bayNameList:
        return

    bayInfoList = []

    for bayName in bayNameList:
        infoDict = {}
        infoDict["typeName"] = 'BAY'
        infoDict["id"] = bayName
        infoDict["name"] = bayName
        infoDict["location"] = ''
        infoDict["oldLoaction"] = ''
        infoDict["modelName"] = ''
        infoDict["parentId"] = 'None'
        infoDict["parentType"] = '0'
        infoDict["healthStatus"] = '0'
        infoDict["runningStatus"] = '0'
        infoDict["startU"] = ''
        infoDict["heightU"] = ''
        infoDict["logicTypeName"] = ""
        infoDict["currentPeerId"] = "--"
        infoDict["suggestPeerId"] = "--"

        bayInfoList.append(infoDict.copy())

    return bayInfoList


def createEncGraph(context, encList):
    """生成指定硬盘框的绘图信息字典

    :param context: 上下文
    :param encList: 硬盘框列表
    :return:
    """
    encInfoList = []
    ENC_HEIGHT = {"EXPSAS2U_25": 2, "EXPSAS4U_24_NEW": 4}
    count = 0  # 辅助生成唯一id
    for enclosure in encList:
        count += 1

        encName = enclosure["name"]
        encType = enclosure["type"]
        startU = enclosure["startU"]
        bayName = enclosure["bay"]  # bay的id和name相同
        encParentId = enclosure["bay"]
        height = ENC_HEIGHT[encType]
        location = '%s.%sU' % (bayName, str(startU))

        infoDict = {}
        infoDict["typeName"] = 'ENCLOSURE'
        infoDict["id"] = str(time.time() + count)
        infoDict["name"] = encName
        infoDict["location"] = location
        infoDict["oldLoaction"] = location
        infoDict["modelName"] = encType
        infoDict["parentId"] = encParentId
        infoDict["parentType"] = '205'
        infoDict["healthStatus"] = '0'
        infoDict["runningStatus"] = '0'
        infoDict["startU"] = str(startU)
        infoDict["heightU"] = str(height)
        infoDict["logicTypeName"] = "EXP"
        infoDict["currentPeerId"] = "--"
        infoDict["suggestPeerId"] = "--"

        encInfoList.append(infoDict.copy())

    return encInfoList


def createEmptyCtrl(bay, startU):
    """在柜上绘制空白控制框,用于高端整柜扩容绘图

    :param bay: 机柜信息
    :param startU: 控制框起始位置
    :return:
    """
    encParentId = bay['id']
    bayName = bay['name']
    height = 4
    location = '%s.%sU' % (bayName, str(height + startU))

    infoDict = {}
    infoDict["typeName"] = 'ENCLOSURE'
    infoDict["id"] = str(time.time())
    infoDict["name"] = "CTRL_4U_EMPTY"
    infoDict["location"] = location
    infoDict["oldLoaction"] = location
    infoDict["modelName"] = "CTRL_4U_EMPTY"
    infoDict["parentId"] = encParentId
    infoDict["parentType"] = '205'
    infoDict["healthStatus"] = '0'
    infoDict["runningStatus"] = '0'
    infoDict["startU"] = str(startU)
    infoDict["heightU"] = str(height)
    infoDict["logicTypeName"] = "CTRL"
    infoDict["currentPeerId"] = "--"
    infoDict["suggestPeerId"] = "--"

    return infoDict


def createEncComponent(encInfoList):
    """绘制指定硬盘框下的组件

    :param encInfoList: 硬盘框信息列表
    :return:
    """
    ENCSLOTNAME_DICT = {'FAN': ['FAN 0', 'FAN 1'],
                        'POWER': ['PSU 0', 'PSU 1'],
                        'EXPBOARD': ['A', 'B'],
                        'SAS_PORT': ['PRI', 'EXP']}
    encCompmentList = []
    expBoardList = []
    for enc in encInfoList:
        parentId = enc['id']
        encName = enc['name']
        encParentId = enc['parentId']

        for componentType in ENCSLOTNAME_DICT.keys():
            if componentType == 'SAS_PORT':
                continue

            modelName = ''
            if componentType == 'EXPBOARD':
                modelName = 'SAS'

            componentList = ENCSLOTNAME_DICT[componentType]
            for component in componentList:
                componentName = encName + '.' + component
                compnentId = encParentId + '.' + componentName
                currentPeerId = '--'
                suggestPeerId = '--'

                infoDict = {}
                infoDict["id"] = compnentId
                infoDict["typeName"] = componentType
                infoDict["name"] = component
                infoDict["location"] = componentName
                infoDict["oldLoaction"] = componentName
                infoDict["modelName"] = modelName
                infoDict["parentId"] = parentId
                infoDict["parentType"] = '206'
                infoDict["healthStatus"] = '1'
                infoDict["runningStatus"] = '27'
                infoDict["startU"] = ''
                infoDict["heightU"] = ''
                infoDict["logicTypeName"] = ''
                infoDict["currentPeerId"] = currentPeerId
                infoDict["suggestPeerId"] = suggestPeerId

                encCompmentList.append(infoDict.copy())
                if componentType == 'EXPBOARD':
                    expBoardList.append(infoDict.copy())

    # 绘制SAS端口
    if len(expBoardList) != 0:
        for expBoard in expBoardList:
            for sasPortType in ENCSLOTNAME_DICT['SAS_PORT']:
                sasPortParentId = expBoard['id']
                sasPortLocation = "%s.%s" % (expBoard['location'], sasPortType)
                sasPortNameId = "%s.%s" % (sasPortParentId.split('.')[0],
                                           sasPortLocation)
                currentPeerId = '--'
                suggestPeerId = '--'

                infoDict = {}
                infoDict["id"] = sasPortNameId
                infoDict["typeName"] = 'SAS_PORT'
                infoDict["name"] = sasPortType
                infoDict["location"] = sasPortLocation
                infoDict["oldLoaction"] = sasPortLocation
                infoDict["modelName"] = ''
                infoDict["parentId"] = sasPortParentId
                infoDict["parentType"] = '208'
                infoDict["healthStatus"] = '1'
                infoDict["runningStatus"] = '27'
                infoDict["startU"] = ''
                infoDict["heightU"] = ''
                infoDict["logicTypeName"] = ''
                infoDict["currentPeerId"] = currentPeerId
                infoDict["suggestPeerId"] = suggestPeerId

                encCompmentList.append(infoDict.copy())
    return encCompmentList


def refreshGraph(context, isBack=False):
    """选择硬盘柜或硬盘框时刷新设备图

    :param context: 上下文对象
    :param isBack: 是否绘制后视图
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备图
        clearGraph(context)

        tlvHardwareList = contextUtil.getItem(context, 'tlvHardwareList')
        newBayList = contextUtil.getItem(context, "newBayList")
        logger.logInfo("%s" % str(newBayList))

        hardwareList = []
        simulateHardwareList = []
        bayHardwareList = createBaysGraph(newBayList)
        simulateHardwareList.extend(bayHardwareList)

        # 绘制空白框，用于显示新增柜
        for bay in bayHardwareList:
            oldCabinetNames = contextUtil.getItem(context,
                                                  "oldCabinetNames")
            if bay["name"] not in oldCabinetNames:
                simulateHardwareList.append(createEmptyCtrl(bay, 19))

        encGraphInfoList = contextUtil.getItem(context, "encGraphInfoList", [])
        encHardwareList = createEncGraph(context, encGraphInfoList)
        encCompHardwareList = createEncComponent(encHardwareList)
        simulateHardwareList.extend(encHardwareList)
        simulateHardwareList.extend(encCompHardwareList)

        hardwareList.extend(tlvHardwareList)
        hardwareList.extend(simulateHardwareList)
        logger.logInfo("hardwareList:%s" % str(hardwareList))

        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {
            "graphConfigPath": devGraphCfg.DEV_GRAPH_CFG_FILE,
            "deviceImgCfgPath": devGraphCfg.DEV_GRAPH_IMG_PATH
        }
        devGraphUtils.buildGraph(cfgMap, hardwareList, isBack, False)
        contextUtil.setItem(context, "simulateHardwareList",
                            simulateHardwareList)
        contextUtil.setItem(context, "hardwareList", hardwareList)

    except (JException, Exception) as exception:
        logger.logException(exception)
    return


def lightDiskEnc(context):
    """点亮所有硬盘框

    :param context: 上下文对象
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)

        devGraphUtils = context.get("DevGraphUtils")
        hardwareList = contextUtil.getItem(context, "hardwareList")
        hardwareType = "ENCLOSURE"
        logicTypeName = "EXP"
        for hardwareInfo in hardwareList:
            if logicTypeName == hardwareInfo["logicTypeName"]:
                try:
                    devGraphUtils.highlightModule(hardwareInfo, 'selected')
                    logger.logInfo("success to highlight hardware"
                                   "(hardwareId:%s, hardwareType:%s)" %
                                   (hardwareInfo["id"], hardwareType))
                except (JException, Exception) as exception:
                    logger.logInfo("fail to highlight hardware(hardwareId:%s,"
                                   " hardwareType:%s)" % (hardwareInfo["id"],
                                                          hardwareType))
                    logger.logException(exception)

    except (JException, Exception) as exception:
        logger.logException(exception)

    return


def lightNewDiskEnclosures(context):
    """高亮所有新增硬盘框，用于融合存储高端整柜

    :param context: 上下文对象
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 获取新增硬盘框
        newEnc2UList = contextUtil.getItem(context, "newEnc2UList")
        newEnc4UList = contextUtil.getItem(context, "newEnc4UList")
        newEnc2UNVMeList = contextUtil.getItem(context, "newEnc2UNVMeList", [])
        allnewEncList = newEnc2UList + newEnc4UList + newEnc2UNVMeList

        hardwareList = contextUtil.getItem(context, "hardwareList")
        # 清除高亮设备
        clearGraph(context)

        # 绘制设备图
        devGraphUtils = context.get("DevGraphUtils")

        cfgMap = {"graphConfigPath": devGraphCfg.DEV_GRAPH_CFG_FILE,
                  "deviceImgCfgPath": devGraphCfg.DEV_GRAPH_IMG_PATH}
        devGraphUtils.buildGraph(cfgMap, hardwareList, False, False)

        # 高亮新增硬盘框
        hardwareType = "ENCLOSURE"
        for encInfo in allnewEncList:
            try:
                location = "%s.%sU" % (encInfo[3], str(encInfo[2]))
                for hardwareInfo in hardwareList:
                    if (hardwareType == hardwareInfo["typeName"]
                            and location == hardwareInfo["location"]):
                        devGraphUtils.highlightModule(hardwareInfo, 'selected')
                        logger.logInfo(
                            "success to highlight hardware(location:%s, "
                            "hardwareType:%s)" % (location, hardwareType))
                        break
            except (JException, Exception) as exception:
                logger.logInfo(
                    "fail to highlight hardware(encInfo:%s)" % encInfo)
                logger.logException(exception)

    except (JException, Exception) as exception:
        logger.logException(exception)
    return


def light_new_disk_enclosure(context):
    """高亮新接入的硬盘框

    :param context: 环境变量
    :return:
    """
    logger = common.getLogger(context.get("logger"), __file__)
    try:
        # 清除高亮设备
        clearGraph(context)

        dev_graph_utils = context.get("DevGraphUtils")
        hardware_list = contextUtil.getItem(
            context, 'hardwareList', [])
        logger.logInfo("hardware_list is: %s" % str(hardware_list))
        # 新接入的硬盘框列表 在上电状态检查项中中保存
        new_enclosure_list = contextUtil.getItem(context, "newInsertedEncList")
        logger.logInfo("new enclosure is: %s" % str(new_enclosure_list))
        hardware_type = "ENCLOSURE"
        logic_type_name = "EXP"
        for hardwareInfo in hardware_list:
            if logic_type_name == hardwareInfo.get("logicTypeName") and \
                    hardwareInfo.get("location") in new_enclosure_list:
                try:
                    dev_graph_utils.highlightModule(hardwareInfo, 'selected')
                    logger.logInfo(
                        "success to highlight (Id:%s, Type:%s)" % (
                            hardwareInfo.get("id"), hardware_type))
                except (JException, Exception) as exception:
                    logger.logInfo(
                        "fail to highlight (Id:%s, Type:%s)" % (
                            hardwareInfo.get("id"), hardware_type))
                    logger.logException(exception)

    except (JException, Exception) as exception:
        logger.logException(exception)

    return
