# -*- coding: UTF-8 -*-

#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import expparser
from cbb.frame.dsl import adapter
from cbb.frame.dsl import offline_adapter
from cbb.frame.dsl.base_operator import BaseOperator
from cbb.frame.dsl.common import DslException
from cbb.frame.dsl.fault_mode import ReturnIfHandler


class BaseExecutor(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.retry_times = 1
        self.reconnected = False
        self.cmd = ""
        self.cli = None

    def get(self):
        """
        exec_cli 'show lun general'
        exec_cli '{}'
        exec_cli '{}' -retry=3 -reconnect -no_log -timeout=300 -all_controller
        :return:
        """
        key = self.get_type()
        a = expparser.CharsNotIn("'")
        b = expparser.Suppress("'")
        cmd_section = expparser.Group(b + expparser.OneOrMore(a) + b).setResultsName("cmd")

        retry = self.get_param("retry", expparser.Word(expparser.nums))
        reconnect = self.get_param("reconnect")
        no_log = self.get_param("no_log")
        timeout = self.get_param("timeout", expparser.Word(expparser.nums))
        param_section = expparser.ZeroOrMore(retry) + expparser.ZeroOrMore(reconnect) + expparser.ZeroOrMore(
            no_log) + expparser.ZeroOrMore(timeout)
        pattern = key + cmd_section + param_section
        pattern.setParseAction(self.base_action)
        return pattern

    def get_type(self):
        return expparser.Literal("exec_cli").setResultsName("type")

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        self.cmd = " ".join(toks.cmd)
        if self.dsl_context.args:
            self.cmd = self.cmd.format(*self.dsl_context.args)

        retry_times = 1
        if toks.retry:
            retry_times = int(toks.retry.value)
        if toks.reconnect:
            self.reconnected = True

        while retry_times > 0:
            retry_times -= 1
            if self.dsl_context.context.get("diagnoseType") != "offline":
                self.cli = self.get_cli()
            try:
                if self.dsl_context.context.get('command_cache_enabled'):
                    flag, result = self.exec_cmd_with_cache()
                else:
                    flag, result = self.exec_cmd()
                self.dsl_context.origin_info.append(result)
                result = ReturnIfHandler(
                    self.dsl_context.kwargs.get("return_if")).process(result)
                if not flag:
                    raise DslException(result)
                return result
            except DslException as e:
                if self.reconnected:
                    self.cli = self.get_cli()
                if retry_times <= 0:
                    self.logger.info("exec cmd={} error={}"
                                     .format(self.cmd, e.get_code()))

    def get_cli(self):
        return adapter.get_con(self.dsl_context.context, True)

    def exec_cmd_with_cache(self):
        current_node = self.dsl_context.context.get("current_exec_node", "")
        cache_data = adapter.get_result(self.sn, self.cmd, current_node)
        if cache_data:
            return True, cache_data
        flag, result = self.exec_cmd()
        if flag is True:
            adapter.save_result(self.sn, self.cmd, result, current_node)
        return flag, result

    def exec_cmd(self):
        raise NotImplementedError


class ExecCliDsl(BaseExecutor):
    def get_type(self):
        return expparser.Literal("exec_cli").setResultsName("type")

    def exec_cmd(self):
        if self.dsl_context.context.get("diagnoseType") == "offline":
            return offline_adapter.exec_cli(self.dsl_context.context, self.cmd)
        return adapter.exec_cli(self.cli, self.cmd)

    def get_cli(self):
        return adapter.get_con(self.dsl_context.context, True, False)


class ExecDiagnoseDsl(BaseExecutor):
    def get_type(self):
        return expparser.Literal("exec_diagnose").setResultsName("type")

    def exec_cmd(self):
        if self.dsl_context.context.get("diagnoseType") == "offline":
            return offline_adapter.exec_diagnose(self.dsl_context.context, self.cmd)
        return adapter.exec_diagnose(self.cli, self.cmd)


class ExecDeveloperDsl(BaseExecutor):
    def get_type(self):
        return expparser.Literal("exec_developer").setResultsName("type")

    def exec_cmd(self):
        if self.dsl_context.context.get("diagnoseType") == "offline":
            return offline_adapter.exec_developer(self.dsl_context.context, self.cmd)
        return adapter.exec_developer(self.cli, self.cmd)

    def get_cli(self):
        return adapter.get_con(self.dsl_context.context, True, False)


class ExecMiniSystemDsl(BaseExecutor):
    def get_type(self):
        return expparser.Literal("exec_mini_system").setResultsName("type")

    def exec_cmd(self):
        if self.dsl_context.context.get("diagnoseType") == "offline":
            return offline_adapter.exec_mini_system(self.dsl_context.context, self.cmd)
        return adapter.exec_mini_system(self.cli, self.cmd)
