# -*- coding: UTF-8 -*-
from cbb.frame.dsl.dsl import Dsl, DslContext
from cbb.frame.faulttree.tree_node import CheckNode
from cbb.frame.dsl import adapter
from cbb.frame.faulttree.echo_adapter import EchoAdapter


class BaseCheckItem(object):
    def __init__(self, context):
        self.logger = adapter.get_logger(context)
        self.lang = adapter.get_lang(context)
        self.context = context
        self.sn = adapter.get_sn(context)
        self.dsl_context = None
        self.echo_adapter = EchoAdapter(context)

        # 用于存放提示信息中用到的变量值
        self.data = {}

        # 最后一次执行的dsl对象
        self.last_dsl = None

        # 最后一个dsl执行的结果
        self.last_data = {}

        # 存放每个节点信息
        self.check_node = None

    def dsl(self, *args, **kwargs):
        if self.dsl_context:
            self.last_dsl = Dsl(self.dsl_context)
        else:
            dsl_context = DslContext(self.context)
            self.last_dsl = Dsl(dsl_context)
        try:
            ret = self.last_dsl.dsl(*args, **kwargs)
        finally:
            if self.last_dsl.origin_info:
                self.check_node.origin_info.extend(self.last_dsl.origin_info)
                self.last_dsl.origin_info = []
        return ret

    def create_check_node(self, node_id):
        """
        每个节点生成一个node实例
        :param node_id:
        :return:
        """
        self.check_node = CheckNode(self.context, node_id, self.data)
        self.logger.info("create check node={}".format(str(self.check_node)))
        return self.check_node

    def set_check_node(self, check_node):
        self.check_node = check_node

    def add_origin_info(self, info):
        self.check_node.origin_info.append(info)

    def get_origin_info(self):
        return self.check_node.origin_info

    def save_data(self, key, value):
        """
        将变量值保存到字典中，后面直接用于提示信息的格式化
        :param key:  键名称
        :param value: 键值
        :return:
        """
        self.check_node.data[key] = value

    def get_data(self, key):
        return self.check_node.data.get(key)

    def set_error_code(self, error_code):
        self.check_node.error_code = error_code

    def cache_result(self, key, value):
        """
        将结果缓存到lucene中
        :param key: 关键字
        :param value: 结果
        :return:
        """
        adapter.save_result(self.sn, key, value)

    def get_cache_result(self, key):
        """
        从缓存中获取结果
        :param key: 关键字
        :return: 缓存的内容
        """
        return adapter.get_result(self.sn, key)

    def get_last_dsl_data(self):
        """
        获取最近一次dsl命中的数据
        :return:
        """
        return self.last_dsl.get_matched_data()

    def get_last_dsl_origin(self):
        return "\n".join(self.last_dsl.origin_info)

    def set_dsl_context(self, context):
        self.dsl_context = DslContext(context)

    def set_table_rows(self, keys, data=None):
        self.logger.info("set table rows")
        key = ",".join(keys)
        if not data:
            data = self.last_dsl.get_matched_data()
        if len(data) > 0:
            self.logger.info("data" + str(data))
            self.check_node.tables[key] = data
