# coding: utf-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

"""
定义Config解析的规则：
提取出满足start_regex到end_regex之间的字符信息，并按照parse_regex解析成结构化数据。
最终返回的是以name为key的字典
遍历一次
"""

import re
import os
import codecs

# Config解析的规则定义
DEFAULT_RULES = [{
    "name": "fc ports",
    "start_regex": "\s*FC Port\s*-+",
    "end_regex": ".*-{10,}",
    "split_regex": "(\s{2,}ID:.*)",
    "parse_regex": "ID:\s*(?P<PortID>[\w\.]+).+?Running Status:\s*(?P<status>[\w -]+).+?Tx power\(uW\):\s*(?P<TxPower>[\d\.-]+).+?Rx power\(uW\):\s*(?P<RxPower>[\d\.-]+)"
}, {
    "name": "sfp module",
    "start_regex": "SFP-{10,}",
    "end_regex": "^\S.* ",
    "parse_regex": "PortID:\s*(?P<PortID>[\w\.]+).+?\s+Running Status:\s*(?P<status>[\w -]+).+?RxPowerReal\(uW\):\s*(?P<RxPower>[\d\.-]+).+?TxPowerReal\(uW\):\s*(?P<TxPower>[\d\.-]+)"
}]


class RunningDataParser(object):
    """
    运行数据解析类：对运行数据文件做一次遍历,解析出需要提取的字符串数据
    每个规则对应一个规则处理器，每个规则处理器各自对每一行进行处理。

    :return: result = {"controller": [["line1", "line2"], ["line3", "line4"]]}
    """
    def __init__(self, config_path, rule=None):
        self.parse_rules = rule or DEFAULT_RULES
        self.all_rule_recorder = {}
        self.config_path = config_path
        self.result = self.parse()

    def parse(self):
        if not self.config_path or not os.path.exists(self.config_path):
            return {}

        for rule in self.parse_rules:
            self.all_rule_recorder[rule.get("name")] = RuleProcessor(rule)

        with codecs.open(self.config_path, encoding='utf-8') as config_f:
            for line_str in config_f:
                for name, rule_recorder in self.all_rule_recorder.items():
                    rule_recorder.add_line(line_str)

        return {rule_name: rule_recorder.get_results()
                for rule_name, rule_recorder in self.all_rule_recorder.items()}

    def get_result_by_name(self, rule_name):
        return self.result.get(rule_name, [])

    def parse_by_rules(self, data_str, rules):
        """
        按自定义规则对传入的数据进行解析
        :param data_str: 待解析数据
        :param rules: 解析规则
        :return: 解析后的数据
        """
        for rule in rules:
            self.all_rule_recorder[rule.get("name")] = RuleProcessor(rule)

        for line in data_str.splitlines():
            for name, rule_recorder in self.all_rule_recorder.items():
                rule_recorder.add_line(line)

        return {rule_name: rule_recorder.get_results()
                for rule_name, rule_recorder in self.all_rule_recorder.items()}


class RuleProcessor(object):
    """
    一个规则处理器：对每到来的行按设定的规则进行处理
    """
    def __init__(self, rule):
        self.rule = rule
        # 开始记录的标记
        self.start_record = False
        self.temp_result = []
        self.results = []

    def add_line(self, line):
        self.check_end_line(line)
        self.check_start_line(line)
        if not self.start_record:
            return
        self.temp_result.append(line)

    def get_results(self):
        return self.results

    def check_start_line(self, line):
        if re.match(self.rule.get("start_regex"), line):
            self.start_record = True

    def check_end_line(self, line):
        if not self.start_record:
            # 模块还没开始，不用判断结束
            return
        if self.rule.get("end_regex") and re.match(self.rule.get("end_regex"), line):
            # 自定义模块结束方式
            self.set_record_end()
            return
        if not line[0].isspace():
            # 默认的结束判断方式
            self.set_record_end()
            return

    def set_record_end(self):
        """
        设置一个模块解析结束：置结束标记
        """
        self.start_record = False
        result_str = "\n".join(self.temp_result)
        if self.rule.get("parse_regex"):
            self.results.extend(self._parse_result(self.rule.get("parse_regex"), result_str))
        else:
            self.results.append(result_str)
        self.temp_result = []

    def _parse_result(self, regex, result):
        data = []
        results = self.split_keep_delimiters(self.rule.get("split_regex"), result) \
            if self.rule.get("split_regex") else [result]
        for item in results:
            data.extend(self.match_group_dict(regex, item))
        return data

    @staticmethod
    def split_keep_delimiters(regex, text):
        texts = re.split(regex, text)
        data = []
        temp_data = []
        for item in texts:
            temp_data.append(item)
            if not re.match(regex, item):
                data.append("".join(temp_data))
                temp_data = []
        return data

    @staticmethod
    def match_group_dict(regex, text):
        iter_obj = re.finditer(regex, text, re.S)
        data = []
        for item in iter_obj:
            data.append(item.groupdict())
        return data
