#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/07/21
@file: check_1822_nic_version.py
@function:
"""
import copy

from Common.base import context_util
from Common.base import entity
from Common.base.constant import MsgKey
from Common.base.context_util import get_mapping_attribute, get_mapping_attribute_url
from Common.base.entity import Compare
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env
CARD_NUM_KEY = "Card num:"
CARD_NAME_KEY = "|----"
TOOL_VERSION_KEY = "1822_NIC_Tool_ver"
DRIVER_VERSION_KEY = "1822_NIC_Driver_ver"
FIRMWARE_VERSION_KEY = "1822_NIC_FW_Ver"

DEV_VERSION_START = ["hinic driver: ", "hinicadm tool: ", "up: ", "ucode: "]
ERR_MSG_ADD_NAMES = ["", "", "up", "ucode"]
MAPPING_VERSION_KEYS = ["1822_NIC_Driver_ver", "1822_NIC_Tool_ver",
                        "1822_NIC_FW_Ver", "1822_NIC_FW_Ver"]
CARE_1822_NIC_DRIVER = "1822_NIC_Driver_ver"
CARE_1822_NIC_TOOL = "1822_NIC_Tool_ver"
CARE_1822_NIC_FW = "1822_NIC_FW_Ver"


def execute(task):
    return check1822NicVersion(task).check()


class check1822NicVersion(object):

    def __init__(self, task):
        self.task = task
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._logger = entity.create_logger(__file__)
        self._ssh_rets = list()
        self._err_msgs = list()
        self.not_support_msg = entity.create_msg(MsgKey.NOT_INVOLVE)
        self._mapping_versions = list()
        self._mapping_url = list()

    def check(self):
        if not context_util.contain_need_check_key(PY_JAVA_ENV, MAPPING_VERSION_KEYS):
            return ResultFactory.create_pass(err_msg=entity.create_msg("not.support"))
        self._obtain_match_versions()
        match_msg = self._get_match_msg()
        try:
            need_check, card_names = self._get_1822_card_names()
            if not need_check:
                return ResultFactory.create_pass(self._ssh_rets,
                                                 "\n".join([entity.create_msg("1822.nic.not.found"), match_msg]))
            res = []
            for name in card_names:
                res.append(self._check_mapping_card_version(name))
            self._err_msgs.append(match_msg)
            if not all(res):
                self._err_msgs.insert(0, entity.build_driver_tool_tips())
                return ResultFactory.create_not_pass(self._ssh_rets,
                                                     self._err_msgs)
            return ResultFactory.create_pass(self._ssh_rets, match_msg)
        except DeployException as e:
            self._logger.error(e.message)
            self._ssh_rets.append(e.origin_info)
            self._err_msgs.append(e.err_msg)
            if e.may_info_miss():
                self.task.openAutoRetry()
            return ResultFactory.create_not_pass(self._ssh_rets,
                                                 self._err_msgs)

    def _obtain_match_versions(self):
        for version_key in MAPPING_VERSION_KEYS:
            self._mapping_versions.append(get_mapping_attribute(
                PY_JAVA_ENV, version_key))
            self._mapping_url.append(get_mapping_attribute_url(PY_JAVA_ENV, version_key))

    def _get_match_msg(self):
        mapping_versions = copy.copy(self._mapping_versions)
        for index in range(len(self._mapping_versions)):
            if not self._mapping_versions[index]:
                mapping_versions[index] = self.not_support_msg
        self._set_not_support(mapping_versions)
        return entity.create_source_file_msg(PY_JAVA_ENV, entity.create_msg("1822netCard.match.version").format(
            mapping_versions[0], mapping_versions[1],
            mapping_versions[2]) + "\n")

    def _set_not_support(self, mapping_versions):
        if not mapping_versions[0]:
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_DRIVER.getKey(), self.not_support_msg)
        if not mapping_versions[1]:
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_FW.getKey(), self.not_support_msg)
        if not mapping_versions[2]:
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_TOOL.getKey(), self.not_support_msg)

    def _contains_1822_nic(self):
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV,
                                                "lspci | grep 1822")
        self._ssh_rets.append(ssh_ret)
        cmd_and_end_line_num = 2
        return len(ssh_ret.splitlines()) > cmd_and_end_line_num

    def _get_1822_card_names(self):
        """
        :return: 是否需要检查， 检查卡名称
        """
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, "hinicadm info")
        card_num = 0
        card_names = list()
        self._ssh_rets.append(ssh_ret)
        if ssh_util.is_invalid_cmd(ssh_ret):
            if self._contains_1822_nic():
                raise DeployException(
                    "invalid cmd",
                    err_msg=entity.create_msg("cmd.not.invalid"))
            return False, card_names
        for line in ssh_ret.splitlines():
            if self._is_card_num_line(line):
                card_num = self._get_card_num(line)
                self._logger.info("1822 card num: {}".format(card_num))
            if self._is_card_name_line(line):
                card_names.append(self._get_card_name(line))
        self._logger.info("1822 card names: {}".format(card_names))
        if card_num != len(card_names):
            raise DeployException(
                "parsed card name failed",
                err_code=DeployException.ErrCode.MAY_INFO_MISS)
        return True, card_names

    def _is_card_num_line(self, line):
        return line.startswith(CARD_NUM_KEY)

    def _get_card_num(self, line):
        num_str = line.strip().split(CARD_NUM_KEY)[1]
        try:
            return int(num_str)
        except ValueError:
            self._logger.error("Get 1882 network card num failed")
            raise DeployException(
                "parsed card num failed",
                err_code=DeployException.ErrCode.MAY_INFO_MISS)

    def _is_card_name_line(self, line):
        return line.startswith(CARD_NAME_KEY)

    def _get_card_name(self, line):
        card_info = line.strip().split(CARD_NAME_KEY)[1]
        return card_info.split("(")[0]

    def _check_mapping_card_version(self, card_name):
        version_match = True
        query_cmd = "hinicadm version -i {}".format(card_name)
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, query_cmd)
        self._ssh_rets.append(ssh_ret)
        if "version command error" in ssh_ret:
            raise DeployException("get card version failed or none")
        dev_versions = list()
        for key in DEV_VERSION_START:
            for line in ssh_ret.splitlines():
                if line.startswith(key):
                    dev_versions.append(line.split(key)[1].split()[0])
        self._logger.info("card versions: {}".format(dev_versions))
        if len(dev_versions) != len(DEV_VERSION_START):
            raise DeployException(
                "get card version failed",
                err_code=DeployException.ErrCode.MAY_INFO_MISS)
        for index in range(len(MAPPING_VERSION_KEYS)):
            mapping_ver = self._mapping_versions[index]
            self._logger.info("compare {} and {}".format(dev_versions[index],
                                                         mapping_ver))
            if mapping_ver and Compare.compare_digital_version(
                    dev_versions[index], mapping_ver) < 0:
                self._set_nic_match_result(index)
                version_match = False
            match_res = entity.create_msg(
                MAPPING_VERSION_KEYS[index]).format(card_name, ERR_MSG_ADD_NAMES[index], dev_versions[index])
            self._err_msgs.append(entity.build_url_error_msg(self._mapping_url[index], match_res))
            self._set_nic_version(dev_versions, index)
        return version_match

    def _set_nic_match_result(self, index):
        if CARE_1822_NIC_DRIVER == MAPPING_VERSION_KEYS[index]:
            self.deploy_node.putResult(context_util.get_version_key_enum().NIC_DRIVER.getKey(),
                                       context_util.get_not_pass_key())
        if CARE_1822_NIC_TOOL == MAPPING_VERSION_KEYS[index]:
            self.deploy_node.putResult(context_util.get_version_key_enum().NIC_TOOL.getKey(),
                                       context_util.get_not_pass_key())
        if CARE_1822_NIC_FW == MAPPING_VERSION_KEYS[index]:
            self.deploy_node.putResult(context_util.get_version_key_enum().NIC_FW.getKey(),
                                       context_util.get_not_pass_key())

    def _set_nic_version(self, dev_versions, index):
        if CARE_1822_NIC_DRIVER == MAPPING_VERSION_KEYS[index]:
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_DRIVER.getKey(), dev_versions[index])
        if CARE_1822_NIC_TOOL == MAPPING_VERSION_KEYS[index]:
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_TOOL.getKey(), dev_versions[index])
        if CARE_1822_NIC_FW == MAPPING_VERSION_KEYS[index]:
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_FW.getKey(), dev_versions[index])
