#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

"""
@time: 2023/02/06
@file: check_1822_v120_nic_version.py
@function:
"""
import re
import xml.etree.ElementTree as ET
from Common.base import context_util
from Common.base import entity
from Common.base.constant import MsgKey
from Common.base.context_util import get_mapping_attribute, get_mapping_attribute_url
from Common.base.entity import Compare
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env
NIC_1822_V120_NORMALIZED_DRIVER_KEY = "1822_v120_normalized_driver_ver"
CMD_AND_END_LINE_NUM = 2


def execute(task):
    return Check1822V120NicVersion(task).check()


class Check1822V120NicVersion(object):

    def __init__(self, task):
        self.task = task
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._logger = entity.create_logger(__file__)
        self._ssh_rets = list()
        self._err_msgs = list()
        self.not_support_msg = entity.create_msg(MsgKey.NOT_INVOLVE)
        self._mapping_version = ""
        self._mapping_url = ""

    def check(self):
        self._logger.info("start to check 1822 v120 nic card.")
        if not context_util.contain_need_check_key(
                PY_JAVA_ENV, [NIC_1822_V120_NORMALIZED_DRIVER_KEY]):
            self._logger.info("not found 1822 v120 mapping version")
            return ResultFactory.create_pass(entity.create_msg("not.support"))
        self._obtain_match_versions()
        match_msg = self._get_match_msg()
        try:
            if not self._is_contains_1822_v120_nic():
                return ResultFactory.create_pass(self._ssh_rets,
                                                 "\n".join([entity.create_msg("1822.v120.nic.not.found"), match_msg]))
            ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV,
                                                    'sh /opt/hinic3_union/hinic3_check_version.sh check_version 0')
            if 'No such file or directory' not in ssh_ret:
                self._ssh_rets.append(ssh_ret)
                if 'result=0' in ssh_ret:
                    return ResultFactory.create_pass(self._ssh_rets,
                                                     "\n".join(
                                                         [entity.create_msg("1822.v120.nic.not.found"), match_msg]))
                actual_version = ssh_ret.split('actual:')[1].split('.')[0].strip()
            else:
                ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV,
                                                        "cat /opt/hinic3_union/version.xml")
                self._ssh_rets.append(ssh_ret.decode('utf-8'))
                if ssh_util.is_invalid_cmd(ssh_ret):
                    return ResultFactory.create_not_pass(self._ssh_rets, "\n".join(
                        [entity.create_msg("no.associated.driver.exist.no.check"),
                         match_msg]))
                pattern = re.compile(
                    r'<FirmwarePackage.*?>(.*?)</FirmwarePackage>', re.DOTALL)
                actual_version = ET.fromstring(
                    pattern.search(ssh_ret).group()).find('Package/Version').text
            is_version_match = self._check_nic_version(actual_version)
            self._err_msgs.append(match_msg)
            if not is_version_match:
                self._err_msgs.insert(0, entity.build_driver_tool_tips())
                return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)
            return ResultFactory.create_pass(self._ssh_rets, self._err_msgs)
        except DeployException as exception:
            self._logger.error(exception.message)
            self._ssh_rets.append(exception.origin_info)
            self._err_msgs.insert(0, match_msg)
            self._err_msgs.append(exception.err_msg)
            if exception.may_info_miss():
                self.task.openAutoRetry()
            return ResultFactory.create_not_pass(self._ssh_rets, self._err_msgs)

    def _obtain_match_versions(self):
        self._mapping_version = get_mapping_attribute(PY_JAVA_ENV, NIC_1822_V120_NORMALIZED_DRIVER_KEY)
        self._mapping_url = get_mapping_attribute_url(PY_JAVA_ENV, NIC_1822_V120_NORMALIZED_DRIVER_KEY)

    def _get_match_msg(self):
        if not self._mapping_version:
            self._mapping_version = self.not_support_msg
            self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_1822_V120.getKey(),
                                        self.not_support_msg)
            mapping_msg = entity.create_msg("1822.v120.nic.match.version").format(self._mapping_version)
        else:
            mapping_msg = entity.build_url_error_msg(self._mapping_url, entity.create_msg(
                "1822.v120.nic.match.version").format(self._mapping_version))
        return entity.create_source_file_msg(PY_JAVA_ENV, mapping_msg)

    def _is_contains_1822_v120_nic(self):
        ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, 'lspci -tv | grep -aiE "Device 0220|Device 0222"')
        self._ssh_rets.append(ssh_ret)
        return len(ssh_ret.splitlines()) > CMD_AND_END_LINE_NUM

    def _check_nic_version(self, version):
        log_compare = "compare {} and {}".format(version, self._mapping_version)
        self._logger.info(log_compare)
        match_res = entity.create_msg("1822.v120.nic.current.version").format(
            version)
        self._err_msgs.append(match_res)
        self.deploy_node.putVersion(context_util.get_version_key_enum().NIC_1822_V120.getKey(), version)
        if self._mapping_version and Compare.compare_digital_version(version, self._mapping_version) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().NIC_1822_V120.getKey(),
                                       context_util.get_not_pass_key())
            return False
        return True
