#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

"""
@time: 2020/12/26
@file: check_bios_firmware_version.py
@function:
"""

from Common.base import context_util
from Common.base import entity
from Common.base.constant import VE_SUPPORTED_NUM, V6_SUPPORTED_NUM
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory, Compare
from Common.protocol.redfish import redfish_util
from Common.service import check_main_board_uids_service

PY_JAVA_ENV = py_java_env
MAIN_BORD_BIOS_FW_VER_KEY = "Main_Bord_BIOS_Ver_{}"
IOB_CARD_BIOS_FW_VER_KEY = "IOB_Card_BIOS_Ver_{}"
OCEANS_MAIN_BORD_FW_VER_KEY = "Oceans_Main_Bord_BIOS_Ver"


def execute(task):
    return CheckBiosFirmwareVersion().check()


class CheckBiosFirmwareVersion(object):

    def __init__(self):
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._login_info = context_util.get_login_info(PY_JAVA_ENV)
        self._logger = entity.create_logger(__file__)
        self._err_msgs = list()
        self._origin_infos = list()
        self._iob_mapping_url = ''
        self._main_board_mapping_url = ''
        self.match_msgs = []

    def check(self):
        try:
            main_board_mapping_version, iob_card_mapping_version = \
                self.get_bios_mapping_version()
            main_board_match = self.is_main_board_match(
                main_board_mapping_version)
            io_board_match = self.is_io_board_match(iob_card_mapping_version)
            self._err_msgs.append(entity.create_source_file_msg(PY_JAVA_ENV, " ".join(self.match_msgs)))
            if main_board_match and io_board_match:
                return ResultFactory.create_pass(self._origin_infos,
                                                 self._err_msgs)
            self._err_msgs.insert(0, entity.build_fw_tool_tips())
            return ResultFactory.create_not_pass(self._origin_infos,
                                                 self._err_msgs)
        except DeployException as e:
            self._origin_infos.append(e.origin_info)
            self._logger.error(e.message)
            self._err_msgs.append(e.err_msg)
            return ResultFactory.create_not_pass(self._origin_infos,
                                                 self._err_msgs)

    def get_bios_mapping_version(self):
        if context_util.is_ocean_platform(PY_JAVA_ENV):
            main_board_mapping_version = context_util.get_mapping_attribute(PY_JAVA_ENV, OCEANS_MAIN_BORD_FW_VER_KEY)
            self._main_board_mapping_url = context_util.get_mapping_attribute_url(
                PY_JAVA_ENV, OCEANS_MAIN_BORD_FW_VER_KEY)
            url_msg = entity.build_url_error_msg(self._main_board_mapping_url,
                                                 entity.create_msg("main.board.bios.match.fw.version").format(
                                                     main_board_mapping_version))
            self.match_msgs.append(url_msg)
            return main_board_mapping_version, ""
        # 查询主板类型
        main_board_num, resource = redfish_util.get_ibmc_main_board_num(
            self._login_info, self._logger)
        self._origin_infos.append(resource)

        # 判断是不是2.0主板
        if check_main_board_uids_service.is_ve_supported_main_board(self._login_info, self._logger):
            main_board_num = main_board_num + VE_SUPPORTED_NUM
        # 判断是不是V6固件
        elif check_main_board_uids_service.is_v6_supported_main_board(self._login_info, self._logger):
            main_board_num += V6_SUPPORTED_NUM
        main_board_mapping_version = \
            context_util.get_mapping_fw_version_with_haiyan(
                PY_JAVA_ENV, MAIN_BORD_BIOS_FW_VER_KEY.format(main_board_num))
        self._main_board_mapping_url = context_util.get_mapping_attribute_url(
            PY_JAVA_ENV, MAIN_BORD_BIOS_FW_VER_KEY.format(main_board_num))
        self.match_msgs.append(entity.build_url_error_msg(self._main_board_mapping_url, (entity.create_msg(
            "main.board.bios.match.fw.version").format(
            main_board_mapping_version))))
        iob_card_mapping_version = context_util.get_mapping_attribute(
            PY_JAVA_ENV, IOB_CARD_BIOS_FW_VER_KEY.format(main_board_num))
        self._iob_mapping_url = context_util.get_mapping_attribute_url(
            PY_JAVA_ENV, IOB_CARD_BIOS_FW_VER_KEY.format(main_board_num))
        if iob_card_mapping_version:
            self.match_msgs.append(entity.build_url_error_msg(self._iob_mapping_url, (entity.create_msg(
                "iob.board.bios.match.fw.version").format(
                iob_card_mapping_version))))
        return main_board_mapping_version, iob_card_mapping_version

    def is_main_board_match(self, main_board_mapping_version):
        version_match = True
        resource = redfish_util.get_systems_info(
            self._login_info, self._logger)
        version = resource.get("BiosVersion")
        self._logger.info("main board bios version: {}".format(version))
        self._origin_infos.append(str(resource))
        version = self.remove_version_useless_description(version)
        self.deploy_node.putVersion(context_util.get_version_key_enum().BIOS_FW.getKey(), version)
        match_res = entity.create_msg("main.bord.bios.fw.current.version").format(version)
        if Compare.compare_digital_version(version, main_board_mapping_version) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().BIOS_FW.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        self._err_msgs.append(match_res)
        return version_match

    def is_io_board_match(self, mapping_version):
        version_match = True
        if not mapping_version:
            return version_match
        versions, origin_info = redfish_util.get_io_board_info(
            self._login_info, "FirmwareVersion", self._logger)
        self._origin_infos.append(origin_info)
        self._logger.info("io board bios version: {}".format(str(versions)))
        for ver in versions:
            ver = self.remove_version_useless_description(ver)
            if Compare.compare_digital_version(ver, mapping_version) < 0:
                version_match = False
            match_res = entity.create_msg("iob.card.bios.fw.current.version").format(ver)
            self._err_msgs.append(match_res)
        return version_match

    def remove_version_useless_description(self, version):
        return version.split("(")[0].strip()
