#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2024. All rights reserved.

"""
@time: 2020/12/26
@file: check_cpld_firmware_version.py
@function:
"""
from Common.base import context_util
from Common.base import entity
from Common.base.constant import VE_SUPPORTED_NUM, V6_SUPPORTED_NUM
from Common.base.entity import DeployException
from Common.base.entity import ResultFactory, Compare
from Common.protocol.redfish import redfish_util
from Common.service import check_main_board_uids_service

PY_JAVA_ENV = py_java_env


class Key(object):
    MAP_VER = "map_ver_key"
    ERR_MSG = "err_msg_key"
    NOT_INVOLVE_MSG = "no_involve_key"
    MATCH_MSG = "match_msg_key"
    DESCRIPTION = "description"
    INVOLVE_DEV_TYPE = "involve_dev_type"
    CPLD_VER = "CPLDVersion"


MAIN_BOARD = {
    Key.MAP_VER: "Main_Bord_CPLD_Ver_{}",
    Key.ERR_MSG: "Main.Bord.CPLD.notpass",
    Key.MATCH_MSG: "Main.Bord.CPLD.match"
}

IOB_CARD = {
    Key.MAP_VER: "IOB_Card_CPLD_Ver_{}",
    Key.ERR_MSG: "IOB.Card.CPLD.notpass",
    Key.MATCH_MSG: "IOB.Card.CPLD.match"
}

RAID_CARD = {
    Key.MAP_VER: "RAID_Card_CPLD_Ver",
    Key.ERR_MSG: "RAID.Card.CPLD.notpass",
    Key.NOT_INVOLVE_MSG: "RAID.Card.CPLD.notInvolve",
    Key.MATCH_MSG: "RAID.Card.CPLD.match"
}

OCEANS_MAIN_BORD_FW_VER_KEY = "Oceans_Main_Bord_CPLD_Ver"
ATLANTIC_SSM_FW_VER_KEY = "Atlantic_Ssm_CPLD_Ver"

# Key.MAP_VER 对应java的内置配套表字段
# Key.DESCRIPTION redfish接口中查出的对应卡的描述
# Key.ERR_MSG 该卡检查不通过时的错误key
# Key.MATCH_MSG 界面展示的配套版本信息key
# Key.INVOLVE_DEV_TYPE 涉及的型号，没有则所有的都涉及
BACK_BOARD = [
    {Key.MAP_VER: "12*3.5_Back_Bord_CPLD_Ver_{}",
     Key.DESCRIPTION: "12*3.5 SAS/SATA",
     Key.ERR_MSG: "12*3.5.Back.Bord.CPLD.notpass",
     Key.MATCH_MSG: "12*3.5.Back.Bord.CPLD.match"},
    {Key.MAP_VER: "4*2.5_NVME_Back_Bord_CPLD_Ver_{}",
     Key.DESCRIPTION: "4*2.5 SAS/SATA/NVME, Rear",
     Key.ERR_MSG: "4*2.5.NVME.Back.Bord.CPLD.notpass",
     Key.MATCH_MSG: "4*2.5.NVME.Back.Bord.CPLD.match"},
    {Key.MAP_VER: "24*3.5_Back_Bord_CPLD_Ver_{}",
     Key.DESCRIPTION: "24*3.5 SAS/SATA",
     Key.ERR_MSG: "24*3.5.Back.Bord.CPLD.notpass",
     Key.MATCH_MSG: "24*3.5.Back.Bord.CPLD.match",
     Key.INVOLVE_DEV_TYPE: ["TaiShan 5280 V2"]},
]


def execute(task):
    return CpldFirmwareVersionCheck().check()


class CpldFirmwareVersionCheck(object):

    def __init__(self):
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._login_info = context_util.get_login_info(PY_JAVA_ENV)
        self._logger = entity.create_logger(__file__)
        self._need_check_back_board = False
        self._err_msgs = list()
        self._origin_infos = list()
        self._map_key_2_map_ver = dict()
        self._map_key_2_map_url = dict()
        self._match_msgs = []

    def check(self):
        try:
            self.obtain_match_msg()
            main_board_match = self.is_main_board_match()
            back_board_match = self.is_back_board_match()
            io_board_match = self.is_io_board_match()
            raid_card_match = self.is_raid_card_match()
            self._err_msgs.append(self._match_msgs)
            if main_board_match and back_board_match and io_board_match and \
                    raid_card_match:
                return ResultFactory.create_pass(self._origin_infos, self._err_msgs)
            self._err_msgs.insert(0, entity.build_fw_tool_tips())
            return ResultFactory.create_not_pass(self._origin_infos, self._err_msgs)
        except DeployException as de:
            self._origin_infos.append(de.origin_info)
            self._logger.error(de.message)
            self._err_msgs.append(de.err_msg)
            return ResultFactory.create_not_pass(self._origin_infos,
                                                 self._err_msgs)

    def obtain_match_msg(self):
        match_msgs = list()
        if context_util.is_ocean_platform(PY_JAVA_ENV):
            main_bord_map_key = OCEANS_MAIN_BORD_FW_VER_KEY
            main_bord_num = "1711"
        elif context_util.is_atlantic_ssm_platform(PY_JAVA_ENV):
            main_bord_map_key = ATLANTIC_SSM_FW_VER_KEY
            main_bord_num = "1711"
        else:
            main_bord_map_key = MAIN_BOARD.get(Key.MAP_VER)
            # 查询主板类型
            main_bord_num, origin_info = redfish_util.get_ibmc_main_board_num(
                self._login_info, self._logger)
            self._origin_infos.append(origin_info)
        # 判断是不是2.0主板
        if check_main_board_uids_service.is_ve_supported_main_board(self._login_info, self._logger):
            main_bord_num = main_bord_num + VE_SUPPORTED_NUM
        # 判断是不是V6固件
        elif check_main_board_uids_service.is_v6_supported_main_board(self._login_info, self._logger):
            main_bord_num += V6_SUPPORTED_NUM
        # 主板
        main_board_map_ver = context_util.get_mapping_fw_version_with_haiyan(
            PY_JAVA_ENV, main_bord_map_key.format(main_bord_num))
        main_board_map_url = context_util.get_mapping_fw_url_with_haiyan(
            PY_JAVA_ENV, main_bord_map_key.format(main_bord_num))
        self._map_key_2_map_ver[main_bord_map_key] = main_board_map_ver
        self._map_key_2_map_url[main_bord_map_key] = main_board_map_url
        main_bord_match_msg = entity.create_msg(MAIN_BOARD.get(
            Key.MATCH_MSG)).format(main_board_map_ver)
        url_msg = entity.build_url_error_msg(main_board_map_url, main_bord_match_msg)
        match_msgs.append(url_msg)

        # io板
        io_board_map_key = IOB_CARD.get(Key.MAP_VER)
        io_board_map_ver = context_util.get_mapping_attribute(
            PY_JAVA_ENV, io_board_map_key.format(io_board_map_key))
        io_board_map_url = context_util.get_mapping_attribute_url(
            PY_JAVA_ENV, io_board_map_key.format(io_board_map_key))
        if io_board_map_ver:
            self._map_key_2_map_ver[io_board_map_key] = io_board_map_ver
            self._map_key_2_map_url[io_board_map_key] = io_board_map_url
            io_bord_url_msg = entity.build_url_error_msg(io_board_map_url, entity.create_msg(IOB_CARD.get(
                Key.MATCH_MSG)).format(io_board_map_ver))
            match_msgs.append(io_bord_url_msg)
        # Raid 卡
        raid_card_map_key = RAID_CARD.get(Key.MAP_VER)
        raid_card_map_ver = context_util.get_mapping_attribute(
            PY_JAVA_ENV, raid_card_map_key)
        raid_card_map_url = context_util.get_mapping_attribute_url(
            PY_JAVA_ENV, raid_card_map_key)
        if raid_card_map_ver:
            self._map_key_2_map_ver[raid_card_map_key] = raid_card_map_ver
            self._map_key_2_map_url[raid_card_map_key] = raid_card_map_url
            raid_url_msg = entity.build_url_error_msg(raid_card_map_url, entity.create_msg(RAID_CARD.get(
                Key.MATCH_MSG)).format(raid_card_map_ver))
            match_msgs.append(raid_url_msg)
        self.obtain_back_board_match_msg(main_bord_num, match_msgs)
        self._match_msgs = entity.create_source_file_msg(PY_JAVA_ENV, "\n".join(match_msgs))

    def obtain_back_board_match_msg(self, main_bord_num, match_msgs):
        # 背板
        for board in BACK_BOARD:
            map_key = board.get(Key.MAP_VER)
            map_ver = context_util.get_mapping_attribute(
                PY_JAVA_ENV, map_key.format(main_bord_num))
            map_url = context_util.get_mapping_attribute_url(
                PY_JAVA_ENV, map_key.format(main_bord_num))
            if map_ver:
                self._map_key_2_map_ver[map_key] = map_ver
                self._map_key_2_map_url[map_key] = map_url
                match_msgs.append(entity.create_msg(board.get(
                    Key.MATCH_MSG)).format(map_ver))
                self._need_check_back_board = True

    def is_main_board_match(self):
        version_match = True
        if context_util.is_atlantic_ssm_platform(PY_JAVA_ENV):
            main_board_cpld_ver, origin_info = redfish_util.get_firmware_inventory_version(
                self._login_info, "MainBoardCPLD", self._logger)
        else:
            main_board_cpld_ver, origin_info = redfish_util.get_main_board_info(
                self._login_info, Key.CPLD_VER, self._logger)
        self._origin_infos.append(origin_info)
        self._logger.info("main board cpld version: {}".format(main_board_cpld_ver))
        if context_util.is_ocean_platform(PY_JAVA_ENV):
            main_bord_map_key = OCEANS_MAIN_BORD_FW_VER_KEY
        elif context_util.is_atlantic_ssm_platform(PY_JAVA_ENV):
            main_bord_map_key = ATLANTIC_SSM_FW_VER_KEY
        else:
            main_bord_map_key = MAIN_BOARD.get(Key.MAP_VER)
        map_ver = self._map_key_2_map_ver.get(main_bord_map_key)
        main_board_cpld_ver = self.remove_version_useless_description(
            main_board_cpld_ver)
        self.deploy_node.putVersion(context_util.get_version_key_enum().MAIN_BOARD_CPLD_FW.getKey(),
                                    main_board_cpld_ver)
        if Compare.compare_digital_version(main_board_cpld_ver, map_ver) < 0:
            self.deploy_node.putResult(context_util.get_version_key_enum().MAIN_BOARD_CPLD_FW.getKey(),
                                       context_util.get_not_pass_key())
            version_match = False
        msg_res = entity.create_msg(MAIN_BOARD.get(Key.ERR_MSG)).format(main_board_cpld_ver)
        self._err_msgs.append(msg_res)
        return version_match

    def is_back_board_match(self):
        if not self._need_check_back_board:
            return True
        back_board_infos, origin_info = redfish_util.get_disk_back_board_info(
            self._login_info, Key.CPLD_VER, self._logger)
        self._origin_infos.append(origin_info)
        self._logger.info("back board cpld version: {}".format(
            str(back_board_infos)))
        back_board_match = True
        for board in BACK_BOARD:
            map_ver = self._map_key_2_map_ver.get(board.get(Key.MAP_VER))
            map_url = self._map_key_2_map_url.get(board.get(Key.MAP_VER))
            cpld_ver = back_board_infos.get(board.get(Key.DESCRIPTION))
            if not cpld_ver or not map_ver:
                self._logger.info("not need check back borad: {}".format(
                    board.get(Key.DESCRIPTION)))
                continue
            # 如果配的有涉及型号，且当前设备型号不在这里面则不需要检查
            if board.get(Key.INVOLVE_DEV_TYPE) and context_util.get_dev_type(
                    PY_JAVA_ENV) not in board.get(Key.INVOLVE_DEV_TYPE):
                self._logger.info("don`t need check back board: {}".format(
                    board.get(Key.DESCRIPTION)))
                continue
            cpld_ver = self.remove_version_useless_description(cpld_ver)
            if Compare.compare_digital_version(cpld_ver, map_ver) < 0:
                back_board_match = False
            msg_res = entity.create_msg(board.get(Key.ERR_MSG)).format(cpld_ver)
            self._err_msgs.append(msg_res)
        return back_board_match

    def is_io_board_match(self):
        io_board_match = True
        map_ver = self._map_key_2_map_ver.get(IOB_CARD.get(Key.MAP_VER))
        map_url = self._map_key_2_map_url.get(IOB_CARD.get(Key.MAP_VER))
        if not map_ver:
            return True
        io_board_versions, origin_info = redfish_util.get_io_board_info(
            self._login_info, Key.CPLD_VER, self._logger)
        self._origin_infos.append(origin_info)
        self._logger.info("io board cpld version: {}".format(
            str(io_board_versions)))
        for cpld_ver in io_board_versions:
            cpld_ver = self.remove_version_useless_description(cpld_ver)
            if Compare.compare_digital_version(cpld_ver, map_ver) < 0:
                io_board_match = False
            msg_res = entity.create_msg(IOB_CARD.get(Key.ERR_MSG)).format(cpld_ver)
            self._err_msgs.append(msg_res)
        return io_board_match

    def is_raid_card_match(self):
        map_ver = self._map_key_2_map_ver.get(RAID_CARD.get(Key.MAP_VER))
        map_url = self._map_key_2_map_url.get(RAID_CARD.get(Key.MAP_VER))
        if not map_ver:
            return True
        raid_card_infos, origin_info = redfish_util.get_raid_card_board_info(
            self._login_info, Key.CPLD_VER, self._logger)
        self._origin_infos.append(origin_info)
        self._logger.info("raid card cpld version: {}".format(
            str(raid_card_infos)))
        raid_card_match = True
        checked = False
        description = ''
        cpld_vers = []
        for description, cpld_ver in raid_card_infos.items():
            cpld_ver = self.remove_version_useless_description(cpld_ver)
            if not cpld_ver:
                continue
            checked = True
            cpld_vers.append(cpld_ver)
            if Compare.compare_digital_version(cpld_ver, map_ver) < 0:
                self.deploy_node.putResult(context_util.get_version_key_enum().RAID_CPLD_FW.getKey(),
                                           context_util.get_not_pass_key())
                raid_card_match = False
            msg_res = entity.create_msg(RAID_CARD.get(Key.ERR_MSG)).format(description, cpld_ver)
            self._err_msgs.append(msg_res)
        self.deploy_node.putVersion(context_util.get_version_key_enum().RAID_CPLD_FW.getKey(), ";".join(set(cpld_vers)))
        if not checked:
            self._err_msgs.append(entity.create_msg(RAID_CARD.get(Key.NOT_INVOLVE_MSG)).format(description))
        return raid_card_match

    def remove_version_useless_description(self, version):
        return version.split("(")[0].strip()
