#!/usr/bin/env python
# coding=UTF-8
# Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import copy
from Common.base import context_util
from Common.base import entity
from Common.base.context_util import get_mapping_attribute, get_mapping_attribute_url
from Common.base.entity import Compare
from Common.base.entity import CheckCommon
from Common.base.constant import Platform
from Common.protocol import ssh_util

PY_JAVA_ENV = py_java_env

MAPPING_VERSION_KEYS = ["SATA", "SAS"]
SATA = "SATA"
SAS = "SAS"
BASE_KEY = "{}_disk_driver_version"


def execute(task):
    return CheckDiskDriverVersion(task).check()


class CheckDiskDriverVersion(CheckCommon):
    def __init__(self, task):
        self.deploy_node = context_util.get_deploy_node(PY_JAVA_ENV)
        self._mapping_versions = []
        self._mapping_urls = []
        self._server_version = context_util.get_mapping_version(PY_JAVA_ENV)
        self._platform_id = context_util.get_platform_id(PY_JAVA_ENV)
        self.not_need_sas_disk_driver = self.not_need_sas()
        super(CheckDiskDriverVersion, self).__init__(task)

    def not_need_sas(self):
        return self._platform_id == Platform.SI_NAN and "9920" in self.deploy_node.getDevType()

    def obtain_match_versions(self):
        for key in MAPPING_VERSION_KEYS:
            version = get_mapping_attribute(PY_JAVA_ENV, BASE_KEY.format(key))
            self._mapping_versions.append(version)
            self.set_not_support_msg(key, version)
            self._mapping_urls.append(get_mapping_attribute_url(PY_JAVA_ENV, BASE_KEY.format(key)))

    def set_not_support_msg(self, key, version):
        if not version:
            self.deploy_node.putVersion(BASE_KEY.format(key), self.not_support_msg)

    def get_match_msg(self):
        mapping_versions = copy.copy(self._mapping_versions)
        for index, val in enumerate(self._mapping_versions):
            if not val:
                mapping_versions[index] = self.not_support_msg
        if self.not_need_sas_disk_driver:
            match_msg = entity.create_msg("disk.driver.version.mapping.match.sata").format(mapping_versions[0])
        else:
            match_msg = entity.create_msg("disk.driver.version.mapping.match").format(*mapping_versions)
        url_msg = entity.build_url_error_msg(self._mapping_urls[0] if self._mapping_urls else "", match_msg)
        return entity.create_source_file_msg(PY_JAVA_ENV, url_msg)

    def check_version(self):
        version_match = True
        msg_list = []
        if self.not_need_sas_disk_driver:
            MAPPING_VERSION_KEYS.remove("SAS")
        for index, key in enumerate(MAPPING_VERSION_KEYS):
            if not context_util.contain_need_check_key(PY_JAVA_ENV, [BASE_KEY.format(key)]):
                continue
            cmd = self.get_cmd_by_key(key)
            ssh_ret = ssh_util.exec_ssh_cmd_nocheck(PY_JAVA_ENV, cmd)
            self._ssh_rets.append(ssh_ret)
            version = ""
            for line in ssh_ret.splitlines():
                if "Version" in line:
                    version = line.split(":")[1].strip()
                    break
            self.deploy_node.putVersion(BASE_KEY.format(key), version)
            if not version:
                no_exist_msg = entity.create_msg("no.associated.disk.driver.exist").format(key)
                msg_list.append(no_exist_msg)
                version_match = False
                self.deploy_node.putVersion(BASE_KEY.format(key), "--")
                continue
            if Compare.compare_digital_version(version, self._mapping_versions[index]) < 0:
                version_match = False
                self.deploy_node.putResult(BASE_KEY.format(key), context_util.get_not_pass_key())
            msg_list.append(entity.create_msg("{}.disk.driver.current.version".format(key)).format(version))
        self._err_msgs.append('\n'.join(msg_list))
        return version_match

    def get_cmd_by_key(self, key):
        if key == "SATA":
            if self._platform_id in ["9550", "9950"] and self._server_version in ["8.1.0", "8.1.1"]:
                cmd = "rpm -qi pangea-sata-driver-euler-4.19.90 |grep -i version"
            else:
                cmd = "rpm -qi pangea-sata-driver-euler |grep -i  version"
        else:
            if self._platform_id == "9550" and self._server_version in ["8.1.0", "8.1.1"]:
                cmd = "rpm -qi pangea-hisisas-driver-euler-4.19.90 | grep -i version"
            else:
                cmd = "rpm -qi pangea-hisisas-driver-euler | grep -i version"
        return cmd
